#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ShowCommand
import Toggles.AbootEosToggleLib

from CliPlugin import BiosCliModel, BiosCliLib
from CliPlugin.BiosInstall import BiosInstaller
from CliToken.Install import installMatcher, sourceMatcher, reloadMatcher, nowMatcher
from CliToken.Platform import platformMatcherForShow
from Url import UrlMatcher

def guardFieldUpdateableAboot( mode, token ):
   # Toggle is disabled
   if not Toggles.AbootEosToggleLib.toggleAbootFirmwareUpdateEnabled():
      return CliParser.guardNotThisEosVersion

   # TODO: This needs tweaking once BUG488401 is done. Leaving this commented out
   # to avoid duplicated work in finding out how to determine if a system is running
   # secureboot when we need this guard.
   #
   # Locked secureboot BIOS without this feature implemented
   # if mode.entityManager.lookup( Cell.path( "aboot/sb/status" ) ).supported:
   #   return CliParser.guardNotThisAbootVersion

   return None

def guardSupervisor( mode, token ):
   '''
   Only display supervisor tokens in the case that they can be used. Must be a
   modular system, and the standby token is only available if the peer is in standby
   mode.
   '''
   entityMibRoot = BiosCliLib.getEntityMibRoot( mode.entityManager )

   if BiosCliLib.isModular( entityMibRoot ):
      if token == 'active':
         return None

      redundancyStatus = BiosCliLib.getRedundancyStatus( mode.entityManager )
      if BiosCliLib.isPeerStandby( redundancyStatus ) and token == 'standby':
         return None

   return CliParser.guardNotThisPlatform

biosMatcherForShow = CliMatcher.KeywordMatcher( 'bios', helpdesc='BIOS info' )
biosMatcherForInstall = CliCommand.guardedKeyword( 'bios', helpdesc='BIOS install',
                                                   guard=guardFieldUpdateableAboot )

supervisorEnum = CliMatcher.EnumMatcher( {
   'active': 'Install only on the active supervisor',
   'standby': 'Install only on the standby supervisor',
} )
supervisorMatcher = CliCommand.Node( matcher=supervisorEnum, guard=guardSupervisor )

abootUpdateDir = 'flash:/aboot/update/'

def showBios( mode, args ):
   ( running, sysName ) = BiosCliModel.getRunningVersionAndSysName( mode )
   result = BiosCliModel.BiosVersions()
   result.sysName = sysName
   result.runningVersion = running

   line = result.runningVersion.line
   abootReader = BiosCliLib.getAbootReader( line )

   ( programmed, fallback ) = \
               abootReader.getProgrammedAndFallbackVersions()
   result.programmedVersion = \
                        BiosCliModel.BiosVersion().toModel( programmed,
                                                            result.runningVersion )
   result.fallbackVersion = \
                        BiosCliModel.BiosVersion().toModel( fallback,
                                                            result.runningVersion )

   if ( not line or not line.startswith( 'norcal' )
        or line in [ 'norcal3', 'norcal5' ] ):
      result.programmedVersion.error += " (not supported)"
      result.fallbackVersion.error += " (not supported)"
   return result

def doInstall( mode, args ):
   src = args[ 'SOURCE' ]
   reboot = 'reload' in args
   now = 'now' in args

   sup = BiosInstaller.ALL
   if 'SUPERVISOR' in args:
      if 'active' in args[ 'SUPERVISOR' ]:
         sup = BiosInstaller.ACTIVE
      elif 'standby' in args[ 'SUPERVISOR' ]:
         sup = BiosInstaller.STANDBY
      else:
         mode.addError( "Supervisor token not accepted" )
         return

   installer = BiosInstaller( mode, sup, reboot, now )
   success = installer.run( src, abootUpdateDir )
   if success == BiosInstaller.SUCCESS:
      mode.addMessage( "BIOS installation succeeded" )
   elif success == BiosInstaller.ABORT:
      mode.addMessage( "BIOS installation aborted" )
   elif success == BiosInstaller.FAIL:
      mode.addError( "BIOS installation failed" )
   else:
      mode.addError( "BIOS installation returned an unknown state" )

#--------------------------------------------------------------------------------
# show platform bios
#--------------------------------------------------------------------------------

class ShowBiosCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform bios'
   data = {
      'platform': platformMatcherForShow,
      'bios': biosMatcherForShow,
   }
   cliModel = BiosCliModel.BiosVersions
   handler = showBios

BasicCli.addShowCommandClass( ShowBiosCmd )

#--------------------------------------------------------------------------------
# install bios source SOURCE [ SUPERVISOR ] [ reload ] [ now ]
#--------------------------------------------------------------------------------
class InstallBiosCmd( CliCommand.CliCommandClass ):
   syntax = 'install bios source SOURCE [ SUPERVISOR ] [ reload ] [ now ]'
   data = {
      'install': installMatcher,
      'bios': biosMatcherForInstall,
      'source': sourceMatcher,
      'SOURCE': UrlMatcher( fsFunc=lambda fs: ( fs.fsType == 'network' or
                                                fs.fsType == 'flash' ),
                            helpdesc='Source path' ),
      'SUPERVISOR': supervisorMatcher,
      'reload': reloadMatcher,
      'now': nowMatcher,
   }
   handler = doInstall

BasicCli.EnableMode.addCommandClass( InstallBiosCmd )
