# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import CliCommand
from CliCommand import CliExpression
import CliMatcher
import CliPlugin.ArBgpCli as ArBgpCli
from CliPlugin.BgpCliHelperCli import convertPeerAddr, convertLargeCommunityValues
from CliPlugin.RoutingBgpCli import configForVrf
from CliPlugin.IpAddrMatcher import ipPrefixMatcher
from CliPlugin.Ip6AddrMatcher import ip6PrefixMatcher
from CliPlugin.IraVrfCli import routeDistinguisherMatcher

from CliPlugin.RouteMapCli import (
        RtSooExtCommCliMatcher,
        LargeCommCliMatcher,
        showLargeCommunityDesc,
)

from CliPlugin.RoutingBgpShowCli import (
        getCommunityValuesScalarList,
)
from CliToken.RoutingBgpShowCliTokens import (
      communityExact,
      longerPrefixes,
      ipGenAddrMatcher,
      showLargeCommunity,
      )
import Tac
from BgpLib import routeTargetToExtCommU64Value
#------------------------------------------------------------------------------
# This file contains common CLI tokens used by MplsVpn and Evpn CliPlugins
#------------------------------------------------------------------------------
# NOTE: This file contains old CLI Parser token as well which are used by Evpn
# CliPlugins. We should remove them once Evpn CliPlugin is also converted for
# new CLI Parser.
showExtCommunity = CliMatcher.KeywordMatcher( "extcommunity",
      helpdesc="Match BGP/VPN extended community list" )

# extcommunity filter
class ExtCommValuesAndExactExpression( CliExpression ):
   expression = "extcommunity { rt RT_VAL } [ exact ]"
   data = {
         "extcommunity" : showExtCommunity,
         "rt" : "Route Target",
         "RT_VAL" : RtSooExtCommCliMatcher( "Route Target" ),
         "exact" : communityExact,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( "extcommunity", None ):
         return
      result = { "extCommunityValues" : args.pop( "RT_VAL" ) }
      if args.pop( "exact", None ):
         result[ "exact" ] = "exact"
      args[ "EXT_COMMUNITIES" ] = result

# large-community filter
class LargeCommValuesAndExactExpression( CliExpression ):
   expression = "large-community { LARGE_COMM_VAL } [ LARGE_COMM_EXACT ]"
   data = {
         "large-community" : showLargeCommunity,
         "LARGE_COMM_VAL" : LargeCommCliMatcher( showLargeCommunityDesc ),
         "LARGE_COMM_EXACT" : communityExact,
      }

   @staticmethod
   def adapter( mode, args, argList ):
      if not args.pop( "large-community", None ):
         return
      result = { "largeCommListVal" : args.get( "LARGE_COMM_VAL" ),
                 "exact" : args.get( "LARGE_COMM_EXACT" ) }
      args[ "LARGE_COMMUNITIES" ] = result

# rd filter
class RdCliExpression( CliExpression ):
   expression = "rd ( RD_VAL )"
   data = { "rd" : "Filter by route distinguisher",
            "RD_VAL" : routeDistinguisherMatcher
         }

# ipv4 prefix filter
class IpPrefixShowExpression ( CliExpression ):
   expression = "( PREFIX [ longer-prefixes ] )"
   data = { "PREFIX" : ipPrefixMatcher,
            "longer-prefixes" : longerPrefixes
          }

# ipv6 prefix filter
class Ipv6PrefixShowExpression ( CliExpression ):
   expression = "( PREFIX [ longer-prefixes ] )"
   data = { "PREFIX" : ip6PrefixMatcher,
            "longer-prefixes" : longerPrefixes
          }

# nexthop filter
class NexthopExpression( CliExpression ):
   expression = "next-hop nexthop-addr"
   data = { "next-hop" : 'Filter by next hop ( IPv4 or IPv6 ) address',
            "nexthop-addr" : ipGenAddrMatcher
          }

# route import keywords
tokenRoute = CliMatcher.KeywordMatcher(
   'route', helpdesc='Route configuration' )
tokenVpnImport = CliMatcher.KeywordMatcher(
   'import', helpdesc='Configure the importing of VPN routes' )

class VpnCliHelperCommand( ArBgpCli.ArBgpCliCommand ):
   def __init__( self, mode, command, nlriAfiSafi, **kwargs ):
      vrfName = kwargs.pop( 'vrfName', None )
      super( VpnCliHelperCommand, self ).__init__(
         mode,
         command,
         vrfName=vrfName,
         nlriAfiSafi=nlriAfiSafi,
         transportAfi=None,
         disableFork=True )

      self.mode = mode

      if self.mode.session_.outputFormat_ == 'json':
         self.addParam( 'json' )

      for k, v in kwargs.iteritems():
         if v:
            self.addParam( k, v )

   def run( self, **kwargs ):
      AgentCommandRequest.runCliPrintSocketCommand( self.entityManager,
                                                    'BgpCliHelper',
                                                    self.command,
                                                    self.paramString(),
                                                    self.mode,
                                                    keepalive=True )

   @staticmethod
   def convertCommunityValues( fromKwargs, toKwargs ):
      value = fromKwargs.get( "communityValues", None )
      commValues = getCommunityValuesScalarList( value )
      toKwargs[ "commValues" ] = "|".join( [ str( c ) for c in commValues ] )
      exact = fromKwargs.get( "exact", None )
      if exact:
         toKwargs[ "standardCommunitiesExactMatch" ] = exact

   @staticmethod
   def convertExtCommunityValues( fromKwargs, toKwargs ):
      """ Input: fromKwargs: cli input params in following format:
      fromKwargs[ "extCommunityValues" ] = [ 1:1 2:2 ]
      toKwargs[ "extCommValues" ]='43189991171031040|21550481834311688|5629542483886'
      toKwargs[ "exact" ] = fromKwargs["exact" ], if it is set."""
      commValues = fromKwargs[ "extCommunityValues" ]
      exact = fromKwargs.get( "exact", None )
      if exact:
         toKwargs[ "extendedCommunitiesExactMatch" ] = exact
      extCommunities = ""
      for extCommValue in commValues:
         if extCommunities != "":
            extCommunities += "|"
         extCommunities += str( routeTargetToExtCommU64Value( extCommValue ) )

      toKwargs[ "extCommValues" ] = extCommunities

   @staticmethod
   def flattenArgs( fromKwargs, toKwargs ):
      C = VpnCliHelperCommand
      for k, v in fromKwargs.items():
         if v is None:
            continue
         if k in [ "prefix", "nlriTypeAndPrefixValues", "prefixValues" ] and (
            isinstance( v, dict ) ):
            C.flattenArgs( v, toKwargs )
         elif k == "commValuesAndExact":
            C.convertCommunityValues( v, toKwargs )
         elif k == "extCommValuesAndExact" and \
               "extCommValues" not in toKwargs:
            C.convertExtCommunityValues( v, toKwargs )
         elif k == "largeCommValuesAndExact":
            convertLargeCommunityValues( fromKwargs=v, kwargs=toKwargs,\
                                         key="largeCommListVal" )
         elif k == "peerAddrValue":
            toKwargs[ "peerAddr" ] = v
            convertPeerAddr( toKwargs )
         elif k == "bgpRouteTypeValue":
            toKwargs[ "routeType" ] = v
         else:
            assert k not in toKwargs
            toKwargs[ k ] = v

#---------------------------------------------------------------------------------
# "[no|default] route import match-failure action discard"
# in the following modes:
#     address-family vpn-ipv4
#     address-family vpn-ipv6
#     address-family evpn
#
# Enables the discarding of VPN paths that won't be imported into any VRF (AID6625).
#---------------------------------------------------------------------------------

# Enables/disables the discarding of unimported VPN paths, or sets it to the
# default value if 'discard' is None
def setRouteImportMatchFailureAction( mode, discard=None ):
   def getVal( defaultValue ):
      return discard if discard is not None else defaultValue

   config = configForVrf( mode.vrfName )
   if mode.addrFamily == 'vpn-ipv4':
      config.vpnPruningEnabledVpnV4 = getVal( config.vpnPruningEnabledVpnV4Default )
   elif mode.addrFamily == 'vpn-ipv6':
      config.vpnPruningEnabledVpnV6 = getVal( config.vpnPruningEnabledVpnV6Default )
   elif mode.addrFamily == 'evpn':
      config.vpnPruningEnabledEvpn = getVal( config.vpnPruningEnabledEvpnDefault )
   else:
      mode.addError( "Not supported in '%s' address-family mode", mode.addrFamily )

class RouteImportMatchFailureDiscardCmd( CliCommand.CliCommandClass ):
   syntax = 'route import match-failure action discard'
   noOrDefaultSyntax = 'route import match-failure action ...'
   data = { 'route' : tokenRoute,
            'import' : tokenVpnImport,
            'match-failure' : 'Configure how to handle VPN routes with no matching '
                              'VRF import Route Targets',
            'action' : 'Configure the action to perform',
            'discard' : 'Discard the route' }

   @staticmethod
   def handler( mode, args ):
      setRouteImportMatchFailureAction( mode, discard=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setRouteImportMatchFailureAction( mode, discard=None )
