# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import ConfigMount
import CliMatcher
import CliCommand
import CliParser
from IpAddrMatcher import ipPrefixMatcher
from Ip6AddrMatcher import ip6PrefixMatcher
from IpLibConsts import DEFAULT_VRF
from RoutingBgpCli import RouterBgpBaseMode, RouterBgpVrfMode, deleteRouterBgpVrfHook
from RoutingBgpCli import bgpMatcherForConfig, ipv4PeerMatcher, ipv6PeerMatcher
import Tac

entityManager = None
vrfTraceFacilityDir = None

def deleteBgpVrfTraceHook( vrfName ):
   if vrfName in vrfTraceFacilityDir.traceFacility:
      del vrfTraceFacilityDir.traceFacility[ vrfName ]

def traceFacilityForVrf( vrfName ):
   return vrfTraceFacilityDir.traceFacility[ vrfName ]

class RouterBgpTraceSharedModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      if self.vrfName not in vrfTraceFacilityDir.traceFacility:
         vrfTraceFacilityDir.traceFacility.newMember( self.vrfName )

RouterBgpBaseMode.addModelet( RouterBgpTraceSharedModelet )
RouterBgpVrfMode.addModelet( RouterBgpTraceSharedModelet )

traceHiddenMatcher = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'trace',
         helpdesc="Configure BGP trace settings" ), hidden=True )

#-------------------------------------------------------------------------------
# [ no | default ] bgp trace neighbor ( <ip> | all )
#-------------------------------------------------------------------------------
def changeNeighborTrace( neighborFacility, vrfName, no=False ):
   traceFacility = traceFacilityForVrf( vrfName )
   addr = Arnet.IpGenAddr( str( neighborFacility ) )
   if not no:
      peerEntry = Tac.Value( 'Routing::Bgp::TraceFacilityPeerEntry', addr )
      peerEntry.enabled = True
      traceFacility.peerEntry.addMember( peerEntry )
   else:
      del traceFacility.peerEntry[ addr ]

def changeNeighborTraceForAll( vrfName, default=None, no=False ):
   traceFacility = traceFacilityForVrf( vrfName )
   peerEntryForAll = Tac.Value( 'Routing::Bgp::TraceFacilityEntryForAll' )
   peerEntryForAll.enabled = not no
   if ( not ( default or vrfName == DEFAULT_VRF ) ) or no:
      peerEntryForAll.isSet = True
   traceFacility.peerEntryForAll = peerEntryForAll
   traceFacility.peerEntry.clear()

class BgpTraceNeighbor( CliCommand.CliCommandClass ):
   syntax = "bgp trace neighbor ( ADDR | ADDR6 | all )"
   noOrDefaultSyntax = syntax
   data = {
         'bgp' : bgpMatcherForConfig,
         'trace' : traceHiddenMatcher,
         'ADDR' : ipv4PeerMatcher,
         'ADDR6' : ipv6PeerMatcher,
         'neighbor' : 'Neighbor trace settings',
         'all' : 'All trace facilities'
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ADDR' in args:
         args[ 'PEER' ] = args[ 'ADDR' ]
      elif 'ADDR6' in args:
         args[ 'PEER' ] = args[ 'ADDR6' ]

   @staticmethod
   def handler( mode, args ):
      if 'all' in args:
         changeNeighborTraceForAll( mode.vrfName )
      else:
         changeNeighborTrace( args[ 'PEER' ], mode.vrfName )

   @staticmethod
   def noHandler( mode, args ):
      if 'all' in args:
         changeNeighborTraceForAll( mode.vrfName, no=True )
      else:
         changeNeighborTrace( args[ 'PEER' ], mode.vrfName, no=True )

   @staticmethod
   def defaultHandler( mode, args ):
      if 'all' in args:
         changeNeighborTraceForAll( mode.vrfName, default=True )
      else:
         changeNeighborTrace( args[ 'PEER' ], mode.vrfName, no=True )

RouterBgpTraceSharedModelet.addCommandClass( BgpTraceNeighbor )


#-------------------------------------------------------------------------------
# [ no | default ] bgp trace route-key ( <prefix> | <afi> [<safi>] | all )
#-------------------------------------------------------------------------------
def setRouteKeyPrefixTrace( vrfName, prefixMatch ):
   traceFacility = traceFacilityForVrf( vrfName )
   prefix = Arnet.IpGenAddrWithMask( str( prefixMatch ) ).subnet
   routeKeyEntry = Tac.Value( 'Routing::Bgp::TraceFacilityRouteKeyEntryForPrefix',
                               prefix )
   routeKeyEntry.enabled = True
   traceFacility.routeKeyEntryForPrefix.addMember( routeKeyEntry )

def noRouteKeyPrefixTrace( vrfName, prefixMatch ):
   traceFacility = traceFacilityForVrf( vrfName )
   prefix = Arnet.IpGenAddrWithMask( str( prefixMatch ) ).subnet
   del traceFacility.routeKeyEntryForPrefix[ prefix ]

def setRouteKeyAfiSafiTrace( vrfName, afi, safi ):
   traceFacility = traceFacilityForVrf( vrfName )
   afiSafi = Tac.Value( "Routing::Bgp::AfiSafi", afi, safi )
   routeKeyEntryForAfiSafi = (
         Tac.Value( 'Routing::Bgp::TraceFacilityRouteKeyEntryForAfiSafi',
                    afiSafi.afiSafiStorage ) )
   routeKeyEntryForAfiSafi.enabled = True
   routeKeyEntryForAfiSafi.isSet = True
   traceFacility.routeKeyEntryForAfiSafi.addMember( routeKeyEntryForAfiSafi )

def noRouteKeyAfiSafiTrace( vrfName, afi, safi, default=False ):
   traceFacility = traceFacilityForVrf( vrfName )
   afiSafi = Tac.Value( "Routing::Bgp::AfiSafi", afi, safi )
   if default or vrfName == DEFAULT_VRF:
      del traceFacility.routeKeyEntryForAfiSafi[ afiSafi.afiSafiStorage ]
   else:
      routeKeyEntryForAfiSafi = (
            Tac.Value( 'Routing::Bgp::TraceFacilityRouteKeyEntryForAfiSafi',
                       afiSafi.afiSafiStorage ) )
      routeKeyEntryForAfiSafi.enabled = False
      routeKeyEntryForAfiSafi.isSet = True
      traceFacility.routeKeyEntryForAfiSafi.addMember( routeKeyEntryForAfiSafi )

def setRouteKeyTraceForAll( vrfName, default=None ):
   traceFacility = traceFacilityForVrf( vrfName )
   routeKeyEntryForAll = Tac.Value( 'Routing::Bgp::TraceFacilityEntryForAll' )
   routeKeyEntryForAll.enabled = True
   if not default and vrfName != DEFAULT_VRF:
      routeKeyEntryForAll.isSet = True
   traceFacility.routeKeyEntryForAll = routeKeyEntryForAll
   traceFacility.routeKeyEntryForPrefix.clear()
   traceFacility.routeKeyEntryForAfiSafi.clear()

def noRouteKeyTraceForAll( vrfName ):
   traceFacility = traceFacilityForVrf( vrfName )
   routeKeyEntryForAll = Tac.Value( 'Routing::Bgp::TraceFacilityEntryForAll' )
   routeKeyEntryForAll.enabled = False
   routeKeyEntryForAll.isSet = True
   traceFacility.routeKeyEntryForAll = routeKeyEntryForAll
   traceFacility.routeKeyEntryForPrefix.clear()
   traceFacility.routeKeyEntryForAfiSafi.clear()

class BgpTraceRouteKey( CliCommand.CliCommandClass ):
   syntax = (
         "bgp trace route-key ( PREFIX | PREFIX6 | ( ipv4 | ipv6 unicast ) | all )" )
   noOrDefaultSyntax = syntax
   data = {
         'bgp' : bgpMatcherForConfig,
         'trace' : traceHiddenMatcher,
         'route-key' : 'Route-key trace settings',
         'neighbor' : 'Neighbor trace settings',
         'PREFIX' : ipPrefixMatcher,
         'PREFIX6' : ip6PrefixMatcher,
         'ipv4' : 'IPv4 related',
         'ipv6' : 'IPv6 related',
         'unicast' : 'Unicast sub-address family',
         'all' : 'All trace facilities'
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'PREFIX' in args:
         args[ 'PEER' ] = args[ 'PREFIX' ]
      elif 'PREFIX6' in args:
         args[ 'PEER' ] = args[ 'PREFIX6' ]
      if 'ipv4' in args:
         args[ 'AFI' ] = 'afiIpv4'
      elif 'ipv6' in args:
         args[ 'AFI' ] = 'afiIpv6'

   @staticmethod
   def handler( mode, args ):
      if 'PEER' in args:
         setRouteKeyPrefixTrace( mode.vrfName, args[ 'PEER' ] )
      elif 'AFI' in args:
         setRouteKeyAfiSafiTrace( mode.vrfName, args[ 'AFI' ], "safiUnicast" )
      else:
         setRouteKeyTraceForAll( mode.vrfName )

   @staticmethod
   def noHandler( mode, args ):
      if 'PEER' in args:
         noRouteKeyPrefixTrace( mode.vrfName, args[ 'PEER' ] )
      elif 'AFI' in args:
         noRouteKeyAfiSafiTrace( mode.vrfName, args[ 'AFI' ], "safiUnicast" )
      else:
         noRouteKeyTraceForAll( mode.vrfName )

   @staticmethod
   def defaultHandler( mode, args ):
      if 'PEER' in args:
         noRouteKeyPrefixTrace( mode.vrfName, args[ 'PEER' ] )
      elif 'AFI' in args:
         noRouteKeyAfiSafiTrace( mode.vrfName, args[ 'AFI' ], "safiUnicast",
                                 default=True )
      else:
         setRouteKeyTraceForAll( mode.vrfName, default=True )

RouterBgpTraceSharedModelet.addCommandClass( BgpTraceRouteKey )


def Plugin( em ):
   global entityManager
   global vrfTraceFacilityDir
   entityManager = em
   vrfTraceFacilityDir = ConfigMount.mount(
         entityManager, 'routing/arbgp/trace',
         'Routing::Bgp::VrfTraceFacilityDir', 'w' )

   deleteRouterBgpVrfHook.addExtension( deleteBgpVrfTraceHook )
