#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher
import ShowCommand
import LazyMount, Cell
from MaintenanceCliLib import showMaintenanceMatcher, defaultMatcher
from MaintenanceCliLib import dynamicUnitName, isDynamicUnit
from MaintenanceCliLib import dynamicUnitComponentName, dynamicUnitVrfName
from CliPlugin.MaintenanceModels import bgpGroupsHook, bgpProfilesHook
from CliPlugin.MaintenanceModels import bgpMaintenanceDebugInfoHook
from CliPlugin.VrfCli import VrfExecCmdDec, generateVrfCliModel
from MaintenanceModels import BgpGroup, BgpGroups, BgpNeighbors, MaintenanceDebugInfo
from MaintenanceModels import BgpProfile, BgpProfiles, MaintenanceEventInfo
from MaintenanceModels import MaintenanceEventInfoEntry
from MaintenanceCliLib import profileNameMatcher
import Tac, CliToken
from MaintenanceModels import showMaintenanceInterfaceHook
from IpLibConsts import DEFAULT_VRF
from MaintenanceModels import MaintenanceInterfaceBgpStatus
from BgpLib import createKey, PeerConfigKey
from BgpMaintenanceModels import BgpMaintenanceDefaultProfile
from RoutingBgpShowCli import getVrfsFunc, allVrfExprFactory
from RoutingBgpCli import bgpNeighborConfig
from BgpMaintenanceModels import BgpMaintenanceStatus, BgpMaintenanceStatusPerPeer
from BgpMaintenanceModels import BgpPeerMaintenanceInfo
from RoutingBgpCli import V4V6PeerKeyCliExpression, PeerCliExpression
from MaintenanceCliLib import toUtc
from MaintenanceCliLib import bgpGroupType, bgpProfileType, defaultProfile
from RouteMapCli import routeMapContainerModel
from RouteMapCliModels import RouteMaps
from MaintenanceCliLib import createGroupKey
from MaintenanceCliLib import maintEnterStageClass, maintExitStageClass
from BgpGroupCli import bgpGroupNameMatcher
from RoutingBgpShowCli import ArBgpShowOutput, EmptyResponseException
from RoutingBgpShowCli import showRibCapiCommand
from DeviceNameLib import eosIntfToKernelIntf
from MaintenanceCliLib import maintRunningShowCliCheck 

allIntfStatusDir = None
ipStatus = None
vrfNbrToGroupDir = None
bgpDefaultProfileDir = None
bgpGroupConfigDir = None
bgpGroupStatusDir = None
maintenanceGroupDir = None
bgpProfileStatusDir = None
mapConfig = None
unitStatusDir = None
defaultInitiatorRmDir = None
defaultReceiverRmDir = None
groupToUnitDir = None
GroupOrigin = Tac.Type( "Group::GroupOrigin" )
maintEnterInstanceLog = None
maintExitInstanceLog = None
defaultRmName = Tac.Type( 'BgpMaintenance::Defaults' ).rmName

def showBgpGroup( bgpGroup ):
   ''' Returns the populated BGP group
       Note: This function expects bgpGroup to be valid
   '''
   ret = BgpGroup()
   ret.origin = bgpGroup.origin
   retPeerGroup = BgpNeighbors()
   retPeerIpv4 = BgpNeighbors()
   retPeerIpv6 = BgpNeighbors()
   for bgpPeer in bgpGroup.neighbor.keys():
      peerStr = bgpPeer.stringValue
      if bgpPeer.type == 'peerGroup':
         retPeerGroup.peers.append( peerStr )
      elif bgpPeer.type == 'peerIpv4':
         retPeerIpv4.peers.append( peerStr )
      elif bgpPeer.type in [ 'peerIpv6', 'peerIpv6Ll' ] :
         retPeerIpv6.peers.append( peerStr )
   bgpPeerDict = { 'peerGroup' : retPeerGroup,
                   'peerIpv4': retPeerIpv4 ,
                   'peerIpv6' : retPeerIpv6 }
   ret.neighbors = bgpPeerDict
   groupName = bgpGroup.groupName
   groupKey = Tac.Value( "Group::GroupKey", bgpGroupType, groupName )
   maintGroup = maintenanceGroupDir.maintenanceGroup.get( groupKey )
   if bgpDefaultProfileDir.profileName == defaultProfile:
      ret.bgpProfile = 'Default'
   else:
      ret.bgpProfile = bgpDefaultProfileDir.profileName
   if maintGroup:
      for profile in maintGroup.profile.keys():
         if profile.type == bgpProfileType: 
            ret.bgpProfile = profile.name
   ret.vrfName = bgpGroup.vrfName
   bgpGroupKey = createGroupKey( bgpGroupType, groupName )
   bgpGroupToUnit = groupToUnitDir.groupToUnit.get( bgpGroupKey ) 
   if bgpGroupToUnit:
      for unit in bgpGroupToUnit.unitName:
         ret.units.append( unit )
   
   return ret

def showBgpGroups_( mode, bgpGroupName=None ):
   allBgpGroups = {}
   if not bgpGroupName:
      for groupStatus in bgpGroupStatusDir.status.values():
         if groupStatus.origin == GroupOrigin.dynamic:
            continue
         bgpGroup = showBgpGroup( groupStatus )
         allBgpGroups[ groupStatus.groupName ] = bgpGroup
   else:
      bgpGroupStatus = bgpGroupStatusDir.status.get( bgpGroupName )
      if bgpGroupStatus:
         bgpGroup = showBgpGroup( bgpGroupStatus )
         allBgpGroups[ bgpGroupName ] = bgpGroup
   return allBgpGroups

@maintRunningShowCliCheck
def showBgpGroups( mode, bgpGroupName=None ):
   allBgpGroups = showBgpGroups_( mode, bgpGroupName )
   return BgpGroups( bgpGroups=allBgpGroups )

def showBgpGroupsHook( mode, groupsModel, bgpGroupName=None ):
   groupsModel.bgpGroups = showBgpGroups_( mode, bgpGroupName )

#-------------------------------------------------------------------------------
# show maintenance groups bgp [ groupName ]
#
# 1. show maintenance groups bgp
# 2. show maintenance groups bgp <groupName>
#-------------------------------------------------------------------------------

bgpMatcher = CliMatcher.KeywordMatcher( 'bgp',
                                        helpdesc='BGP groups information' )

groupsMatcher = CliMatcher.KeywordMatcher( 'groups',
                                           helpdesc='Groups information' )

class ShowMaintenanceGroupsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance groups bgp [ GROUP_NAME ]"
   data = { "maintenance": showMaintenanceMatcher,
            "groups": groupsMatcher,
            "bgp": bgpMatcher,
            "GROUP_NAME": bgpGroupNameMatcher }

   @staticmethod
   def handler( mode, args ):
      return showBgpGroups( mode, bgpGroupName=args.get( "GROUP_NAME" ) )
   cliModel = BgpGroups

BasicCli.addShowCommandClass( ShowMaintenanceGroupsCmd )

def showBgpProfile( profileStatus ):
   ''' Returns the populated Bgp Profile
       Note: This function expects the profileStatus to be valid
   '''
   assert( profileStatus )
   bgpProfile = BgpProfile()
   bgpProfile.routeMapIn = profileStatus.routeMapIn
   bgpProfile.routeMapOut = profileStatus.routeMapOut
   bgpProfile.routeMapInOut = profileStatus.routeMapInOut

   return bgpProfile

def showBgpProfiles_( mode, bgpProfileName=None ):
   allBgpProfiles = {}
   if not bgpProfileName:
      for profile in bgpProfileStatusDir.status:
         if profile == defaultProfile:
            continue
         profileStatus = bgpProfileStatusDir.status.get( profile )
         if profileStatus:
            bgpProfile = showBgpProfile( profileStatus )
            allBgpProfiles[ profile ] = bgpProfile
   else:
      profileStatus = bgpProfileStatusDir.status.get( bgpProfileName )
      if profileStatus:
         bgpProfile = showBgpProfile( profileStatus )
         allBgpProfiles[ bgpProfileName ] = bgpProfile
   return allBgpProfiles

@maintRunningShowCliCheck
def showBgpProfiles( mode, bgpProfileName=None ):
   allBgpProfiles = showBgpProfiles_( mode, bgpProfileName )
   return BgpProfiles( bgpProfiles=allBgpProfiles )

def showBgpProfilesHook( mode, profilesModel, bgpProfileName=None ):
   profilesModel.bgpProfiles = showBgpProfiles_( mode, bgpProfileName )

#-------------------------------------------------------------------------------
# show maintenance profiles bgp [ profileName ]
#
# 1. show maintenance profiles bgp
# 2. show maintenance profiles bgp <profileName>
#-------------------------------------------------------------------------------

bgpProfileNameMatcher = profileNameMatcher( lambda mode:
                                            bgpProfileStatusDir.status.keys() )

profilesMatcher = CliMatcher.KeywordMatcher( 'profiles',
                                             helpdesc='Profiles information' )

class ShowMaintenanceProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance profiles bgp [ PROFILE_NAME ]"
   data = { "maintenance": showMaintenanceMatcher,
            "profiles": profilesMatcher,
            "bgp": bgpMatcher,
            "PROFILE_NAME": bgpProfileNameMatcher }

   @staticmethod
   def handler( mode, args ):
      return showBgpProfiles( mode, bgpProfileName=args.get( "PROFILE_NAME" ) )
   cliModel = BgpProfiles

BasicCli.addShowCommandClass( ShowMaintenanceProfilesCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp ip all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
allMatcher = CliMatcher.KeywordMatcher( 'all',
                                        helpdesc="All the bgp neighbors" )

desc = "Per VRF BGP peer maintenance status information"
BgpMaintenanceStatusVrf = generateVrfCliModel( BgpMaintenanceStatus, desc )

def _doShowMaintenanceBgp( mode, ipv6, peer, vrfName,
                           statusModel=BgpMaintenanceStatus ):
   args = { 'ipv6' : ipv6 }
   cmd = 'MIO_DGET_BGP_MAINTENANCE'
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName
   if peer is not None:
      peerConfigKey = PeerConfigKey( peer )
      if peerConfigKey.type == 'peerIpv4':
         args[ 'peerv4' ] = peerConfigKey.stringValue
      elif peerConfigKey.type in [ 'peerIpv6' ]:
         args[ 'peerv6' ] = peerConfigKey.stringValue
      elif peerConfigKey.type in [ 'peerIpv6Ll' ]:
         args[ 'peerv6' ] = peerConfigKey.v6Addr.stringValue + '%' + \
                             eosIntfToKernelIntf( peerConfigKey.llIntf )
   try:
      return showRibCapiCommand()( mode, statusModel, cmd, args,
                                   clientName='BGP' )
   except EmptyResponseException():
      return None

@maintRunningShowCliCheck
@ArBgpShowOutput( 'doShowMaintenanceBgpIpAll' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=BgpMaintenanceStatusVrf )
def doShowMaintenanceBgpIpAll( mode, vrfName=None ):
   return _doShowMaintenanceBgp( mode, False, None, vrfName )

class ShowMaintenanceBgpIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp ip all [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "ip": CliToken.Ip.ipMatcherForShow,
            "all": allMatcher,
            "VRF": allVrfExprFactory, }

   @staticmethod
   def handler( mode, args ):
      return doShowMaintenanceBgpIpAll( mode, vrfName=args.get( "VRF" ) )

   cliModel = BgpMaintenanceStatusVrf

BasicCli.addShowCommandClass( ShowMaintenanceBgpIpCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp ipv6 all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@maintRunningShowCliCheck
@ArBgpShowOutput( 'doShowMaintenanceBgpIp6All' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=BgpMaintenanceStatusVrf )
def doShowMaintenanceBgpIpv6All( mode, vrfName=None ):
   return _doShowMaintenanceBgp( mode, True, None, vrfName )

class ShowMaintenanceBgpIPv6Cmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp ipv6 all [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "ipv6": CliToken.Ipv6.ipv6MatcherForShow,
            "all": allMatcher,
            "VRF": allVrfExprFactory, }

   @staticmethod
   def handler( mode, args ):
      return doShowMaintenanceBgpIpv6All( mode, vrfName=args.get( "VRF" ) )
   cliModel = BgpMaintenanceStatusVrf

BasicCli.addShowCommandClass( ShowMaintenanceBgpIPv6Cmd )

#-------------------------------------------------------------------------------
# "show maintenance profiles bgp default"
#-------------------------------------------------------------------------------

@maintRunningShowCliCheck
def doShowMaintenanceBgpDefaultProfile( mode, args ):
   bgpDefaultProfile = BgpMaintenanceDefaultProfile()
   defaultProfileName = bgpDefaultProfileDir.profileName
   if defaultProfileName == defaultProfile:
      bgpDefaultProfile.profileName = 'Default'
      bgpDefaultProfile.routeMap = defaultRmName
      bgpDefaultProfile.routeMapIn = defaultRmName
      bgpDefaultProfile.routeMapOut = defaultRmName

      bgpDefaultProfile.routeMapInfo = routeMapContainerModel(
         defaultInitiatorRmDir, defaultRmName )
      bgpDefaultProfile.routeMapInfoIn = routeMapContainerModel(
         defaultInitiatorRmDir, defaultRmName )
      bgpDefaultProfile.routeMapInfoOut = routeMapContainerModel(
         defaultInitiatorRmDir, defaultRmName )
   else:
      bgpDefaultProfile.profileName = defaultProfileName
      bgpDefaultProfileStatus = bgpProfileStatusDir.status.get(
         defaultProfileName, None )
      if not bgpDefaultProfileStatus:
         bgpDefaultProfile.routeMapIn = ""
         bgpDefaultProfile.routeMapOut = ""
         bgpDefaultProfile.routeMap = ""
      else:
         bgpDefaultProfile.routeMapIn = bgpDefaultProfileStatus.routeMapIn
         bgpDefaultProfile.routeMapOut = bgpDefaultProfileStatus.routeMapOut
         if bgpDefaultProfile.routeMapIn == bgpDefaultProfile.routeMapOut:
            bgpDefaultProfile.routeMap = bgpDefaultProfile.routeMapIn
      bgpDefaultProfile.routeMapInfo = routeMapContainerModel(
         mapConfig, bgpDefaultProfile.routeMap )
      bgpDefaultProfile.routeMapInfoIn = routeMapContainerModel(
         mapConfig, bgpDefaultProfile.routeMapIn )
      bgpDefaultProfile.routeMapInfoOut = routeMapContainerModel(
         mapConfig, bgpDefaultProfile.routeMapOut )

   return bgpDefaultProfile

class ShowDefaultProfilesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance profiles bgp default"
   data = { "maintenance": showMaintenanceMatcher,
            "profiles": profilesMatcher,
            "bgp": bgpMatcher,
            "default": defaultMatcher }
   handler = doShowMaintenanceBgpDefaultProfile
   cliModel = BgpMaintenanceDefaultProfile

BasicCli.addShowCommandClass( ShowDefaultProfilesCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp <addr> [vrf <vrfName>]"
#-------------------------------------------------------------------------------

def getBgpMaintInfo( mode, peer, vrfName, statusModel=BgpMaintenanceStatus ):
   ipv6 = peer.type in [ 'peerIpv6', 'peerIpv6Ll' ]
   return _doShowMaintenanceBgp( mode, ipv6, peer, vrfName, statusModel )

def getBgpGroupProfileInfo( mode, peer, vrfName, bgpPeerMaintInfo ):
   peerKey = peer
   if vrfName not in vrfNbrToGroupDir.vrfBgpNeighborToGroup:
      return
   nbrToGroupDir = vrfNbrToGroupDir.vrfBgpNeighborToGroup[ vrfName ]
   if not nbrToGroupDir:
      return
   if not peerKey in nbrToGroupDir.bgpNeighborToGroup.keys():
      # Look if the peer group has any groups associated
      peerConfig = bgpNeighborConfig( peerKey, vrfName, False )
      if peerConfig and peerConfig.isPeerGroupPeer:
         peerKey = peerConfig.peerGroupKey
         if not peerKey in nbrToGroupDir.bgpNeighborToGroup.keys():
            return
      else:
         return
   groups = nbrToGroupDir.bgpNeighborToGroup[ peerKey ].group.keys()
   for group in groups:
      bgpGroup = bgpGroupStatusDir.status.get( group.name )
      if bgpGroup.origin == GroupOrigin.dynamic:
         continue
      bgpPeerMaintInfo.groups.append( group.name )

   selectedProfile = nbrToGroupDir.bgpNeighborToGroup[ peerKey ].selectedProfile
   systemDefaultProfileName = defaultProfile
   # TODO: If the selected profile is not set, the best profile from the configured
   # groups needs to be shown as the best profile.
   if bgpProfileType in selectedProfile:
      profile = selectedProfile[ bgpProfileType ]
      bgpPeerMaintInfo.selectedProfile = profile.name
   if bgpPeerMaintInfo.selectedProfile == systemDefaultProfileName:
      if bgpDefaultProfileDir.profileName == defaultProfile:
         bgpPeerMaintInfo.selectedProfile = 'Default'
      else:
         bgpPeerMaintInfo.selectedProfile = bgpDefaultProfileDir.profileName

desc = "Per VRF BGP peer maintenance information"
BgpPeerMaintenanceInfoVrf = generateVrfCliModel( BgpPeerMaintenanceInfo,
                                                 desc, revision=5 )
initiatorRM = Tac.Value( "Routing::Bgp::DefaultMaintenanceInitiatorRmap" ).name

@maintRunningShowCliCheck
@ArBgpShowOutput( 'doShowMaintenanceBgp' )
@VrfExecCmdDec( getVrfsFunc=getVrfsFunc, cliModel=BgpPeerMaintenanceInfoVrf )
def doShowMaintenanceBgp( mode, peer, vrfName=None ):
   maintBgp = BgpPeerMaintenanceInfo()

   maintBgp.peer = peer.stringValue
   bgpPeerStatus = getBgpMaintInfo( mode, peer, vrfName,
                                    statusModel=BgpMaintenanceStatusPerPeer )
   rmName = None
   rmNameIn = None
   rmNameOut = None
   if bgpPeerStatus and bgpPeerStatus.peers.get( peer.stringValue ):
      maintBgp.status = bgpPeerStatus
      rmName = bgpPeerStatus.peers[ peer.stringValue ].appliedRouteMap
      rmNameIn = bgpPeerStatus.peers[ peer.stringValue ].appliedRouteMapIn
      rmNameOut = bgpPeerStatus.peers[ peer.stringValue ].appliedRouteMapOut
   if rmName:
      if rmName.lower() == defaultRmName.lower():
         maintBgp.routeMapInfo = routeMapContainerModel( defaultInitiatorRmDir,
                                                         defaultRmName )
      else:
         maintBgp.routeMapInfo = routeMapContainerModel( mapConfig, rmName )
   if rmNameIn:
      if rmNameIn.lower() == defaultRmName.lower():
         maintBgp.routeMapInfoIn = routeMapContainerModel( defaultInitiatorRmDir,
                                                           defaultRmName )
      else:
         maintBgp.routeMapInfoIn = routeMapContainerModel( mapConfig, rmNameIn )
   if rmNameOut:
      if rmNameOut.lower() == defaultRmName.lower():
         maintBgp.routeMapInfoOut = routeMapContainerModel( defaultInitiatorRmDir,
                                                            defaultRmName )
      else:
         maintBgp.routeMapInfoOut = routeMapContainerModel( mapConfig, rmNameOut )

   getBgpGroupProfileInfo( mode, peer, vrfName, maintBgp )
   return maintBgp

class ShowAddrVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp PEER [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "PEER": V4V6PeerKeyCliExpression,
            "VRF": allVrfExprFactory, }

   @staticmethod
   def handler( mode, args ):
      return doShowMaintenanceBgp( mode, args[ "PEER" ], vrfName=args.get( "VRF" ) )
   cliModel = BgpPeerMaintenanceInfoVrf

BasicCli.addShowCommandClass( ShowAddrVrfCmd )

def getVrf( intfId ):
   ipIntfStatus = ipStatus.ipIntfStatus.get( intfId.stringValue )
   if not ipIntfStatus:
      return None
   return ipIntfStatus.vrf

@ArBgpShowOutput( 'bgpShowMaintenanceIntf' )
def bgpShowMaintenanceIntf( mode, intfId, status ):
   intfStatus = allIntfStatusDir.intfStatus.get( intfId )
   if not intfStatus or intfStatus.deviceName == "":
      return None
   args = { 'if_name' : intfStatus.deviceName }

   vrfName = getVrf( intfId )
   if not vrfName:
      return None
   if vrfName != DEFAULT_VRF:
      args[ 'vrfName' ] = vrfName

   try:
      cmd = 'MIO_DGET_BGP_MAINTENANCE_IF'
      status.bgpStatus = showRibCapiCommand()( mode, MaintenanceInterfaceBgpStatus,
                                               cmd, args, clientName='BGP' )
   except EmptyResponseException():
      return None

#-------------------------------------------------------------------------------
# "show maintenance debug bgp"
#-------------------------------------------------------------------------------

def populateEventInfo( maintEvents, logs ):
   for logEntry in logs.itervalues():
      entry = MaintenanceEventInfoEntry()
      entry.timeStamp = toUtc( logEntry.timestamp )
      entry.eventInfo = logEntry.msg
      maintEvents.append( entry )

def populateDebugInfo( unitStatus ):
   maintenanceEventInfo = MaintenanceEventInfo()
   instanceLogs = maintEnterInstanceLog.instanceLog.get(
      unitStatus.maintEnterInstance, None )
   if instanceLogs:
      populateEventInfo( maintenanceEventInfo.maintEnterEvents,
                         instanceLogs.log )
   instanceLogs = maintExitInstanceLog.instanceLog.get(
      unitStatus.maintExitInstance, None )
   if instanceLogs:
      populateEventInfo( maintenanceEventInfo.maintExitEvents,
                         instanceLogs.log )

   return maintenanceEventInfo

@maintRunningShowCliCheck
def showMaintenanceAllBgpDebugInfo( mode, ignoreDisplay=False ):
   allDebugInfo = {}
   for unitStatus in unitStatusDir.status.values():
      if isDynamicUnit( unitStatus ):
         group = unitStatus.group.keys()[ 0 ]
         peerString = createKey( dynamicUnitComponentName( unitStatus ) ).stringValue
         if group.type == bgpGroupType:
            vrf = dynamicUnitVrfName( unitStatus )
            allDebugInfo[ ( 'Bgp Peer %s(vrf-%s)' %( peerString, vrf ) ) ] = \
                populateDebugInfo( unitStatus )
   return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

debugMatcher = CliMatcher.KeywordMatcher( 'debug',
                                          helpdesc='Debug information' )

class ShowMaintenanceDebugBgpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance debug bgp"
   data = { "maintenance": showMaintenanceMatcher,
            "debug": debugMatcher,
            "bgp": bgpMatcher }

   @staticmethod
   def handler( mode, args ):
      return showMaintenanceAllBgpDebugInfo( mode )
   cliModel = MaintenanceDebugInfo

BasicCli.addShowCommandClass( ShowMaintenanceDebugBgpCmd )

#-------------------------------------------------------------------------------
# show maintenance debug bgp <peer> [ vrf <vrf_name> ]
#-------------------------------------------------------------------------------

@maintRunningShowCliCheck
def showMaintenanceBgpDebugInfo( mode, peer, vrf=None ):
   allDebugInfo = {}
   if not vrf:
      vrf = 'default'
   peerKey = createKey( peer )
   peerString = peerKey.stringValue
   peerUnitName = dynamicUnitName( peerString, vrf )
   peerUnitStatus = unitStatusDir.status.get( peerUnitName )
   if peerUnitStatus:
      allDebugInfo[ ( 'Bgp Peer %s(vrf-%s)' %( peerString, vrf ) ) ] = \
            populateDebugInfo( peerUnitStatus )
   return MaintenanceDebugInfo( maintenanceEventInfo=allDebugInfo )

class ShowPeerDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance debug bgp PEER [ VRF ]"
   data = { "maintenance": showMaintenanceMatcher,
            "debug": debugMatcher,
            "bgp": bgpMatcher,
            "PEER": PeerCliExpression,
            "VRF": allVrfExprFactory, }

   @staticmethod
   def handler( mode, args ):
      return showMaintenanceBgpDebugInfo( mode, args[ "PEER" ],
                                          vrf=args.get( "VRF" ) )
   cliModel = MaintenanceDebugInfo

BasicCli.addShowCommandClass( ShowPeerDebugCmd )

#-------------------------------------------------------------------------------
# show maintenance bgp receiver route-map
#-------------------------------------------------------------------------------
receiverMatcher = CliMatcher.KeywordMatcher( 'receiver',
                                             helpdesc='Show receiver route map' )
routemapMatcher = CliMatcher.KeywordMatcher( 'route-map',
                                             helpdesc='Show receiver route map' )

@maintRunningShowCliCheck
def doShowMaintenanceBgpReceiverRm( mode, args ):
   return routeMapContainerModel( defaultReceiverRmDir, defaultRmName )   

class ShowReceiverRouteCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show maintenance bgp receiver route-map"
   data = { "maintenance": showMaintenanceMatcher,
            "bgp": bgpMatcher,
            "receiver": receiverMatcher,
            "route-map": routemapMatcher }
   handler = doShowMaintenanceBgpReceiverRm
   cliModel = RouteMaps

BasicCli.addShowCommandClass( ShowReceiverRouteCmd )


def Plugin( entityManager ):
   global bgpGroupConfigDir, bgpGroupStatusDir, bgpProfileStatusDir
   global allIntfStatusDir, ipStatus, unitStatusDir
   global vrfNbrToGroupDir, bgpDefaultProfileDir
   global maintenanceGroupDir, mapConfig, defaultInitiatorRmDir
   global defaultReceiverRmDir, groupToUnitDir
   global maintEnterInstanceLog, maintExitInstanceLog

   bgpGroupsHook.addExtension( showBgpGroupsHook )
   bgpProfilesHook.addExtension( showBgpProfilesHook )
   bgpMaintenanceDebugInfoHook.addExtension( showMaintenanceAllBgpDebugInfo )

   unitStatusDir = LazyMount.mount( entityManager,
                                    'maintenance/unit/status',
                                    'Maintenance::Unit::StatusDir', 'r' )
   bgpGroupConfigDir = LazyMount.mount( entityManager,
                                        'group/config/bgp',
                                        'BgpGroup::ConfigDir', 'r' )
   bgpGroupStatusDir = LazyMount.mount( entityManager,
                                        'group/status/bgp',
                                        'BgpGroup::StatusDir', 'r' )
   bgpProfileStatusDir = LazyMount.mount( entityManager,
                                          'maintenance/profile/status/bgp',
                                          'BgpMaintenanceProfile::StatusDir', 'r' )
   maintEnterLogPath = Cell.path( 'stage/' + maintEnterStageClass + '/log' )
   maintExitLogPath = Cell.path( 'stage/' + maintExitStageClass + '/log' )
   maintEnterInstanceLog = LazyMount.mount( entityManager, maintEnterLogPath,
                                            'Stage::Log', 'r' )
   maintExitInstanceLog = LazyMount.mount( entityManager, maintExitLogPath,
                                           'Stage::Log', 'r' )
   bgpDefaultProfileDir = LazyMount.mount(
      entityManager,
      'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   vrfNbrToGroupDir = LazyMount.mount( entityManager,
         'maintenance/mapping/member/bgp',
         'BgpMaintenance::VrfBgpNeighborToGroupDir', 'r' )
   maintenanceGroupDir = LazyMount.mount(
      entityManager, 'maintenance/group/config',
      'Maintenance::MaintenanceGroup::ConfigDir', 'r' )
   defaultInitiatorRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/initiatorRM/',
      'Routing::RouteMap::Config', 'r' )
   defaultReceiverRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/receiverRM/',
      'Routing::RouteMap::Config', 'r' )
   mapConfig = LazyMount.mount(
      entityManager, 'routing/routemap/config', 'Routing::RouteMap::Config', 'r' )
   groupToUnitDir = LazyMount.mount( entityManager,
                                     'maintenance/mapping/group',
                                     'Maintenance::GroupToUnitDir', 'r' )
   showMaintenanceInterfaceHook.addExtension( bgpShowMaintenanceIntf )
