# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import BasicCli
import CliParser
import CliMatcher
import ConfigMount
import CliCommand
import LazyMount
import re
from BgpGroupCli import (
   BgpGroupConfigMode,
   BgpBuiltinGroupConfigMode,
)
from CliMode.BgpMaintenanceProfile import BgpMaintenanceProfileMode
from CliMode.BgpMaintenanceMode import BgpMaintenanceMode
from BgpLib import PeerConfigKey
from CliPlugin.IraIpCli import getAllVrfNames
from CliPlugin.RouteMapCli import mapNameMatcher
from CliPlugin.RoutingBgpCli import PeerCliExpression
from CliPlugin.MaintenanceCliLib import (
   bgpGroupType,
   bgpProfileType,
   defaultProfile,
   DynamicComponentRe,
   dynamicGroupName,
   dynamicUnitName,
   dynamicUnitType,
   linecardBuiltinGroupPrefix,
   maintenanceKwMatcher,
   Profile,
   profileMatcher,
   profileNameMatcher,
   quiesceMatcher,
   reservedProfileName,
)
from CliPlugin.MaintenanceMode import MaintenanceConfigMode
from CliPlugin.MaintenanceModels import profilesCleanupHook
from CliPlugin.MaintenanceGroupCli import MaintenanceGroup
from CliPlugin.IntfGroupLib import (
   IntfBuiltinGroupConfigMode,
   IntfGroupConfigMode,
)
from CliPlugin.IntfGroupCli import IntfBuiltinGroup

globalBgpGroupConfigDir = None
globalBgpProfileConfigDir = None
globalMaintenanceUnitConfigDir = None
globalMaintenanceUnitStatusDir = None
globalMaintenanceUnitInputDir = None
globalDefaultBgpProfile = None

GroupOrigin = Tac.Type( "Group::GroupOrigin" )

matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
                                        helpdesc='Configure BGP' )
matcherVrf = CliMatcher.KeywordMatcher( 'vrf', helpdesc='VRF name' )
matcherBgpExec = CliMatcher.KeywordMatcher( 'bgp', helpdesc='Bgp' )

#-------------------------------------------------------------------------------
# [no] maintenance profile bgp <profileName>
# in "config-group-bgp-<groupName>" mode
#-------------------------------------------------------------------------------
bgpProfileNameMatcher = profileNameMatcher( lambda mode:
                                            globalBgpProfileConfigDir.config.keys() )

class MaintenanceProfileBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'maintenance profile bgp PROFILE_NAME'
   noOrDefaultSyntax = 'maintenance profile bgp [ PROFILE_NAME ]'
   data = {
      'maintenance': maintenanceKwMatcher,
      'profile': profileMatcher,
      'bgp': matcherBgp,
      'PROFILE_NAME': bgpProfileNameMatcher,
   }

   @staticmethod
   def _handleLineCardProfile( mode, profileName, add ):
      """
      Linecard Interface group has list of builtin Interface groups.
      Add/Remove profile for all maintenance groups in the list.
      """
      if add:
         profileKey = Tac.Value( 'Maintenance::Profile::ProfileKey',
                                 type='profileTypeBgp', name=profileName )
      for _group in mode.group():
         assert( isinstance( _group, IntfBuiltinGroup ) and
                 _group.name().startswith( linecardBuiltinGroupPrefix ) )
         maintenanceGroup = MaintenanceGroup( _group.key() )
         if add:
            maintenanceGroup.addProfile( profileKey )
         else:
            maintenanceGroup.remProfile( bgpProfileType, profileName )

   @staticmethod
   def _addProfile( mode, profileName ):
      if profileName.lower() == reservedProfileName:
         mode.addError( "The profile name '%s' is reserved." % profileName )
         return
      if isinstance( mode.group(), list ):
         MaintenanceProfileBgpCmd._handleLineCardProfile( mode,
                                                          profileName,
                                                          True )
      else:
         profileKey = Tac.Value( 'Maintenance::Profile::ProfileKey',
                                 type='profileTypeBgp', name=profileName )
         maintenanceGroup = MaintenanceGroup( mode.group().key() )
         maintenanceGroup.addProfile( profileKey )

   @staticmethod
   def _remProfile( mode, profileName ):
      if isinstance( mode.group(), list ):
         MaintenanceProfileBgpCmd._handleLineCardProfile( mode,
                                                          profileName,
                                                          False )
      else:
         maintenanceGroup = MaintenanceGroup( mode.group().key() )
         maintenanceGroup.remProfile( bgpProfileType, profileName )

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      MaintenanceProfileBgpCmd._addProfile( mode, profileName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args.get( 'PROFILE_NAME' )
      MaintenanceProfileBgpCmd._remProfile( mode, profileName )

IntfGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )
IntfBuiltinGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )
BgpGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )
BgpBuiltinGroupConfigMode.addCommandClass( MaintenanceProfileBgpCmd )

#-------------------------------------------------------------------------------
# [no] profile bgp <profileName>
# in "config-maintenance" mode
#-------------------------------------------------------------------------------

class BgpMaintenanceProfileConfigMode( BgpMaintenanceProfileMode,
   BasicCli.ConfigModeBase ):
   name = "Bgp Maintenance profile configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, bgpProfile ):
      self.bgpProfile_ = bgpProfile
      self.session_ = session
      BgpMaintenanceProfileMode.__init__( self, self.bgpProfile_.name() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#
# BgpProfile class: Holds cli state for each configured BGP profile
#
class BgpProfile( Profile ):
   def __init__( self, profileName ):
      self.name_ = profileName
      Profile.__init__( self )

   def addProfile( self, profileName ):
      if profileName not in globalBgpProfileConfigDir.config:
         profileConfig = globalBgpProfileConfigDir.newConfig( profileName )
         Profile.addProfile( self, profileConfig.key )

   def remProfile( self, profileName ):
      if profileName in globalBgpProfileConfigDir.config.keys():
         profileConfig = globalBgpProfileConfigDir.config.get( profileName )
         Profile.remProfile( self, profileConfig.key )
         del globalBgpProfileConfigDir.config[ profileName ]

   def name( self ):
      return self.name_

class ProfileBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'profile bgp PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'profile': profileMatcher,
      'bgp': matcherBgp,
      'PROFILE_NAME': bgpProfileNameMatcher,
   }

   @staticmethod
   def delBgpProfileConfig_( mode, profileName ):
      bgpProfile = BgpProfile( profileName )
      bgpProfile.remProfile( profileName )

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == reservedProfileName:
         mode.addError( "The profile name '%s' is reserved." % profileName )
         return
      bgpProfile = BgpProfile( profileName )
      bgpProfile.addProfile( profileName )
      childMode = mode.childMode( BgpMaintenanceProfileConfigMode,
                                  bgpProfile=bgpProfile )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      ProfileBgpCmd.delBgpProfileConfig_( mode, profileName )

MaintenanceConfigMode.addCommandClass( ProfileBgpCmd )

#---------------------------------------------------------------------------------
# [ no | default ] initiator route-map MAPNAME ( inout | in | out )
# and
# [ no | default ] initiator route-map -- removes all the route-map names
# in "maint-profile-bgp" mode
#---------------------------------------------------------------------------------
class InitiatorRouteMapCmd( CliCommand.CliCommandClass ):
   syntax = 'initiator route-map MAPNAME DIRECTION'
   noOrDefaultSyntax = 'initiator route-map [ MAPNAME DIRECTION ]'
   data = {
      'initiator': 'Maintenance Mode Initiator',
      'route-map':
      'Configure route-map to override default inbound and outbound policy',
      'MAPNAME': mapNameMatcher,
      'DIRECTION': CliMatcher.EnumMatcher( {
         'in': 'Apply inbound only',
         'out': 'Apply outbound only',
         'inout': 'Apply inbound and outbound',
      } ),
   }

   @staticmethod
   def _setMaintenanceModeRouteMap( mode, mapName=None, mapDirectionAttr=None ):
      profileName = mode.bgpProfile_.name_
      profileConfig = globalBgpProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return
      setattr( profileConfig, mapDirectionAttr, mapName )

   @staticmethod
   def _noMaintenanceModeRouteMap( mode, mapName=None, mapDirectionAttr=None ):
      profileName = mode.bgpProfile_.name_
      profileConfig = globalBgpProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return
      if getattr( profileConfig, mapDirectionAttr ) == mapName:
         setattr( profileConfig, mapDirectionAttr, profileConfig.routeMapDefault )

   @staticmethod
   def _delAllMaintenanceModeRouteMaps( mode ):
      profileName = mode.bgpProfile_.name_
      profileConfig = globalBgpProfileConfigDir.config.get( profileName )
      if not profileConfig:
         return
      profileConfig.routeMapInOut = profileConfig.routeMapDefault
      profileConfig.routeMapIn = profileConfig.routeMapDefault
      profileConfig.routeMapOut = profileConfig.routeMapDefault

   @staticmethod
   def adapter( mode, args, argsList ):
      directionToAttr = {
         'in': 'routeMapIn',
         'out': 'routeMapOut',
         'inout': 'routeMapInOut',
      }
      if 'DIRECTION' in args:
         args[ 'directionAttr' ] = directionToAttr[ args[ 'DIRECTION' ] ]

   @staticmethod
   def handler( mode, args ):
      InitiatorRouteMapCmd._setMaintenanceModeRouteMap(
         mode, mapName=args[ 'MAPNAME' ], mapDirectionAttr=args[ 'directionAttr' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if args.get( 'MAPNAME' ):
         InitiatorRouteMapCmd._noMaintenanceModeRouteMap(
            mode, mapName=args.get( 'MAPNAME' ),
            mapDirectionAttr=args.get( 'directionAttr' ) )
      else:
         InitiatorRouteMapCmd._delAllMaintenanceModeRouteMaps( mode )

BgpMaintenanceProfileConfigMode.addCommandClass( InitiatorRouteMapCmd )

#-------------------------------------------------------------------------------
# [no] bgp <peerName> [ vrf <vrfName> ]
# in "config-maintenance" mode
#-------------------------------------------------------------------------------

class BgpMaintenanceConfigMode( BgpMaintenanceMode,
                                BasicCli.ConfigModeBase ):
   name = "Bgp Maintenance Mode configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, bgpPeer, vrfName=None ):
      if vrfName:
         self.bgpPeer_ = dynamicUnitName( bgpPeer, vrfName )
      else:
         self.bgpPeer_ = dynamicUnitName( bgpPeer, 'default' )

      self.session_ = session
      BgpMaintenanceMode.__init__( self, self.bgpPeer_ )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class BgpMaintenanceConfigModeCmd( CliCommand.CliCommandClass ):
   syntax = 'bgp PEER [ vrf VRFNAME ]'
   noOrDefaultSyntax = syntax
   data = {
      'bgp': matcherBgpExec,
      'PEER': PeerCliExpression,
      'vrf': matcherVrf,
      'VRFNAME': CliMatcher.DynamicNameMatcher(
         getAllVrfNames,
         helpdesc='VRF name'
      ),
   }

   @staticmethod
   def handler( mode, args ):
      peer = args[ 'PEER' ]
      vrfName = args.get( 'VRFNAME', 'default' )
      peerKey = PeerConfigKey( peer )
      peerString = peerKey.stringValue
      # Create a new dynamic unit and group and add it to input
      # Not having any profile in the dynamic group would mean default profile

      unitName = dynamicUnitName( peerString, vrfName=vrfName )
      groupName = dynamicGroupName( peerString, vrfName=vrfName )

      dynamicUnit = globalMaintenanceUnitConfigDir.newConfig( unitName )
      dynamicUnit.unitType = dynamicUnitType
      dynamicGroup = globalBgpGroupConfigDir.newConfig( groupName )
      dynamicGroup.origin = GroupOrigin.dynamic
      dynamicGroup.vrfName = vrfName

      dynamicGroup.neighbor[ peerKey ] = True
      groupKey = Tac.Value( 'Group::GroupKey', bgpGroupType, groupName )
      dynamicUnit.group[ groupKey ] = True

      if not isinstance( peer, str ):
         peer = peer.stringValue

      childMode = mode.childMode( BgpMaintenanceConfigMode,
                                  bgpPeer=peer, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      peer = args[ 'PEER' ]
      vrfName = args.get( 'VRFNAME', 'default' )
      peerString = PeerConfigKey( peer ).stringValue
      unitName = dynamicUnitName( peerString, vrfName=vrfName )
      groupName = dynamicGroupName( peerString, vrfName=vrfName )
      del globalMaintenanceUnitConfigDir.config[ unitName ]
      del globalBgpGroupConfigDir.config[ groupName ]
      if globalMaintenanceUnitInputDir.unit.has_key( unitName ):
         del globalMaintenanceUnitInputDir.unit[ unitName ]

MaintenanceConfigMode.addCommandClass( BgpMaintenanceConfigModeCmd )

#-------------------------------------------------------------------------------
# [ no | default ] quiesce
# in "config-maint-bgp-<peerName>" mode
#-------------------------------------------------------------------------------
class QuiesceCmd( CliCommand.CliCommandClass ):
   syntax = 'quiesce'
   noOrDefaultSyntax = syntax
   data = {
      'quiesce': quiesceMatcher,
   }

   @staticmethod
   def _enterMaintenanceModeBgp( mode, peer, vrfName=None ):
      if not vrfName:
         vrfName = 'default'
      peerString = PeerConfigKey( peer ).stringValue
      unitName = dynamicUnitName( peerString, vrfName=vrfName )
      globalMaintenanceUnitInputDir.unit[ unitName ] = True

   @staticmethod
   def _exitMaintenanceModeBgp( mode, peer, vrfName=None ):
      if not vrfName:
         vrfName = 'default'
      peerString = PeerConfigKey( peer ).stringValue
      unitName = dynamicUnitName( peerString, vrfName=vrfName )
      if globalMaintenanceUnitInputDir.unit.has_key( unitName ):
         del globalMaintenanceUnitInputDir.unit[ unitName ]

   @staticmethod
   def handler( mode, args ):
      bgpVrfRe = "<Dynamic.+><.+><vrf-(.+)>"
      vrf = re.search( bgpVrfRe, mode.bgpPeer_ ).group( 1 )
      peer = re.search( DynamicComponentRe, mode.bgpPeer_ ).group( 1 )
      if vrf != 'default':
         QuiesceCmd._enterMaintenanceModeBgp( mode, peer, vrf )
      else:
         QuiesceCmd._enterMaintenanceModeBgp( mode, peer )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      bgpVrfRe = "<Dynamic.+><.+><vrf-(.+)>"
      vrf = re.search( bgpVrfRe, mode.bgpPeer_ ).group( 1 )
      peer = re.search( DynamicComponentRe, mode.bgpPeer_ ).group( 1 )
      if vrf != 'default':
         QuiesceCmd._exitMaintenanceModeBgp( mode, peer, vrf )
      else:
         QuiesceCmd._exitMaintenanceModeBgp( mode, peer )

BgpMaintenanceConfigMode.addCommandClass( QuiesceCmd )

#-------------------------------------------------------------------------------
# "[no] profile bgp <profileName> default"
# in "config-maintenance" mode
#-------------------------------------------------------------------------------
class ProfileBgpDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'profile bgp PROFILE_NAME default'
   noOrDefaultSyntax = syntax

   data = {
      'maintenance': maintenanceKwMatcher,
      'profile': profileMatcher,
      'bgp': matcherBgp,
      'PROFILE_NAME': bgpProfileNameMatcher,
      'default': 'default profile',
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName.lower() == reservedProfileName:
         mode.addError( "The profile name '%s' is reserved." % profileName )
         return
      globalDefaultBgpProfile.profileName = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE_NAME' ]
      if profileName == globalDefaultBgpProfile.profileName:
         globalDefaultBgpProfile.profileName = defaultProfile

MaintenanceConfigMode.addCommandClass( ProfileBgpDefaultCmd )

def bgpProfileCleanup( mode ):
   for profileName in globalBgpProfileConfigDir.config:
      if profileName != defaultProfile:
         ProfileBgpCmd.delBgpProfileConfig_( mode, profileName )
   globalDefaultBgpProfile.profileName = defaultProfile

def Plugin( entityManager ):
   global globalBgpGroupConfigDir
   global globalBgpProfileConfigDir
   global globalDefaultBgpProfile
   global globalMaintenanceUnitConfigDir
   global globalMaintenanceUnitStatusDir
   global globalMaintenanceUnitInputDir
   profilesCleanupHook.addExtension( bgpProfileCleanup )
   globalBgpGroupConfigDir = ConfigMount.mount( entityManager,
                                                'group/config/bgp',
                                                'BgpGroup::ConfigDir', 'w' )
   globalBgpProfileConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/bgp',
      'BgpMaintenanceProfile::ConfigDir', 'w' )
   globalDefaultBgpProfile = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'w' )
   globalMaintenanceUnitConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/unit/config',
      'Maintenance::Unit::ConfigDir', 'w' )
   globalMaintenanceUnitStatusDir = LazyMount.mount(
      entityManager, 'maintenance/unit/status',
      'Maintenance::Unit::StatusDir', 'r' )
   globalMaintenanceUnitInputDir = ConfigMount.mount( 
      entityManager, 'maintenance/unit/input/cli',
      'Maintenance::Unit::Input', 'w' )
