#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# CliPlugin module for Bgp Mac Vrf commands
import CliCommand
import CliParser
import BasicCli
import Ark
import MultiRangeRule
import ConfigMount
import LazyMount
from CliMatcher import (
   EnumMatcher,
   IntegerMatcher,
   KeywordMatcher,
   PatternMatcher,
   StringMatcher,
)
import CliPlugin.RouteDistinguisher as RouteDistinguisher
from CliPlugin.RoutingBgpCli import RouterBgpBaseMode, deleteRouterBgpMacVrfHook
from CliPlugin.RouteMapCli import RtSooExtCommCliMatcher, AsNumCliExpr
from CliMode.BgpMacVrfConfigMode import BgpMacVrfMode, BgpMacVrfVpwsPwMode
from BgpLib import getVpwsName, routeTargetToExtCommU64Value
from Toggles.ArBgpToggleLib import (
   toggleVpwsFlowLabelToggleEnabled,
   toggleArBgpL2EvpnOverMulticastEnabled,
   toggleRedistRouterMacPrimaryCliEnabled,
)
from Toggles.PseudowireToggleLib import toggleEvpnVpwsFxc1Enabled
import Tac
from BridgeIdHelper import vlanIdToBrId
from TypeFuture import TacLazyType


#-----------------------------------------------------------------------------
# Globals
#-----------------------------------------------------------------------------
VNI_MAX_VALUE = 0xFFFFFF
U32_MAX_VALUE =  0xFFFFFFFF
bgpMacVrfConfigDir = None 
bgpMacVrfRdMap = None
bridgingHwCapabilities = None
entityManager = None
pwHw = "Pseudowire::Hardware::"
BgpControlWordOverride = TacLazyType( pwHw + "BgpControlWordOverride" )
EvpnEtid = TacLazyType( "Evpn::Etid" )
MacVrfTypeEnum = TacLazyType( "Routing::Bgp::MacVrfType" )
MacVrfSubTypeEnum = TacLazyType( "Routing::Bgp::MacVrfSubType" )
MacVrfConfig = TacLazyType( "Routing::Bgp::MacVrfConfig" )

MPLS = 'MPLS transport configuration'

def vniMacVrfSupported( mode, token ):
   platform = Ark.getPlatform()
   if platform and platform != 'veos':
      return 'vni-aware-bundle cannot be enabled on a physical switch'
   return None

def deleteBgpMacVrfConfigHook():
   bgpMacVrfConfigDir.config.clear()

#-----------------------------------------------------------------------------
# MacVrfConfig Mode
#------------------------------------------------------------------------------

def RtToU64( rt ):
   return routeTargetToExtCommU64Value( rt )

class BgpMacVrfConfigModeBase( BgpMacVrfMode, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, macVrfName, isBundle, isVniMacVrf, isVpws ):

      # create new macVrfConfig
      self.name = macVrfName 
      self.session_ = session
      self.macVrfType = None
      if isVpws:
         assert not isBundle
         assert not isVniMacVrf
         self.macVrfType = MacVrfTypeEnum.macVrfTypeVpws
      elif isVniMacVrf:
         self.macVrfType = MacVrfTypeEnum.macVrfTypeVni
      else:
         self.macVrfType = MacVrfTypeEnum.macVrfTypeVlan
      self.macVrfSubType = MacVrfSubTypeEnum.macVrfSubTypeNone
      bgpMacVrfConfigDir.config.newMember( self.name, isBundle, self.macVrfType,
                                           self.macVrfSubType )
      BgpMacVrfMode.__init__( self, self.name, isBundle, isVniMacVrf, isVpws )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      
   def config( self ):
      assert self.name in bgpMacVrfConfigDir.config, "MacVrf not found in Config"
      return bgpMacVrfConfigDir.config[ self.name ]

   def setRouteDistinguisher( self, args ):
      rd = args.get( 'RD' )
      self.config().autoRd = False
      rdToVrfList = bgpMacVrfRdMap.rdToVrfList.get( rd )
      if rdToVrfList and self.name not in rdToVrfList.vrfName:
         self.addWarning( "RD %s already used by other MAC-VRF(s)" % rd )
      self.config().rd = rd

   def noRouteDistinguisher( self, args ):
      self.config().rd = 'INVALID'
      self.config().autoRd = False
      
   def setRouteTargetImport( self, rt ):
      self.config().importRtList[ RtToU64( rt ) ] = True
      del self.config().bothRtList[ RtToU64( rt ) ]
      del self.config().importExportRtList[ RtToU64( rt ) ]

   def noRouteTargetImport( self, rt ):
      del self.config().bothRtList[ RtToU64( rt ) ]
      del self.config().importExportRtList[ RtToU64( rt ) ]
      del self.config().importRtList[ RtToU64( rt ) ]

   def setRouteTargetExport( self, rt ):
      self.config().exportRtList[ RtToU64( rt ) ] = True
      del self.config().bothRtList[ RtToU64( rt ) ]
      del self.config().importExportRtList[ RtToU64( rt ) ]

   def noRouteTargetExport( self, rt ):
      del self.config().bothRtList[ RtToU64( rt ) ]
      del self.config().importExportRtList[ RtToU64( rt ) ]
      del self.config().exportRtList[ RtToU64( rt ) ]

   def setRouteTargetConcatImportExport( self, rt, **kwargs ):
      u64Rt = RtToU64( rt )
      self.config().importRtList[ u64Rt ] = True
      self.config().exportRtList[ u64Rt ] = True
      self.config().importExportRtList[ u64Rt ] = True
      del self.config().bothRtList[ u64Rt ]

   def noRouteTargetConcatImportExport( self, rt, **kwargs ):
      u64Rt = RtToU64( rt )
      del self.config().importRtList[ u64Rt ]
      del self.config().exportRtList[ u64Rt ]
      del self.config().bothRtList[ u64Rt ]
      del self.config().importExportRtList[ u64Rt ]

   @staticmethod
   def _computeImportExport( importExport ):
      """
      Handle the importExport argument from "ruleImportExport"

      It can match one of 'import', 'export' or 'both'.
      """
      if importExport == 'both':
         setImport = True
         setExport = True
      else:
         setImport = importExport == 'import'
         setExport = importExport == 'export'
      return setImport, setExport

   def setRouteTargetImportExport( self, importExport, rt ):
      setImport, setExport = self._computeImportExport( importExport )
      u64Rt = RtToU64( rt )
      if setImport:
         self.config().importRtList[ u64Rt ] = True
      if setExport:
         self.config().exportRtList[ u64Rt ] = True
      if importExport == 'both':
         self.config().bothRtList[ u64Rt ] = True
         del self.config().importExportRtList[ u64Rt ]
      elif setImport and setExport:
         del self.config().bothRtList[ u64Rt ]
         self.config().importExportRtList[ u64Rt ] = True
      else:
         del self.config().bothRtList[ u64Rt ]
         del self.config().importExportRtList[ u64Rt ]

   def noRouteTargetImportExport( self, importExport, rt ):
      clearImport, clearExport = self._computeImportExport( importExport )
      u64Rt = RtToU64( rt )
      if clearImport:
         del self.config().importRtList[ u64Rt ]
      if clearExport:
         del self.config().exportRtList[ u64Rt ]
      del self.config().bothRtList[ u64Rt ]
      del self.config().importExportRtList[ u64Rt ]

   def noRouteTargetAll( self ):
      self.config().importRtList.clear()
      self.config().exportRtList.clear()
      self.config().bothRtList.clear()
      self.config().importExportRtList.clear()

   def setRouteTargetImportExportAuto( self, args ):
      config = self.config()
      asn = args.get( 'AS_NUM' )
      # pylint: disable-msg=simplifiable-if-statement
      if asn is None:
         config.autoExportRt = True
      else:
         config.autoImportRtList[ int( asn ) ] = True

   def noRouteTargetImportExportAuto( self, args ):
      config = self.config()
      asn = args.get( 'AS_NUM' )
      if asn is None:
         config.autoExportRt = False
      else:
         del config.autoImportRtList[ int( asn ) ]

   def setRedistribute( self, source ):
      self.config().addSource( source )

   def noRedistribute( self, source ):
      self.config().removeSource( source )

   def defaultRedistribute( self, source ):
      self.config().defaultSource( source )

   def setRouteDistinguisherAuto( self, args ):
      self.config().rd = 'INVALID'
      self.config().autoRd = True

   def setMaxRoutes( self, args ):
      numRoutes = args.get( 'NUM' )
      self.config().maxRoutes = numRoutes

   def noSetMaxRoutes( self, args ):
      self.config().maxRoutes = 0

   def setControlWord( self, args ):
      """Use default when value=None"""
      self.config().mplsControlWord = True

   def noControlWord( self, args ):
      self.config().mplsControlWord = False

   def setFlowLabel( self, args ):
      self.config().flowLabel = True

   def noFlowLabel( self, args ):
      self.config().flowLabel = False

   def setMtu( self, args ):
      self.config().pseudowireMtu = args.get( 'MTU', 0 )

   def setFxcSignaling( self, args ):
      self.config().fxcMode = 'defaultFxc'

   def noOrDefaultFxcSignaling( self, args ):
      self.config().fxcMode = 'notFxc'

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vni-aware-bundle <bundle name>
#------------------------------------------------------------------------------

class BgpMacVrfVniBundleMode( BgpMacVrfConfigModeBase ):
   # Attributes required for every Mode Class
   name = 'vnimacvrf-bundle'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName, 
            isBundle=True, isVniMacVrf=True, isVpws=False )

   def setRedistVxlan( self, args ):
      enable = not CliCommand.isNoOrDefaultCmd( args )
      self.config().redistributeVcs = enable
      
def getVniBundleName( bundleName ):
   return "vnibundle." + bundleName

def gotoBgpMacVrfVniBundleMode( mode, args ):
   vnibundleName = args[ 'NAME' ]
   macVrfName = getVniBundleName( vnibundleName )

   # Allow only one VNI bundle
   vniBundleExists = False
   macVrfs = bgpMacVrfConfigDir.config.keys()
   if macVrfs:
      if any( 'vnibundle.' in mv for mv in macVrfs ):
         bundle = [ b for b in macVrfs if 'vnibundle.' in b ]
         if bundle[ 0 ] != macVrfName:
            vniBundleExists = True
            err = 'Only one VNI bundle configuration allowed.\n' + \
                  'VNI-Aware-Bundle %s already exists' % \
                  bgpMacVrfConfigDir.config.keys()[ 0 ].split( '.' )[ 1 ]
            mode.addError( err )

   if not vniBundleExists:
      childMode = mode.childMode( BgpMacVrfVniBundleMode, macVrfName=macVrfName )
      mode.session_.gotoChildMode( childMode )

def deleteBgpMacVrfVniBundleMode( mode, args ):
   vnibundleName = args[ 'NAME' ]
   macVrfName = getVniBundleName( vnibundleName )
   del bgpMacVrfConfigDir.config[ macVrfName ]

class BgpVniBundleCommand( CliCommand.CliCommandClass ):
   syntax = 'vni-aware-bundle NAME'
   noOrDefaultSyntax = syntax
   data = {
       'vni-aware-bundle' : 'Configure MAC VRF BGP for multiple VNI support',
       'NAME' : StringMatcher( helpdesc='Unique name to identify VNI Aware Bundle',
                               helpname='VNI Bundle Name' )
   }

   handler = gotoBgpMacVrfVniBundleMode
   noOrDefaultHandler = deleteBgpMacVrfVniBundleMode

RouterBgpBaseMode.addCommandClass( BgpVniBundleCommand )

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vlan <vlan id>
#------------------------------------------------------------------------------

class BgpMacVrfStandAloneMode( BgpMacVrfConfigModeBase ):
   # Attributes required for every Mode Class
   name = 'macvrf-standalone'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName, 
            isBundle=False, isVniMacVrf=False, isVpws=False )
      cfg = bgpMacVrfConfigDir.config[ self.name ]
      vlanId = int( self.macVrfId )
      brId = vlanIdToBrId( vlanId ) 
      cfg.brIdToEtId[ brId  ] = 0
      
def getStandAloneName( vlanId ):
   return "vlan.%d" % vlanId

def gotoBgpMacVrfStandAloneMode( mode, args ):
   vlanId = args[ 'VLAN_ID' ]
   macVrfName = getStandAloneName( vlanId )
   childMode = mode.childMode( BgpMacVrfStandAloneMode, macVrfName=macVrfName )
   mode.session_.gotoChildMode( childMode )

def deleteBgpMacVrfStandAloneMode( mode, args ):
   vlanId = args[ 'VLAN_ID' ]
   macVrfName = getStandAloneName( vlanId )
   del bgpMacVrfConfigDir.config[ macVrfName ]

class BgpMacVrfStandAloneCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN_ID'
   noOrDefaultSyntax = syntax
   data = {
       'vlan' : 'Configure MAC VRF BGP for single VLAN support',
       'VLAN_ID' : IntegerMatcher( 1, 4094, helpdesc='Identifier for a Virtual LAN' )
   }

   handler = gotoBgpMacVrfStandAloneMode
   noOrDefaultHandler = deleteBgpMacVrfStandAloneMode

RouterBgpBaseMode.addCommandClass( BgpMacVrfStandAloneCommand )

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vlan-aware-bundle <bundle name>
#------------------------------------------------------------------------------

class BgpMacVrfVlanBundleMode( BgpMacVrfConfigModeBase ):
   # Attributes required for every Mode Class
   name = 'macvrf-bundle'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName, 
            isBundle=True, isVniMacVrf=False, isVpws=False )
      
   def setVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      current = set( brId.vlanId for brId, etid in
            macVrfConfig.brIdToEtId.iteritems() if etid == 0 )
      newSet = set( vlanSet )
      removeSet = current.difference( newSet )
      addSet = newSet.difference( current )
      for vlanId in removeSet:
         brId = vlanIdToBrId( vlanId ) 
         del macVrfConfig.brIdToEtId[ brId ]
      for vlanId in addSet:
         brId = vlanIdToBrId( vlanId ) 
         macVrfConfig.brIdToEtId[ brId ] = 0

   def noVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      for vlanId in vlanSet:
         brId = vlanIdToBrId( vlanId ) 
         del macVrfConfig.brIdToEtId[ brId ]

   def setVlanToEtidMap( self, args ):
      vlanId = args.get( 'VLAN_ID' )
      etid = args.get( 'ET_ID' )
      brId = vlanIdToBrId( vlanId ) 
      macVrfConfig = self.config()
      # check for duplicates
      for v, e in macVrfConfig.brIdToEtId.iteritems():
         if e == etid and v != brId:
            self.session_.addWarning( "Duplicated ETID assignment detected" )
      macVrfConfig.brIdToEtId[ brId ] = etid

   def addVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      currSet = set( brId.vlanId for brId in macVrfConfig.brIdToEtId.keys() )
      diffSet = set( vlanSet ).difference( currSet )
      for vlanId in diffSet:
         brId = vlanIdToBrId( vlanId )
         macVrfConfig.brIdToEtId[ brId ] = 0

   def removeVlanRange( self, vlanSet ):
      macVrfConfig = self.config()
      for vlanId in vlanSet:
         brId = vlanIdToBrId( vlanId )
         del macVrfConfig.brIdToEtId[ brId ]
   
def getBundleName( bundleName ):
   return "vlanbundle." + bundleName 

def gotoBgpMacVrfBundleMode( mode, args ):
   bundleName = args[ 'NAME' ]
   macVrfName = getBundleName( bundleName )
   childMode = mode.childMode( BgpMacVrfVlanBundleMode, macVrfName=macVrfName )
   mode.session_.gotoChildMode( childMode )

def deleteBgpMacVrfBundleMode( mode, args ):
   bundleName = args[ 'NAME' ]
   macVrfName = getBundleName( bundleName )
   del bgpMacVrfConfigDir.config[ macVrfName ]

class BgpMacVrfBundleCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan-aware-bundle NAME'
   noOrDefaultSyntax = syntax
   data = {
       'vlan-aware-bundle' : 'Configure MAC VRF BGP for multiple VLAN support',
       'NAME' : StringMatcher( helpdesc='Unique name to identify VLAN Aware Bundle',
                               helpname='Bundle Name' )
   }

   handler = gotoBgpMacVrfBundleMode
   noOrDefaultHandler = deleteBgpMacVrfBundleMode

RouterBgpBaseMode.addCommandClass( BgpMacVrfBundleCommand )

#------------------------------------------------------------------------------
# (config-router-bgp)# [no|default] vpws <name>
#------------------------------------------------------------------------------

class BgpMacVrfVpwsMode( BgpMacVrfConfigModeBase ):
   name = 'macvrf-vpws'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, macVrfName ):
      BgpMacVrfConfigModeBase.__init__( self, parent, session, macVrfName,
            isBundle=False, isVniMacVrf=False, isVpws=True )

def gotoBgpMacVrfVpwsMode( mode, args ):
   vpwsEviName = args[ 'VPWS' ]
   macVrfName = getVpwsName( vpwsEviName )
   childMode = mode.childMode( BgpMacVrfVpwsMode, macVrfName=macVrfName )
   mode.session_.gotoChildMode( childMode )

def deleteBgpMacVrfVpwsMode( mode, args ):
   vpwsEviName = args[ 'VPWS' ]
   macVrfName = getVpwsName( vpwsEviName )
   del bgpMacVrfConfigDir.config[ macVrfName ]

class BgpMacVrfVpwsCommand( CliCommand.CliCommandClass ):
   syntax = 'vpws VPWS'
   noOrDefaultSyntax = syntax
   data = {
      'vpws' : 'Configure EVPN instance for VPWS',
      'VPWS' : PatternMatcher(
         pattern='.+',
         helpdesc='Unique name to identify the EVPN instance',
         helpname='WORD' ),
   }

   handler = gotoBgpMacVrfVpwsMode
   noHandler = deleteBgpMacVrfVpwsMode
   defaultHandler = deleteBgpMacVrfVpwsMode

RouterBgpBaseMode.addCommandClass( BgpMacVrfVpwsCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] vlan [add|remove] <range>
#------------------------------------------------------------------------------
def vlanIdListFunc( mode, grList ):
   return grList.values()

bundleVlanSet = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, 4094 ),
   False,
   'VLAN IDs of the MAC VRF',
   value=vlanIdListFunc )

vlanKwMatcher = KeywordMatcher( 'vlan',
      helpdesc='Set of VLANs that forms a MAC VRF' )

class VlanRangeCmd( CliCommand.CliCommandClass ):
   syntax = ' vlan [add|remove] <range> '
   noOrDefaultSyntax = 'vlan <range> ...'
   data = {
      'vlan' : vlanKwMatcher,
      'add' : 'Add VLANs to the current list',
      'remove' : 'Remove VLANs from the current list',
      '<range>' : bundleVlanSet
   }

   @staticmethod
   def handler( mode, args ):
      vlanSet = args.get( '<range>' )
      if args.get( "add" ):
         mode.addVlanRange( vlanSet )
      elif args.get( "remove" ):
         mode.removeVlanRange( vlanSet )
      else:
         mode.setVlanRange( vlanSet )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanSet = args.get( '<range>' )
      mode.noVlanRange( vlanSet )

BgpMacVrfVlanBundleMode.addCommandClass( VlanRangeCmd )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] vlan <vlanId> etid <etid> ]
#------------------------------------------------------------------------------
# Restrict ETID range to 1-4094 in vlan-aware bundle mode. ETID is used as
# dot1q tag in Lfib and anything outside this range will cause ArBgp to crash.
# Per RFC, ETID can either be 24-bit or 12-bit. In our EVPN MPLS
# implementation, it is not useful to have ETID range greater than vlan range.

class VlanToEtidMapCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan VLAN_ID etid ET_ID'
   data = {
       'vlan' : vlanKwMatcher,
       'VLAN_ID' : IntegerMatcher( 1, 4094,
          helpdesc='Identifier for a Virtual LAN' ),
       'etid' : 'ETID mapping for VLAN',
       'ET_ID' : IntegerMatcher( 1, 4094, helpdesc='ETID' )
   }

   handler = BgpMacVrfVlanBundleMode.setVlanToEtidMap

BgpMacVrfVlanBundleMode.addCommandClass( VlanToEtidMapCommand )
# Note: The noOrDefault command is taken care of by [ no|default ] vlan <range>

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] rd <auto | route distinguisher>
#------------------------------------------------------------------------------

rdKwMatcher = KeywordMatcher( 'rd', helpdesc='BGP route distinguisher' )
rdMatcher = RouteDistinguisher.RdDistinguisherMatcher(
      helpdesc='BGP route distinguisher' )

class RdCommand( CliCommand.CliCommandClass ):
   syntax = 'rd RD'
   noOrDefaultSyntax = 'rd ...'
   data = {
         'rd' : rdKwMatcher,
         'RD' : rdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setRouteDistinguisher( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noRouteDistinguisher( args )

BgpMacVrfStandAloneMode.addCommandClass( RdCommand )
BgpMacVrfVlanBundleMode.addCommandClass( RdCommand )
BgpMacVrfVniBundleMode.addCommandClass( RdCommand )
BgpMacVrfVpwsMode.addCommandClass( RdCommand )

class RdAutoCommand( CliCommand.CliCommandClass ):
   syntax = 'rd auto'
   data = {
         'rd' : rdKwMatcher,
         'auto' : 'Auto Generate Route Distinguisher',
   }

   handler = BgpMacVrfStandAloneMode.setRouteDistinguisherAuto

# rd auto is only applicable for VLAN-based service
BgpMacVrfStandAloneMode.addCommandClass( RdAutoCommand )

#---------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] route-target 
#           < import | export | import export | both | all > < auto > < rt | asn >
# e.g.
#   route-target import <rt>
#   route-target export <rt>
#   route-target import export <rt>
#   route-target both <rt>
#---------------------------------------------------------------------------------

importKwMatcher = KeywordMatcher( 'import', helpdesc='Route Import' )
exportKwMatcher = KeywordMatcher( 'export', helpdesc='Route Export' )

class RtCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target ( ( ( import [ export ] ) | export | both ) RT )'
   noOrDefaultSyntax = syntax + ' | all'
   data = {
      'route-target' : 'Route target',
      'import' : importKwMatcher,
      'export' : exportKwMatcher,
      'both' : 'Both Route Import and Export',
      'RT' : RtSooExtCommCliMatcher( "Route target" ),
      'all' : 'removes all import and export route targets',
   }

   @staticmethod
   def handler( mode, args ):
      if 'import' in args and 'export' in args:
         mode.setRouteTargetConcatImportExport( args.get( 'RT' ) )
      else:
         for importExport in [ 'both', 'import', 'export' ]:
            if importExport in args:
               mode.setRouteTargetImportExport( importExport, args.get( 'RT' ) )
               return

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'import' in args and 'export' in args:
         mode.noRouteTargetConcatImportExport( args.get( 'RT' ) )
      elif 'all' in args:
         mode.noRouteTargetAll()
      else:
         for importExport in [ 'both', 'import', 'export' ]:
            if importExport in args:
               mode.noRouteTargetImportExport( importExport, args.get( 'RT' ) )
               return

BgpMacVrfStandAloneMode.addCommandClass( RtCommand )
BgpMacVrfVlanBundleMode.addCommandClass( RtCommand )
BgpMacVrfVniBundleMode.addCommandClass( RtCommand )

class RtAutoCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target ( ( export auto ) | ( import auto AS_NUM ) )'
   noOrDefaultSyntax = syntax
   data = {
     'route-target' : 'Route target',
     'import' : importKwMatcher,
     'export' : exportKwMatcher,
     'auto' : KeywordMatcher( 'auto', 'Auto Generate Route Target' ),
     'AS_NUM' : AsNumCliExpr
   }

   handler = BgpMacVrfStandAloneMode.setRouteTargetImportExportAuto
   noOrDefaultHandler = BgpMacVrfStandAloneMode.noRouteTargetImportExportAuto

# route-target export auto is only applicable for VLAN-based service
BgpMacVrfStandAloneMode.addCommandClass( RtAutoCommand )

class VpwsRouteTargetCommand( CliCommand.CliCommandClass ):
   syntax = 'route-target ( ( import [ export ] ) | export ) evpn RT'
   noOrDefaultSyntax = (
      'route-target ( ( ( import [ export ] ) | export ) evpn RT ) | all' )
   data = {
      'route-target' : 'Route target',
      'import' : CliCommand.Node( KeywordMatcher( 'import', 'Route import' ),
                                  maxMatches=1 ),
      'export' : CliCommand.Node( KeywordMatcher( 'export', 'Route export' ),
                                  maxMatches=1 ),
      'evpn' : 'Import or export only EVPN VPWS routes',
      'RT' : RtSooExtCommCliMatcher( "Route target" ),
      'all' : 'Remove all import and export route targets',
   }

   @staticmethod
   def handler( mode, args ):
      rtVal = args[ 'RT' ]
      setImport = 'import' in args
      setExport = 'export' in args
      if setImport:
         if setExport:
            mode.setRouteTargetConcatImportExport( rtVal )
         else:
            mode.setRouteTargetImport( rtVal )
      else:
         mode.setRouteTargetExport( rtVal )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'all' in args:
         mode.noRouteTargetAll()
      else:
         rtVal = args[ 'RT' ]
         if 'import' in args:
            mode.noRouteTargetImport( rtVal )
         if 'export' in args:
            mode.noRouteTargetExport( rtVal )

BgpMacVrfVpwsMode.addCommandClass( VpwsRouteTargetCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<id>)# [no|default] redistribute
#      < learned [ remote ] | static | dot1x | link-local ipv6 |
#        router-mac [ system | ( next-hop vtep primary ) ] | host-route | igmp >
#------------------------------------------------------------------------------

def vtepClassificationBridgingSupportGuard( mode, token ):
   if ( bridgingHwCapabilities.vxlanVtepToVtepBridgingSupported and
        bridgingHwCapabilities.vtepClassificationBridgingSupported ):
      return None
   return CliParser.guardNotThisPlatform

redistKwMatcher = KeywordMatcher( 'redistribute',
      helpdesc='Redistribute routes in to MAC-VRF' )

class MacVrfRedistributeCommand( CliCommand.CliCommandClass ):
   syntax = ( 'redistribute '
              '( ( learned [ remote ] ) '
              '| static '
              '| dot1x '
              '| ( link-local ipv6 ) '
              '| ( router-mac '
                  '[ system '
                  '| ( next-hop vtep primary ) ] ) '
              '| host-route '
              '| igmp )' )
   data = {
      'redistribute' : redistKwMatcher,
      'learned' : 'locally learned MACs',
      'static' : 'statically defined MACs',
      'router-mac' : 'routerMACs',
      'system' : 'System MAC address',
      'host-route' : 'host route using symmetric IRB',
      'dot1x' : 'dot1x MACs',
      'igmp' : 'IGMP proxy for EVPN',
      'link-local' : 'link local adresses',
      'ipv6' : 'Ipv6 link local adresses',
      'next-hop' : 'configure the advertised next-hop',
      'vtep' : 'associate next-hop with a VTEP',
      'primary' : 'use the primary VTEP IP',
      'remote' : CliCommand.Node(
                    matcher=KeywordMatcher(
                       'remote',
                       helpdesc='datapath learned remote MACs' ),
                    guard=vtepClassificationBridgingSupportGuard ),
   }

   if not toggleArBgpL2EvpnOverMulticastEnabled():
      syntax = syntax.replace( ' | igmp', '' )
      del data[ 'igmp' ]

   if not toggleRedistRouterMacPrimaryCliEnabled():
      syntax = syntax.replace( '| ( next-hop vtep primary )', '' )
      del data[ 'next-hop' ]
      del data[ 'vtep' ]
      del data[ 'primary' ]

   noOrDefaultSyntax = syntax

   @staticmethod
   def _handlerCommon( fn, args ):
      valueDict = {
         'learned' : 'redistributeLearned',
         'static' : 'redistributeStatic',
         'router-mac' : 'redistributeRouterMac',
         'host-route' : 'redistributeHostRoute',
         'dot1x' : 'redistributeDot1x',
         'igmp' : 'redistributeIgmp',
         'link-local' : 'redistributeLinkLocal',
         'system' : 'redistributeSysMac',
         'remote' : 'redistributeRemoteLearned',
         'primary' : 'redistributeRouterMacPrimary',
      }
      for arg in args:
         if arg in valueDict.keys():
            if arg == 'router-mac' and ( 'system' in args or 'next-hop' in args ):
               continue
            if arg == 'learned' and 'remote' in args:
               continue
            fn( valueDict[ arg ] )

   @staticmethod
   def handler( mode, args ):
      MacVrfRedistributeCommand._handlerCommon( mode.setRedistribute, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if CliCommand.isNoCmd( args ):
         MacVrfRedistributeCommand._handlerCommon( mode.noRedistribute, args )
      elif CliCommand.isDefaultCmd( args ):
         MacVrfRedistributeCommand._handlerCommon( mode.defaultRedistribute, args )

BgpMacVrfStandAloneMode.addCommandClass( MacVrfRedistributeCommand )
BgpMacVrfVlanBundleMode.addCommandClass( MacVrfRedistributeCommand )

#------------------------------------------------------------------------------
# (config-macvrf-name)# [no|default] redistribute service vxlan
#------------------------------------------------------------------------------
# This redist command only applies to the vni-macvrfs

class RedistVxlanCommand( CliCommand.CliCommandClass ):
   syntax = 'redistribute service vxlan'
   noOrDefaultSyntax = 'redistribute service vxlan'
   data = {
      'redistribute' : redistKwMatcher,
      'service' : 'Redistribute routes from a service into a MAC-VRF',
      'vxlan' : 'Redistribute routes from VXLAN service into a MAC-VRF'
   }

   handler = BgpMacVrfVniBundleMode.setRedistVxlan
   noOrDefaultHandler = BgpMacVrfVniBundleMode.setRedistVxlan

BgpMacVrfVniBundleMode.addCommandClass( RedistVxlanCommand )

#------------------------------------------------------------------------------
# (config-macvrf-name)# [no|default] maximum-routes <num>
#------------------------------------------------------------------------------

class MaxRoutesCommand( CliCommand.CliCommandClass ):
   syntax = 'maximum-routes NUM'
   noOrDefaultSyntax = 'maximum-routes ...'
   data = {
      'maximum-routes' : 'Maximum number of routes accepted on this MAC-VRF',
      'NUM' : IntegerMatcher( 0, 4294967294,
         helpdesc='Maximum number of accepted routes (0 means unlimited)' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setMaxRoutes( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noSetMaxRoutes( args )

BgpMacVrfStandAloneMode.addCommandClass( MaxRoutesCommand )
BgpMacVrfVlanBundleMode.addCommandClass( MaxRoutesCommand )
BgpMacVrfVniBundleMode.addCommandClass( MaxRoutesCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# [no|default] mpls control-word
#------------------------------------------------------------------------------

class VpwsMplsControlWordCommand( CliCommand.CliCommandClass ):
   syntax = 'mpls control-word'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': MPLS,
      'control-word': 'Enable control word',
   }
   handler = BgpMacVrfVpwsMode.setControlWord
   noOrDefaultHandler = BgpMacVrfVpwsMode.noControlWord

BgpMacVrfVpwsMode.addCommandClass( VpwsMplsControlWordCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# [no|default] label flow
#------------------------------------------------------------------------------

class VpwsFlowLabelCommand( CliCommand.CliCommandClass ):
   syntax = 'label flow'
   noOrDefaultSyntax = syntax
   data = {
      'label' : 'Label operation',
      'flow' : 'Enable flow label',
   }
   handler = BgpMacVrfVpwsMode.setFlowLabel
   noOrDefaultHandler = BgpMacVrfVpwsMode.noFlowLabel

if toggleVpwsFlowLabelToggleEnabled():
   BgpMacVrfVpwsMode.addCommandClass( VpwsFlowLabelCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# [no|default] mtu (ignore|<num>)
#------------------------------------------------------------------------------

class VpwsMtuCommand( CliCommand.CliCommandClass ):
   syntax = 'mtu ( ignore | MTU )'
   noOrDefaultSyntax = 'mtu ...'
   data = {
      'mtu' : 'MTU value to use for VPWS signaling and validation',
      'ignore' : 'Do not send MTU and ignore received MTU',
      'MTU' : IntegerMatcher( 1, 65535,
         helpdesc='Maximum transmission unit in bytes' ),
   }
   handler = BgpMacVrfVpwsMode.setMtu
   noOrDefaultHandler = handler

BgpMacVrfVpwsMode.addCommandClass( VpwsMtuCommand )

#------------------------------------------------------------------------------
# (config-macvrf) [no|default] flexible-cross-connect [vlan]
# Note: [vlan] will be added later
#------------------------------------------------------------------------------

class FxcSignalingCommand( CliCommand.CliCommandClass ):
   syntax = 'flexible-cross-connect'
   noOrDefaultSyntax = 'flexible-cross-connect ...'
   data = {
      'flexible-cross-connect': 'Flexible cross-connect (FXC) configuration',
      #'vlan': 'Use VLAN-signaled FXC cross-connect',
   }
   handler = BgpMacVrfVpwsMode.setFxcSignaling
   noOrDefaultHandler = BgpMacVrfVpwsMode.noOrDefaultFxcSignaling

if toggleEvpnVpwsFxc1Enabled():
   BgpMacVrfVpwsMode.addCommandClass( FxcSignalingCommand )

#------------------------------------------------------------------------------
# (config-macvrf-<name>)# pseudowire <name>
#------------------------------------------------------------------------------

class BgpMacVrfVpwsPwConfigMode( BgpMacVrfVpwsPwMode, BasicCli.ConfigModeBase ):
   name = 'BGP VPWS pseudowire configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, pwName ):
      BgpMacVrfVpwsPwMode.__init__( self, pwName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.parent = parent
      configColl = parent.config().pseudowireConfig
      configColl.newMember( pwName )

   def config( self ):
      configColl = self.parent.config().pseudowireConfig
      if self.pwName in configColl:
         return configColl[ self.pwName ]
      else:
         orphanParent = Tac.Type( 'Routing::Bgp::MacVrfConfig' )(
               'orphanMacVrfConfig', False, MacVrfTypeEnum.macVrfTypeVpws,
               MacVrfSubTypeEnum.macVrfSubTypeNone )
         return orphanParent.pseudowireConfig.newMember( self.pwName )

   def setVpwsId( self, args ):
      local = args[ 'LOCAL' ]
      remote = args[ 'REMOTE' ]
      if local != remote:
         self.config().vpwsIdLocal = local
         self.config().vpwsIdRemote = remote
      else:
         self.addError(
               "Local and remote VPWS service instance identifiers must be "
               "different" )

   def noOrDefaultVpwsId( self, args ):
      self.config().vpwsIdLocal = EvpnEtid.max
      self.config().vpwsIdRemote = EvpnEtid.max

   cwDirMap = { 'receive': 'rxOverride', 'transmit': 'txOverride' }
   cwStatusMap = { 'always': 'cwAlways', 'disabled': 'cwDisabled' }

   def setDataPlaneCw( self, direction, status ):
      override = Tac.nonConst( self.config().controlWordOverride )
      setattr( override, self.cwDirMap[ direction ], self.cwStatusMap[ status ] )
      self.config().controlWordOverride = override

   def noOrDefaultDataPlaneCw( self, direction=None ):
      direction = [ direction ] if direction else self.cwDirMap.keys()
      override = Tac.nonConst( self.config().controlWordOverride )
      for clearDirection in direction:
         setattr( override, self.cwDirMap[ clearDirection ], 'noOverride' )
      self.config().controlWordOverride = override

class BgpMacVrfVpwsPwCommand( CliCommand.CliCommandClass ):
   syntax = 'pseudowire NAME'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowire' : 'Configure VPWS pseudowire for this EVPN instance',
      'NAME' : PatternMatcher(
         pattern='.+',
         helpdesc='Unique name to identify the pseudowire',
         helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      pwName = args[ 'NAME' ]
      childMode = mode.childMode( BgpMacVrfVpwsPwConfigMode, pwName=pwName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # NB: "mode" here is the parent mode, not the PW child mode
      pwName = args[ 'NAME' ]
      del mode.config().pseudowireConfig[ pwName ]

BgpMacVrfVpwsMode.addCommandClass( BgpMacVrfVpwsPwCommand )

#------------------------------------------------------------------------------
# (config-vpws-pw-<name>)# evpn vpws id local <LOCAL> remote <REMOTE>
#------------------------------------------------------------------------------
vpwsIdMatcher = IntegerMatcher( 1, EvpnEtid.max - 1,
                                helpdesc='VPWS service instance identifier' )

class VpwsPwEvpnVpwsIdCommand( CliCommand.CliCommandClass ):
   syntax = 'evpn vpws id local LOCAL remote REMOTE'
   noOrDefaultSyntax = 'evpn vpws id ...'
   data = {
      'evpn' : 'EVPN pseudowire parameters',
      'vpws' : 'EVPN VPWS pseudowire parameters',
      'id' : 'EVPN VPWS service instance identifiers',
      'local' : 'VPWS service instance identifier advertised to peer',
      'LOCAL' : vpwsIdMatcher,
      'remote' : 'VPWS service instance identifier advertised by peer',
      'REMOTE' : vpwsIdMatcher,
   }
   handler = BgpMacVrfVpwsPwConfigMode.setVpwsId
   noOrDefaultHandler = BgpMacVrfVpwsPwConfigMode.noOrDefaultVpwsId

BgpMacVrfVpwsPwConfigMode.addCommandClass( VpwsPwEvpnVpwsIdCommand )

#------------------------------------------------------------------------------
# (config-vpws-pw-<name>)#
# mpls control-word data-plane (receive|transmit) (always|disabled)
#------------------------------------------------------------------------------
directionMatcher = EnumMatcher( {
   'receive': 'Receive (decapsulation) direction',
   'transmit': 'Transmit (encapsulation) direction',
} )
statusMatcher = EnumMatcher( {
   'always': 'Control word is always used in this direction',
   'disabled': 'Control word is never used in this direction',
} )

class VpwsPwControlWordDataPlaneCommand( CliCommand.CliCommandClass ):
   syntax = 'mpls control-word data-plane DIRECTION STATUS'
   noOrDefaultSyntax = 'mpls control-word data-plane [ DIRECTION ] ...'
   data = {
      'mpls': MPLS,
      'control-word': 'Override control word behavior',
      'data-plane': 'Data-plane configuration',
      'DIRECTION': directionMatcher,
      'STATUS': statusMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      direction = args[ 'DIRECTION' ]
      status = args[ 'STATUS' ]
      mode.setDataPlaneCw( direction, status )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      direction = args.get( 'DIRECTION' )
      mode.noOrDefaultDataPlaneCw( direction=direction )

BgpMacVrfVpwsPwConfigMode.addCommandClass( VpwsPwControlWordDataPlaneCommand )

#------------------------------------------------------------------------------
# CliPlugin init hook
#------------------------------------------------------------------------------
def Plugin( em ):
   global entityManager
   global bgpMacVrfConfigDir
   global bgpMacVrfRdMap
   global bridgingHwCapabilities
   entityManager = em
   bgpMacVrfConfigDir = ConfigMount.mount( 
      entityManager, 'routing/bgp/macvrf', 
      'Routing::Bgp::MacVrfConfigDir', 'w' )
   bgpMacVrfRdMap = LazyMount.mount(
      entityManager, 'routing/bgp/macvrfrdmap',
      'Routing::Bgp::RdToVrfListDir', 'r' )
   bridgingHwCapabilities = LazyMount.mount(
      entityManager, 'bridging/hwcapabilities',
      'Bridging::HwCapabilities', 'r' )

   deleteRouterBgpMacVrfHook.addExtension( deleteBgpMacVrfConfigHook )
