#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from ArnetModel import (
   Ip4Address,
   Ip6Address,
   IpGenericPrefix,
)
from CliModel import (
   Bool,
   DeferredModel,
   Enum,
   GeneratorDict,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from CliPlugin.BgpCliModels import (
   BgpRouteASPathEntry,
   BgpRouteDetailEntry,
   BgpRouteTypeEntry,
   REASON_NOT_BEST_LIST,
)
from CliPlugin.VrfCli import generateVrfCliModel

BGP_LS_NLRI_TYPES = (
   'node',
   'link',
   'ipv4Prefix',
   'ipv6Prefix',
)

BGP_LS_PROTOCOL_IDS = (
   'isisLevel1',
   'isisLevel2',
)

BGP_LS_IGP_ROUTER_ID_TYPES = (
   'isisSystemId',
   'isisPseudoNodeId',
)

class BgpLsNodeDescriptor( Model ):
   igpRouterIdType = Enum( values=BGP_LS_IGP_ROUTER_ID_TYPES,
                           help='Type of IGP router identifier' )
   # optional because in the future for Ospf this field will be absent. For now,
   # it'll always be present
   isisSystemId = Str( help='IS-IS system Identifier', optional=True )
   isisPnodeId = Str( help='IS-IS pseudonode Identifier', optional=True )
   # When we add Ospf support we can add ospfRouterId and ospfDrRouterId, ospfDrIntf
   asn = Int( help='Autonomous System Number of BGP-LS producer', optional=True )

class BgpLsLinkDescriptor( Model ):
   localIpv4Addrs = List( valueType=Ip4Address,
                          help='List of local IPv4 interface addresses' )
   localIpv6Addrs = List( valueType=Ip6Address,
                          help='List of local IPv6 interface addresses' )
   remoteIpv4Addrs = List( valueType=Ip4Address,
                          help='List of remote IPv4 interface addresses' )
   remoteIpv6Addrs = List( valueType=Ip6Address,
                          help='List of remote IPv6 interface addresses' )

class BgpLsPrefixDescriptor( Model ):
   prefix = IpGenericPrefix( help='Prefix originated in IGP' )

class BgpLsRoutePath( DeferredModel ):
   asPathEntry = Submodel( valueType=BgpRouteASPathEntry,
                           help='AS path information' )
   # no MED, localPref, weight as we don't populate them for producer paths
   # We can add them as optional attrs later when we do RR support
   # No peerEntry in Producer only role
   routeType = Submodel( valueType=BgpRouteTypeEntry, help='Route type' )
   routeDetail = Submodel( valueType=BgpRouteDetailEntry, optional=True,
                           help='Route details' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

class BgpLsRouteEntry( DeferredModel ):
   # Report the total number of paths as a separate attribute because the path list
   # may be filtered by peer.
   nlriType = Enum( values=BGP_LS_NLRI_TYPES,
                    help='NLRI type' )
   protocolId = Enum( values=BGP_LS_PROTOCOL_IDS,
                      help='IGP source protocol' )
   identifier = Int( help='IGP Instance Identifier' )
   localNodeDescriptor = Submodel( valueType=BgpLsNodeDescriptor,
                                   help='Identity of local node' )
   remoteNodeDescriptor = Submodel( valueType=BgpLsNodeDescriptor, optional=True,
                                    help='Identity of remote node for link NLRIs' )
   linkDescriptor = Submodel( valueType=BgpLsLinkDescriptor, optional=True,
                              help='Identity of link for link NLRIs' )
   # Using a sub model as in the future with MT support it'll have the MT ID
   # Also a structure containing descriptors is the natural representation of a
   # BGP-LS NLRI
   prefixDescriptor = Submodel( valueType=BgpLsPrefixDescriptor, optional=True,
                           help='Identity of prefix for IPv4 and IPv6 prefix NLRIs' )
   # Report the total number of paths as a separate attribute because the path list
   # may be filtered by peer.
   totalPaths = Int( help='Total number of paths for this BGP-LS route' )
   paths = List( valueType=BgpLsRoutePath,
                 help='BGP-LS paths' )
   _routeKey = Str( help='String representation of the BGP-LS NLRI' )

class BgpLsRoutes( DeferredModel ):
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   bgpLsRoutes = GeneratorDict( keyType=str, valueType=BgpLsRouteEntry,
                                help='Dictionary of BGP Link State route entries '
                                'keyed by a string representation of the NLRI' )

BgpLsRoutesVrfModel = generateVrfCliModel( BgpLsRoutes,
                                           'BGP Link State routes' )
