# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliModel
from CliPlugin.TechSupportCli import ( techSupportKwMatcher,
                                      extendedKwMatcher,
                                      ShowTechModel,
                                      showTechSupportExec )
from CliPlugin.ArBgpCli import (
      doShowIpBgpSummaryAcrImpl,
      ArBgpCliCommand,
      )
from CliPlugin.BgpCliModels import BgpSummary
from CliPlugin.BgpLsCliModels import BgpLsRoutesVrfModel
from CliPlugin.BgpCliHelperCli import convertPeerAddr
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput
from CliToken.RoutingBgpShowCliTokens import (
         bgpAfterShow,
         detail,
         summary,
         neighbors,
         advertised,
      )
from CliToken.BgpLs import (
      BgpLsNlriType,
      tokenLinkState,
      )
import BasicCli
from CliMatcher import KeywordMatcher
from ShowCommand import ShowCliCommandClass
from IpLibConsts import DEFAULT_VRF
import AgentCommandRequest

bgpKwMatch = KeywordMatcher( 'bgp',
      helpdesc='Show tech-support extended bgp command' )
#----------------------------------------------------------
# 'show bgp link-state summary'
#----------------------------------------------------------
class ShowBgpLsSummary( ShowCliCommandClass ):
   syntax = "show bgp link-state summary"
   data = {
         "bgp": bgpAfterShow,
         "link-state": tokenLinkState,
         "summary": summary,
         }
   cliModel = BgpSummary

   @staticmethod
   @ArBgpShowOutput( 'ShowBgpLsSummary', arBgpModeOnly=True )
   def handler( mode, args ):
      return doShowIpBgpSummaryAcrImpl( mode, vrfName=DEFAULT_VRF,
            nlriAfiSafi='linkState' )

BasicCli.addShowCommandClass( ShowBgpLsSummary )

class BgpLsCliHelperCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      ArBgpCliCommand.__init__( self, mode, command, vrfName=DEFAULT_VRF,
            nlriAfiSafi='linkState', disableFork=True )
      self.mode = mode
      if self.mode.session_.outputFormat_ == 'json':
         self.addParam( 'json' )
      if 'routeType' not in kwargs:
         kwargs[ 'routeType' ] = 'routes'
      for k, v in kwargs.iteritems():
         if v:
            self.addParam( k, v )

   def run( self, **kwargs ):
      AgentCommandRequest.runCliPrintSocketCommand(
            self.entityManager, 'BgpCliHelper', self.command, self.paramString(),
            self.mode, keepalive=True )
      return CliModel.cliPrinted( BgpLsRoutesVrfModel )

def doShowBgpLs( mode, **kwargs ):
   return BgpLsCliHelperCommand( mode, 'show bgp link-state', **kwargs ).run()

#---------------------------------------------------------
# 'show bgp link-state [ node | link | ipv4-prefix | ipv6-prefix ] [ detail ]'
#---------------------------------------------------------
class ShowBgpLsNlri( ShowCliCommandClass ):
   syntax = "show bgp link-state [ NLRI_TYPE ] [ detail ]"
   data = {
         "bgp": bgpAfterShow,
         "link-state": tokenLinkState,
         "NLRI_TYPE": BgpLsNlriType,
         "detail": detail,
         }
   cliModel = BgpLsRoutesVrfModel

   @staticmethod
   @ArBgpShowOutput( 'ShowBgpLsNlri', arBgpModeOnly=True )
   def handler( mode, args ):
      return doShowBgpLs( mode,
                          nlriType=args.get( 'NLRI_TYPE' ),
                          detail=args.get( "detail" ) )

#----------------------------------------------------------------
# 'show bgp neighbors <neigh_addr> link-state advertised-routes [ nlriType ]'
#----------------------------------------------------------------
class ShowBgpLsAdvRoutes( ShowCliCommandClass ):
   syntax = "show bgp neighbors PEER_ADDR link-state advertised-routes [ NLRI_TYPE ]"
   data = {
         "bgp": bgpAfterShow,
         "neighbors": neighbors,
         "PEER_ADDR": V4V6PeerKeyCliExpression,
         "link-state": tokenLinkState,
         "advertised-routes": advertised,
         "NLRI_TYPE": BgpLsNlriType,
      }
   cliModel = BgpLsRoutesVrfModel

   @staticmethod
   @ArBgpShowOutput( 'ShowBgpLsAdvRoutes', arBgpModeOnly=True )
   def handler( mode, args ):
      kwargs = {}
      kwargs[ 'routeType' ] = 'advertised-routes'
      kwargs[ 'peerAddr' ] = args.get( "PEER" )
      kwargs[ 'nlriType' ] = args.get( 'NLRI_TYPE' )
      convertPeerAddr( kwargs )
      return doShowBgpLs( mode, **kwargs )

# List of show commands to run with 'show tech extended bgp link-state'
def _showTechLsCmds():
   return [
            'show isis neighbors detail',
            'show isis database traffic-engineering',
            'show bgp link-state summary',
            'show bgp link-state detail',
          ]

#-----------------------------------------------------------------------------------
# show tech-support extended bgp link-state
#-----------------------------------------------------------------------------------
# Creating a seperate show command class for this since there are 2 keyword
# tokens following extended keyword and the existing 'ShowTechSupport'
# cannot accommodate that syntax
class ShowTechExtBgpLsSupport( ShowCliCommandClass ):
   syntax = ( 'show tech-support extended bgp link-state' )
   data = {
            'tech-support': techSupportKwMatcher,
            'extended': extendedKwMatcher,
            'bgp': bgpKwMatch,
            'link-state': tokenLinkState,
          }
   cliModel = ShowTechModel
   privileged = True

   @staticmethod
   def handler( mode, args ):
      cmd = 'show tech-support extended bgp link-state'
      # inCmdCallbacks is expected to be a list of tuples containing ( timestamp,
      # commandList function ). The timestamp is used to order between the
      # multiple show command list. In our case, there will just be one entry. So
      # pass 0 for the timestamp.
      return showTechSupportExec( mode, inputCmd=cmd,
                                  inCmdCallbacks=[ ( 0, _showTechLsCmds ) ] )

BasicCli.addShowCommandClass( ShowBgpLsNlri )
BasicCli.addShowCommandClass( ShowBgpLsAdvRoutes )
BasicCli.addShowCommandClass( ShowTechExtBgpLsSupport )
