# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from itertools import chain
import BasicCli
import CliParser
import Tac
import BasicCliModes
import ConfigMount
import LazyMount
import CliCommand
from CliMode.BgpGroup import BgpGroupMode, BgpBuiltinGroupMode
from CliPlugin.GroupLib import Group, GroupConfigMode
from CliPlugin.MaintenanceMode import MaintenanceUnitConfigMode
from CliPlugin.RoutingBgpCli import PeerCliExpression
from CliPlugin import IraIpCli
from CliPlugin.MaintenanceGroupCli import MaintenanceGroup
from CliPlugin.MaintenanceCliLib import (
   bgpGroupType,
   bgpNeighVrfBuiltinGroupPrefix,
   builtinGroupType,
   groupNameMatcher,
)
from CliPlugin.MaintenanceModels import bgpMaintenanceGroupCleanupHook
from CliPlugin.VrfCli import vrfMatcher
from BgpLib import createKey
from IpLibConsts import VRFNAMES_RESERVED

globalBgpConfig = None
globalBgpGroupConfigDir = None
globalBgpGroupStatusDir = None
allVrfConfig = None

class BgpBuiltinGroupConfigMode( GroupConfigMode, BgpBuiltinGroupMode,
                                 BasicCli.ConfigModeBase ):
   name = "Bgp Builtin Group configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, bgpBuiltinGroup ):
      GroupConfigMode.__init__( self, session, bgpBuiltinGroup )
      BgpBuiltinGroupMode.__init__( self, bgpBuiltinGroup.name() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class BgpGroupConfigMode( GroupConfigMode, BgpGroupMode,
                          BasicCli.ConfigModeBase ):
   name = "Bgp Group configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, bgpGroup ):
      GroupConfigMode.__init__( self, session, bgpGroup )
      BgpGroupMode.__init__( self, bgpGroup.name() )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#
# BgpBuiltinGroup class: Holds cli state for each builtin peer group
#
class BgpBuiltinGroup( Group ):
   def __init__( self, groupName ):
      self.name_ = groupName
      groupKey = Tac.Value( 'Group::GroupKey', bgpGroupType, groupName )
      Group.__init__( self, groupKey )
      globalBgpGroupConfigDir.newConfig( self.name_ )
      group = globalBgpGroupConfigDir.config.get( self.name_ )
      group.origin = builtinGroupType
      Group.addGroup( self )

   def name( self ):
      return self.name_

   def remGroup( self ):
      maintenanceGroup = MaintenanceGroup( self.key_ )
      maintenanceGroup.remGroup()
      if self.name_ in globalBgpGroupConfigDir.config:
         del globalBgpGroupConfigDir.config[ self.name_ ]
         Group.remGroup( self )
#
# BgpGroup class: Holds cli state for each configured peer group
#
class BgpGroup( Group ):
   def __init__( self, groupName ):
      self.name_ = groupName
      groupKey = Tac.Value( 'Group::GroupKey', bgpGroupType, groupName )
      Group.__init__( self, groupKey )
   
   def addGroup( self ):
      globalBgpGroupConfigDir.newConfig( self.name_ )
      Group.addGroup( self )

   def remGroup( self ):
      maintenanceGroup = MaintenanceGroup( self.key_ )
      maintenanceGroup.remGroup()
      if self.name_ in globalBgpGroupConfigDir.config:
         del globalBgpGroupConfigDir.config[ self.name_ ]
         Group.remGroup( self )

   def name( self ):
      return self.name_

   def neighborKey( self, peerOrAddr ):
      return createKey( peerOrAddr )

   def addPeer( self, peerOrAddr ):
      key = self.neighborKey( peerOrAddr )
      config = globalBgpGroupConfigDir.config.get( self.name_ )
      if not config:
         return
      config.neighbor[ key ] = True

   def remPeer( self, peerOrAddr ):
      key = self.neighborKey( peerOrAddr )
      config = globalBgpGroupConfigDir.config.get( self.name_ )
      if not config:
         return
      del config.neighbor[ key ]

   def addProfile( self, profileKey ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      groupConfig.profile[ profileKey ] = True

   def remProfile( self, profileType ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      for profile in groupConfig.profile:
         if profile.type == profileType:
            profileKey = Tac.Value( "Group::ProfileKey", type = profileType,
                                    name = profile.name )
            del groupConfig.profile[ profileKey ]

   def addVrf( self, vrfName ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if groupConfig:
         groupConfig.vrfName = vrfName

   def vrfName( self ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if groupConfig:
         return groupConfig.vrfName
      else:
         return None

bgpGroupNameMatcher = groupNameMatcher( lambda mode:
                                        globalBgpGroupConfigDir.config )

bgpGroupNameWithBuiltinMatcher = groupNameMatcher(
   lambda mode: chain( globalBgpGroupConfigDir.config,
                       ( bgpNeighVrfBuiltinGroupPrefix + vrf for vrf in
                         IraIpCli.getAllVrfNames( mode ) ) ) )

#--------------------------------------------------------------------------------
# [ no | default ] group bgp BGP_GROUP_NAME in global config mode
#--------------------------------------------------------------------------------
class GroupBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'group bgp BGP_GROUP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'group': 'Configure Group',
      'bgp': 'Configure BGP Group',
      'BGP_GROUP_NAME': bgpGroupNameWithBuiltinMatcher,
   }

   @staticmethod
   def delBgpGroupConfig_( mode, groupName ):
      if groupName.startswith( bgpNeighVrfBuiltinGroupPrefix ):
         bgpBuiltinGroup = BgpBuiltinGroup( groupName )
         bgpBuiltinGroup.remGroup()
         return
      bgpGroup = BgpGroup( groupName )
      bgpGroup.remGroup()

   @staticmethod
   def handler( mode, args ):
      groupName = args[ 'BGP_GROUP_NAME' ]
      if groupName.startswith( bgpNeighVrfBuiltinGroupPrefix ):
         bgpBuiltinGroup = BgpBuiltinGroup( groupName )
         bgpBuiltinGroup.addGroup()
         childMode = mode.childMode( BgpBuiltinGroupConfigMode,
                                     bgpBuiltinGroup=bgpBuiltinGroup )
         mode.session_.gotoChildMode( childMode )
         return
      bgpGroup = BgpGroup( groupName )
      bgpGroup.addGroup()
      childMode = mode.childMode( BgpGroupConfigMode, bgpGroup=bgpGroup )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      GroupBgpCmd.delBgpGroupConfig_( mode, args[ 'BGP_GROUP_NAME' ] )

BasicCliModes.GlobalConfigMode.addCommandClass( GroupBgpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor PEER
#--------------------------------------------------------------------------------
class BgpGroupNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor PEER'
   noOrDefaultSyntax = syntax
   data = {
      'neighbor': 'Configure member',
      'PEER': PeerCliExpression,
   }

   @staticmethod
   def handler( mode, args ):
      mode.group().addPeer( args[ 'PEER' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.group().remPeer( args[ 'PEER' ] )

BgpGroupConfigMode.addCommandClass( BgpGroupNeighborCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vrf [ VRFNAME ]
#--------------------------------------------------------------------------------
class BgpGroupVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = 'vrf [ VRF_NAME ]'
   data = {
      'vrf': 'VRF name',
      'VRF_NAME': vrfMatcher,
   }

   @staticmethod
   def _changeVrf( mode, vrfName ):
      groupConfig = globalBgpGroupConfigDir.config.get( mode.group().name() )
      if not groupConfig:
         return
      if len( vrfName ) > Tac.Type( "L3::VrfName" ).maxLength:
         mode.addError( "'%s' too long: must be no more than %d characters"
                        % ( vrfName, Tac.Type( "L3::VrfName" ).maxLength ) )
         return
      IraIpCli.warnIfRoutingDisabled( mode, vrfName )
      if groupConfig.vrfName == vrfName:
         return
      mode.group().addVrf( vrfName )

   @staticmethod
   def handler( mode, args ):
      vrfName = args[ 'VRF_NAME' ]
      if vrfName in VRFNAMES_RESERVED:
         mode.addError( "vrf name %s is reserved." % vrfName )
         return
      BgpGroupVrfCmd._changeVrf( mode, vrfName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrfName = args.get( 'VRF_NAME' )
      if mode.group().vrfName() == vrfName or not vrfName:
         BgpGroupVrfCmd._changeVrf( mode, 'default' )

BgpGroupConfigMode.addCommandClass( BgpGroupVrfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] group bgp GROUPNAME in maintenance-unit mode
#--------------------------------------------------------------------------------
class MaintenanceUnitGroupBgpCmd( CliCommand.CliCommandClass ):
   syntax = 'group bgp BGP_GROUP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'group': 'Configure Group',
      'bgp': 'Configure BGP Group',
      'BGP_GROUP_NAME': bgpGroupNameWithBuiltinMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      groupName = args[ 'BGP_GROUP_NAME' ]
      groupKey = Tac.Value( "Group::GroupKey", type=bgpGroupType,
                            name=groupName )
      mode.maintenanceUnit_.addGroup( groupKey )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groupName = args[ 'BGP_GROUP_NAME' ]
      groupKey = Tac.Value( "Group::GroupKey", type=bgpGroupType,
                            name=groupName )
      mode.maintenanceUnit_.remGroup( groupKey )

MaintenanceUnitConfigMode.addCommandClass( MaintenanceUnitGroupBgpCmd )

def Plugin( entityManager ):
   global globalBgpGroupConfigDir, globalBgpConfig, allVrfConfig
   global globalBgpGroupStatusDir
   bgpMaintenanceGroupCleanupHook.addExtension( GroupBgpCmd.delBgpGroupConfig_ )
   globalBgpGroupConfigDir = ConfigMount.mount( entityManager,
      'group/config/bgp', 'BgpGroup::ConfigDir', 'w' )
   globalBgpConfig = LazyMount.mount( entityManager, 'routing/bgp/config',
      'Routing::Bgp::Config', 'r' )
   globalBgpGroupStatusDir = LazyMount.mount( entityManager,
      'group/status/bgp', 'BgpGroup::StatusDir', 'r' )
   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
