#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
from BasicCli import addShowCommandClass
from CliModel import cliPrinted
from CliPlugin.RoutingBgpShowCli import ( ArBgpShowOutput,
                                          allVrfExprFactory,
)
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
from CliPlugin.RoutingBgpShowCli import summaryVrfModel
from CliPlugin.BgpFlowspecCliModels import bgpFlowspecRulesVrfModel
from CliPlugin.ArBgpCli import doShowIpBgpSummaryAcrImpl, ArBgpAsyncCliCommand
from CliPlugin.FlowspecShowCli import flowSpecMatcherForShow
from CliPlugin.FlowspecMatcher import ruleStringMatcher
from CliPlugin.BgpCliHelperCli import convertPeerAddr
from CliToken.RoutingBgpShowCliTokens import (
   bgpAfterShow,
   detail,
   neighbors,
   summary,
   routeTypeMatcher,
)
from CliToken.Flowspec import (
   ruleMatcher,
   identifierMatcher,
   ruleIdMatcher,
   destinationMatcher,
   sourceMatcher,
   ipv4AddrMatcherForShow,
   ipv6AddrMatcherForShow
)
from CliToken.Ipv4 import ipv4MatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow

from ShowCommand import ShowCliCommandClass

#-------------------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------------------

@ArBgpShowOutput( 'doShowBgpFlowspec', arBgpModeOnly=True, vrfLookup=True )
def doShowBgpFlowspec( mode, **kwargs ):
   convertPeerAddr( kwargs )
   kwargs = dict( ( k, v ) for k, v in kwargs.iteritems() if v != None )
   ArBgpAsyncCliCommand( mode, 'show flowspec', **kwargs ).run()
   return cliPrinted( bgpFlowspecRulesVrfModel )

# The output of "show bgp flow-spec ipv4|ipv6 summary" and "show ip bgp summary"
# are the same, so we can use the existing doShowIpBgpSummaryAcrImpl.
@ArBgpShowOutput( 'doShowBgpFlowspecSummary', arBgpModeOnly=True,
                  vrfLookup=True )
def doShowBgpFlowspecSummary( mode, nlriAfiSafi, vrfName=None ):
   return doShowIpBgpSummaryAcrImpl( mode, nlriAfiSafi=nlriAfiSafi,
                                     vrfName=vrfName )

#-------------------------------------------------------------------------------
# "show bgp flow-spec ..."
#-------------------------------------------------------------------------------
# Define common syntax for the ipv4 and ipv6 versions of these commands.
# We can have a common handler, but need to register separate command classes
# because the address-family of the command affects which matchers should be used
# for matching prefix args.
# Define syntax and data in this base class, the derived classes need to update the
# syntax and data to add address-family specific matchers.
#
class ShowBgpFlowspecBaseClass( object ):
   syntax = (
      'show bgp flow-spec %s '
      '[ ( rule ( STRING | ( identifier RULEID ) ) ) ] '
      '[ destination DSTPREFIX ] [ source SRCPREFIX ] '
      '[ detail ] [ VRF ]'
   )
   data = {
      'bgp' : bgpAfterShow,
      'flow-spec' : flowSpecMatcherForShow,
      # af : needs to be updated in each specific show command,
      'rule' : ruleMatcher,
      'STRING' : ruleStringMatcher,
      'identifier' : identifierMatcher,
      'RULEID' : ruleIdMatcher,
      'source' : sourceMatcher,
      'destination' : destinationMatcher,
      # 'SRCPREFIX' : needs to be updated in each specific show command,
      # 'DSTPREFIX' : needs to be updated in each specific show command,
      'detail' : detail,
      'VRF' : allVrfExprFactory,
   }

   neighborSyntax = (
      'show bgp neighbors PEER_ADDR flow-spec %s '
      '( ROUTE_TYPE ) '
      '[ ( rule ( STRING | ( identifier RULEID ) ) ) ] '
      '[ destination DSTPREFIX ] [ source SRCPREFIX ] '
      '[ detail ] [ VRF ]'
   )

   neighborData = data.copy()
   neighborData.update( {
      'neighbors' : neighbors,
      'PEER_ADDR' : V4V6PeerKeyCliExpression,
      'ROUTE_TYPE' : routeTypeMatcher,
   } )

   @staticmethod
   def handler( mode, args ):
      nlriType = 'ipv4Flowspec' if 'ipv4' in args else 'ipv6Flowspec'
      return doShowBgpFlowspec( mode,
                                nlriType=nlriType,
                                peerAddr=args.get( 'PEER' ),
                                routeType=args.get( 'ROUTE_TYPE' ),
                                rule=args.get( 'STRING' ),
                                ruleId=args.get( 'RULEID' ),
                                destPrefix=args.get( 'DSTPREFIX' ),
                                srcPrefix=args.get( 'SRCPREFIX' ),
                                detail=args.get( 'detail' ),
                                vrfName=args.get( 'VRF' ) )

#-------------------------------------------------------------------------------
# "show bgp flow-spec ipv4 [ rule ( string | identifier id ) ]
#                          [ destination <ipv4-prefix> ]
#                          [ source <ipv4-prefix> ]
#                          [ detail ] [vrf <vrfName/all]"
#-------------------------------------------------------------------------------
# Show BGP flow-spec rules, optionally filtering by rule, rule id,
# source or destination prefix.
class ShowBgpFlowspecV4Cmd( ShowCliCommandClass,
                            ShowBgpFlowspecBaseClass ):
   syntax = ShowBgpFlowspecBaseClass.syntax % 'ipv4'
   data = ShowBgpFlowspecBaseClass.data.copy()
   data.update( {
      'ipv4' : ipv4MatcherForShow,
      'SRCPREFIX' : ipv4AddrMatcherForShow,
      'DSTPREFIX' : ipv4AddrMatcherForShow,
   } )

   cliModel = bgpFlowspecRulesVrfModel

BasicCli.addShowCommandClass( ShowBgpFlowspecV4Cmd )

#-------------------------------------------------------------------------------
# "show bgp flow-spec ipv6 [ rule ( string | identifier id ) ]
#                          [ destination <ipv6-prefix> ]
#                          [ source <ipv6-prefix> ]
#                          [ detail ] [vrf <vrfName/all]"
#-------------------------------------------------------------------------------
class ShowBgpFlowspecV6Cmd( ShowCliCommandClass,
                            ShowBgpFlowspecBaseClass ):
   syntax = ShowBgpFlowspecBaseClass.syntax % 'ipv6'
   data = ShowBgpFlowspecBaseClass.data.copy()
   data.update( {
      'ipv6' : ipv6MatcherForShow,
      'SRCPREFIX' : ipv6AddrMatcherForShow,
      'DSTPREFIX' : ipv6AddrMatcherForShow,
   } )

   cliModel = bgpFlowspecRulesVrfModel

BasicCli.addShowCommandClass( ShowBgpFlowspecV6Cmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <peerAddr> flow-spec ipv4
#                          ( routes | received-routes | advertised-routes )
#                          [ rule ] [ rule identifier ]
#                          [ destination <ipv4-prefix> ]
#                          [ source <ipv4-prefix> ]
#                          [ detail ] [vrf <vrfName/all]"
#-------------------------------------------------------------------------------
# Show BGP flow-spec ipv4 rules received or advertised to a particular peer,
# optionally filtering by rule, rule id, source or destination prefix.
class ShowBgpNeighborsFlowspecV4Cmd( ShowCliCommandClass,
                                     ShowBgpFlowspecBaseClass ):
   syntax = ShowBgpFlowspecBaseClass.neighborSyntax % 'ipv4'
   data = ShowBgpFlowspecBaseClass.neighborData.copy()
   data.update( {
      'ipv4' : ipv4MatcherForShow,
      'SRCPREFIX' : ipv4AddrMatcherForShow,
      'DSTPREFIX' : ipv4AddrMatcherForShow,
   } )

   cliModel = bgpFlowspecRulesVrfModel

BasicCli.addShowCommandClass( ShowBgpNeighborsFlowspecV4Cmd )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> flow-spec ipv6
#                          ( routes | received-routes | advertised-routes )
#                          [ rule ] [ rule identifier ]
#                          [ destination <ipv6-prefix> ]
#                          [ source <ipv6-prefix> ]
#                          [ detail ] [vrf <vrfName/all]"
#-------------------------------------------------------------------------------
# Show BGP flow-spec ipv6 rules received or advertised to a particular peer,
# optionally filtering by rule, rule id, source or destination prefix.
class ShowBgpNeighborsFlowspecV6Cmd( ShowCliCommandClass,
                                     ShowBgpFlowspecBaseClass ):
   syntax = ShowBgpFlowspecBaseClass.neighborSyntax % 'ipv6'
   data = ShowBgpFlowspecBaseClass.neighborData.copy()
   data.update( {
      'ipv6' : ipv6MatcherForShow,
      'SRCPREFIX' : ipv6AddrMatcherForShow,
      'DSTPREFIX' : ipv6AddrMatcherForShow,
   } )

   cliModel = bgpFlowspecRulesVrfModel

BasicCli.addShowCommandClass( ShowBgpNeighborsFlowspecV6Cmd )

#------------------------------------------------------
# "show bgp flow-spec ipv4 summary [vrf <vrfName/all]"
#------------------------------------------------------
class ShowBgpFlowspecSummaryCmd( ShowCliCommandClass ):
   syntax = 'show bgp flow-spec ( ipv4 | ipv6 ) summary [ VRF ]'
   data = {
      'bgp' : bgpAfterShow,
      'flow-spec' : flowSpecMatcherForShow,
      'ipv4' : ipv4MatcherForShow,
      'ipv6' : ipv6MatcherForShow,
      'summary' : summary,
      'VRF' : allVrfExprFactory,
   }
   cliModel = summaryVrfModel

   @staticmethod
   def handler( mode, args ):
      nlriAfiSafi = 'ipv4Flowspec' if 'ipv4' in args else 'ipv6Flowspec'
      vrfName = args.get( 'VRF' )
      return doShowBgpFlowspecSummary( mode, nlriAfiSafi, vrfName=vrfName )

addShowCommandClass( ShowBgpFlowspecSummaryCmd )
