#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand
import CliMatcher
from CliPlugin.RoutingBgpCli import (
   RouterBgpSharedModelet,
   RouterBgpAfBaseModelet,
   afModeExtensionHook,
   configForVrf,
)
from CliPlugin.RoutingBgpInstanceCli import (
   MissingPolicyAfSharedCmd,
   resetBgpAfModeConfig,
)
from CliPlugin.RoutingBgpNeighborCli import RouterBgpAfActivateCommand
from CliPlugin.ArBgpCliCommon import getBgpExtCommParserContainerSm
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliMode.BgpCommon import RoutingBgpBaseAfMode, RoutingBgpVrfAfMode
import CliToken.RoutingBgp as bgpTokens
import BasicCli
import CliParser
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType
from BgpLib import updateConfigAttrsAfMap
import Tac

# pkgdeps: rpm BgpFlowspec-lib

updateConfigAttrsAfMap( { 'flow-spec ipv4' : [ 'V4Flowspec', 'AfV4Flowspec' ],
                          'flow-spec ipv6' : [ 'V6Flowspec', 'AfV6Flowspec' ] },
                          True )

class RouterBgpBaseAfFlowspecMode( RoutingBgpBaseAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP address family Flow Specification configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily ):
      self.vrfName = DEFAULT_VRF
      RoutingBgpBaseAfMode.__init__( self, addrFamily,
                                     modeKey='router-bgp-af-flow-spec' )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpVrfAfFlowspecMode( RoutingBgpVrfAfMode, BasicCli.ConfigModeBase ):
   name = 'BGP VRF address family Flow Specification configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, addrFamily, vrfName ):
      self.vrfName = vrfName
      RoutingBgpVrfAfMode.__init__( self, addrFamily, vrfName )
      # set cli prompt in vrf af config mode. TODO: get it cli-reviewed
      self.longModeKey = 'router-bgp-vrf-%s-af-flow-spec' % ( vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterBgpAfFlowspecModelet( RouterBgpAfBaseModelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      RouterBgpAfBaseModelet.__init__( self, mode )

RouterBgpBaseAfFlowspecMode.addModelet( RouterBgpAfFlowspecModelet )
RouterBgpVrfAfFlowspecMode.addModelet( RouterBgpAfFlowspecModelet )

#-------------------------------------------------------------------------------
# "[ no | default ] address-family flow-spec ( ipv4 | ipv6 )"
#-------------------------------------------------------------------------------
flowspecKwMatcher = CliMatcher.KeywordMatcher(
   'flow-spec',
   helpdesc='Flow Specification' )
ipv4FlowspecKwMatcher = CliMatcher.KeywordMatcher(
   'ipv4',
   helpdesc='IPv4 Flow Specification' )
ipv6FlowspecKwMatcher = CliMatcher.KeywordMatcher(
   'ipv6',
   helpdesc='IPv6 Flow Specification' )

class EnterBgpFlowspecAfMode( CliCommand.CliCommandClass ):
   syntax = 'address-family flow-spec ( ipv4 | ipv6 )'
   noOrDefaultSyntax = 'address-family flow-spec ( ipv4 | ipv6 ) ...'

   data = { 'address-family' : bgpTokens.addrFamily,
            'flow-spec' : flowspecKwMatcher,
            'ipv4' : ipv4FlowspecKwMatcher,
            'ipv6' : ipv6FlowspecKwMatcher }

   @staticmethod
   def adapter( mode, args, argsList ):
      if 'ipv4' in args:
         args[ 'af' ] = 'ipv4'
      elif 'ipv6' in args:
         args[ 'af' ] = 'ipv6'

   @staticmethod
   def handler( mode, args ):
      if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
         mode.addWarning( "Routing protocols model multi-agent must be "
                          "configured for the BGP Flow Specification "
                          "address-family" )
      addrFamily = 'flow-spec %s' % args[ 'af' ]
      if mode.vrfName == DEFAULT_VRF:
         assert addrFamily in afModeExtensionHook.afModeExtensions()
         childMode = mode.childMode(
            afModeExtensionHook.afModeExtension[ addrFamily ],
            addrFamily=addrFamily )
      else:
         assert addrFamily in afModeExtensionHook.vrfAfModeExtensions()
         childMode = mode.childMode(
            afModeExtensionHook.vrfAfModeExtension[ addrFamily ],
            addrFamily=addrFamily, vrfName=mode.vrfName )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      addrFamily = 'flow-spec %s' % args[ 'af' ]
      bgpConfig = configForVrf( mode.vrfName )
      resetBgpAfModeConfig( bgpConfig, addrFamily, mode.vrfName )

RouterBgpSharedModelet.addCommandClass( EnterBgpFlowspecAfMode )

#-------------------------------------------------------------------------------
# "[no|default] neighbor <ipaddr|peerGroup> activate"
#-------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( RouterBgpAfActivateCommand )

#-------------------------------------------------------------------------------
# "[no|default] bgp missing-policy [include sub-route-map] direction [ in | out ]
# action" command, in BgpFlowspec mode.
#-------------------------------------------------------------------------------
RouterBgpAfFlowspecModelet.addCommandClass( MissingPolicyAfSharedCmd )

def Plugin( entityManager ):
   for af in [ 'flow-spec ipv4', 'flow-spec ipv6' ]:
      afModeExtensionHook.addAfModeExtension( af, RouterBgpBaseAfFlowspecMode )
      afModeExtensionHook.addVrfAfModeExtension( af, RouterBgpVrfAfFlowspecMode )
   # We need to register our BgpFlowspec specific parsing subrules for extended
   # communities, because they are reparsed in the Cli client to generate the
   # user facing string representations.
   # initialize the global ext comm parser state machinery, or get the existing one
   bgpExtCommParserContainerSm = getBgpExtCommParserContainerSm()
   # create Flowspec-specific ext comm parsing SM and add it to the container
   parseExtCommFlowspecSm = Tac.newInstance(
      "Routing::Bgp::FlowspecPlugin::FlowspecParseExtCommSm",
      bgpExtCommParserContainerSm.parseExtCommRuleCollections )
   bgpExtCommParserContainerSm.extCommParserSms.enq( parseExtCommFlowspecSm )
