# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=wrong-import-position
from Toggles import RoutingLibToggleLib
# pylint: enable=wrong-import-position

import Arnet
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from ArnetModel import IpGenericAddress
from BgpLib import PeerConfigKey
import Tac
from CliCommon import CliModelNotDegradable
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import GeneratorDict
from CliModel import GeneratorList
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from CliModel import DeferredModel
from CliModel import _TacAttributeType
from HumanReadable import formatTimeInterval
from TableOutput import createTable, Format
from TunnelModels import TunnelTableEntry, Via
from CliPlugin.TunnelCli import getNhAndIntfStrs, getViaModelFromViaDict
from CliPlugin.IpRibLibCliModels import (
      ResolutionRibProfileConfig,
      ResolutionRib
      )
from ArnetLib import asnStrToNum
import re
import ctypes
from SrTePolicyCommonLib import tacEnlp

#-------------------------------------------------------------------------------
# "show ip[v6] bgp summary"
#-------------------------------------------------------------------------------

PEER_STATE_MAP = { 'NoIf' : 'NoInterface', 'GrpAdmin' : 'GroupAdmin' }
PEER_STATE_MAP_INV = { v : k for k, v in PEER_STATE_MAP.items() }
IDLE_RE = re.compile( r'\w+\((\w+)\)' )
summaryFormatStr = '%s %s%-16s %-1s  %-10s %9s %9s %4s %4s %8s %-7s %-6s %s'
summaryFormatStrDes = '%s %-24s %-16s %-1s  %-10s %9s %9s %4s %4s %8s %-7s %-6s %s'
MAX_LEN_OF_DES = 24

stateHelp = '''Current peer state.
  Idle - Ignoring everything
  Connect - Trying to connect
  Active - Waiting for a connection
  OpenSent - Open packet has been sent
  OpenConfirm - Waiting for a keepalive or notify
  Established - Connection has been established'''

idleReasonHelp = '''Used when peer state is "Idle".
  NoInterface - The peer's interface is down
  Admin - The peer is disabled
  GroupAdmin - The peer's group is disabled
  MaxPath - The peer has exceeded its max route limit
  Deleted - The peer is deleted and waiting for deffered cleanup
  NoActivate - The peer is not activated in any address-family mode
  NoLocalNexthops - The peer is missing local addresses for activated AFI/SAFIs
  UnsupportedCaps - The peer has not negotiated required capabilities
  NoRemoteAs - Missing remote-as for peer
  AsuShutdown - ASU2 shutdown in progress
  IbgpLinkLocalUnsupported - IBGP peering over IPv6 link-local address not supported
  MaxAcceptedPath - The peer has exceeded its max accepted route limit
  Other - The peer is idled for some other reason
  PeerNotification - The peer is pending notification message to be sent
  EbgpLinkLocalMultihopUnsupported - EBGP multihop session over IPv6 link-local
                                     addresses not supported
  MaxAdvertisedPath - The peer has exceeded its max advertised route limit
  Damping - The peer is waiting for BFD connection to be stable'''

capitalizedStrings = [ 'OSPF', 'OSPF3', 'RIP', 'IS-IS', 'IGP', 'EGP',
                       'BGP', 'INCOMPLETE' ]
# Map the capitalized string in value to camel style if the string is part of
# 'capitalizedStrings'. eg: 'OSPF3' to 'Ospf3' and 'RIP' to 'Rip'
def maybeCamelize( value ):
   if value in capitalizedStrings:
      return value.title()
   else:
      return value

# Map the capitalized string in camel style to normal all capitalized style if the
# string is part of 'capitalizedStrings'. eg: 'Ospf3' to 'OSPF3' and 'Rip' to 'RIP'
def maybeDecamelize( value ):
   if value and value.upper() in capitalizedStrings:
      return value.upper()
   else:
      return value

def trimIntfName( addr ):
   return addr if '%' not in addr else addr.split( '%' )[ 0 ]

def peerKeyFromData( data ):
   """
   Given a dictionary (de-serialized from an AMI response), determine the
   PeerConfigKey used to key the dictionary, if present.
   """
   if 'addrv6' in data and data[ 'addrv6' ]:
      key = data[ 'addrv6' ]
      del data[ 'addrv6' ]
   if 'addrv4' in data and data[ 'addrv4' ]:
      key = data[ 'addrv4' ]
      del data[ 'addrv4' ]
   assert key is not None
   return PeerConfigKey( key ).stringValue

class PeerIdleStateModel( Model ):
   noIf = Int( optional=True,
               help='Number of peers in idle state because '
                    'the peer\'s interface is down' )
   admin = Int( optional=True,
                help='Number of peers in idle state because '
                     'the peer is disabled' )
   grpAdmin = Int( optional=True,
                   help='Number of peers in idle state because '
                        'the peer\'s group is disabled' )
   noActivate = Int( optional=True,
                     help='Number of peers in idle state because '
                          'the peer is not activated in '
                          'any address-family mode' )
   noLocalNexthops = Int( optional=True,
                          help='Number of peers in idle state because '
                               'the peer is missing local addresses '
                               'for activated AFI/SAFIs' )
   noRemoteAs = Int( optional=True,
                     help='Number of peers in idle state because '
                          'remote-as for peer is missing' )
   maxPath = Int( optional=True,
                  help='Number of peers in idle state because '
                       'the peer has exceeded its max route limit' )
   deleted = Int( optional=True,
                  help='Number of peers in idle state because '
                       'the peer is deleted and waiting for deffered cleanup' )
   unsupportedCaps = Int( optional=True,
                          help='Number of peers in idle state because the '
                               'peer has not negotiated required capabilities' )
   asuShutdown = Int( optional=True,
                      help='Number of peers in idle state because '
                           'ASU2 shutdown is in progress' )
   maxAcceptedPath = Int( optional=True,
                          help='Number of peers in idle state because '
                               'the peer has exceeded its max accepted route limit' )
   pendingNotification = Int( optional=True,
                              help='Number of peers in idle state because '
                              'the peer is pending notification message to be sent' )
   ebgpLinkLocalMultihopUnsupported = Int( optional=True,
                                           help='Number of EBGP ipv6 link local '
                                           'peers in idle state because multihop'
                                           ' is configured' )
   maxAdvertisedPath = Int( optional=True,
                            help='Number of peers in idle state because '
                            'the peer has exceeded its max advertised route limit' )
   damping = Int( optional=True,
                  help='Number of peers in idle state because '
                       'of BFD flap damping' )
   other = Int( optional=True,
                help='Number of peers in idle state '
                     'because of some unknown reason' )

   def form( self, data ):
      if data > 1:
         return "s are"
      else:
         return " is"

   def renderIdleEntry( self ):
      if self.noIf is not None:
         print "%d neighbor%s in Idle(NoIf) state" \
               % ( self.noIf, self.form( self.noIf ) )
      if self.admin is not None:
         print "%d neighbor%s in Idle(Admin) state" \
               % ( self.admin, self.form( self.admin ) )
      if self.grpAdmin is not None:
         print "%d neighbor%s in Idle(GrpAdmin) state" \
               % ( self.grpAdmin, self.form( self.grpAdmin ) )
      if self.noActivate is not None:
         print "%d neighbor%s in Idle(NoActivate) state" \
               % ( self.noActivate, self.form( self.noActivate ) )
      if self.noLocalNexthops is not None:
         print "%d neighbor%s in Idle(NoLocalNexthops) state" \
               % ( self.noLocalNexthops, self.form( self.noLocalNexthops ) )
      if self.noRemoteAs is not None:
         print "%d neighbor%s in Idle(NoRemoteAs) state" \
               % ( self.noRemoteAs, self.form( self.noRemoteAs ) )
      if self.maxPath is not None:
         print "%d neighbor%s in Idle(MaxPath) state" \
               % ( self.maxPath, self.form( self.maxPath ) )
      if self.deleted is not None:
         print "%d neighbor%s in Idle(Deleted) state" \
               % ( self.deleted, self.form( self.deleted ) )
      if self.unsupportedCaps is not None:
         print "%d neighbor%s in Idle(UnsupportedCaps) state" \
               % ( self.unsupportedCaps, self.form( self.unsupportedCaps ) )
      if self.asuShutdown is not None:
         print "%d neighbor%s in Idle(AsuShutdown) state" \
               % ( self.asuShutdown, self.form( self.asuShutdown ) )
      if self.maxAcceptedPath is not None:
         print "%d neighbor%s in Idle(MaxAcceptedPath) state" \
               % ( self.maxAcceptedPath, self.form( self.maxAcceptedPath ) )
      if self.pendingNotification is not None:
         print "%d neighbor%s in Idle(PendingNotification) state" \
            % ( self.pendingNotification, self.form( self.pendingNotification ) )
      if self.ebgpLinkLocalMultihopUnsupported is not None:
         print "%d neighbor%s in Idle(EbgpLinkLocalMultihopUnsupported) state" \
            % ( self.ebgpLinkLocalMultihopUnsupported,
                  self.form( self.ebgpLinkLocalMultihopUnsupported ) )
      if self.maxAdvertisedPath is not None:
         print "%d neighbor%s in Idle(MaxAdvertisedPath) state" \
               % ( self.maxAdvertisedPath, self.form( self.maxAdvertisedPath ) )
      if self.damping is not None:
         print "%d neighbor%s in Idle(Damping) state" \
               % ( self.damping, self.form( self.damping ) )
      if self.other is not None:
         print "%d neighbor%s in Idle(Other) state" \
               % ( self.other, self.form( self.other ) )
      
class PeerStateModel( Model ):
   idle = Submodel( optional=True, valueType = PeerIdleStateModel,
                    help='Number of peers in idle state for different idle reasons' )
   connect = Int( optional=True,
                  help='Number of peers in connect state' )
   active = Int( optional=True,
                 help='Number of peers in Active state' )
   openSent = Int( optional=True,
                   help='Number of peers in OpenSent state' )
   openConfirm = Int( optional=True,
                      help='Number of peers in OpenConfirm state' )
   established = Int( optional=True,
                      help='Number of peers in Established state' )
   notNegotiated = Int( optional=True,
                        help='Number of peers in NotNegotiated state' )

   def form( self, data ):
      if data > 1:
         return "s are"
      else:
         return " is"
  
   def renderEntry( self ):
      if self.idle is not None :
         self.idle.renderIdleEntry()
      if self.connect is not None:
         print "%d neighbor%s in Connect state" \
               % ( self.connect, self.form( self.connect ) )
      if self.active is not None:
         print "%d neighbor%s in Active state" \
               % ( self.active, self.form( self.active ) )
      if self.openSent is not None:
         print "%d neighbor%s in OpenSent state" \
               % ( self.openSent, self.form( self.openSent ) )
      if self.openConfirm is not None:
         print "%d neighbor%s in OpenConfirm state " \
               % ( self.openConfirm, self.form( self.openConfirm ) )
      if self.established is not None:
         print "%d neighbor%s in Established state" \
               % ( self.established, self.form( self.established ) )
      if self.notNegotiated is not None:
         print "%d neighbor%s in NotNegotiated state" \
               % ( self.notNegotiated, \
               self.form( self.notNegotiated ) )

class BgpStatistics( Model ):
   asn = Str( help='Autonomous System Number' )
   routerId = Ip4Address( help='Bgp Router Identity' )
   numOfPeersByState = Submodel( valueType=PeerStateModel,
                                 help='Number of peers in each  state' )
   _vrf = Str( help='Vrf name' )
   
   def setVrf( self, data ):
      self._vrf = data
   
   def render( self ):
      print "BGP statistics information for VRF %s" % ( self._vrf )
      print "BGP router identifier %s, local AS number %s" % \
            ( self.routerId, self.asn )
      self.numOfPeersByState.renderEntry()      

class NegotiatedMpCapabilities( Model ):
   ipv4Unicast = Bool( help='Has IPv4 unicast capability' )
   ipv6Unicast = Bool( help='Has IPv6 unicast capability' )
   ipv4Multicast = Bool( help='Has IPv4 multicast capability' )
   ipv6Multicast = Bool( help='Has IPv6 multicast capability' )
   ipv4MplsLabels = Bool( help='Has IPv4 labeled-unicast capability' )
   ipv6MplsLabels = Bool( help='Has IPv6 labeled-unicast capability' )
   ipv4SrTe = Bool( help='Has IPv4 SR-TE capability' )
   ipv6SrTe = Bool( help='Has IPv6 SR-TE capability' )
   ipv4MplsVpn = Bool( help='Has VPN-IPv4 capability' )
   ipv6MplsVpn = Bool( help='Has VPN-IPv6 capability' )
   ipv4FlowSpec = Bool( help='Has IPv4 flow specification capability' )
   ipv6FlowSpec = Bool( help='Has IPv6 flow specification capability' )
   l2VpnEvpn = Bool( help='Has L2VPN EVPN capability' )
   linkState = Bool( help='Has BGP Link State capability' )
   dps = Bool( help='Has dynamic path selection capability' )
   rtMembership = Bool( help='Has RT Membership capability' )

AFI_SAFI_LIST = (
   'ipv4Unicast',
   'ipv6Unicast',
   'ipv4Multicast',
   'ipv6Multicast',
   'ipv4MplsLabels',
   'ipv6MplsLabels',
   'ipv4SrTe',
   'ipv6SrTe',
   'ipv4MplsVpn',
   'ipv6MplsVpn',
   'ipv4FlowSpec',
   'ipv6FlowSpec',
   'l2VpnEvpn',
   'linkState',
   'dps',
   'rtMembership',
)

class BgpPeerGroupEntry( Model ):
   peerState = Enum( values=( 'Idle', 'Connect', 'Active', 'OpenSent',
                              'OpenConfirm', 'Established', 'NotNegotiated' ),
                     help=stateHelp )
   negotiatedMpCapabilities = Submodel( valueType=NegotiatedMpCapabilities,
                                        help='Negotiated MP capabilities' )
   vrf = Str( help='VRF name' )

class BgpPeerGroupListenRangeSubnet( Model ):
   asNumber = Str( help='AS number' )
   peerFilter = Str( help='Peer filter' )
   vrf = Str( help='VRF name' )

class BgpPeerGroup( Model ):
   staticPeers = GeneratorDict( keyType=str, valueType=BgpPeerGroupEntry,
                    help='A dictionary of static peers keyed by IP address' )
   listenRangeSubnets = GeneratorDict( keyType=str,
                    valueType=BgpPeerGroupListenRangeSubnet,
                    help='A dictionary of listen-range subnets keyed by IP prefix' )
   dynamicPeers = GeneratorDict( keyType=str, valueType=BgpPeerGroupEntry,
                    help='A dictionary of dynamic peers keyed by IP address' )

   _vrf = Str( help='VRF name' )

class BgpPeerGroups( Model ):
   peerGroups = GeneratorDict( keyType=str, valueType=BgpPeerGroup,
                               help='A dictionary of peer groups keyed by name' )

class BgpSummaryPeerEntry( Model ):
   __revision__ = 2
   asn = Str( help='Autonomous System Number' )
   version = Int( help='BGP version' )
   peerState = Enum( values=( 'Idle', 'Connect', 'Active', 'OpenSent',
                              'OpenConfirm', 'Established', 'NotNegotiated' ),
                     help=stateHelp )
   peerStateIdleReason = Enum( values=( 'NoInterface', 'Admin', 'GroupAdmin',
                                        'NoActivate', 'NoLocalNexthops',
                                        'NoRemoteAs', 'MaxPath', 'Deleted', 'Other',
                                        'UnsupportedCaps', 'MaxAcceptedPath',
                                        'AsuShutdown', 'IbgpLinkLocalUnsupported',
                                        'EbgpLinkLocalMultihopUnsupported',
                                        'PendingNotification',
                                        'MaxAdvertisedPath', 'Damping' ),
                               optional=True, help=idleReasonHelp )
   msgReceived = Int( help='Number of messages received' )
   msgSent = Int( help='Number of messages sent' )
   inMsgQueue = Int( help='Number of messages waiting to be processed' )
   outMsgQueue = Int( help='Number of messages waiting to be sent' )
   prefixAccepted = Int( help='Number of prefixes accepted' )
   prefixReceived = Int( help='Number of prefixes receieved from peer' )
   rulesAccepted = Int( help='Number of flowspec rules accepted from peer',
                        optional=True )
   rulesReceived = Int( help='Number of flowspec rules received from peer',
                        optional=True )
   prefixInBestEcmp = Int( help='Number of paths end in best ecmp group' )
   prefixInBest = Int( help='Number of paths select as best' )
   upDownTime = Float( help='Timestamp of last up/down transition' )
   underMaintenance = Bool( optional=True,
                            help='Peering under maintenance' )
   description = Str( optional=True, help="Peer Description" )


   def getKey( self, data ):
      return peerKeyFromData( data )

   def processData( self, data ):
      data[ 'upDownTime' ] = Tac.utcNow() - data[ 'upDownTime' ]

      state = data[ 'peerState' ]
      if 'Idle' in state:
         m = IDLE_RE.match( state )
         if m:
            reason = m.group( 1 )
            if reason in PEER_STATE_MAP:
               reason = PEER_STATE_MAP[ reason ]
            state = 'Idle'
         else:
            reason = 'Other'
         data[ 'peerStateIdleReason' ] = reason
      data[ 'peerState' ] = state

      if state != 'Established':
         data[ 'prefixReceived' ] = 0
         data[ 'prefixAccepted' ] = 0

      if 'NotNegotiated' in state:
         data[ 'peerState' ] = 'NotNegotiated'

      data[ 'underMaintenance' ] = ord( data[ 'maintenance' ] ) != 0
      del data[ 'maintenance' ]

      return data

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         asnInStr = dictRepr[ 'asn' ]
         dictRepr[ 'asn' ] = asnStrToNum( asnInStr ) 
         return dictRepr
      return dictRepr

   def renderEntry( self, peer, showDes=False ):
      state = self.peerState
      if state == 'Established':
         state = 'Estab'
         received = self.prefixReceived
         accepted = self.prefixAccepted
      else:
         received = ''
         accepted = ''

      if state == 'NotNegotiated':
         state = 'Estab(NotNegotiated)'

      reason = self.peerStateIdleReason
      if state == 'Idle' and reason:
         if reason in PEER_STATE_MAP_INV:
            reason = PEER_STATE_MAP_INV[ reason ]
         state = 'Idle(%s)' % reason

      flags = ' '
      if self.underMaintenance:
         flags = 'm'

      time = int( Tac.utcNow() - self.upDownTime )
      time = formatTimeInterval( time )

      description = ( self.description or '' )[ : MAX_LEN_OF_DES ]
      formatStr = summaryFormatStrDes if showDes else summaryFormatStr
      print formatStr % ( flags, description, peer, self.version,
                        self.asn, self.msgReceived, self.msgSent, self.inMsgQueue,
                        self.outMsgQueue, time, state, received, accepted )

def _degradeToIpGenAddress( peerKey ):
   # The peerKey could be None for originated routes
   if peerKey is None:
      return None
   try:
      Arnet.IpGenAddr( str( peerKey ) )
   except ( IndexError, ValueError ):
      raise CliModelNotDegradable( "An IPv6 link-local address cannot be"
                                   " degraded to an IpGenAddr." )
   return peerKey

def _degradePeersDict( dictRepr ):
   """
   The revision change simply changed the type of 'peerAddress' IpGenAddress => str.
   The string representation of the peer key did not change, only its type.
   Therefore, the degrade routine only attempts to check the type of the key against
   the old model's type (IpGenAddress). If it cannot create an instance of this type,
   CliModelNotDegradable is raised as the client must be using an IPv6 link-local
   peer type which cannot be represented accurately as an IpGenAddress.
   """
   if dictRepr and 'peers' in dictRepr:
      dictRepr[ 'peers' ] = { _degradeToIpGenAddress( pKey ): peer \
                              for pKey, peer in dictRepr[ 'peers' ].iteritems() }
   return dictRepr

def _degradeSubmodel( modelClass, dictRepr, key, revision ):
   if dictRepr and key in dictRepr:
      revision = max( 1, revision )
      return modelClass().degrade( dictRepr[ key ], revision )
   else:
      return dictRepr

helpStr = 'Dictionary of BGP peer entries indexed by the peer address'
class BgpSummary( Model ):
   __revision__ = 3
   asn = Str( help='Autonomous System Number' )
   routerId = Ip4Address( help='BGP Router Identity' )
   vrf = Str( help='VRF Name' )
   peers = GeneratorDict( valueType=BgpSummaryPeerEntry, help=helpStr )

   def processData( self, data ):
      data[ 'asn' ] = data[ 'as' ]
      data[ 'routerId' ] = data[ 'routerid' ]
      del data[ 'as' ]
      del data[ 'routerid' ]
      return data

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         dictRepr = _degradePeersDict( dictRepr )
      if revision < 2:
         asnInStr = dictRepr[ 'asn' ]
         dictRepr[ 'asn' ] = asnStrToNum( asnInStr ) 
      return dictRepr

   def render( self ):
      print 'BGP summary information for VRF', self.vrf
      print 'Router identifier %s, local AS number %s' % ( self.routerId,
                                                           self.asn )
      print 'Neighbor Status Codes: m - Under maintenance'
      # As self.peers is of GeneratorDict we cannot sort it in line, we must generate
      # the entire dictionary, then sort the keys.
      peers = { addr : peer for addr, peer in self.peers }
      showDes = False
      for key in peers:
         des = peers[ key ].description
         if des != '' and des is not None:
            showDes = True
            break
      formatStr = summaryFormatStrDes if showDes else summaryFormatStr
      description = 'Description' if showDes else ''
      print formatStr % ( ' ', description, 'Neighbor', 'V', 'AS', 'MsgRcvd',
            'MsgSent', 'InQ', 'OutQ', 'Up/Down', 'State', 'PfxRcd', 'PfxAcc' )
      for key in sorted( PeerConfigKey( addr ) for addr in peers ):
         peers[ key.stringValue ].renderEntry( key.stringValue, showDes )

class BgpPeerListBase( Model ):
   # NOTE: As 'BgpPeer' has undergone an incompatible type change and had its
   # revision incremented accordingly, we must also increment the refcount on
   # BgpPeerListBase (BgpPeerList's parent) from 1 => 2. However, the degrade
   # method is a no-op as degrading the GeneratorList is done per element.
   __revision__ = 2

#-------------------------------------------------------------------------------
# show ip(v6) bgp [vrf name|all|default]
# show ip(v6) bgp <addr> [vrf name|all|default]
# show ip(v6) bgp <prefix> [longer-prefixes] [detail] [installed] [not-installed] 
#       [vrf name|all|default]
# show ip(v6) bgp detail [vrf name|all|default]
# show ip(v6) bgp community [aa:nn] [community-num] [internet] [local-as] 
#       [no-advertise] [no-export] [detail] [vrf name|all|default]
# show ip(v6) bgp regexp <AS-PATH-REGEX>
# show ip(v6) bgp neighbors <peer> [ipv4|ipv6 unicast] [advertised-routes | 
#       routes | received-routes] [vrf name|all|default]
# show ip(v6) bgp neighbors <peer> [advertised-routes | routes | received-routes] 
#       addr [vrf name|all|default]
# show ip(v6) bgp neighbors <peer> [advertised-routes | routes | received-routes] 
#       prefix [longer-prefixes] [vrf name|all|default]
# show ip(v6) bgp neighbors <peer> [advertised-routes | routes | received-routes] 
#       community [aa:nn] [community-num] [internet] [local-as] [no-advertise] 
#       [no-export]
# show ip(v6) bgp neighbors <peer> [ipv4|ipv6 unicast] [advertised-routes | routes | 
#       received-routes] detail [vrf name|all|default]
# show ip(v6) bgp neighbors <peer> [advertised-routes | routes | received-routes] 
#       regexp <AS-PATH-REGEX>
#-------------------------------------------------------------------------------

# Make sure the following definitions are in sync with the definitions in 
# /src/gated/gated-ctk/src/mioagt/bgp_api.h
BGP_RT_ACTIVE = 0x01
BGP_RT_VALID = 0x02
BGP_RT_SUPPRESSED = 0x04
BGP_RT_ECMP = 0x08
BGP_RT_ECMP_HEAD = 0x10
BGP_RT_ATOMICAGG = 0x20
BGP_RT_STALE = 0x40
BGP_RT_ECMP_CONTRIB = 0x80
BGP_RT_QUEUED = 0x100
BGP_RT_BACKUP = 0x200
BGP_RT_LOCAL_AGG = 0x400
BGP_RT_INSTALL_DENIED = 0x800
BGP_RT_BEST_INACTIVE = 0x1000
BGP_RT_UCMP = 0x2000
BGP_RT_LU = 0x4000
BGP_RT_LAB_RT_PRESENT = 0x8000

LINE_INDENT = 7 * ' '

BGP_SUMMARY_DETAIL_HEADER = 0x01
BGP_SUMMARY_PFX_DETAIL_HEADER = 0x02

BGP_PENDING_OUT_DELAY_TIME = 0xffff

BGP_OUTDELAY_APPLY_INITIAL = 0
BGP_OUTDELAY_APPLY_CHANGES = 1
BGP_OUTDELAY_APPLY_CHANGES_WITHDRAWALS = 2

flagsLength = 9
routeEntryFmtStr = "%-" + str( flagsLength ) + "s%-19s %-16s %-7s %-7s %-7s %-s"
srTePolicyEntryFmtStr = \
      "%-" + str( flagsLength ) + "s%-19s %-7s %-13s %-16s %-7s %-7s %-7s %-s"
routeEntryDetailFmtStr = \
      "      Origin %s, metric %s, localpref %s, IGP metric %s, %s%s, %s, " + \
      "%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s"

# These are reasons that are common to gated and ArBgp. (Some of the reasons may
# not be used by ArBgp for example, routerId). Also ArBgp doesn't care about
# the order in which these are defined, but gated does, so do NOT reorder them.
# Make sure the following definitions are in sync with the definitions in 
# /src/gated/gated-ctk/src/aspath/aspath.h
# aspath.h holds the values as Enums (ints) bgp_reason_not_bestpath_value_t
# The ArBgp definitions are present BgpShowCliNotBestReason.tac and used in sub
# classes of BgpBestpathStep in the lossReasonJson attribute and must match what's
# defined here.
REASON_NOT_BEST_LIST_GATED_ARBGP_COMMON = [
   'invalid',
   'infMed',
   'weight',
   'preference',
   'localpref',
   'aspathLength',
   'origin',
   'med',
   'ebgp',
   'igpCost',
   'aspathDetails',
   'ecmpFast',
   'routerId',
   'routerIdTB',
   'originatorId',
   'originatorIdTB',
   'clusterlistLen',
   'clusterlistLenTB',
   'peerAddress',
   'pathId',
   'routeNotReady',
   'age',
   'ageTB',
   'routeUnusable',
   'noBest',
   'noReason',
   'redist',
]

# These reasons are either specific to ArBgp (for axample noLocalLabel)
# or they're present because of a difference in the ArBgp
# implementation where it cannot used of the gated reasons (for example
# originatorIdOrRouterId)
# The ArBgp show command implementations do not create the model, they emit the json
# directly. So if we create new subclasses of BgpBestpathStep, we must update
# REASON_NOT_BEST_LIST_ARBGP and augment REASON_NOT_BEST_OUTPUT_MAP as well
REASON_NOT_BEST_LIST_ARBGP = [
   'originatorIdOrRouterId',
   'localPath',
   'noLocalLabel',
   'nexthopVrf',
   'macMobilityStickyMac',
   'macMobilitySeqNum',
   'macMobilityEsi',
   'macMobilityTepAddr'
   'domainPathLength',
   'originAsValidity',
]

# Must keep REASON_NOT_BEST_LIST_GATED_ARBGP_COMMON first so that the
# reason string position in the list and enum value bgp_reason_not_bestpath_value_t
# match
REASON_NOT_BEST_LIST = REASON_NOT_BEST_LIST_GATED_ARBGP_COMMON + \
                       REASON_NOT_BEST_LIST_ARBGP

REASON_NOT_BEST_MAP = dict( enumerate( REASON_NOT_BEST_LIST, start=0 ) )
# asserts to check that the REASON_NOT_BEST_LIST has the correct order
# These asserts will be executed when loading / importing this python module
assert REASON_NOT_BEST_MAP[ 0 ] == 'invalid'
assert REASON_NOT_BEST_MAP[
   len( REASON_NOT_BEST_LIST_GATED_ARBGP_COMMON ) - 1 ] == 'redist'
# This is to prevent adding new reasons to REASON_NOT_BEST_LIST_GATED_ARBGP_COMMON or
# bgp_reason_not_bestpath_value_t with a value that's BGP_ROUTE_NOT_BEST_REDIST +1
# since that's used by 'originatorIdOrRouterId', so any new reasons in gated
# will need to have an enum value greater than the last entry in
# REASON_NOT_BEST_LIST. There's a comment to this effect in
# /src/gated/gated-ctk/src/aspath/aspath.h as well.
assert REASON_NOT_BEST_MAP[
   len( REASON_NOT_BEST_LIST_GATED_ARBGP_COMMON ) ] == 'originatorIdOrRouterId'

# For ArBgp the value portion of this dict, (ie., the rendered/text reason) is
# defined in nameOfStep/lossReason attribute of subclasses of BgpBestpathStep.
# (The reason strings are defined in BgpShowCliNotBestReason.tac)
# The values defined there must match the values defined in this map.
REASON_NOT_BEST_OUTPUT_MAP = { 'invalid' : 'Unknown',
                               'infMed' : 'Another route from the same '
                               'AS is a better BGP route',
                               'weight' : 'Path weight',
                               'localpref' : 'Local preference',
                               'domainPathLength': 'Domain path length',
                               'aspathLength': 'AS path length',
                               'origin' : 'Origin',
                               'med' : 'Path MED',
                               'ebgp' : 'eBGP path preferred',
                               'igpCost' : 'IGP cost',
                               'aspathDetails' : 'AS path details',
                               'ecmpFast' : 'ECMP-Fast configured',
                               'routerId' : 'Router ID',
                               'routerIdTB' : 'Router ID tie-break configured',
                               'originatorId' : 'Originator ID',
                               'originatorIdTB' : 
                                  'Originator ID tie-break configured',
                               'clusterlistLen' : 'Cluster list length',
                               'clusterlistLenTB' : 
                                  'Cluster list length tie-break configured',
                               'peerAddress' : 'Peer IP address',
                               'pathId' : 'Path ID',
                               'routeNotReady' : 'Peer not ready',
                               'age' : 'Age',
                               'ageTB' : 'Age tie-break configured',
                               'routeUnusable' : 'Unusable',
                               'noBest' : '',
                               'noReason' : '',
                               'redist': 'Redistributed route exists',
                               'originatorIdOrRouterId': 'Originator/Router ID',
                               'localPath': 'Local path',
                               'noLocalLabel':
                                  'A valid local label has not been allocated',
                               'nexthopVrf': 'Nexthop VRF',
                               'macMobilityStickyMac': 'MAC mobility sticky MAC',
                               'macMobilitySeqNum': 'MAC mobility sequence number',
                               'macMobilityEsi': 'MAC mobility ESI',
                               'macMobilityTepAddr':
                                  'MAC mobility tunnel endpoint address',
                               'originAsValidity': 'Origin AS validity',
}

class BgpRouteASPathEntry( Model ):
   asPath = Str( optional=True, help='AS path string (if absent,  \
                 then the route was originated locally)' )
   asPathType = Enum( values=( 'Internal', 'External', 
                               'Confed-External', 'Local', 
                               'Invalid' ), 
                      help='AS path type: \
                            Internal - originated by I-BGP \
                            External - originated by E-BGP \
                            Confed-External - originated by a E-BGP confederation \
                            Local - originated locally \
                            Invalid - AS path is invalid' )

class BgpRouteTypeEntry ( Model ):
   stale = Bool( default=False, help='Route is stale' )
   valid = Bool( default=False, help='Route is valid' )
   suppressed = Bool( default=False, 
                      help='Route is suppressed from entering the Rib' )
   active = Bool( default=False, help='Route is active' )
   backup = Bool( default=False, 
                  help='Route is backup' )
   ecmpHead = Bool( default=False, help='Route is the ECMP head' )
   ecmp = Bool( default=False, help='Route is an ECMP route' )
   ucmp = Bool( default=False, help='Route is an UCMP route' )
   ecmpContributor = Bool( default=False, 
                           help='Route contributes to ECMP' )
   atomicAggregator = Bool( default=False, 
                            help='Route is an atomic-aggregate' )
   queued = Bool( default=False, 
                  help='Route is queued for advertisement' )
   localAgg = Bool( optional=True, help='Route is locally aggregated' )
   luRoute = Bool( default=False, help='Route is an LU route' )
   notInstalledReason = Enum( values=( 'routeBestInactive', 
                              'routeInstallDenied', 'labeledRoutePresent' ),
                              optional=True,
                              help="Reason for route not being installed" )
   origin = Enum( optional=True, values=( 'Igp', 'Egp', 'Incomplete' ),
                                 help = 'Route origin' )
   waitForConvergence = Bool( optional=True,
                              help='Route is pending BGP convergence' )
   unknownMetric = Bool( optional=True,
                         help='Route is pending resolution' )
   sixPeRoute = Bool( optional=True,
                      help='Route is an IPv6 provider edge (6PE) route' )
   originAsValidity = Enum( optional=True, values=( 'valid', 'invalid',
                                                    'notFound', 'notValidated' ),
                            help='Result of validating origin AS' )
   def get( self, attribute ):
      return getattr( self, attribute )

class BgpRoutePeerEntry( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         dictRepr[ 'peerAddr' ] = _degradeToIpGenAddress( dictRepr[ 'peerAddr' ] )
      return dictRepr

   peerAddr = Str( help='Peer address for the route' )
   peerRouterId = Ip4Address( help='Peer router Identity' )

class BgpRouteContributor( Model ):
   asPath = Str( optional=True, help='AS path string (if absent,  \
                 then the route was originated locally)' )
   contributorAddr = IpGenericAddress( help='Contributor address' )
   contributorMaskLen = Int( help='Contributor mask length' )
   largeCommunities = List( valueType=str, help='Large community for route' )
   extCommunities = List( valueType=str, help='Extended community for route' )
   communities = List( valueType=str, help='Route community' )
   origin = Enum( values=( 'Igp', 'Egp', 'Incomplete' ), 
                           help = 'Contributor Route origin' )

   # The source protocol from which the contributing route was learned.
   #
   # Note that these strings need to be kept in sync with the following:
   #     Gated: /src/gated/gated-ctk/src/bgp/bgp_dget_route.c: redist_proto_str()
   #     ArBgp: /src/ArBgp/BgpShowCliHelperCommon.cpp: redistProtoShowString()
   # Note these are usually transformed by maybeCamelize() (except for BGP-Aggregate)
   redistProtocol = Enum( values=( 'Connected', 'Static', 'Ospf3', 'Ospf', 'Rip',
                                   'Is-Is', 'Bgp', 'BGP-Aggregate', 'Dynamic',
                                   'unknown' ),
                          optional=True,
                          help='Protocol from which the contributor was learned' )

   def render( self ):
      if self.asPath:
         asList = self.asPath
      else:
         asList = "Local"

      if self.origin:
         origin = maybeDecamelize( self.origin )
      else:
         origin = "INCOMPLETE"

      prefixStr = "%s/%d" % ( self.contributorAddr,
                              self.contributorMaskLen )
      print "        %-18s Proto: %-13s Origin: %-12s AS Path: %s  " % \
         ( prefixStr, maybeDecamelize( self.redistProtocol ), origin, asList )

      indent = " " * 10
      if self.communities and \
            len( self.communities ):
         commList = ' '.join( self.communities )
         commStr = indent + "Community: %-s" % commList
         print commStr

      if self.extCommunities and \
            len( self.extCommunities ):
         extCommList = ' '.join( self.extCommunities )
         extCommStr = indent + "Extended Community: %-s" % extCommList
         print extCommStr

      if self.largeCommunities and len( self.largeCommunities ):
         largeCommList = ' '.join( self.largeCommunities )
         largeCommStr = indent + "Large Community: %-s" % largeCommList
         print largeCommStr

   def processData( self, data ):
      if 'contributor' in data or 'contributorv6' in data:
         if 'contributor' in data:
            self.contributorAddr = Arnet.IpGenAddr( data.pop( 'contributor' ) )
         else:
            self.contributorAddr = Arnet.IpGenAddr( data.pop( 'contributorv6' ) )

         self.contributorMaskLen = data.pop( 'contributor_mlen' )

      if 'aspath' in data:
         self.asPath = data.pop( 'aspath' )

      if 'origin' in data:
         self.origin = maybeCamelize( data.pop( 'origin' ) )

      if 'community' in data:
         commlist = data[ 'community' ].strip().split( ' ' )
         for comm in commlist:
            self.communities.append( comm.strip() )
         data.pop( 'community' )

      if 'extcommunity' in data:
         extCommlist = data[ 'extcommunity' ].strip().split( ' ' )
         for ecomm in extCommlist:
            self.extCommunities.append( ecomm.strip() )
         data.pop( 'extcommunity' )

      if 'largecommunity' in data:
         largeCommlist = data[ 'largecommunity' ].strip().split( ' ' )
         for lcomm in largeCommlist:
            self.largeCommunities.append( lcomm.strip() )
         data.pop( 'largecommunity' )

      if 'redist_proto' in data:
         self.redistProtocol = maybeCamelize( data.pop( 'redist_proto' ) )

class EncapLabel( DeferredModel ):
   encapType = Enum( values=( 'VXLAN', 'MPLS' ), help='Encapsulation type' ) 
   value = Str( help='Label value' )

class BgpSrTePolicySegment( Model ):
   mplsLabelSid = Int( help='Segment Identifier in the form of MPLS Label',
                       optional=True )

class BgpSrTePolicySegmentList( Model ):
   weight = Int( optional=True, help='Weight of the segment list' )
   segments = List( valueType=BgpSrTePolicySegment,
                    help='Segments that make up the segment list' )

   def processData( self, data ):
      if 'segmentCount' in data:
         data.pop( 'segmentCount' )
      if 'segmentListWeight' in data:
         self.weight = data.pop( 'segmentListWeight' )
      if 'segments' in data:
         segmentValues = data.pop( 'segments' )
         for label in segmentValues:
            segmentValue = BgpSrTePolicySegment()
            segmentValue.mplsLabelSid = label
            self.segments.append( segmentValue )
   def render( self ):
      slStr = ' '.join( "%u" % seg.mplsLabelSid for seg in self.segments )
      slStr = "Label Stack: [" + slStr + "]"
      if self.weight is not None:
         slStr += ", Weight: %d" % ( self.weight )
      print "         Segment List: %s" % ( slStr )

class BgpSrTePolicyBindingSid( Model ):
   bindingSidType = Enum( values=( 'mpls', 'ipv6' ),
                          help='Binding Segment Identifier Type' )
   mplsLabelSid = Int( help='Binding Segment Identifier Value in the '
                            'form of MPLS Label',
                       optional=True )

class BgpSrTePolicyEnlp( Model ):
   enlpType = Enum( values=( 'ipv4', 'ipv6', 'ipv4AndIpv6', 'none', 'unknown' ),
                    help='Explicit null label policy type' )
   value = Int( help='Explicit null label policy value' )

class BgpSrTePolicyTunnelEncap( Model ):
   preference = Int( optional=True, help='Preference of the SR Policy' )
   bindingSid = Submodel( optional=True, valueType=BgpSrTePolicyBindingSid,
                          help='Binding Segment Identifier' )
   enlp = Submodel( optional=True, valueType=BgpSrTePolicyEnlp,
                    help='Explicit null label policy' )
   segmentLists = List( valueType=BgpSrTePolicySegmentList,
                        help='Segment lists' )
   def render( self ):
      print "      Tunnel encapsulation attribute: SR Policy"
      if self.preference:
         print "         Preference: %u" % ( self.preference )
      if self.bindingSid:
         print "         Binding SID: %u" % ( self.bindingSid.mplsLabelSid )
      if self.enlp:
         valueStr = tacEnlp.valueToShowString( self.enlp.value )
         if self.enlp.enlpType == 'unknown':
            valueStr += '(%u)' % ( self.enlp.value )
         print "         Explicit null label policy: %s" % ( valueStr )
      for sl in self.segmentLists:
         sl.render()

class PmsiPimTunnelId( Model ):
   source = IpGenericAddress(
         help='Source address used in tunnel encapsulation' )
   group = IpGenericAddress(
         help='Destination address used in tunnel encapsulation' )

class PmsiTunnelId( Model ):
   ingressReplication = IpGenericAddress( optional=True, help="Replication address" )
   pim = Submodel( optional=True, valueType=PmsiPimTunnelId,
                   help="PIM tree information" )

class PmsiTunnel( Model ):
   tunnelType = Enum( values=( 'noInfo', 'rsvpTeP2mpLsp', 'mLdpP2mpLsp',
                               'pimSsmTree', 'pimSmTree', 'bidirPimTree',
                               'ingressReplication', 'mLdpMp2mpLsp',
                               'transportTunnel', 'assistedReplication', 'bier',
                               'wildcardTransport', 'unknown' ),
                      help='Tunnel technology used to establish PMSI tunnel' )
   mplsLabel = Int( default=0, help="MPLS label" )
   leafInformationRequired = Bool( default=False, help="Leaf information required" )
   tunnelId = Submodel( optional=True, valueType=PmsiTunnelId,
                        help="Tunnel information" )

class BgpLsNodeFlags( Model ):
   isisOverload = Bool( default=False, help="IS-IS node is overloaded" )
   isisAttached = Bool( default=False, help="IS-IS node is attached to L2 node in a"
                        " different area" )

class SrCapabilitySrgb( Model ):
   srgbBase = Int( help="SRGB base" )
   srgbRange = Int( help="SRGB range" )

class SrCapabilityFlags( Model ):
   mplsV4 = Bool( help="SR capability MPLS IPv4 Flag" )
   mplsV6 = Bool( help="SR capability MPLS IPv6 Flag" )

class SrCapabilityModel( Model ):
   flags = Submodel( valueType=SrCapabilityFlags, help="SR Capability Flags" )
   srCapabilitySrgb = GeneratorList( valueType=SrCapabilitySrgb,
                            help="Segment Routing Global Blocks" )

class SrlbRange( Model ):
   srlbBase = Int( help="SRLB base" )
   srlbRange = Int( help="SRLB range" )

class SrlbModel( Model ):
   srlbRanges = List( valueType=SrlbRange,
                      help="Segment Routing Local Blocks" )

class BgpLsNodeAttributes( Model ):
   # We can mark the flags non optional because the booleans inside will be at their
   # defaults.
   flags = Submodel( valueType=BgpLsNodeFlags, help="Node flags" )
   # Pretty much everything here has to be optional as these may not be advertised
   # by an IS-IS speaking node
   ipv4RouterId = Ip4Address( optional=True,
                              help="IPv4 router ID of node" )
   ipv6RouterId = Ip6Address( optional=True,
                              help="IPv6 router ID of node" )
   # Empty list is valid, so not optional
   isisAreaIds = List( valueType=str,
                      help="List of IS-IS area identifiers" )
   nodeName = Str( optional=True,
                   help="Symbolic name of node" )
   srCapabilities = Submodel( optional=True, valueType=SrCapabilityModel,
                                   help="SR Capabilities" )
   srlb = Submodel( optional=True, valueType=SrlbModel,
                    help="SR Local Block Information" )

class BgpLsSrAdjSidFlags( Model ):
   ipv6 = Bool( help="SID is used for forwarding IPv6 traffic" )
   backup = Bool( help="SID is eligible for protection such as using "
      "IPFRR or MPLS-FRR" )
   value = Bool( help="SID carries a value instead of an index" )
   local = Bool( help="Value/index carried by SID has local significance" )
   adjacencySet = Bool( help="SID refers to set of adjacencies and may also be "
      "assigned to other adjacencies" )
   persistent = Bool( help="SID is persistent across router restart/interface "
      "flap." )

class BgpLsSrAdjacencySid( Model ):
   sid = Int( help="SR Adjacency Index/Label SID" )
   flags = Submodel( valueType=BgpLsSrAdjSidFlags,
         help="SR Adjacency SID Flags" )
   weight = Int( default=0,
                 help="Weight of the Adj-SID for load balancing" )

class BgpLsLinkAttributes( Model ):
   # Simply storing the integer metric is sufficient. We don't need to know the
   # type of metric (ospf, isisWide, isisNarrow) as that only affects the on wire
   # encoding in TLV 1095.
   igpMetric = Int( help="IGP metric of link" )
   localIpv4RouterId = Ip4Address( optional=True,
                                   help="IPv4 Router ID of local node" )
   localIpv6RouterId = Ip6Address( optional=True,
                                   help="IPv6 Router ID of local node" )
   remoteIpv4RouterId = Ip4Address( optional=True,
                                   help="IPv4 Router ID of remote node" )
   remoteIpv6RouterId = Ip6Address( optional=True,
                                   help="IPv6 Router ID of remote node" )
   administrativeGroup = Int( optional=True,
                              help="Administrative Group(color) of link" )
   maxLinkBw = Int( optional=True,
                    help="Maximum link bandwidth (bps)" )
   maxReservableLinkBw = Int( optional=True,
                              help="Maximum bandwidth (bps) that can be reserved "
                              "on link" )
   unreservedBw = Dict( optional=True, valueType=long,
                        help="Unreserved bandwith(bps) keyed by priority" )
   srlgIds = List( optional=True, valueType=int, help='Link SRLG values' )
   teDefaultMetric = Int( optional=True,
                          help="Traffic engineering metric of link" )
   srAdjacencySids = List( optional=True, valueType=BgpLsSrAdjacencySid,
                          help="List of SR Adjacency SIDs" )

class BgpLsPrefixFlags( Model ):
   isisDown = Bool( default=False,
                    help="A prefix has been leaked from a higher level to a "
                    "lower level" )

class BgpLsSrPrefixFlags( Model ):
   readvertised = Bool( help="Corresponding prefix has been re-advertised from "
      "another level or redistribution " )
   nodeSid = Bool( help="SID refers to the router identified by the prefix, "
      "typically attached to router loopback address" )
   noPhp = Bool( help="Penultimate hop must not pop SID before sending packet to "
      "advertising node" )
   explicitNull = Bool( help="Upstream neighbor must replace with an Explicit NULL "
      "value SID before sending packet " )
   value = Bool( help="SID carries a value instead of an index" )
   local = Bool( help="Value/index carried by SID has local significance" )

class BgpLsSrPrefixSid( Model ):
   sid = Int( help="SR Prefix Index/Label SID" )
   flags = Submodel( valueType=BgpLsSrPrefixFlags,
         help="SR Prefix SID Flags" )
   algorithm = Int( default=0,
                    help="Algorithm identifier to compute reachability of prefix" )

class BgpLsPrefixAttributes( Model ):
   # Using a submodel because there are 3 ospf flags which we may need in the future
   flags = Submodel( valueType=BgpLsPrefixFlags,
                     help="Prefix flags" )
   # This marked optional because it's possible for it to be absent.
   # In IS-IS it's always present
   # https://tools.ietf.org/html/draft-ietf-idr-rfc7752bis-02#section-4.3.3.4
   metric = Int( optional=True,
                 help="Metric of prefix as known in the IGP topology" )

   srPrefixSid = Submodel( optional=True, valueType=BgpLsSrPrefixSid,
                           help="SR Prefix SID" )

class BgpRouteDetailEntry( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         if 'peerEntry' in dictRepr:
            dictRepr[ 'peerEntry' ] = \
                  BgpRoutePeerEntry().degrade( dictRepr[ 'peerEntry' ], 1 )
      return dictRepr

   aggregator = Str( optional=True, help='Aggregator of the route' )
   nextHopVrfName = Str( optional=True,
                         help='VRF Name of the nexthop for the route' )
   communityList = List( valueType=str, help='Route community' )
   extCommunityList = List( valueType=str, help='Extended community for route' )
   largeCommunityList = List( valueType=str, help='Large community for route' )
   rxPathId = Int( optional=True, help='Received path ID of this route' )
   txPathId = Int( optional=True, help='Advertised path ID for this route' )
   rxSafi = Str( optional=True, help='Received SAFI of this route' )
   origin = Enum( values=( 'Igp', 'Egp', 'Incomplete' ), help = 'Route origin' )
   originator = Str( optional=True, 
                     help='Router ID of the originator of this route' )
   clusterList = Str( optional=True, 
                       help='Cluster list for the route' )
   recvdFromRRClient = Bool( default=False, 
                             help='Route received from route reflector client' )
   seqNumber = Int( optional=True, help='Route sequence number' )
   pendingTimeToAdv = Float( optional=True, 
                             help='Timestamp of route advertisement' )

   # The protocol from which the route was redistributed into BGP. Both "Bgp" and
   # "BGP-Aggregate" are listed here purely for completness, even though such routes
   # won't ever be explicitly redistributed into BGP.
   #
   # Note that these strings need to be kept in sync with the following:
   #     Gated: /src/gated/gated-ctk/src/bgp/bgp_dget_route.c: redist_proto_str()
   #     ArBgp: /src/ArBgp/BgpShowCliHelperCommon.cpp: redistProtoShowString()
   # Note these are usually transformed by maybeCamelize() (except for BGP-Aggregate)
   redistributionProtocol = Enum( values=( 'Connected', 'Static', 'Ospf3', 'Ospf',
                                           'Rip', 'Is-Is', 'Bgp', 'BGP-Aggregate',
                                           'Dynamic', 'unknown' ),
                                  optional=True,
                                  help='Protocol from which the route was \
                                     redistributed into BGP' )
   isLeaked = Bool( default=False, optional=True,
                    help="Route leaked from a different VRF" )
   labelStack = List( optional=True, valueType=int, 
                      help='MPLS label stack information' )
   tunnelRibEligible = Bool( default=False,
                             help='Route eligible to be installed in Tunnel RIB' )
   localLabel = Int( optional=True, help='Local MPLS label' )
   srLabelIndex = Int( optional=True, help='Prefix SID Label Index' )
   bgpContributors = List( optional=True,
                     valueType=BgpRouteContributor,
                     help='List of contributors to this aggregated route' )
   igpMetric = Int( optional=True, help='IGP metric' )
   remoteLabel = Submodel( optional=True, valueType=EncapLabel, help='Remote label' )
   srTeTunnelEncap = Submodel( optional=True, valueType=BgpSrTePolicyTunnelEncap,
                   help='Bgp Segment Routing Policy Tunnel Encapsulation Attribute' )
   domainPath = List( optional=True, valueType=str, help="Domain path attribute" )
   pmsiTunnel = Submodel( optional=True, valueType=PmsiTunnel,
                          help="P-Multicast Service Interface Tunnel Attribute" )
   bgpLsNodeAttributes = Submodel( optional=True, valueType=BgpLsNodeAttributes,
                                   help='Attributes of a BGP Link State node' )
   bgpLsLinkAttributes = Submodel( optional=True,
         valueType=BgpLsLinkAttributes,
         help='Attributes a BGP Link State link' )
   bgpLsPrefixAttributes = Submodel( optional=True,
         valueType=BgpLsPrefixAttributes,
         help='Attributes of a BGP Link State prefix' )
   if RoutingLibToggleLib.toggleAccumulatedIgpMetricEnabled():
      aigpDistance = Int( optional=True,
                          help='Accumulated IGP Distance for the route' )
      aigpMetric = Int( optional=True,
                        help='Accumulated IGP metric for the route' )

class BgpRoutePath( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2 :
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         if 'nextHop' in dictRepr:
            dictRepr[ 'nextHop' ] = _degradeToIpGenAddress( dictRepr[ 'nextHop' ] )
         if 'routeDetail' in dictRepr:
            dictRepr[ 'routeDetail' ] = \
                  BgpRouteDetailEntry().degrade( dictRepr[ 'routeDetail' ], 1 )
      return dictRepr

   nextHop = Str( optional=True, help='Route next hop address' )
   asPathEntry = Submodel( valueType=BgpRouteASPathEntry, 
                           help='AS path information' )
   med = Int( optional=True,
              help='Multi Exit Discriminator for the route' )
   aigpDistance = Int( optional=True,
                       help='Accumulated IGP Distance for the route' )
   aigpMetric = Int( optional=True, help='Accumulated IGP metric for the route' )
   localPreference = Int( optional=True,
                          help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=BgpRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   timestamp = Int( optional=True,
                    help="UTC seconds since epoch when the route was received.\
                          Only returned with 'show ip bgp detail'" )
   routeDetail = Submodel( valueType=BgpRouteDetailEntry, optional=True, 
                           help='Route details' )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True, 
                         help='Peer information for the route' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )

   def processData( self, data ):
      if 'detail' in data:
         self.routeDetail = BgpRouteDetailEntry()
         data.pop( 'detail' )

      if 'nhv4' in data:
         self.nextHop = PeerConfigKey( data.pop( 'nhv4' ) ).stringValue
      if 'nhv6' in data:
         self.nextHop = PeerConfigKey( data.pop( 'nhv6' ) ).stringValue

      if 'aspath' in data or 'peer_route_type' in data:
         self.asPathEntry = BgpRouteASPathEntry()
         if 'aspath' in data:
            self.asPathEntry.asPath = data.pop( 'aspath' )

         prt = data.get( 'peer_route_type' )
         if prt == 'internal':
            self.asPathEntry.asPathType = 'Internal'
         if prt == 'external':
            self.asPathEntry.asPathType = 'External'
         if prt == 'confed-external':
            self.asPathEntry.asPathType = 'Confed-External'
         if prt == 'local':
            self.asPathEntry.asPathType = 'Local'
         if prt == 'invalid':
            self.asPathEntry.asPathType = 'Invalid'
         if prt:
            data.pop( 'peer_route_type' )

      if 'metric' in data:
         self.med = data.pop( 'metric' )
         med = self.med
         if med < 0:
            self.med = ctypes.c_ulong( med ).value

      if 'metric2' in data:
         self.localPreference = data.pop( 'metric2' )
         lp = self.localPreference
         if lp < 0:
            self.localPreference = ctypes.c_ulong( lp ).value

      self.routeType = BgpRouteTypeEntry()
      rt = data.get( 'route_type' )
      if rt:
         if rt & BGP_RT_ACTIVE:
            self.routeType.active = True
         if rt & BGP_RT_BACKUP:
            self.routeType.backup = True
         if rt & BGP_RT_VALID:
            self.routeType.valid = True
         if rt & BGP_RT_SUPPRESSED:
            self.routeType.suppressed = True
         if rt & BGP_RT_ECMP:
            self.routeType.ecmp = True
         if rt & BGP_RT_UCMP:
            self.routeType.ucmp = True
         if rt & BGP_RT_ECMP_HEAD:
            self.routeType.ecmpHead = True
         if rt & BGP_RT_ATOMICAGG:
            self.routeType.atomicAggregator = True
         if rt & BGP_RT_STALE:
            self.routeType.stale = True
         if rt & BGP_RT_ECMP_CONTRIB:
            self.routeType.ecmpContributor = True
         if rt & BGP_RT_QUEUED:
            self.routeType.queued = True
         if rt & BGP_RT_LOCAL_AGG:
            self.routeType.localAgg = True
         if rt & BGP_RT_LU:
            self.routeType.luRoute = True
         if rt & BGP_RT_INSTALL_DENIED:
            self.routeType.notInstalledReason = "routeInstallDenied"
         elif rt & BGP_RT_BEST_INACTIVE:
            self.routeType.notInstalledReason = "routeBestInactive"
         elif rt & BGP_RT_LAB_RT_PRESENT:
            self.routeType.notInstalledReason = "labeledRoutePresent"

         data.pop( 'route_type' )

      if 'community' in data:
         commlist = data[ 'community' ].strip().split( ' ' )
         for comm in commlist:
            self.routeDetail.communityList.append( comm.strip() )
         data.pop( 'community' )

      if 'extcommunity' in data:
         extCommlist = data[ 'extcommunity' ].strip().split( ' ' )
         for ecomm in extCommlist:
            self.routeDetail.extCommunityList.append( ecomm.strip() )
         data.pop( 'extcommunity' )

      if 'largecommunity' in data:
         largeCommlist = data[ 'largecommunity' ].strip().split( ' ' )
         for lcomm in largeCommlist:
            self.routeDetail.largeCommunityList.append( lcomm.strip() )
         data.pop( 'largecommunity' )

      if 'rx_pathid' in data:
         self.routeDetail.rxPathId = data.pop( 'rx_pathid' )

      if 'tx_pathid' in data:
         self.routeDetail.txPathId = data.pop( 'tx_pathid' )

      if 'rx_safi' in data:
         self.routeDetail.rxSafi = data.pop( 'rx_safi' )

      if 'aggregator' in data:
         self.routeDetail.aggregator = data.pop( 'aggregator' )

      if 'origin' in data:
         # Camelcase the routeDetail.origin into Capi Enum
         self.routeDetail.origin = maybeCamelize( data.pop( 'origin' ) )
         self.routeType.origin = self.routeDetail.origin

      if 'originator' in data:
         self.routeDetail.originator = data.pop( 'originator' )

      if 'cluster_list' in data:
         self.routeDetail.clusterList = data.pop( 'cluster_list' )

      if 'recvd_from_rr_client' in data:
         self.routeDetail.recvdFromRRClient = \
               data.pop( 'recvd_from_rr_client' ) == 1 
     
      if 'igp_metric' in data:
         self.routeDetail.igpMetric = data.pop( 'igp_metric' )

      if 'weight' in data:
         self.weight = data.pop( 'weight' )

      if 'timestamp' in data:
         self.timestamp = data.pop( 'timestamp' )

      if 'peer_addrv4' in data or 'peer_addrv6' in data or 'rt_id' in data:
         self.peerEntry = BgpRoutePeerEntry()
         if 'peer_addrv4' in data:
            self.peerEntry.peerAddr = \
                  PeerConfigKey( data.pop( 'peer_addrv4' ) ).stringValue
         if 'peer_addrv6' in data:
            self.peerEntry.peerAddr = \
                  PeerConfigKey( data.pop( 'peer_addrv6' ) ).stringValue
         if 'rt_id' in data:
            self.peerEntry.peerRouterId = data.pop( 'rt_id' )

      if 'vtime' in data:
         self.routeDetail.seqNumber = data.pop( 'vtime' )

      if 'pending_time' in data:
         self.routeDetail.pendingTimeToAdv = \
             Tac.utcNow() + data.pop( 'pending_time' )

      if 'redist_proto' in data:
         self.routeDetail.redistributionProtocol = \
             maybeCamelize( data.pop( 'redist_proto' ) )

      if self.routeDetail and 'label_stack' in data:
         self.routeDetail.labelStack = data.pop( 'label_stack' ) 

      if self.routeDetail:
         if 'reason_not_bestpath' in data:
            reasonNotBestpath = data.pop( 'reason_not_bestpath' )
            if reasonNotBestpath in REASON_NOT_BEST_MAP:
               self.reasonNotBestpath = REASON_NOT_BEST_MAP[ reasonNotBestpath ]
         else:
            self.reasonNotBestpath = None
     
      if self.routeDetail and 'tunnel_rib_eligible' in data:
         self.routeDetail.tunnelRibEligible = \
               data.pop( 'tunnel_rib_eligible' ) == 1 

      if self.routeDetail and 'sr_te_bsid' in data:
         value = data.pop( 'sr_te_bsid' )
         if not self.routeDetail.srTeTunnelEncap:
            self.routeDetail.srTeTunnelEncap = BgpSrTePolicyTunnelEncap()
         self.routeDetail.srTeTunnelEncap.bindingSid = BgpSrTePolicyBindingSid()
         self.routeDetail.srTeTunnelEncap.bindingSid.bindingSidType = 'mpls'
         self.routeDetail.srTeTunnelEncap.bindingSid.mplsLabelSid = value

      if self.routeDetail and 'sr_te_enlp' in data:
         value = data.pop( 'sr_te_enlp' )
         if not self.routeDetail.srTeTunnelEncap:
            self.routeDetail.srTeTunnelEncap = BgpSrTePolicyTunnelEncap()
         enlp = BgpSrTePolicyEnlp()
         enlp.enlpType, enlp.value = tacEnlp.valueToStrep( value ), value
         self.routeDetail.srTeTunnelEncap.enlp = enlp

      if self.routeDetail and 'sr_te_preference' in data:
         if not self.routeDetail.srTeTunnelEncap:
            self.routeDetail.srTeTunnelEncap = BgpSrTePolicyTunnelEncap()
         self.routeDetail.srTeTunnelEncap.preference = data.pop( 'sr_te_preference' )

      if self.routeDetail and 'sr_te_seg_list_cnt' in data:
         value = data.pop( 'sr_te_seg_list_cnt' )
         if not self.routeDetail.srTeTunnelEncap:
            self.routeDetail.srTeTunnelEncap = BgpSrTePolicyTunnelEncap()

      return data

   def renderTable( self, pfx, advRoutes, srTeSafi ):
      # If you add or remove a flag please
      # update flagsLength variable to reflect
      # number of flags +1
      if self.routeType.get( 'stale' ):
         flags = 'S'
      else:
         flags = ' '
      if self.routeType.get( 'valid' ):
         flags += '*'
      else:
         flags += ' '
      if self.routeType.get( 'suppressed' ):
         flags += 's'
      else:
         flags += ' '
      if self.routeType.get( 'notInstalledReason' ):
         flags += "#"
      elif self.routeType.get( 'active' ):
         flags += '>'
      elif self.routeType.get( 'waitForConvergence' ):
         flags += '%'
      elif self.routeType.get( 'backup' ):
         flags += 'b'
      else:
         flags += ' '
      if self.routeType.get( 'ecmpHead' ):
         flags += 'E'
      elif self.routeType.get( 'ecmp' ):
         flags += 'e'
      else:
         flags += ' '
      if self.routeType.get( 'ecmpContributor' ):
         flags += 'c'
      else:
         flags += ' '
      if self.routeType.get( 'queued' ) and advRoutes:
         flags += 'q'
      else:
         flags += ' '
      if self.routeType.get( 'luRoute' ):
         flags += 'L'
      else:
         flags += ' '

      flags += ' '
      if self.asPathEntry and self.asPathEntry.asPath:
         asList = self.asPathEntry.asPath
      else:
         asList = ''

      routeCommonInfo = (
               self.nextHop if self.nextHop is not None else "-",
               self.med if self.med is not None else "-",
               self.localPreference if self.localPreference is not None else "-",
               self.weight if self.weight is not None else "-",
               asList )
      if not srTeSafi:
         print routeEntryFmtStr % ( ( flags, pfx ) + routeCommonInfo )
      else:
         policy = pfx.split( '|' )
         print srTePolicyEntryFmtStr % ( ( flags, policy[ 0 ], policy[ 1 ],
                                           policy[ 2 ] ) + routeCommonInfo )

   def renderDetail( self, pfx, detailRequested, lpSeqNum ):
      indent = " " * 6
      asPath = self.asPathEntry.asPath if self.asPathEntry else None
      print "  %s%s%s" % \
            ( asPath if asPath else "Local",
              self.routeDetail.aggregator if \
                    self.routeDetail.aggregator is not None else "",
              " (Received from a RR-client)" if self.routeDetail.recvdFromRRClient \
                    else "" )
      labelstackStr = "" 
      if self.routeDetail.labelStack:
         labelstackStr = "labels [ "
         for i in self.routeDetail.labelStack:
            if i:
               labelstackStr = labelstackStr + "%d " % i
         labelstackStr = labelstackStr + "] "

      print "    %s %sfrom %s (%s)" % \
            ( self.nextHop if self.nextHop is not None else "-", 
               labelstackStr, 
               self.peerEntry.peerAddr if \
                    self.peerEntry is not None and \
                    self.peerEntry.peerAddr is not None else "-",
              self.peerEntry.peerRouterId if \
                    self.peerEntry is not None and \
                    self.peerEntry.peerRouterId is not None else "-" )

      weightStr = "weight %u" % self.weight if self.weight is not None \
                                               else "weight -"
      ageStr = ""
      if self.timestamp is not None:
         deltaAge = Tac.utcNow() - self.timestamp
         if deltaAge < 0:
            deltaAge = 0
         ageStr = ", received %s ago" % formatTimeInterval( deltaAge )

      redistStr = ", redistributed (%s)" % \
                     maybeDecamelize( self.routeDetail.redistributionProtocol ) \
                 if self.routeDetail.redistributionProtocol is not None else ""
      # We want the CLI output print for routeDetail origin to be in all Caps,
      # hence decamelize
      print routeEntryDetailFmtStr % \
            ( maybeDecamelize( self.routeDetail.origin ) if self.routeDetail.origin \
                 is not None else "INCOMPLETE",
              str( self.med ) if \
                    self.med is not None else '-', 
              str( self.localPreference ) if self.localPreference is not None \
                                          else '-',
              str( self.routeDetail.igpMetric ) if self.routeDetail.igpMetric \
                                          is not None else '-',
              weightStr,
              ageStr,
              "valid" if self.routeType.get( 'valid' ) else "invalid",
              self.asPathEntry.asPathType.lower() if self.asPathEntry else '',
              ", suppressed" if self.routeType.get( 'suppressed' ) else "",
              ", ECMP head" if self.routeType.get( 'ecmpHead' ) else "",
              ", ECMP" if self.routeType.get( 'ecmp' ) else "",
              ", UCMP" if self.routeType.get( 'ucmp' ) else "",
              ", best" if self.routeType.get( 'active' ) else "",
              ", not installed (denied by install-map)" if \
                    self.routeType.get( "notInstalledReason" ) == \
                    'routeInstallDenied' else \
              (", not installed (better AD route present)" if \
                    self.routeType.get( "notInstalledReason" ) == \
                    'routeBestInactive' else \
              (", not installed (labeled-route present)" if \
                    self.routeType.get( "notInstalledReason" ) == \
                    'labeledRoutePresent' else "") ),
              ", ECMP contributor" if self.routeType.get( 'ecmpContributor' ) \
                    else "",
              ", backup" if self.routeType.get( 'backup' ) else "",
              ", local-aggregate"  if self.routeType.get( 'localAgg' ) else "",
              ", atomic-aggregate" if self.routeType.get( 'atomicAggregator' ) \
                    else "",
              ", stale" if self.routeType.get( 'stale' ) else "",
              ", pending BGP convergence" if \
                           self.routeType.get( 'waitForConvergence' ) else "",
              ", queued" if self.routeType.get( 'queued' ) else "",
              redistStr)

      if self.routeDetail.clusterList is not None: 
         cList = self.routeDetail.clusterList
      else:
         cList = ""

      if self.routeDetail.originator or self.routeDetail.clusterList:
         print indent + "%s%s%s" % (
                  self.routeDetail.originator \
                        if self.routeDetail.originator is not None else "",
                  ", " if self.routeDetail.originator is not None and \
                        self.routeDetail.clusterList is not None else "",
                  cList )

      if self.routeDetail.communityList and len( 
            self.routeDetail.communityList ):
         commList = ' '.join( self.routeDetail.communityList )
         print indent + "Community: %-s" % commList

      if self.routeDetail.extCommunityList and len( 
            self.routeDetail.extCommunityList ):
         extCommList = ' '.join( self.routeDetail.extCommunityList )
         print indent + "Extended Community: %-s" % extCommList

      if self.routeDetail.largeCommunityList and len(
            self.routeDetail.largeCommunityList ):
         largeCommList = ' '.join( self.routeDetail.largeCommunityList )
         print indent + "Large Community: %-s" % largeCommList

      strings = []
      if self.routeDetail.rxPathId is not None:
         strings.append( 'Rx path id: 0x%x' % self.routeDetail.rxPathId )
      if self.routeDetail.txPathId is not None:
         strings.append( 'Tx path id: 0x%x' % self.routeDetail.txPathId )
      if strings:
         print indent + ', '.join( strings )

      if self.routeDetail.rxSafi :
         print indent + "Rx SAFI: %s" % self.routeDetail.rxSafi

      if self.reasonNotBestpath is not None and \
             detailRequested == BGP_SUMMARY_PFX_DETAIL_HEADER:
         reasonFieldStr = 'Not best: '
         reasonStr = ( 6 * ' ' ) + reasonFieldStr + \
             REASON_NOT_BEST_OUTPUT_MAP[ self.reasonNotBestpath ]
         if len( REASON_NOT_BEST_OUTPUT_MAP[ self.reasonNotBestpath ] ):
            print reasonStr

      remoteLabel = self.routeDetail.remoteLabel
      if remoteLabel is not None:
         labelType = 'MPLS label' if remoteLabel.encapType == 'MPLS' else 'VNI'
         print indent + "Remote %s: %s" % ( labelType, remoteLabel.value )

      if self.routeDetail.seqNumber or self.routeDetail.pendingTimeToAdv:
         if self.routeDetail.seqNumber:
            print indent + "Sequence number: %d" % self.routeDetail.seqNumber,
            if self.routeDetail.seqNumber > lpSeqNum:
               print "(Ack pending)",
         if self.routeDetail.pendingTimeToAdv:
            if self.routeDetail.seqNumber:
               print ",",
            pendingTime = \
                int( round( self.routeDetail.pendingTimeToAdv - Tac.utcNow() ) )
            if pendingTime == BGP_PENDING_OUT_DELAY_TIME:
               print "Out-delay pending",
            else:
               print "%d seconds remaining" % pendingTime,
         print ""
      
      if self.routeDetail.tunnelRibEligible:
         print indent + "Tunnel RIB eligible"

      if self.routeDetail.bgpContributors:
         print indent + "%d Contributing routes:" % \
                 len( self.routeDetail.bgpContributors )
         for contribRoute in self.routeDetail.bgpContributors:
            contribRoute.render()

      if self.routeDetail.srTeTunnelEncap:
         self.routeDetail.srTeTunnelEncap.render()

   def renderEntry( self, prefix, detail, advRoutes, lpSeqNum, srTeSafi ):
      if detail:
         self.renderDetail( prefix, detail, lpSeqNum )
      else:
         self.renderTable( prefix, advRoutes, srTeSafi )

class BgpRouteAdvertisedPeerListEntry( Model ):
   peerList = List( valueType=str,
      help='List of peers the route is advertised to' )

ADV_TO_PEERS_ROW_LEN = 3

peerListObject = None

notUcmpReasonHelp = '''Used when ECMP could be converted to UCMP.
  NoLinkBWOnAllPaths - All contributing paths do not have link bandwidth available
  maxUcmpExceeded -  Max UCMP configuration is less than or equal to the 
                     number of nexthops preventing UCMP application effectively'''

UCMP_NO_LINK_BANDWIDTH = 1
UCMP_MAX_EXCEEDED = 2

class BgpRouteEntry( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # See BgpCommon/CliPlugin/BgpCliModels.py:_degradePeersDict for details
         bgpRoutePaths = [ BgpRoutePath().degrade( rtPath, 1 ) \
                           for rtPath in dictRepr[ 'bgpRoutePaths' ] ]
         dictRepr[ 'bgpRoutePaths' ] = bgpRoutePaths
      return dictRepr

   address = IpGenericAddress(
         help='Route address; For SR-TE SAFI, this is same as srTeEndpoint' )
   maskLength = Int( help='Route mask length' )
   totalPaths = Int( optional=True, help='Total number of paths for this route' )
   bgpRoutePaths = List( valueType=BgpRoutePath, 
                              help='List of BGP route ECMP paths' )
   totalAdvertisedPeers = Int( optional=True,
                               help='Number of peers this route is advertised to' )
   bgpAdvertisedPeers = Submodel( optional=True, 
                     valueType=BgpRouteAdvertisedPeerListEntry,
                     help='List of peers this route is advertised to' )
   bgpAdvertisedPeerGroups = Dict( optional=True, keyType=str,
                     valueType=BgpRouteAdvertisedPeerListEntry,
      help='List of peers this route is advertised to indexed by peer-group name' )
   notUcmpEligibleReason = Enum( values=( 'noLinkBWOnAllPaths', 'maxUcmpExceeded' ),
                               optional=True, help=notUcmpReasonHelp )
   srTeEndpoint = IpGenericAddress( optional=True,
                                    help='Endpoint address of SR TE policy' )
   srTeColor = Int( optional=True, help='Color of SR TE policy' )
   srTeDistinguisher = Int( optional=True, help='Distinguisher of SR TE policy' )
   _routeKey = Str( help='Route Key; For SR-TE SAFI, this is represented as'
                         ' srTeEndpoint|srTeColor|srTeDistinguisher and for other'
                         ' SAFIs it is represented as address/maskLength' )

   def getSrTeKey( self, data ):
      srTeAddr = data.get( 'addrSrTeV4' )
      if srTeAddr is None:
         srTeAddr = data.get( 'addrSrTeV6' )
      # SR-TE Address is in the format 'endpoint|color|distinguisher'.
      return srTeAddr

   def getKey( self, data ):
      ipAddr = data.get( 'addrv6' )
      if ipAddr is None:
         ipAddr = data.get( 'addrv4' )
         if ipAddr is None:
            return self.getSrTeKey( data )
      return ipAddr + '/' + str( data[ 'mlen' ] )

   def overrideSrTeAddress( self, data ):
      if data[ 'mlen' ]:
         del data[ 'mlen' ]
      srTeAddr = data.get( 'addrSrTeV4' )
      if srTeAddr is not None:
         del data[ 'addrSrTeV4' ]
         self.maskLength = 32
      else:
         srTeAddr = data.get( 'addrSrTeV6' )
         if srTeAddr is not None:
            del data[ 'addrSrTeV6' ]
            self.maskLength = 128
      if srTeAddr is not None:
         splitAddr = srTeAddr.split( '|' )
         self.srTeEndpoint = splitAddr[ 0 ]
         self.srTeColor = int( splitAddr[ 1 ] )
         self.srTeDistinguisher = int( splitAddr[ 2 ] )
         self.address = self.srTeEndpoint

   def overrideHierarchy( self, data ):
      global peerListObject
      readNext = True

      if 'reason_not_ucmp' in data:
         reason = data.pop( 'reason_not_ucmp' )
         if reason == UCMP_NO_LINK_BANDWIDTH:
            self.notUcmpEligibleReason = 'noLinkBWOnAllPaths'
         elif reason == UCMP_MAX_EXCEEDED:
            self.notUcmpEligibleReason = 'maxUcmpExceeded'

      if not self.bgpRoutePaths:
         if self._routeKey is None:
            self._routeKey = self.getKey( data )
         ipAddr = data.get( 'addrv6' )
         if ipAddr is not None:
            self.address = ipAddr
            del data[ 'addrv6' ]
            self.maskLength = data.pop( 'mlen' )
         else:
            ipAddr = data.get( 'addrv4' )
            if ipAddr is not None:
               self.address = data[ 'addrv4' ]
               del data[ 'addrv4' ]
               self.maskLength = data.pop( 'mlen' )
            else:
               self.overrideSrTeAddress( data )
         if 'total_paths' in data:
            self.totalPaths = data.pop( 'total_paths' )
         else:
            pathEntry = BgpRoutePath()
            pathEntry.processData( data )
            self.bgpRoutePaths.append( pathEntry )
         return ( data, readNext )
      elif 'contributor' in data or 'contributorv6' in data:
         pathEntry = self.bgpRoutePaths[-1]
         contributor = BgpRouteContributor()
         contributor.processData( data )
         pathEntry.routeDetail.bgpContributors.append( contributor )
         return ( data, readNext )
      elif 'segmentCount' in data:
         pathEntry = self.bgpRoutePaths[ -1 ]
         segmentList = BgpSrTePolicySegmentList()
         segmentList.processData( data )
         pathEntry.routeDetail.srTeTunnelEncap.segmentLists.append( segmentList )
         return ( data, readNext )
      elif self._routeKey == self.getKey( data ):
         if 'total_peers' in data:
            self.totalAdvertisedPeers = data.pop( 'total_peers' )
            peerListObject = None
         elif 'advertised_to_peer_v4' in data or 'advertised_to_peer_v6' in data:
            if 'advertised_to_peer_v4' in data:
               peerAddr = data.pop( 'advertised_to_peer_v4' )
            elif 'advertised_to_peer_v6' in data:
               peerAddr = data.pop( 'advertised_to_peer_v6' )
            routeTypeQueued = False
            rt = data.get( 'route_type' )
            if rt:
               if rt & BGP_RT_QUEUED:
                  routeTypeQueued = True
               data.pop( 'route_type' )

            nAddPaths = None
            if 'n_addpaths' in data:
               nAddPaths = data.pop( 'n_addpaths' )
            if 'advertised_to_pg_name' in data:
               pgName = data.pop( 'advertised_to_pg_name' )
               if not pgName in self.bgpAdvertisedPeerGroups:
                  self.bgpAdvertisedPeerGroups[ pgName ] = \
                        BgpRouteAdvertisedPeerListEntry()
               peerListObject = self.bgpAdvertisedPeerGroups[ pgName ]
            elif peerAddr and peerListObject is None:
               if not self.bgpAdvertisedPeers:
                  self.bgpAdvertisedPeers = BgpRouteAdvertisedPeerListEntry()
               peerListObject = self.bgpAdvertisedPeers

            if peerListObject:
               if nAddPaths:
                  peerAddr += " (%d add-path%s)" % \
                        (nAddPaths, 's' if nAddPaths > 1 else '')
               if routeTypeQueued:
                  peerListObject.peerList.append( '[' + peerAddr + ']' )
               else:
                  peerListObject.peerList.append( peerAddr )
         else:
            pathEntry = BgpRoutePath()
            pathEntry.processData( data )
            self.bgpRoutePaths.append( pathEntry )
         return ( data, readNext )
      else:
         return ( data, False )

   def renderEntry( self, prefix, detail, advRoutes, lpSeqNum, srTeSafi ):
      if detail and self.totalPaths:
         if srTeSafi:
            print "BGP routing table entry for " \
                  "Endpoint: %s, Color: %d, Distinguisher: %d" \
                  % ( self.srTeEndpoint, self.srTeColor, self.srTeDistinguisher )
         else:
            print "BGP routing table entry for %s" % prefix
         print " Paths: %u available" % self.totalPaths
      for pathModel in self.bgpRoutePaths:
         pathModel.renderEntry( prefix, detail, advRoutes, lpSeqNum, srTeSafi )
      if detail:
         if self.totalAdvertisedPeers is not None:
            if self.totalAdvertisedPeers > 0:
               print " Advertised to %d peers:" % self.totalAdvertisedPeers
            else:
               print " Not advertised to any peer"

         if self.bgpAdvertisedPeers and self.bgpAdvertisedPeers.peerList:
            for i in xrange( 0, len(self.bgpAdvertisedPeers.peerList),
                             ADV_TO_PEERS_ROW_LEN):
               pList = LINE_INDENT.join( self.bgpAdvertisedPeers.
                                         peerList[i:i+ADV_TO_PEERS_ROW_LEN] )
               print "    " + pList
         for pgName in self.bgpAdvertisedPeerGroups:
            print "  peer-group %s:" % pgName
            for i in xrange( 0,
                             len(self.bgpAdvertisedPeerGroups[ pgName ].peerList),
                             ADV_TO_PEERS_ROW_LEN):
               pList = LINE_INDENT.join( self.bgpAdvertisedPeerGroups[ pgName ].
                                         peerList[i:i+ADV_TO_PEERS_ROW_LEN] )
               print "    " + pList

         if self.notUcmpEligibleReason is not None:
            if self.notUcmpEligibleReason == 'noLinkBWOnAllPaths':
               print " Not UCMP: not all paths have link bandwidth information"
            elif self.notUcmpEligibleReason == 'maxUcmpExceeded':
               print " Not UCMP: number of resolved nexthops exceeds max UCMP config"

bgpRouteEntriesHelp = """Dictionary of BGP route entries indexed by the route prefix.
For SR-TE SAFI, the index is in the format:
   'Endpoint|Color|Distinguisher' """

class BgpRouteHeader( Model ):
   __revision__ = 3
   # At scale bgpRouteEntries can have millions of keys and so must be streamed. See
   # BUG182766 and BUG250019.
   __streamable__ = True

   asn = Str( help='Autonomous System Number' )
   routerId = Ip4Address( help='BGP Router Identity' )
   vrf = Str( help='VRF name' )
   lastProcessedSeqNum = Int( optional=True, 
         help='Last route sequence number acknowledged' )
   currentSeqNum = Int( optional=True, 
         help='Current route sequence number' )
   _detail = Int( optional=True, help='Detailed output is requested' )
   _advRoutes = Bool( optional=True, help='Advertised routes output is requested' )
   _srTeSafi = Bool( optional=True,
         help='Segment Routing Traffic Engineering SAFI output is requested' )
   bgpRouteEntries = GeneratorDict( keyType=str, valueType=BgpRouteEntry,
         help=bgpRouteEntriesHelp )

   def processData( self, data ):
      self.asn = data.pop( 'as' )
      self.routerId = data.pop( 'routerid' )
      if 'lp_vtime' in data:
         self.lastProcessedSeqNum = data.pop( 'lp_vtime' )
      if 'curr_vtime' in data:
         self.currentSeqNum = data.pop( 'curr_vtime' )
      if 'output_type' in data:
         outputType = data.pop( 'output_type' )
         if outputType == 1:
            self._advRoutes = True
         elif outputType == 2:
            self._srTeSafi = True
      if 'detail' in data:
         self._detail = data.pop( 'detail' )
      return data

   def degrade( self, dictRepr, revision ):
      if revision < 3:
         dictRepr[ 'bgpRouteEntries' ] = { \
               key: BgpRouteEntry().degrade( rtEntry, 1 ) for key, rtEntry in \
               dictRepr[ 'bgpRouteEntries' ].iteritems() }
      if revision < 2:
         asnInStr = dictRepr[ 'asn' ]
         dictRepr[ 'asn' ] = asnStrToNum( asnInStr ) 
      return dictRepr

   def renderForSrTeSafi( self, flagDescStr, originDescStr, asPathDescStr ):
      if not self._detail:
         print "Policy status codes: %s" % ( flagDescStr )
         print originDescStr
         print asPathDescStr
         print ""
         print srTePolicyEntryFmtStr % ( "", "Endpoint", "Color", "Distinguisher",
                                         "Next Hop", "Metric", "LocPref", "Weight",
                                         "Path" )

   def render( self ):
      print "BGP routing table information for VRF %s" % self.vrf
      print "Router identifier %s, local AS number %s" % ( self.routerId, self.asn )

      flagDescStr = "s - suppressed, * - valid, > - active, # - not installed, " \
                    "E - ECMP head, e - ECMP"
      originDescStr = "Origin codes: i - IGP, e - EGP, ? - incomplete"
      asPathDescStr = "AS Path Attributes: Or-ID - Originator ID, C-LST -" \
                      "Cluster List, LL Nexthop - Link Local Nexthop"

      if self._srTeSafi:
         self.renderForSrTeSafi( flagDescStr, originDescStr, asPathDescStr )
      elif not self._detail:
         print "Route status codes: %s" % ( flagDescStr )
         if self._advRoutes is None:
            print "                    S - Stale, c - Contributing to ECMP,", \
                  " b - backup, L = labeled-unicast"
            print "                    % - Pending BGP convergence"
         else:
            print "                    S - Stale, c - Contributing to ECMP,", \
                  " b - backup, L = labeled-unicast, q - Queued for advertisement"
         print originDescStr
         print asPathDescStr
         print ""
         print routeEntryFmtStr % ( "", "Network", "Next Hop", "Metric", "LocPref",
                                    "Weight", "Path" )
      elif self._advRoutes is not None:
         if self._advRoutes:
            print "Update wait-install is enabled"
            print "Last processed sequence number: %d," % self.lastProcessedSeqNum, \
                  " Current sequence number: %d" % self.currentSeqNum
         else:
            print "Update wait-install is disabled"
      elif self._detail == BGP_SUMMARY_PFX_DETAIL_HEADER:
         print "Route status: [a.b.c.d] - Route is ", \
               "queued for advertisement to peer."

      for prefix, model in self.bgpRouteEntries:
         model.renderEntry( prefix, self._detail, self._advRoutes,
               self.lastProcessedSeqNum, self._srTeSafi )

#-------------------------------------------------------------------------------
# "show ip bgp paths"
#-------------------------------------------------------------------------------
pathEntryFmtStr = "%-8s %-10s %s"
class BgpASPathEntry( Model ):
   referenceCount = Int( help='Number of references to the AS path' )
   med = Int( help='Multi Exit Discriminator for the route' )
   asPathInfo = Str( help='AS path string' )

   def processData( self, data ):
      if data.get( 'referenceCount' ) is None:
         self.referenceCount = 0
      if data.get( 'med' ) is None:
         self.med = 0
      return data

   def render( self ):
      print pathEntryFmtStr % ( self.referenceCount,
                                self.med,
                                self.asPathInfo )

class BgpASPathList( Model ):
   asPathList = GeneratorList( valueType=BgpASPathEntry,
                               help='List of AS path entries' )

   def render( self ):
      print pathEntryFmtStr % ( "Refcount", "Metric", "Path" )
      #"Refcount Metric     Path"
      for model in self.asPathList:
         model.render()

#-------------------------------------------------------------------------------
# "show bgp labeled-unicast tunnel [<tunnel-id>]"
#-------------------------------------------------------------------------------
DyTunIntfId = Tac.Type( 'Arnet::DynamicTunnelIntfId' )
class BgpLuTunnelTableEntry( TunnelTableEntry ):
   vias = List( valueType=Via, help="List of nexthops" )
   labels = List( valueType=str, help="Label stack" )
   contribution = Str( help="Contribution of tunnel entry to Tunnel RIB" )
   bgpMetric = Int( help="BGP Metric for tunnel entry" ) 
   bgpMetric2 = Int( help="BGP Metric 2 for tunnel entry" ) 
   bgpPref = Int( help="BGP Preference for tunnel entry" ) 
   bgpPref2 = Int( help="BGP Preference 2 for tunnel entry" ) 

   def getPrefStr( self ):
      return 'Yes' if self.contribution == 'contributing' else 'No'

   def renderBgpLuTunnelTableEntry( self, table, tunnelIndex ):
      labelsStr = '[ ' + ' '.join( self.labels ) + ' ]'
      nhStr = intfStr = '-'
      if self.vias:
         firstVia = self.vias[ 0 ]
         nhStr, intfStr = getNhAndIntfStrs( firstVia )
         # the desired label stack will either be in the entry CAPI model label 
         # stack or in the vias in the case of an LU Push entry via, but not both
         if 'labels' in self.vias[ 0 ]:
            labelsStr = '[ ' + ' '.join( self.vias[ 0 ][ 'labels' ] ) + ' ]'
      table.newRow( tunnelIndex, str( self.endpoint ), nhStr, intfStr, labelsStr,
                    self.getPrefStr(), str( self.bgpMetric ), str( self.bgpMetric2 ),
                    str( self.bgpPref ), str( self.bgpPref2 ) )

      # For vias after the first via, print a dash for these fields:
      # 'Index', 'Endpoint', 'Contributing', 'Metric', 'Metric2', 'Pref', and 'Pref2'
      # since they are common for each tunnel. For the 'Labels' field, we have to
      # check if the label-stack is stored in the tunnel entry's 'labels' attribute
      # or in each individual via. If each via has a label-stack, we have to print
      # each label-stack per via. Otherwise, we can assume the tunnel entry has
      # one label-stack, which we print only for the first via when rendering.
      for via in self.vias[ 1 : ]:
         nhStr, intfStr = getNhAndIntfStrs( via )
         if 'labels' in via:
            labelsStr = '[ ' + ' '.join( via[ 'labels' ] ) + ' ]'
         else:
            labelsStr = '-'
         table.newRow( '-', '-', nhStr, intfStr, labelsStr, '-', '-', '-', '-', '-' )

class BgpLuTunnelTable( Model ):
   __revision__ = 2
   entries = Dict( keyType=long, valueType=BgpLuTunnelTableEntry,
                   help="BGP LU tunnel table entries keyed by tunnel index" )

   def render( self ):
      headings = ( "Index", "Endpoint", "Nexthop/Tunnel Index", "Interface",
                   "Labels", "Contributing", "Metric", "Metric 2",
                   "Pref", "Pref 2" )
      fl = Format( justify='left' )
      table = createTable( headings )
      table.formatColumns( fl, fl, fl, fl, fl, fl, fl, fl, fl, fl )
      for tunnelIndex, bgpLuTunnelTableEntry in sorted( self.entries.iteritems() ):
         bgpLuTunnelTableEntry.renderBgpLuTunnelTableEntry( table, tunnelIndex )
        
      print table.output()

   # degrade function is called for,
   # 'show bgp labeled-unicast tunnel [<tunnel-idx>] | json revision 1'
   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for entry in dictRepr[ 'entries' ].itervalues():
            for via in entry[ 'vias' ]:
               if via: 
                  getViaModelFromViaDict( via ).degradeToV1( via )
      return dictRepr

#-------------------------------------------------------------------------------
# "show bgp labeled-unicast push [<tunnel-id>]"
#-------------------------------------------------------------------------------
class BgpLuPushTunnelTableEntry( Model ):
   vias = List( valueType=Via, help="List of nexthops" )
   labels = List( valueType=str, help="Label stack" )

   def renderBgpLuPushTunnelTableEntry( self, table, tunnelIndex ):
      labelsStr = '[ ' + ' '.join( self.labels ) + ' ]'
      nhStr = intfStr = '-'
      if self.vias:
         firstVia = self.vias[ 0 ]
         nhStr, intfStr = getNhAndIntfStrs( firstVia )
         # the desired label stack will either be in the entry CAPI model label 
         # stack or in the vias in the case of an LU Push entry via, but not both
         if 'labels' in self.vias[ 0 ]:
            labelsStr = '[ ' + ' '.join( self.vias[ 0 ][ 'labels' ] ) + ' ]'
      table.newRow( tunnelIndex, nhStr, intfStr, labelsStr )
      for via in self.vias[ 1 : ]:
         nhStr, intfStr = getNhAndIntfStrs( via )
         if 'labels' in via:
            labelsStr = '[ ' + ' '.join( via[ 'labels' ] ) + ' ]'
         table.newRow( '-', nhStr, intfStr, labelsStr )

class BgpLuPushTunnelTable( Model ):
   __revision__ = 2
   entries = Dict( keyType=long, valueType=BgpLuPushTunnelTableEntry,
                   help="BGP LU Push entries keyed by tunnel index" )
   
   def render( self ):
      headings = ( "Index", "Nexthop", "Interface", "Labels" )
      fl = Format( justify='left' )
      table = createTable( headings )
      table.formatColumns( fl, fl, fl, fl )
      for tunnelIndex, lsTunnelTableEntry in sorted( self.entries.iteritems() ):
         lsTunnelTableEntry.renderBgpLuPushTunnelTableEntry( table, tunnelIndex )
        
      print table.output()

   # degrade function is called for,
   # 'show bgp labeled-unicast push [<tunnel-idx>] | json revision 1'
   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for entry in dictRepr[ 'entries' ].itervalues():
            for via in entry[ 'vias' ]:
               if via: 
                  getViaModelFromViaDict( via ).degradeToV1( via )
      return dictRepr

#-------------------------------------------------------------------------------
# "show ip[v6] bgp neighbor [<peerip>] update-error"
#-------------------------------------------------------------------------------
class BgpUpdateErrorPeerEntry( Model ):
   withdrawAttr = Str( help='Withdraw Attribute', optional=True )
   ignoreAttr = Str( help='Ignore Attribute', optional=True )
   disableAfiSafiAttr = Str( help='Disable AFI/SAFI Attribute', optional=True )
   withdrawBuffer = Str( help='Withdraw Update Buffer', optional=True )
   ignoreBuffer = Str( help='Ignore Attribute Update Buffer', optional=True )
   disableAfiSafiBuffer = Str( help='Disable AFI SAFI Update Buffer', optional=True )

   def getKey( self, data ):
      return peerKeyFromData( data )

   def processData( self, data ):
      if 'withdraw_attr' in data:
         data[ 'withdrawAttr' ] = data[ 'withdraw_attr' ]
         del data[ 'withdraw_attr' ]
      if 'ignore_attr' in data:
         data[ 'ignoreAttr' ] = data[ 'ignore_attr' ]
         del data[ 'ignore_attr' ]
      if 'disable_afi_safi_attr' in data:
         data[ 'disableAfiSafiAttr' ] = data[ 'disable_afi_safi_attr' ]
         del data[ 'disable_afi_safi_attr' ]
      if 'withdraw_buffer' in data:
         data[ 'withdrawBuffer' ] = data[ 'withdraw_buffer' ]
         del data[ 'withdraw_buffer' ]
      if 'attr_ignore_buffer' in data:
         data[ 'ignoreBuffer' ] = data[ 'attr_ignore_buffer' ]
         del data[ 'attr_ignore_buffer' ]
      if 'disable_afi_safi_buffer' in data:
         data[ 'disableAfiSafiBuffer' ] = data[ 'disable_afi_safi_buffer' ]
         del data[ 'disable_afi_safi_buffer' ]
      return data

   def renderEntry( self, peer ):
      print 'Neighbor:', peer
      if self.withdrawAttr:
         print 'Malformed Action: Withdraw-Route Attribute:', self.withdrawAttr
         if self.withdrawBuffer:
            print 'Update Buffer:', self.withdrawBuffer
      if self.ignoreAttr:
         print 'Malformed Action: Attribute-Ignore Attribute:', self.ignoreAttr
         if self.ignoreBuffer:
            print 'Update Buffer:', self.ignoreBuffer
      if self.disableAfiSafiAttr:
         print 'Malformed Action: Disable AFI/SAFI Attribute:' \
               , self.disableAfiSafiAttr
         if self.disableAfiSafiBuffer:
            print 'Update Buffer:', self.disableAfiSafiBuffer

helpStr = 'Dictionary of BGP Update Error entries indexed by the peer address'
class BgpNeighborUpdateError( Model ):
   __revision__ = 2

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         dictRepr = _degradePeersDict( dictRepr )
      return dictRepr

   vrf = Str( help='VRF Name' )
   peers = GeneratorDict( valueType=BgpUpdateErrorPeerEntry, help=helpStr )

   def render( self ):
      print 'BGP update-error information for VRF', self.vrf
      # As self.peers is of GeneratorDict we cannot sort it in line, we must generate
      # the entire dictionary, then sort the keys.
      peers = { addr : peer for addr, peer in self.peers }
      for key in sorted( PeerConfigKey( addr ) for addr in peers ):
         peers[ key.stringValue ].renderEntry( key.stringValue )

#---------------------------------------------------------------------------------
# "show bgp update-group [index | addr] [vrf]
#---------------------------------------------------------------------------------
class BgpUpdateGroupPeerEntry( Model ):
   __revision__ = 1
   inSync = Bool( optional=True, help='Peer is in-sync' )
   queuedAdv = Int( optional=True, help='Number of queued advertisements' )

   def getKey( self, data ):
      return peerKeyFromData( data )

   def processData( self, data ):
      if 'inSync' in data:
         data[ 'inSync' ] = True
      return data

helpStr = 'Dictionary of BGP Update Group entries indexed by peer address'
outDelayApplyHelp = '''Announcment types to which out delay is applied.
  newRoutes   - Out delay is applied only to new routes
  newRoutesChanges - Out delay is applied to to new routes and changes
  newRoutesChangesWithdrawals - Out delay is applied to new routes, changes,
and withdrawals'''
class BgpUpdateGroup( Model ):
   __revision__ = 1
   peerType = Enum( values=( 'external', 'internal' ), help='Peer type' )
   confedExternalPeer = Bool( optional=True, help='eBGP confederation peer' )
   membersTotal = Int( help='Number of members' )
   membersInSync = Int( help='Number of in-sync members' )
   routeMapOut = Str( optional=True, help='Outbound policy' )
   routeMapOutV4Uni = Str( optional=True, help='Outbound policy for IPv4 unicast' )
   routeMapOutV6Uni = Str( optional=True, help='Outbound policy for IPv6 unicast' )
   routeMapOutVpn4 = Str( optional=True, help='Outbound policy for VPN4' )
   routeMapOutDefOri = Str( optional=True,
                            help='Outbound policy of default originate' )
   routeMapOutDefOriV4Uni = \
                  Str( optional=True,
                       help='Outbound policy of default originate for IPv4 unicast' )
   routeMapOutDefOriV6Uni = \
                  Str( optional=True,
                       help='Outbound policy of default originate for IPv6 unicast' )
   routeMapOutMaint = Str( optional=True,
                           help='Outbound policy for maintenance mode' )
   prefixList = Str( optional=True, help='Prefix list' )
   prefixListV4Uni = Str( optional=True,
                          help='Prefix List for IPv4 unicast' )
   prefixListV6Uni = Str( optional=True,
                          help='Prefix List for IPv6 unicast' )
   outDelay = Int( optional=True, help='Out delay time' )
   outDelayApply = Enum( values=( 'newRoutes', 'newRoutesChanges',
                                  'newRoutesChangesWithdrawals' ),
                         optional=True, help=outDelayApplyHelp )
   outdelayLbw = Int( optional=True, help='Out delay time for linkband width' )
   exportLocalPref = Int( optional=True, help='Local preference to send' )
   metricOut = Int( optional=True, help='Metric to send' )
   rrClient = Bool( optional=True, help='Route Reflector Client' )
   nexthopSelf = Bool( optional=True, help='Export self as nexthop' )
   prependOwnDisabled = Bool( optional=True, help='Do not prepend own AS number' )
   removePrivateAs = Bool( optional=True, help='Remove private as numbers' )
   removePrivateAsAlways = \
         Bool( optional=True, help='Always remove private AS numbers' )
   removePrivateAsReplace = \
         Bool( optional=True,
               help='Replace private AS number with local AS number' )
   sendCommunity = Bool( optional=True, help='Send community attribute' )
   sendExtendedCommunity = \
         Bool( optional=True, help='Send extended community attribute' )
   sendLargeCommunity = Bool( optional=True, help='Send large community attribute' )
   routeRefreshCap = Bool( optional=True, help='Route refresh capability' )
   localAs = Int( optional=True, help='Local as number' )
   mpBgpAfiSafiCap = Enum( values=AFI_SAFI_LIST,
                           optional=True, help='MPBGP AFI SAFI capability' )
   mpBgpV4UniCap = Bool( optional=True, help='MPBGP capability IPv4 unicast' )
   mpBgpV6UniCap = Bool( optional=True, help='MPBGP capability IPv6 unicast' )
   mpBgpV4LabelCap = Bool( optional=True,
                           help='MPBGP capability IPv4 label unicast' )
   mpBgpV6LabelCap = Bool( optional=True,
                           help='MPBGP capability IPv6 label unicast' )
   mpBgpV4Vpn4Cap = Bool( optional=True, help='MPBGP capability IPv4 VPN4' )
   apSendCap = Bool( optional=True,
                          help='Additional path send capability' )
   apSendV4UniCap = Bool( optional=True,
                          help='IPv4 unicast additional path send capability' )
   apSendV6UniCap = Bool( optional=True,
                       help='IPv6 unicast additional path send capability' )
   defaultOriginate = \
         Enum( values=( 'enabled', 'always' ),
               optional=True, help='Advertise a default route to the neighbor' )
   sixPe = Bool( optional=True, help='6PE activated for the neighbor' )
   reflectedRoutesAttributesState = Enum( values=(
      'notPreserved',
      'preserved',
      'alwaysPreserved',
      ), optional=True, help='State of attributes of the reflected routes' )
   nexthopUnchanged = \
      Bool( optional=True,
         help='Preserve original next hop while advertising routes to eBGP peers' )
   negotiated4byteAs = Bool( optional=True, help='Four-byte ASN negotiated' )
   extendedNextHop = Enum( values=(
      'disabled',
      'enabled',
      'enabledWithOriginate',
      ), optional=True, help='Extended next hop capability' )
   rtMembershipConstrain = \
      Bool( optional=True,
            help='Constrain route distribution based on peer RT membership' )
   sendOriginAsValidity = \
         Bool( optional=True, help='Send origin AS validation state' )
   missingPolicyAction = \
         Enum( values=( 'permit', 'deny', ),
               optional=True, help='Missing policy action' )
   neighborAs = Int( optional=True, help='Neighbor AS number' )
   replaceAs = Int( optional=True, help='Replace AS number' )
   defaultMed = Int( optional=True, help='Default MED' )
   advRoutes = Int( help='Number of advertised routes' )
   queuedAdv = Int( help='Number of queued route advertisements' )
   peers = GeneratorDict( valueType=BgpUpdateGroupPeerEntry, help=helpStr )

   def getKey( self, data ):
      if 'ugIndex' in data:
         key = data[ 'ugIndex' ]
      assert key is not None
      del data[ 'ugIndex' ]
      return key

   def processData( self, data ):
      if 'peerType' in data:
         data[ 'peerType' ] = 'internal' if data[ 'peerType' ] == '\x01' \
                              else 'external'
      if 'rrClient' in data:
         data[ 'rrClient' ] = True
      if 'nexthopSelf' in data:
         data[ 'nexthopSelf' ] = True
      if 'removePrivateAs' in data:
         data[ 'removePrivateAs' ] = True
      if 'sendCommunity' in data:
         data[ 'sendCommunity' ] = True
      if 'routeRefreshCap' in data:
         data[ 'routeRefreshCap' ] = True
      if 'mpBgpV4UniCap' in data:
         data[ 'mpBgpV4UniCap' ] = True
      if 'mpBgpV4LabelCap' in data:
         data[ 'mpBgpV4LabelCap' ] = True
      if 'mpBgpV6LabelCap' in data:
         data[ 'mpBgpV6LabelCap' ] = True
      if 'mpBgpV4Vpn4Cap' in data:
         data[ 'mpBgpV4Vpn4Cap' ] = True
      if 'mpBgpV6UniCap' in data:
         data[ 'mpBgpV6UniCap' ] = True
      if 'apSendV4UniCap' in data:
         data[ 'apSendV4UniCap' ] = True
      if 'apSendV6UniCap'in data:
         data[ 'apSendV6UniCap' ] = True
      odApply = data.get( 'outDelayApply' )
      if odApply is not None:
         odApplyStr = 'newRoutes'
         if odApply != BGP_OUTDELAY_APPLY_INITIAL:
            odApplyStr += 'Changes'
            if odApply == BGP_OUTDELAY_APPLY_CHANGES_WITHDRAWALS:
               odApplyStr += 'Withdrawals'
         data[ 'outDelayApply' ] = odApplyStr
      if 'prependOwnDisabled' in data:
         data[ 'prependOwnDisabled' ] = True
      return data

helpStr = 'Dictionary of Bgp update groups by update group index'

class BgpUpdateGroups( Model ):
   __revision__ = 1
   updateGroups = GeneratorDict( keyType=long, valueType=BgpUpdateGroup,
                                 help=helpStr )

# Model support for:
# show ip bgp neighbors <ip> advertised-routes queued-withdrawals ...
# show ipv6 bgp peers <ip> advertised-routes queued-withdrawals ...
class BgpQueuedWithdrawal( Model ):
   address = IpGenericAddress( help='Address of route being withdrawn' )
   maskLength = Int( help='Mask length of route being withdrawn' )
   localLabel = Int( optional=True,
                     help='Local MPLS label of route being withdrawn' )
   pathId = Int( optional=True, help='Path ID of route being withdrawn' )
   rxAfiSafi = Str( help='Received AFI SAFI of route being withdrawn' )
   secondsRemaining = Int( help='Seconds remaining until route is withdrawn' )

class BgpQueuedWithdrawals( Model ):
   __streamable__ = True

   localAsn = Str( help='Local Autonomous System Number' )
   localRouterId = Ip4Address( help='Local BGP Router Identity' )
   vrf = Str( help='VRF name' )
   peerAddress = Str( help='BGP peer address' )
   bgpQueuedWithdrawals = GeneratorList( valueType=BgpQueuedWithdrawal,
                                         help="List of paths pending withdrawal "
                                         "due to out-delay" )

#########
# Models for "show bgp neighbors history"
#########

class NeighborHistoryRecordModel( Model ):
   """
   Models a single event.
   """
   event = Enum( values=( 'connectFailure', ), help='Type of recorded event' )
   connectionType = Enum( values=( 'static', 'dynamic', 'interface' ),
                          help='Type of peer connection' )
   asn = Int( help='Peer remote AS number' )
   time = Float( help='UTC timestamp of event' )
   message = Str( help='Event description' )

class NeighborHistoryPeerModel( Model ):
   """
   Models a peer along with its events.
   """
   events = List( valueType=NeighborHistoryRecordModel, help='Event history' )

class NeighborHistoryVrfModel( Model ):
   """
   Models a VRF in the history
   """
   peers = Dict( keyType=IpGenericAddress, valueType=NeighborHistoryPeerModel,
      help='A dictionary of peer histories keyed by peer address' )

class ShowNeighborsHistoryModel( Model ):
   """
   Models the whole peer history. 
   """
   vrfs = Dict( valueType=NeighborHistoryVrfModel,
      help='A dictionary of peer histories keyed by VRF name.' )

##########
# Models for "show bgp convergence"
##########

class ShowBgpConvergenceVrfConfigModel( Model ):
   """
   Configured options for VRF convergence
   """
   convergenceTimeout = Int( help="Convergence timeout in seconds." )
   slowPeerTimeout = Int( help="Slow peer timeout in seconds." )
   afiSafiWaitForConvergence = Dict( valueType=bool,
      help="Wait for convergence enabled for each AFI/SAFI" )
   convergenceUpdateSyncEnabled = Bool( help="Convergence based update "
                                             "synchronisation is enabled",
                                        optional=True )

class ShowBgpConvergenceVrfPeerStatsModel( Model ):
   """
   Counts peers on each state.
   """
   totalPeers = Int( help="Number of total peers" )
   establishedPeers = Int( help="Number of established peers" )
   pendingPeers = Int( help="Number of pending peers" )
   disabledPeers = Int( help="Number of disabled peers" )

class ShowBgpConvergenceVrfStatusModel( Model ):
   """
   Current status of VRF convergence
   """
   convergenceState = Enum( [
      "notInitialized",
      "convergenceIdle",
      "convergenceActive",
      "timeout",
      "converged"
      ], help="BGP convergence states" )
   afiSafiConvergenceStates = Dict( valueType=str,
                                    help="Convergence state per AFI/SAFI type",
                                    optional=True )
   lastEventTime = Float( help="Timestamp of last event since epoch",
                          optional=True )
   timeToConverge = Float( help="Seconds taken to converge",
                           optional=True )
   firstPeerUp = Bool( help="First peer is up", optional=True )
   firstPeerUpTime = Float( help="Timestamp since epoch when first peer came up",
                            optional=True )
   convergenceTimer = Bool( help="Convergence timer is running",
                            optional=True )
   convergenceTimerExpires = Float( help="Time since epoch when timer expires",
                                    optional=True )
   convergenceTimeout = Float( help="Convergence timeout in seconds.",
                               optional=True )
   slowPeerTimeout = Float( help="Slow peer timeout in seconds.",
                            optional=True )
   expectedPeersUp = Bool( help="Expected peers are up", optional=True )
   dynamicPeerGroups = Bool( help="Dynamic peering is configured",
                             optional=True )
   allIgpConverged = Bool( help="All IGP protocols have converged",
                           optional=True )
   outstandingEors = Int( help="Number of outstanding EORs", optional=True )
   outstandingKeepalives = Int( help="Number of outstanding keepalives",
                                optional=True )
   previouslyEstablishedPeers = Int( help="Number of previously established peers",
                                     optional=True )
   peers = Submodel( ShowBgpConvergenceVrfPeerStatsModel,
                     help="Convergence statistics" )

class ShowBgpConvergenceVrfModel( Model ):
   """
   Models VRF within "show bgp convergence"
   """
   config = Submodel( ShowBgpConvergenceVrfConfigModel,
      help="Convergence configration for VRF" )
   status = Submodel( valueType=ShowBgpConvergenceVrfStatusModel,
      help="Current convergence status for VRF" )
   peersNotConverged = Dict( keyType=IpGenericAddress, valueType=str,
      help="Current state of peers that still haven't converged in VRF,"
      " keyed by peer address" )

   def getKey( self, data ):
      key = data.pop( 'convergenceIndex', None )
      assert key is not None
      return key

   def overrideHierarchy( self, data ):
      if self.config is None:
         # Config
         config = ShowBgpConvergenceVrfConfigModel()
         config.convergenceTimeout = data.pop( 'confTimeout' )
         config.slowPeerTimeout = data.pop( 'confSlowPeerTimeout' )
         # Per afi safi wait for convergence only supported by ArBgp
         config.afiSafiWaitForConvergence = {}
         # Update sync enabled if confUpdateSync is '0x00' in data
         config.convergenceUpdateSyncEnabled = ord(
                                                data.pop( 'confUpdateSync' ) ) == 0
         self.config = config

      if self.status is None:
         # Status
         status = ShowBgpConvergenceVrfStatusModel()
         status.convergenceState = data.pop( 'convergenceState' )
         if 'lastEventTime' in data:
            status.lastEventTime = float( data.pop( 'lastEventTime' ) )
         if 'timeToConverge' in data:
            status.timeToConverge = float( data.pop( 'timeToConverge' ) )
         if 'firstPeerUp' in data:
            status.firstPeerUp = data.pop( 'firstPeerUp' )
         if 'firstPeerUpTime' in data:
            status.firstPeerUpTime = float( data.pop( 'firstPeerUpTime' ) )
         if 'convergenceTimer' in data:
            status.convergenceTimer = ord( data.pop( 'convergenceTimer' ) ) != 0
         if 'convergenceTimerExpires' in data:
            status.convergenceTimerExpires = float( data.pop(
                                                      'convergenceTimerExpires' ) )
         if 'convergenceTimeout' in data:
            status.convergenceTimeout = float( data.pop( 'convergenceTimeout' ) )
         if 'slowPeerTimeout' in data:
            status.slowPeerTimeout = float( data.pop( 'slowPeerTimeout' ) )
         if 'expectedPeersUp' in data:
            status.expectedPeersUp = ord( data.pop( 'expectedPeersUp' ) ) != 0
         if 'dynamicPeerGroups' in data:
            status.dynamicPeerGroups = ord( data.pop( 'dynamicPeerGroups' ) ) != 0
         if 'allIgpConverged' in data:
            # allIgpConverged is True if allIgpConverged is '0x01' in data
            status.allIgpConverged = ord( data.pop( 'allIgpConverged' ) ) == 1
         if 'outstandingEors' in data:
            status.outstandingEors = data.pop( 'outstandingEors' )
         if 'outstandingKeepalives' in data:
            status.outstandingKeepalives = data.pop( 'outstandingKeepalives' )
         if 'previouslyEstablishedPeers' in data:
            status.previouslyEstablishedPeers = data.pop(
                                                   'previouslyEstablishedPeers' )
         # Status Peer Statistics
         stats = ShowBgpConvergenceVrfPeerStatsModel()
         stats.totalPeers = data.pop( 'totalPeers' )
         stats.pendingPeers = data.pop( 'pendingPeers' )
         stats.disabledPeers = data.pop( 'disabledPeers' )
         stats.establishedPeers = data.pop( 'establishedPeers' )
         status.peers = stats

         self.status = status

         return ( data, True )
      elif 'notConvergedAddrv4' in data or 'notConvergedAddrv6' in data:
         if 'notConvergedAddrv6' in data:
            addr = data.pop( 'notConvergedAddrv6' )
         else:
            addr = data.pop( 'notConvergedAddrv4' )
         self.peersNotConverged[ addr ] = data.pop( 'notConvergedState', None )
         return ( data, True )
      else:
         return ( data, False )

class ShowBgpConvergenceModel( Model ):
   """
   Models for "show bgp convergence"
   """
   vrfs = Dict( valueType=ShowBgpConvergenceVrfModel,
      help="A dictionary of convergence status keyed by VRF name." )

##########
# Models for "show bgp instance"
##########

class ShowBgpInstanceRedistProtoModel( Model ):
   """
   Models for redistributed routes.
   """
   proto = Str( help="Source protocol name" )
   routeMap = Str( optional=True, help="Route map name" )
   includeLeaked = Bool( optional=True,
                   help='Include leaked routes in redistribution' )

class ShowBgpInstanceAfiSafiRouteLists( Model ):
   """
   Possible route lists for Target Imports/Exports
   """
   v4u = List( valueType=str, optional=True,
      help="List of routes for IPv4 Unicast." )
   v4m = List( valueType=str, optional=True,
      help="List of routes for IPv4 Multicast." )
   v4lu = List( valueType=str, optional=True,
      help="List of routes for IPv4 Labeled Unicast." )
   v4SrTe = List( valueType=str, optional=True,
      help="List of routes for IPv4 SR-TE." )
   mplsVpnV4u = List( valueType=str, optional=True,
      help="List of routes for IPv4 MPLS VPN." )
   v4Flowspec = List( valueType=str, optional=True,
      help="List of routes for IPv4 Flowspec." )
   v6u = List( valueType=str, optional=True,
      help="List of routes for IPv6 Unicast." )
   v6m = List( valueType=str, optional=True,
      help="List of routes for IPv6 Multicast." )
   v6lu = List( valueType=str, optional=True,
      help="List of routes for IPv6 Labeled Unicast." )
   v6SrTe = List( valueType=str, optional=True,
      help="List of routes for IPv6 SR-TE." )
   mplsVpnV6u = List( valueType=str, optional=True,
      help="List of routes for IPv6 MPLS VPN." )
   v6Flowspec = List( valueType=str, optional=True,
      help="List of routes for IPv6 Flowspec." )
   evpn = List( valueType=str, optional=True,
      help="List of routes for EVPN." )
   linkState = List( valueType=str, optional=True,
      help="List of routes for BGP Link State." )
   dps = List( valueType=str, optional=True,
      help="List of routes for dynamic path selection." )

class BgpLsImportDbEntry( Model ):
   """
   Models Link State Import Database configs
   """
   source = Enum( values=( 'isis', ), help="Source protocol" )
   vrf = Str( help="VRF of source protocol's instance" )

class ShowBgpInstanceProducerStatsModel( Model ):
   """
   Models producer summary stats
   """
   numNodes = Int( help="Number of LSDB nodes." )
   numLinks = Int( help="Number of LSDB links." )
   numIpv4Prefixes = Int( help="Number of LSDB IPv4 prefixes." )
   numIpv6Prefixes = Int( help="Number of LSDB IPv6 prefixes." )

class NhResolutionRibProfile( Model ):
   """
   This is essentially the CAPI model for a Rib::ResolutionProfile, but with some
   addons (route-map, defaults) that are specific to BGP.
   """
   resolutionMethods = List( valueType=ResolutionRib, optional=True,
         help="List of resolution methods" )
   routeMap = Str( help="Route map to determine resolution profile", optional=True )
   defaults = List( valueType=ResolutionRib, optional=True,
                    help="List of default resolution methods" )

   @staticmethod
   def degradeResolutionMethods( dictRepr ):
      """
      Convert a resolution RIB JSON object into the canonical string representation
      for CAPI degradation and simpler test validation. E.g.

         "resolutionRibs": {
            "resolutionMethods": [
               {
                  "ribType": "tunnel",
                  "colored": true,
                  "name": "system-colored-tunnel-rib"
               },
               {
                  "ribType": "tunnel",
                  "colored": false,
                  "name": "system-tunnel-rib"
               },
               {
                  "ribType": "ip",
                  "name": "system-unicast-rib"
               }
            ]
         }

      Becomes:

         [
         "tunnel-rib colored system-colored-tunnel-rib",
         "tunnel-rib system-tunnel-rib",
         "system-unicast-rib"
         ]
      """
      methods = dictRepr.get( 'resolutionMethods', [] )
      config = ResolutionRibProfileConfig.fromJson( methods )
      assert config is not None
      return [ method.stringValue()
               for method in config.resolutionMethod.values()
               if method.stringValue() ]

class ShowBgpInstanceAfiSafiConfigModel( Model ):
   """
   Models configuration per AFI/SAFI
   """
   __revision__ = 2
   waitForConvergence = Bool( optional=True,
      help="Convergence based update synchornization enable." )
   routeDistinguisher = Int( optional=True, help="Route distinguisher." )
   routeTargetImports = Submodel( valueType=ShowBgpInstanceAfiSafiRouteLists,
      optional=True, help="Route targets to import." )
   routeTargetExports = Submodel( valueType=ShowBgpInstanceAfiSafiRouteLists,
      optional=True, help="Route targets to export." )
   routeMapImports = Dict( valueType=str, optional=True,
      help="Route maps to apply on import, keyed by AFI/SAFI name." )
   routeMapExports = Dict( valueType=str, optional=True,
      help="Route maps to apply on export, keyed by AFI/SAFI name." )
   ipLookupLocalLabel = Str( optional=True,
      help="Local IP lookup MPLS VRF label." )
   additionalPathsInstallation = Bool( optional=True,
      help="Additional paths installation enabled." )
   additionalPathsInstallationEcmpPrimary = Bool( optional=True,
      help="Allow additional path with ECMP primary path." )
   resolutionRibs = Submodel( valueType=NhResolutionRibProfile, optional=True,
         help="Resolution RIBs." )
   mplsResolutionRibs = Submodel( valueType=NhResolutionRibProfile, optional=True,
      help="MPLS resolution RIBs." )
   sixPeResolutionRibs = Submodel( valueType=NhResolutionRibProfile, optional=True,
      help="6PE resolution RIBs." )
   vxlanResolutionRibs = Submodel( valueType=NhResolutionRibProfile, optional=True,
      help="VXLAN resolution RIBs." )
   extendedNextHopCapability = Bool( optional=True,
      help="Extended next-hop capability" )
   ipLuTargetRibs = Str( optional=True,
      help="IP LU Target RIBs." )
   ipLuRibRouteMap = Str( optional=True,
      help="IP LU RIB route map." )
   ipLuTunnelRibRouteMap = Str( optional=True,
      help="IP LU tunnel RIB route map." )
   redistributedRoutes = List( valueType=ShowBgpInstanceRedistProtoModel,
      optional=True, help="Redistributed routes into BGP." )
   aigpSessionIbgp = Bool( optional=True,
      help="AIGP session is enabled for iBGP peers." )
   aigpSessionConfed = Bool( optional=True,
      help="AIGP session is enabled for confed peers." )
   aigpSessionEbgp = Bool( optional=True,
      help="AIGP session is enabled for eBGP peers." )
   roles = List( optional=True, valueType=str,
        help="Link State roles. Can be any combination of 'producer', "
        "'propagator', and 'consumer'. Currently only 'producer' is supported "
        "in EOS" )
   importedDatabases = GeneratorDict( optional=True, keyType=str,
                           valueType=BgpLsImportDbEntry,
                           help='Dictionary of BGP Link State imported LSDBs '
                             'keyed by IGP instance name' )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # Degrade the follow attributes:
         #
         #    * resolutionRibs
         #    * mplsResolutionRibs
         #    * sixPeResolutionRibs
         #    * vxlanResolutionRibs
         #
         # From the following Submodel:
         #
         # <attr> : {
         #       'resolutionMethods': [
         #                    { 'colored': True,
         #                      'name': 'system-colored-tunnel-rib',
         #                      'ribType': 'tunnel' },
         #                      { 'colored': False,
         #                      'name': 'system-tunnel-rib',
         #                      'ribType': 'tunnel' },
         #                      { 'name': 'system-unicast-rib',
         #                      'ribType': 'ip' }
         #                      ]
         #              }
         #
         #
         # To the original list representation:
         #
         # <attr> : [
         #            'tunnel-rib colored system-colored-tunnel-rib',
         #            'tunnel-rib system-tunnel-rib',
         #            'system-unicast-rib',
         #          ]
         degradedKeys = [
               'resolutionRibs',
               'mplsResolutionRibs',
               'sixPeResolutionRibs',
               'vxlanResolutionRibs'
               ]
         for key in degradedKeys:
            if key not in dictRepr:
               continue
            dictRepr[ key ] = \
                  NhResolutionRibProfile.degradeResolutionMethods( dictRepr[ key ] )
      return dictRepr

class ShowBgpInstanceAfiSafiStatusModel( Model ):
   """
   Models status per AFI/SAFI
   """
   producerStats = Submodel( optional=True,
         valueType=ShowBgpInstanceProducerStatsModel,
         help="Producer statistics." )

class ShowBgpInstanceAfiSafiConvergenceModel( Model ):
   """
   Models peer convergence status
   """
   converged = Bool( help="AFI/SAFI converged." )
   timeToConverge = Float( optional=True,
      help="Time taken to converge." )
   outstandingEors = Int( help="Number of outstanding End-of-RIB's." )
   outstandingKeepalives = Int( help="Number of outstanding Keepalives." )

class ShowBgpInstancePeersModel( Model ):
   """
   Models peer statistics
   """
   totalPeers = Int( help="Number of peers." )
   staticPeers = Int( help="Number of static peers." )
   dynamicPeers = Int( help="Number of dynamic peers." )
   establishedPeers = Int( help="Number of established peers." )

class ShowBgpInstanceVrfModel( Model ):
   """
   Models each VRF within "show bgp instance".
   """
   __revision__ = 2
   adjRibInsAwaitingCleanup = Int( help="Number of Adj-RIB-Ins awaiting cleanup." )
   afiSafiConfig = Dict( valueType=ShowBgpInstanceAfiSafiConfigModel,
      help="Configuration keyed by AFI/SAFI name." )
   afiSafiState = Dict( optional=True, valueType=ShowBgpInstanceAfiSafiStatusModel,
      help="Status keyed by AFI/SAFI name." )
   afiSafiConvergence = Dict( valueType=ShowBgpInstanceAfiSafiConvergenceModel,
      help="Convergence status keyed by AFI/SAFI name.", optional=True )
   bgpIpv4Listening = Bool( help="BGP is listening on IPv4." )
   bgpIpv4ListenPort = Int( help="Port BGP is listening on IPv4",
      optional=True )
   bgpIpv6Listening = Bool( help="BGP is listening on IPv6." )
   bgpIpv6ListenPort = Int( help="Port BGP is listening on IPv6",
      optional=True )
   converged = Bool( help="BGP has converged." )
   convergenceTimeout = Int( optional=True, help="Convergence timeout in seconds." )
   convergenceTimer = Bool( help="Convergence timer enabled." )
   convergenceTimerExpires = Float( optional=True,
      help="Convergence expiration time since epoch." )
   eorTimeout = Int( help="End-of-RIB timeout in seconds." )
   fourByteAsn = Bool( help="four-byte ASN enabled." )
   gracefulRestart = Bool( help="Graceful restart mode enabled." )
   gracefulRestartHelper = Bool( "Graceful restart helper mode." )
   gracefulRestartTimeout = Int( help="Graceful restart timeout in seconds." )
   localAs = Int( help="Local AS number." )
   outstandingKeepalives = Int( help="Number of outstanding keepalives." )
   outstandingEors = Int( help="Number of outsdanding End-of-RIB's." )
   peerMacResolutionTimeout = Int( help="Peer MAC resolution in seconds." )
   peers = Submodel( ShowBgpInstancePeersModel, help="Peer statistics." )
   reflectedRoutesAttributesState = Enum( [
      "notPreserved",
      "preserved",
      "alwaysPreserved",
      "unknown"
   ], help="State of attributes of the reflected routes." )
   routerId = Ip4Address( help="Router ID." )
   slowPeerTimeout = Int( help="Slow peer timeout in seconds." )
   timeToConverge = Float( help="Time taken to BGP convergence in seconds.",
      optional=True )
   ucmpMode = Enum( [ "ucmpMode1", "ucmpDisabled" ], help="UCMP mode." )
   ucmpMaximumPaths = Int( help="UCMP maximum paths.", optional=True )
   ucmpDeviation = Float( help="UCMP deviation.", optional=True )
   ucmpLinkBwDelay = Int( help="UCMP link bandwidth delay.", optional=True )

class ShowBgpInstanceModel( Model ):
   """
   Models for "show bgp instance".
   """
   __revision__ = 2
   vrfs = Dict( valueType=ShowBgpInstanceVrfModel,
                help="Instance information, keyed by VRF name." )

class RouteTarget( _TacAttributeType ):
   '''A route target.'''
   tacType = Tac.Type( 'Routing::Bgp::RouteTarget' )
   realType = ( tacType, str )

   def _createTacValue( self, value ):
      rt = self.tacType()
      rt.stringValue = value
      return rt
