# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import Arnet
from ArnetModel import IpGenericPrefix
import ArnetLib
from CliModel import Model, Dict, DeferredModel, Int, cliPrinted
from CliToken.RoutingBgpShowCliTokens import (
      bgpAfterShow,
      detail,
      neighbors,
      routeTypeMatcher,
)
import Tracing
from BgpLib import PeerConfigKey
import Toggles.BgpCommonToggleLib

from RoutingBgpShowCli import (
        ArBgpShowOutput,
        arBgpShowCmdDict,
        getCommunityValuesScalarList,
        routeSummaryVrfModel,
        validatePosixRegex,
        validateAspRegex,
)

from ArBgpCli import (
        ArBgpCliCommand,
        doShowAfBgpInternal,
        flattenShowBgpNlriAf,
        mplsLabelSafiArg,
        rtMembershipAfterShowBgp,
)
import BasicCli
import ShowCommand
from CliPlugin.RoutingBgpCli import V4V6PeerKeyCliExpression
from CliPlugin.RtMembershipCliModels import rtMembershipRoutesVrfModel

traceHandle = Tracing.Handle( 'ArBgpCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

class BgpCliHelperCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      assert 'keepalive' not in kwargs
      vrfName = kwargs.pop( 'vrfName', None )
      nlriAfiSafi = kwargs.pop( 'afi', '' )
      safi = kwargs.pop( 'safi', '' )
      if safi == 'unicast':
         nlriAfiSafi += "Unicast"
      elif safi == 'mplslabel':
         nlriAfiSafi += "Lu"
      elif safi == 'multicast':
         nlriAfiSafi += "Multicast"
      extraNlriAfiSafis = kwargs.pop( 'extraNlriAfiSafis', None )
      super( BgpCliHelperCommand, self ).__init__( mode,
                                               command,
                                               vrfName=vrfName,
                                               nlriAfiSafi=nlriAfiSafi,
                                               extraNlriAfiSafis=extraNlriAfiSafis,
                                               transportAfi=None,
                                               disableFork=True )
      addr = kwargs.pop( 'addr', None )
      if addr is not None:
         self.addParam( 'addr', str( addr ) )

      self.mode = mode

      if self.mode.session_.outputFormat_ == 'json':
         self.addParam( 'json' )

      for k, v in kwargs.iteritems():
         if v:
            self.addParam( k, v )

   def addNlriAfiSafi( self, nlriAfiSafi ):
      """Overloaded method from ArBgpCliCommand  base class"""
      self.addParam( 'nlriAfiSafi', nlriAfiSafi )

   def run( self, **kwargs ):
      AgentCommandRequest.runCliPrintSocketCommand( self.entityManager,
                                                    'BgpCliHelper',
                                                    self.command,
                                                    self.paramString(),
                                                    self.mode,
                                                    keepalive=True,
                                                    **kwargs )
      # Return the correct capi model for converted commands and return
      # a "dummy" capi model for all unconverted commands.
      if self.command == 'show bgp segment-routing':
         return PrefixSegments
      elif 'show bgp rt-membership' in self.command:
         return cliPrinted( rtMembershipRoutesVrfModel )
      else:
         return cliPrinted( routeSummaryVrfModel )

def dictReplace( what, where ):
   for k, v in what.iteritems():
      assert k in where, "key {} not found in {}".format( k, where )
      where[ v ] = where[ k ]
      del where[ k ]

def registerShowBgpNlriAf( cmdName, argReplace=None, argsAdd=None ):
   def callback( mode, *args, **kwargs ):
      if argsAdd:
         kwargs.update( argsAdd )
      if isinstance( argReplace, dict ):
         dictReplace( argReplace, kwargs )
      elif callable( argReplace ):
         argReplace( kwargs )
      elif argReplace is not None:
         raise Exception( "Unexpected argReplace={}".format( argReplace ) )

      # BgpCliHelper expects pfxAddr as addr
      pfxAddr = kwargs.pop( "pfxAddr", None )
      if pfxAddr is not None:
         kwargs[ "addr" ] = pfxAddr

      mplsLabelSafiArg( kwargs )

      internal = kwargs.pop( "internal", False )
      if internal:
         return doShowAfBgpInternal( mode, **kwargs )

      if kwargs.get( 'afi' ) is None:
         return False

      convertPeerAddr( kwargs )
      if not validateRegex( mode, kwargs ):
         return False
      convertCommunityValues( mode, kwargs )
      convertLargeCommunityValues( kwargs, key="largeCommValues" )
      if 'routeType' not in kwargs:
         kwargs[ 'routeType' ] = 'routes'

      # Set ACR read timeout to 10m. In scale setup, BgpCliHelper can
      # iterate a huge number of paths without printing anything
      # and the default 120s timeout before ConfigAgent gives up
      # reading from the socket is not enough.
      bchCmd = BgpCliHelperCommand( mode, 'show bgp <af>', **kwargs )
      return bchCmd.run( timeout=600 )

   callback.__name__ = cmdName
   arBgpShowCmdDict[ cmdName ] = callback

def convertCommunityValues( mode, kwargs ):
   key = 'commValues'
   value = kwargs.get( key, None )
   if not value:
      return
   commValues = getCommunityValuesScalarList( value )
   # Using a custom separator that is not intepreted by the ACR
   kwargs[ key ] = "|".join( [ str( c ) for c in commValues ] )
   exact = kwargs.pop( "exact", None )
   if exact:
      kwargs[ "standardCommunitiesExactMatch" ] = exact

def convertLargeCommunityValues( kwargs, fromKwargs=None, key=None ):
   def asplain( largeCommValue ):
      """ Return large community value in AS-plain format """
      tokens = largeCommValue.split( ':' )
      tokens[ 0 ] = str( ArnetLib.asnStrToNum( tokens[ 0 ] ) )
      return ":".join( tokens )
   if fromKwargs is None:
      fromKwargs = kwargs
   value = fromKwargs.get( key, None )
   if not value:
      return
   # Using a custom separator that is not intepreted by the ACR
   kwargs[ 'largeCommValues' ] = "|".join( [ asplain( c ) for c in value ] )
   exact = fromKwargs.pop( "exact", None )
   if exact:
      kwargs[ "largeCommunitiesExactMatch" ] = exact

def convertPeerAddr( kwargs ):
   key = 'peerAddr'
   peerAddr = kwargs.get( key, None )
   llIntf = ''
   if peerAddr is not None:
      peerKey = PeerConfigKey( peerAddr )
      if peerKey.type == 'peerIpv4':
         peerAddr = peerKey.v4Addr
      elif peerKey.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         peerAddr = peerKey.v6Addr.stringValue
         llIntf = Arnet.IntfId( peerKey.llIntf ).stringValue
         kwargs[ 'llIntf' ] = llIntf
      kwargs[ key ] = peerAddr

def validateRegex( mode, kwargs ):
   commReg = kwargs.get( 'commRegex', None )
   aspReg = kwargs.get( 'aspRegex', None )
   validatedComm = validatePosixRegex( mode, commReg ) if commReg else True
   if aspReg:
      # If we have a valid as path regex, set which mode we are in too
      validatedAsp = validateAspRegex( mode, aspReg )
   else:
      validatedAsp = True
   return validatedComm and validatedAsp

registerShowBgpNlriAf(
        cmdName="doShowIpBgp",
        argsAdd={ "afi": "ipv4" } )

registerShowBgpNlriAf(
        cmdName="doShowIpBgpRoutesDetail",
        argsAdd={ "afi": "ipv4" } )

registerShowBgpNlriAf(
        cmdName="doShowIpBgpRoutesCommunities",
        argsAdd={ "afi": "ipv4" },
        argReplace={ "values": "commValues" } )

registerShowBgpNlriAf(
        cmdName="doShowIpBgpRoutesCommunitiesRegex",
        argReplace={ "regex" : "commRegex" },
        argsAdd={ "afi" : "ipv4" } )

registerShowBgpNlriAf(
        cmdName="doShowIpBgpLargeCommunity",
        argsAdd={ "afi" : "ipv4" },
        argReplace={ "values" : "largeCommValues" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6Bgp",
        argsAdd={ "afi": "ipv6" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6BgpRoutesDetail",
        argsAdd={ "afi": "ipv6" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6BgpRoutesCommunities",
        argsAdd={ "afi": "ipv6" },
        argReplace={ "values": "commValues" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6BgpLargeCommunity",
        argsAdd={ "afi" : "ipv6" },
        argReplace={ "values" : "largeCommValues" } )

#-------------------------------------------------------------------------------
# show ip[v6] bgp community-list <name> [exact] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
registerShowBgpNlriAf(
        cmdName="doShowIpBgpRoutesCommunityList",
        argsAdd={ "afi" : "ipv4" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6BgpRoutesCommunityList",
        argsAdd={ "afi" : "ipv6" } )

#-------------------------------------------------------------------------------
# show ip[v6] bgp large-community-list <name> [exact] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
registerShowBgpNlriAf(
        cmdName="doShowIpBgpRoutesLargeCommunityList",
        argsAdd={ "afi" : "ipv4" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6BgpRoutesLargeCommunityList",
        argsAdd={ "afi" : "ipv6" } )

#-------------------------------------------------------------------------------
# "show ip[v6] bgp regexp <regex> [vrf <vrfName>]
#-------------------------------------------------------------------------------
registerShowBgpNlriAf(
        cmdName="doShowIpBgpRegexp",
        argReplace={ "regex" : "aspRegex" },
        argsAdd={ "afi" : "ipv4" } )

registerShowBgpNlriAf(
        cmdName="doShowIpv6BgpRegexp",
        argReplace={ "regex" : "aspRegex" },
        argsAdd={ "afi" : "ipv6" } )

registerShowBgpNlriAf(
        cmdName="doShowIpBgpAndRegexp",
        argsAdd={ "afi" : "ipv4" } )

#-------------------------------------------------------------------------------
# "show (ip | ipv6) bgp neighbors <ip> [(ipv4 unicast) | (ipv6 unicast) |
#       (ipv4 multicast) | (ipv4 labeled-unicast) | (ipv6 labeled-unicast)]
#       (routes | advertised-routes | received-routes [filtered])
#       [<ip> | <prefix>] [longer-prefixes] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------

def handleShowIpBgpNeighborsRoutesArgs( kwargs ):
   if 'afi' not in kwargs:
      kwargs[ 'afi' ] = 'ipv4'
   if 'nbrAddr' in kwargs:
      dictReplace( { "nbrAddr" : "peerAddr" }, kwargs )
   addr = kwargs.pop( 'addr', None )
   if addr:
      kwargs[ 'peerAddr' ] = addr
   flattenShowBgpNlriAf( kwargs )
   if 'regex' in kwargs:
      dictReplace( { "regex" : "aspRegex" }, kwargs )
   values = kwargs.pop( 'values', None )
   if values:
      kwargs[ 'commValues' ] = values

registerShowBgpNlriAf(
      cmdName="doShowIpBgpNeighborsRoutes",
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpv6BgpNeighborsRoutes",
      argsAdd={ "afi" : "ipv6" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpBgpNeighborsReceivedRoutes",
      argsAdd={ "routeType": "received-routes" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpv6BgpNeighborsReceivedRoutes",
      argsAdd={ "afi" : "ipv6", "routeType" : "received-routes" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpBgpNeighborsPrefixAddr",
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpv6BgpNeighborsPrefixAddr",
      argsAdd={ "afi" : "ipv6" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpBgpNeighborsPrefixAddrRecvdRts",
      argsAdd={ "routeType": "received-routes" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpv6BgpNeighborsPrefixAddrRecvdRts",
      argsAdd={ "afi" : "ipv6", "routeType" : "received-routes" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

#-------------------------------------------------------------------------------
# "show (ip | ipv6) bgp neighbors <ip>
#       (routes | advertised-routes | received-routes [filtered])
#       community <communities> [exact] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
registerShowBgpNlriAf(
      cmdName="doShowIpBgpNeighborsCommunityRoutes",
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpv6BgpNeighborsCommunityRoutes",
      argsAdd={ "afi" : "ipv6" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpBgpNeighborsCommunityRecvdRoutes",
      argsAdd={ "routeType": "received-routes" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
      cmdName="doShowIpv6BgpNeighborsCommunityRecvdRoutes",
      argsAdd={ "afi" : "ipv6", "routeType" : "received-routes" },
      argReplace=handleShowIpBgpNeighborsRoutesArgs )

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> { received-routes [ filtered ] | routes |
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------

registerShowBgpNlriAf(
   cmdName="doShowIpBgpNbrRtsRegex",
   argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
   cmdName="doShowIpBgpNbrRecvdRtsRegex",
   argsAdd={ "routeType" : "received-routes" },
   argReplace=handleShowIpBgpNeighborsRoutesArgs )

#-------------------------------------------------------------------------------
# Deprecated command:
# "show ipv6 bgp neighbors <ip> { received-routes [ filtered ] | routes |
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#
# New version:
# "show ipv6 bgp peers <ip> { received-routes [ filtered ] | routes |
# advertised-routes } regexp <regex> [vrf <vrfName>]"
#-------------------------------------------------------------------------------

registerShowBgpNlriAf(
   cmdName="doShowIpv6BgpNbrRtsRegex",
   argsAdd={ "afi" : "ipv6" },
   argReplace=handleShowIpBgpNeighborsRoutesArgs )

registerShowBgpNlriAf(
   cmdName="doShowIpv6BgpNbrRecvdRtsRegex",
   argsAdd={ "afi" : "ipv6", "routeType" : "received-routes" },
   argReplace=handleShowIpBgpNeighborsRoutesArgs )

#-------------------------------------------------------------------------------
# "show bgp ( ipv4 | ipv6 ) unicast [ <ip> | <prefix> [longer-prefixes] ]
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#
# "show bgp ipv4 multicast [ <ip> | <prefix> [longer-prefixes] ]
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
registerShowBgpNlriAf(
       cmdName="doShowBgpNlriAf",
       argReplace=flattenShowBgpNlriAf )

#-------------------------------------------------------------------------------
# "show bgp neighbors <ip> [(ipv4 unicast) | (ipv6 unicast) | (ipv4 multicast) |
#       (ipv4 labeled-unicast) | (ipv6 labeled-unicast)]
#       ( routes | advertised-routes | received-routes [filtered] )
#       [ <ip> | <prefix> [longer-prefixes] ]
#       [ community [<communities>] [exact] ] [detail] [vrf <vrfName>]
#-------------------------------------------------------------------------------
registerShowBgpNlriAf(
       cmdName="doShowBgpNeighborsNlriAfRoutes",
       argReplace=handleShowIpBgpNeighborsRoutesArgs )
registerShowBgpNlriAf(
       cmdName="doShowBgpNeighborsNlriAfPrefixAddr",
       argReplace=handleShowIpBgpNeighborsRoutesArgs )

#-----------------------------------------
# show bgp segment-routing prefix-segments
#-----------------------------------------
class PrefixSegment( Model ):
   sid = Int( help='The segment ID of the route' )
   localLabel = Int( help='The generated local label of the route' )

class PrefixSegments( DeferredModel ):
   prefixSegments = Dict( keyType=IpGenericPrefix, valueType=PrefixSegment,
         help='Mapping of IPv4 or IPv6 prefixes to corresponding route segments' )

class ShowBgpSegmentRoutingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp segment-routing prefix-segments'
   data = {
      'bgp' : 'BGP information',
      'segment-routing' : 'Segment Routing Information',
      'prefix-segments' : 'Prefix Segment Information',
   }

   cliModel = PrefixSegments

   @staticmethod
   def handler( mode, args ):
      extraNlriAfiSafis = [ 'ipv6Lu' ]
      return BgpCliHelperCommand( mode, 'show bgp segment-routing', afi='ipv4',
                              safi='mplslabel', routeType='routes',
                              extraNlriAfiSafis=extraNlriAfiSafis ).run()

BasicCli.addShowCommandClass( ShowBgpSegmentRoutingCmd )

#------------------------------------------------------------------------------
#  show bgp rt-membership [ detail ]
#------------------------------------------------------------------------------

class ShowBgpRtMembership( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp rt-membership [ detail ]"
   data = {
         "bgp" : bgpAfterShow,
         "rt-membership" : rtMembershipAfterShowBgp,
         "detail" : detail,
   }
   cliModel = rtMembershipRoutesVrfModel

   @staticmethod
   @ArBgpShowOutput( 'doShowBgpRtMembership', arBgpModeOnly=True )
   def handler( mode, args ):
      return BgpCliHelperCommand( mode, 'show bgp rt-membership',
                                  afi='rt-membership',
                                  detail=args.get( 'detail' ) ).run()

if Toggles.BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   BasicCli.addShowCommandClass( ShowBgpRtMembership )

#------------------------------------------------------------------------------
#  show bgp neighbors <ip> rt-membership [ advertised-routes |
#  received-routes | routes ]
#------------------------------------------------------------------------------

class ShowBgpNeighborsRtMembership( ShowCommand.ShowCliCommandClass ):
   syntax = "show bgp neighbors PEER_ADDR rt-membership ROUTE_TYPE"
   data = {
         "bgp" : bgpAfterShow,
         "neighbors" : neighbors,
         "PEER_ADDR" : V4V6PeerKeyCliExpression,
         "rt-membership" : rtMembershipAfterShowBgp,
         "ROUTE_TYPE" : routeTypeMatcher,
   }
   cliModel = rtMembershipRoutesVrfModel

   @staticmethod
   @ArBgpShowOutput( 'doShowBgpNeighborsRtMembership', arBgpModeOnly=True )
   def handler( mode, args ):
      peerAddr = { 'peerAddr' : args.get( 'PEER' ) }
      convertPeerAddr( peerAddr )
      return BgpCliHelperCommand( mode, 'show bgp rt-membership',
                                  afi='rt-membership',
                                  peerAddr=peerAddr.get( 'peerAddr' ),
                                  routeType=args.get( 'ROUTE_TYPE' ) ).run()

if Toggles.BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   BasicCli.addShowCommandClass( ShowBgpNeighborsRtMembership )

def Plugin( entityManager ):
   return
