#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import os
import re
import socket
import sys
from collections import namedtuple
from itertools import chain
from struct import ( pack, unpack )
import ArPyUtils
import Cell
import BasicCli
import CliCommand
from CliCommand import guardedKeyword, singleKeyword, singleNode
import CliMatcher
from CliMatcher import EnumMatcher
import ShowCommand
import CliParser
import CliPlugin.AclCli as AclCli
from CliPlugin.AclCliModel import AllAclList
import CliPlugin.BfdModel as BfdModel
from CliPlugin.BfdModel import ( BfdNeighbors,
                                 PeerHistory,
                                 RbfdStats,
                                 SbfdStats,
                                 BfdSummary,
                                 BfdDebug,
                                 BfdHwAccel,
                                 BfdHwResourceList )
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
import CliPlugin.TechSupportCli as TechSupportCli
from CliPlugin.VrfCli import ALL_VRF_NAME, VrfExprFactory, getAllVrfNames, vrfExists
from CliToken.Ip import ipMatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
from Intf.IntfRange import IntfRangeMatcher
import LazyMount
import Smash
import SmashLazyMount
import Tac
import Tracing
from Toggles.BfdToggleLib import toggleHwBfdEnabled, toggleBfdTelemetryEnabled
from IpLibConsts import DEFAULT_VRF
from BfdLib import ( diagEnumToReason,
                     authType,
                     authEnumTypeToText,
                     authEnumNumToText,
                     operSessionEnumToType, )
from DeviceNameLib import eosIntfToKernelIntf

traceHandle = Tracing.defaultTraceHandle()
t0 = traceHandle.trace0

aclCpConfig = None
aclStatus = None
aclCheckpoint = None
appConfigDir = None
bfdConfigGlobal = None
bfdConfigIntf = None
bfdHwStatus = None
bfdHwConfig = None
bfdStatusPeer = None
bfdConfigInfo = None
bfdConfigPeer = None
bfdOperInfoPeer = None
bfdStatsSmash = None
configBfdLag = None
allIntfStatusDir = None
bfdHwResourceConfig = None
redundancyConfig = None
ipConfig = None

matcherBfd = CliMatcher.KeywordMatcher( 'bfd',
      helpdesc='Status for the BFD protocol' )
matcherDebug = CliMatcher.KeywordMatcher( 'debug',
      helpdesc='Bfd debug information' )
matcherDestIp = CliMatcher.KeywordMatcher( 'dest-ip',
      helpdesc='BFD neighbor IP address' )
matcherNbrDetail = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Detailed neighbor information' )
matcherHistory = CliMatcher.KeywordMatcher( 'history',
      helpdesc='Neighbor transition history' )
matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Interface keyword' )
matcherInterval = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Transmit rate in milliseconds' )
matcherMin_RxDeprecated = singleKeyword( 'min_rx',
      helpdesc='Expected minimum incoming rate in milliseconds',
      hidden=True )
matcherMinRx = CliMatcher.KeywordMatcher( 'min-rx',
      helpdesc='Expected minimum incoming rate in milliseconds' )
matcherMultiplier = CliMatcher.KeywordMatcher( 'multiplier',
      helpdesc='BFD multiplier' )
neighborsDeprecatedNode = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'neighbors',
                  helpdesc='BFD Neighbor information' ),
      deprecatedByCmd='show bfd peers' )
matcherPeers = CliMatcher.KeywordMatcher( 'peers',
      helpdesc='BFD Neighbor information' )
matcherSrcIp = CliMatcher.KeywordMatcher( 'src-ip',
      helpdesc='BFD source IP address' )
matcherSummary = CliMatcher.KeywordMatcher( 'summary',
      helpdesc='BFD summary' )
nodeDetail = guardedKeyword( 'detail',
      helpdesc='Access list detail',
      guard=AclCli.countersPerChipEnabledGuard )
matcherHardware = CliMatcher.KeywordMatcher( 'hardware',
      helpdesc='Hardware-specific BFD parameters' )
emptyResourceCompletion = [ CliParser.Completion( 'RESOURCE', 'Resource name',
                                                  literal=False ) ]
matcherResource = CliMatcher.DynamicKeywordMatcher(
      lambda mode: [ r for rConfig in bfdHwResourceConfig.itervalues()
         for r in rConfig.maxHwResourceSessions.iterkeys() ],
      emptyTokenCompletion=emptyResourceCompletion )
matcherProtocol = CliMatcher.KeywordMatcher( 'protocol',
      helpdesc='Subscribing protocol' )
matcherSbfd = CliMatcher.KeywordMatcher( 'sbfd',
      helpdesc='Status for the SBFD sessions only', )
matcherInitiators = CliMatcher.KeywordMatcher( 'initiators',
      helpdesc='SBFD initiators' )
matcherReflectors = CliMatcher.KeywordMatcher( 'reflectors',
      helpdesc='SBFD reflectors' )
vrfExprFactory = VrfExprFactory(
      helpdesc='BFD information of VRF',
      maxMatches=1 )

AddressFamily = Tac.Type( 'Arnet::AddressFamily' )
configSessionType = Tac.Type( 'Bfd::SessionType' )
operSessionType = Tac.Type( 'Bfd::OperSessionType' )
historyInfo = Tac.Value( 'Bfd::HistoryInfo'  )
operState = Tac.Type( 'Bfd::OperState'  )

sbfdReflectorGlobal = namedtuple(
   "sbfdReflectorGlobal",
   [ 'rxDrop', 'txSend', 'rxValid', 'rxTruncated', 'rxInvalidVer',
     'rxInvalidLen', 'rxInvalidSrcPort', 'rxDemandUnset', 'rxInvalidMulti',
     'rxInvalidMyDisc', 'rxYourDiscUnmatch', 'rxInvalidBfdState',
     'rxUnsupported', 'rxInvalidPktFormat', 'txSendToFail',
     'txInvalidLen', 'rxHashMiss', 'rxHashHit', 'rxHashTrim' ] )

sbfdReflectorSess = namedtuple(
   "sbfdReflectorSess",
   [ 'disc', 'srcIp', 'minTxInt',
     'minRxInt', 'minEchoRxInt', 'minPeriod', 'maxPeriod',
     'lastPktYourDisc', 'srcPort', 'detectMulti', 'bfdState', 'lastBfdState',
     'diag', 'version', 'demand', 'final', 'poll', 'lastPktLen',
     'sentBfdState', 'lastSentBfdState', 'sentDiag',
     'pad1', 'pad2', 'rxNumReceived', 'rxTotalPeriod',
     'rxLastRxTimeSec', 'rxLastRxTimeNsec',
     'rxLastStateChgSec', 'rxLastStateChgNsec',
     'txNumSent',
     'txLastTxTimeSec', 'txLastTxTimeNsec',
     'txLastStateChgSec', 'txLastStateChgNsec' ] )

# Reflector Size constants
sbfdReflectorCacheSize = { 'global'  : 88,
                           'session' : 136,
                           'maxSess' : 512,
}

def getReflectorStats():
   globalStats = {}
   sessions = {}
   output = os.popen( 'ps -ef | grep sbfdreflectord' ).read()
   if 'sbfdreflectord -d' not in output:
      # sbfdReflectord is not configured and running
      return globalStats, sessions
   shmName = '/dev/shm/sbfdreflectord'
   nspath = os.getenv( 'NSPATH' )
   if nspath:
      shmName = shmName + nspath.replace( '/', '-' )
   if not os.path.exists( shmName ):
      # sbfdReflectord is not configured and never run
      return globalStats, sessions
   cacheSize = sbfdReflectorCacheSize[ 'global' ] + \
            sbfdReflectorCacheSize[ 'session' ] * sbfdReflectorCacheSize[ 'maxSess' ]
   if os.path.getsize( shmName ) != cacheSize:
      t0( "sbfdReflector statistics are of wrong size, expected %d" % cacheSize )
      return globalStats, sessions
   with open( shmName, 'rb' ) as f:
      buf = f.read( sbfdReflectorCacheSize[ 'global' ] )
      globalStats = sbfdReflectorGlobal( *unpack( '3Q16I', buf ) )
      for _ in range( sbfdReflectorCacheSize[ 'maxSess' ] ):
         buf = f.read( sbfdReflectorCacheSize[ 'session' ] )
         sess = sbfdReflectorSess( *unpack( '8IH14B11Q', buf ) )
         if sess.disc:
            ipStr = socket.inet_ntoa( pack( "<I", sess.srcIp ) )
            sessions[ ( ipStr, sess.disc ) ] = sess
   return globalStats, sessions

def getReflectorPeerStatus( sess, localIpAddr=None ):
   ipStr = socket.inet_ntoa( pack( "<I", sess.srcIp ) )
   ip = Tac.Value( 'Arnet::IpGenAddr', ipStr )
   peer = Tac.Value( 'Bfd::Peer', ip, DEFAULT_VRF )
   peer.type = configSessionType.sbfdReflector
   peerStatus = Tac.newInstance( 'Bfd::PeerStatus', peer,
                                 operSessionType.sessionTypeSbfdReflector )
   peerStatus.localDisc = bfdConfigGlobal.sbfdReflectorLocalDisc
   peerStatus.remoteDisc = sess.disc
   peerStatus.status = stateName[ sess.sentBfdState ]
   peerStatus.authType = authType.authNone
   peerStatus.localIp = localIpAddr
   return peerStatus

def showBfdNeighborHistory( vrfNames=None, intfSet=None,
                            destIpAddr=None, srcIpAddr=None,
                            minTx=None, minRx=None, multiplier=None, proto=None,
                            af=None, bfdOnly=False, sbfdOnly=False ):
   seq = bfdStatusPeer.currentSeq
   peerHistoryModel = BfdModel.PeerHistory()
   for _ in range( len( bfdStatusPeer.peerHistory ) ):
      # Currently printing most recent entry first. This might need to change.
      if seq == 0:
         seq = historyInfo.maxEntries
      seq -= 1
      peerHistory = bfdStatusPeer.peerHistory.get( seq, None )
      if peerHistory is None:
         continue
      # peerHistory is sorted and the loop starts from the most recent entry,
      # so we can return as soon as genId != historyGeneration
      if peerHistory.genId != bfdConfigInfo.historyGeneration:
         return peerHistoryModel
      pst = peerHistory.peerSnapshot
      if ( peerMatchesFilter( pst.peer, vrfNames=vrfNames, intfSet=intfSet,
                              destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                              minTx=minTx, minRx=minRx, multiplier=multiplier,
                              proto=proto, addrFamily=af, 
                              bfdOnly=bfdOnly, sbfdOnly=sbfdOnly ) ):
         peerSnapshotModel = printNeighborHistory( pst )
         # pylint: disable=no-member
         peerHistoryModel.peerSnapshots.append( peerSnapshotModel )
   return peerHistoryModel

MAX_U64 = 0xFFFFFFFFFFFFFFFF
MAX_U32 = 0xFFFFFFFF

def convertBfdToSecs( sec, nsec ):
   return float( sec + nsec / 1000000000 )

def getSmashTxStats( peerStatus, echo=False ):
   disc = peerStatus.localDisc
   if echo:
      disc = peerStatus.echoDisc

   if not disc in bfdStatsSmash.txStats:
      return None

   txStatsSmash = bfdStatsSmash.txStats[ disc ]
   txStats = Tac.newInstance( 'BfdPyUtils::TxStatsEntry',
         peerStatus.localDisc )
   txStats.fromSmashStats( txStatsSmash )
   return txStats

def generateTxStatsModel( txStats ):
   txStatsModel = BfdModel.TxStats()
   txStatsModel.numSent = txStats.bfdMsgTxStats.numSent
   txStatsModel.minLateness = txStats.bfdMsgTxStats.minLateness
   txStatsModel.maxLateness = txStats.bfdMsgTxStats.maxLateness
   txStatsModel.avgLateness = txStats.bfdMsgTxStats.avgLateness
   txStatsModel.lastTxTime = convertBfdToSecs( txStats.bfdMsgTxStats.lastTxTimeSec,
                                               txStats.bfdMsgTxStats.lastTxTimeNsec )
   txStatsModel.sentWithin1p = txStats.bfdMsgTxStats.sentWithin1p
   txStatsModel.sentWithin2p = txStats.bfdMsgTxStats.sentWithin2p
   txStatsModel.sentWithin3p = txStats.bfdMsgTxStats.sentWithin3p
   txStatsModel.sentAfter3p = txStats.bfdMsgTxStats.sentAfter3p
   return txStatsModel

def generateRxStatsModel( rxStats ):
   rxStatsModel = BfdModel.RxStats()
   rxStatsModel.numReceived = rxStats.bfdMsgRxStats.numReceived
   rxStatsModel.minPeriod = rxStats.bfdMsgRxStats.minPeriod
   rxStatsModel.maxPeriod = rxStats.bfdMsgRxStats.maxPeriod
   rxStatsModel.avgPeriod = rxStats.bfdMsgRxStats.avgPeriod
   rxStatsModel.lastRxTime = convertBfdToSecs( rxStats.bfdMsgRxStats.lastRxTimeSec,
                                               rxStats.bfdMsgRxStats.lastRxTimeNsec )

   return rxStatsModel

def updatePeerStatsDetailReflector( peer, peerStatus, peerStatsModel,
                                    peerStatsDetailModel, reflectorSess ):
   sess = reflectorSess.get( ( str( peer.ip ), peerStatus.remoteDisc ) )
   if not sess:
      t0( "session (%s, %d) doesn't exist" % ( peer.ip, peerStatus.remoteDisc ) )
      return False
   peerStatsModel.localDisc = peerStatus.localDisc
   peerStatsModel.remoteDisc = peerStatus.remoteDisc
   peerStatsModel.status = peerStatus.status
   peerStatsModel.sessType = peerStatus.type
   peerStatsModel.destPort = sess.srcPort
   peerStatsModel.lastDiag = diagToEnum[ sess.sentDiag ]
   peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
   peerStatsModel.authProfileName = ''
   peerStatsModel.l3intf = ''
   if peerStatus.status == operState.up:
      peerStatsModel.lastUp = float( sess.txLastStateChgSec +
                                     sess.txLastStateChgNsec / 1000000000 )
      peerStatsModel.lastDown = 0.0
   else:
      peerStatsModel.lastUp = 0.0
      peerStatsModel.lastDown = float( sess.txLastStateChgSec +
                                       sess.txLastStateChgNsec / 1000000000 )
   peerStatsModel.rxStats = BfdModel.RxStats()
   peerStatsModel.rxStats.numReceived = sess.rxNumReceived
   peerStatsModel.rxStats.minPeriod = sess.minPeriod
   peerStatsModel.rxStats.maxPeriod = sess.maxPeriod
   if sess.rxNumReceived > 1:
      peerStatsModel.rxStats.avgPeriod = long( sess.rxTotalPeriod /
                                               ( sess.rxNumReceived - 1 ) )
   else:
      peerStatsModel.rxStats.avgPeriod = 0
   peerStatsModel.rxStats.lastRxTime = float( sess.rxLastRxTimeSec +
                                              sess.rxLastRxTimeNsec / 1000000000 )
   peerStatsModel.rxStats.lastRxStateChg = float(
      sess.rxLastStateChgSec + sess.rxLastStateChgNsec / 1000000000 )
   peerStatsModel.rxStats.lastRxState = stateName[ sess.lastBfdState ]
   peerStatsModel.txStats = BfdModel.TxStats()
   peerStatsModel.txStats.numSent = sess.txNumSent
   peerStatsModel.txStats.lastTxTime = float( sess.txLastTxTimeSec +
                                              sess.txLastTxTimeNsec / 1000000000 )
   peerStatsDetailModel.operRxInterval = bfdConfigGlobal.sbfdReflectorMinRx
   peerStatsDetailModel.localIpAddr = str( peerStatus.localIp )

   # Last rx packet info
   peerStatsDetailModel.lastVersion = sess.version
   peerStatsDetailModel.lastDiag = sess.diag
   peerStatsDetailModel.lastState = stateName[ sess.bfdState ]
   peerStatsDetailModel.lastDemand = bool( sess.demand )
   peerStatsDetailModel.lastPoll = bool( sess.poll )
   peerStatsDetailModel.lastFinal = bool( sess.final )
   peerStatsDetailModel.lastDetectMult = sess.detectMulti
   peerStatsDetailModel.lastLength = sess.lastPktLen
   peerStatsDetailModel.lastMyDiscU32 = sess.disc
   peerStatsDetailModel.lastYourDiscU32 = sess.lastPktYourDisc
   peerStatsDetailModel.lastMinTxIntv = sess.minTxInt
   peerStatsDetailModel.lastMinRxIntv = sess.minRxInt
   peerStatsDetailModel.lastEchoRxIntv = sess.minEchoRxInt
   peerStatsModel.peerStatsDetail = peerStatsDetailModel
   return True

def updatePeerStatsDetail( peer, peerStatus, peerStatsModel, peerStatsDetailModel, 
                           operInfo, netlinkStub, apps, sessType, destIpAddr=None,
                           reflectorSess=None ):
   if peerStatus.type == operSessionType.sessionTypeSbfdReflector:
      return updatePeerStatsDetailReflector( peer, peerStatus, peerStatsModel,
                                             peerStatsDetailModel, reflectorSess )

   l3intf = peer.intf
   if sessType == operSessionType.sessionTypeMicroLegacy or\
      sessType == operSessionType.sessionTypeMicroRfc7130:
      l3intf = lagIntfIdForIntfId( peer.intf )
      if l3intf is None:
         # unable to look up which port-channel this physical port is
         # configured in. The safe thing to do here to return without printing
         # anything.
         return False
   
   peerStatsModel.status = str( peerStatus.status ) 
   peerStatsModel.sessType = sessType
   peerStatsModel.localDisc = peerStatus.localDisc
   peerStatsModel.remoteDisc = peerStatus.remoteDisc
   peerStatsModel.lastUp = operInfo.lastUp
   peerStatsModel.lastDown = operInfo.lastDown
   peerStatsModel.lastDiag = diagToEnum[ operInfo.lastDiag ]
   peerStatsModel.l3intf = l3intf
   peerStatsModel.tunnelId = peer.tunnelId
   peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
   peerStatsModel.authProfileName = peerStatus.secretProfileName
   peerStatsDetailModel.localIpAddr = str( peerStatus.localIp )
   peerStatsDetailModel.apps = (
         [ mapAppName( a, showDetail=True ) for a in set( apps ) ] )
   peerStatsDetailModel.echoOn = peerStatus.echoOn
   peerStatsDetailModel.echoActive = peerStatus.echoActive
   peerStatsDetailModel.echoDisc = peerStatus.echoDisc
   peerStatsDetailModel.operEchoTxRxInterval = \
      peerStatus.operEchoTxRxIntervalInMicrosecond
   peerStatsDetailModel.operTxInterval = peerStatus.operTxIntervalInMicrosecond
   peerStatsDetailModel.operRxInterval = peerStatus.operRxIntervalInMicrosecond
   peerStatsDetailModel.detectMult = peerStatus.detectMult
   peerStatsDetailModel.remoteMinRxInterval = \
      peerStatus.remoteMinRxIntervalInMicrosecond
   peerStatsDetailModel.remoteDetectMult = peerStatus.remoteDetectMult
   peerStatsDetailModel.detectTime = peerStatus.detectTimeInMicrosecond

   # Get stats for this particular peer, if destIpAddr is not None
   if destIpAddr:
      netlinkStub.sendGetStats( peerStatus.localDisc, False, False, False )
      netlinkStub.bfdRead()
   rxStats = netlinkStub.receivedRxStats.get( peerStatus.localDisc )
   if bfdHwStatus.isHwAccelerated( peerStatus.localDisc ):
      txStats = getSmashTxStats( peerStatus, echo=False )
      hwAccelAsync = True
   else:
      txStats = netlinkStub.receivedTxStats.get( peerStatus.localDisc )
      hwAccelAsync = False
   if toggleHwBfdEnabled():
      peerStatsDetailModel.hwAcceleratedStates[ "Async" ] = hwAccelAsync

   if rxStats:
      peerStatsModel.rxStats = generateRxStatsModel( rxStats )

   if txStats:
      peerStatsModel.txStats = generateTxStatsModel( txStats )

   # Get echo stats for this particular peer, if destIpAddr is not None
   if destIpAddr:
      netlinkStub.sendGetStats( peerStatus.echoDisc, False, False, False )
      netlinkStub.bfdRead()
   rxStats = netlinkStub.receivedRxStats.get( peerStatus.echoDisc )
   if bfdHwStatus.isHwAccelerated( peerStatus.echoDisc ):
      txStats = getSmashTxStats( peerStatus, echo=True )
      hwAccelEcho = True
   else:
      txStats = netlinkStub.receivedTxStats.get( peerStatus.echoDisc )
      hwAccelEcho = False
   if toggleHwBfdEnabled():
      peerStatsDetailModel.hwAcceleratedStates[ "Echo" ] = hwAccelEcho

   if rxStats:
      peerStatsModel.echoRxStats = generateRxStatsModel( rxStats )

   if txStats:
      peerStatsModel.echoTxStats = generateTxStatsModel( txStats )
      
   peerStatsDetailModel.lastVersion = peerStatus.lastVersion
   peerStatsDetailModel.lastDiag = peerStatus.lastDiag
   peerStatsDetailModel.lastState = stateName[ peerStatus.lastState ]
   peerStatsDetailModel.lastDemand = peerStatus.lastDemand
   peerStatsDetailModel.lastPoll = peerStatus.lastPoll
   peerStatsDetailModel.lastFinal = peerStatus.lastFinal
   peerStatsDetailModel.lastDetectMult = peerStatus.lastDetectMult
   peerStatsDetailModel.lastLength = peerStatus.lastLength
   peerStatsDetailModel.lastMyDiscU32 = peerStatus.lastMyDiscU32
   peerStatsDetailModel.lastYourDiscU32 = peerStatus.lastYourDiscU32
   peerStatsDetailModel.lastMinTxIntv = peerStatus.lastMinTxIntvU32
   peerStatsDetailModel.lastMinRxIntv = peerStatus.lastMinRxIntvU32 
   peerStatsDetailModel.lastEchoRxIntv = peerStatus.lastEchoRxIntvU32

   if toggleHwBfdEnabled():
      if peerStatus.microBfdPeer:
         localSsoReady = True
         for microPeer in peerStatus.microBfdPeer:
            microPeerStatus = bfdStatusPeer.peerStatus.get( microPeer )
            if not microPeerStatus:
               continue
            if not bfdHwStatus.isHwAccelerated( microPeerStatus.localDisc ):
               localSsoReady = False
               break

         # pylint: disable-msg=protected-access
         peerStatsDetailModel._localSsoReady = localSsoReady

   peerStatsModel.peerStatsDetail = peerStatsDetailModel

   return True

def updateNeighborsModel( peer, peerStatus, bfdNeighborsModel, peerStatsModel ):
   bfdNeighborVrfModel = bfdNeighborsModel.vrfs.setdefault( peer.vrf, 
                         BfdModel.BfdNeighborVrf() )
   if peerStatus.type == operSessionType.sessionTypeSbfdInitiator:
      if peerStatus.localIp.af == 'ipv4':
         neighbors = bfdNeighborVrfModel.ipv4InitiatorNeighbors
      else:
         neighbors = bfdNeighborVrfModel.ipv6InitiatorNeighbors
      sbfdInitiatorTunnelModel = neighbors.setdefault( str( peer.ip ),
            BfdModel.SbfdInitiatorTunnel() )
      sbfdInitiatorTunnelModel.peerStats[ long( peer.tunnelId ) ] = \
            peerStatsModel
   elif peerStatus.type == operSessionType.sessionTypeSbfdReflector:
      if peerStatus.localIp.af == 'ipv4':
         neighbors = bfdNeighborVrfModel.ipv4ReflectorNeighbors
      else:
         neighbors = bfdNeighborVrfModel.ipv6ReflectorNeighbors
      sbfdReflectorModel = neighbors.setdefault( str( peer.ip ),
            BfdModel.SbfdReflector() )
      sbfdReflectorModel.peerStats[ long( peerStatus.remoteDisc ) ] = peerStatsModel
   else:
      if peer.ip.af == 'ipv4':
         neighbors = bfdNeighborVrfModel.ipv4Neighbors
      else:
         neighbors = bfdNeighborVrfModel.ipv6Neighbors
      bfdNeighborInterfaceModel = neighbors.setdefault( str( peer.ip ),
            BfdModel.BfdNeighborInterface() )
      bfdNeighborIntfPeerModel = bfdNeighborInterfaceModel.peers.setdefault(
            peer.intf, BfdModel.BfdNeighborIntfPeer() )
      bfdNeighborSrcAddrModel = bfdNeighborIntfPeerModel.types.setdefault(
            operSessionEnumToType[ peerStatus.type ], BfdModel.BfdNeighborSrcAddr() )
      bfdNeighborSrcAddrModel.peerStats[ str( peerStatus.localIp ) ] = \
            peerStatsModel

def printBfdNeighborDetailEntry( peerStatus, bfdNeighborsModel, netlinkStub,
                                 destIpAddr=None, reflectorSess=None ):
   try:
      peer = peerStatus.peer
      peerStatsModel = BfdModel.PeerStats()
      peerStatsDetailModel = BfdModel.PeerStatsDetail()
      if peerStatus.type == operSessionType.sessionTypeSbfdReflector:
         operInfo = None
         apps = None
      else:
         peerOperInfo = bfdOperInfoPeer.peerOperInfo[ peer ]
         operInfo = peerOperInfo.info
         apps = getApp( peer )
         if apps is None:
            return
   except KeyError:
      # This peer has been removed during the show command, so skip it.
      return
   sessType = peerStatus.type
   if not updatePeerStatsDetail( peer, peerStatus, peerStatsModel, 
                                 peerStatsDetailModel, operInfo, netlinkStub, 
                                 apps, sessType, destIpAddr, reflectorSess ):
      # failed to update peerStatsModel, so skip it
      return
   
   updateNeighborsModel( peer, peerStatus, bfdNeighborsModel, peerStatsModel )

def showBfdNeighborHelper( bfdNeighborsModel, displayType, addrFamily=None, 
                           vrfNames=None, intfSet=None, destIpAddr=None, 
                           srcIpAddr=None, minTx=None, minRx=None, multiplier=None,
                           resource=None, proto=None, 
                           bfdOnly=False, sbfdOnly=False,
                           sbfdInitiatorOnly=False, sbfdReflectorOnly=False,
                           reflectorSess=None ):
   # pylint: disable=too-many-nested-blocks
   if not sbfdReflectorOnly:
      if displayType == 'detail':
         netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
         netlinkStub.createFd( False )
         if destIpAddr is None:
            # We get stats for all peers.
            netlinkStub.sendGetStats( 0, False, False, False )
            netlinkStub.bfdRead()

      for peerStatus in bfdStatusPeer.peerStatus.itervalues():
         if ( peerMatchesFilter( peerStatus.peer, addrFamily=addrFamily,
                                 vrfNames=vrfNames, intfSet=intfSet,
                                 destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                 minTx=minTx, minRx=minRx, multiplier=multiplier,
                                 resource=resource, proto=proto, bfdOnly=bfdOnly,
                                 sbfdOnly=sbfdOnly ) ):
            if displayType == 'detail':
               printBfdNeighborDetailEntry( peerStatus, bfdNeighborsModel,
                                            netlinkStub, destIpAddr,
                                            reflectorSess=reflectorSess )
               continue

            try:
               peer = peerStatus.peer
               intfToPeerMap = bfdStatusPeer.intfToPeerMap.get( peer.intf, None )
               peerOperInfo = bfdOperInfoPeer.peerOperInfo[ peer ].info

               peerStatsModel = BfdModel.PeerStats()
               peerStatsModel.localDisc = peerStatus.localDisc
               peerStatsModel.remoteDisc = peerStatus.remoteDisc
               peerStatsModel.status = peerStatus.status
               peerStatsModel.sessType = peerStatus.type
               peerStatsModel.lastUp = peerOperInfo.lastUp
               peerStatsModel.lastDown = peerOperInfo.lastDown
               peerStatsModel.lastDiag = diagToEnum[ peerOperInfo.lastDiag ]
               peerStatsModel.kernelIfIndex = ( intfToPeerMap.kernelIfIndex
                                                if intfToPeerMap else 0 )
               peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
               peerStatsModel.authProfileName = peerStatus.secretProfileName
               l3intf = peer.intf
               if ( peerStatus.type == operSessionType.sessionTypeMicroLegacy or
                    peerStatus.type == operSessionType.sessionTypeMicroRfc7130 ):
                  l3intf = lagIntfIdForIntfId( peer.intf )
                  if not l3intf:
                     # unable to look up which port-channel this physical port is
                     # configured in. The safe thing to do here to return without
                     # printing anything.
                     return
               peerStatsModel.l3intf = l3intf
               peerStatsModel.tunnelId = peer.tunnelId

               updateNeighborsModel( peer, peerStatus, bfdNeighborsModel,
                                     peerStatsModel )

            except KeyError:
               # This peer has been removed during the show command, so skip it.
               continue

   if ( not bfdOnly and not sbfdInitiatorOnly and
        addrFamily and AddressFamily.ipv4 in addrFamily and
        vrfNames and DEFAULT_VRF in vrfNames and
        not minTx and not multiplier and not resource and not proto ):
      ipIntfConfig = ipConfig.ipIntfConfig.get( bfdConfigGlobal.sbfdLocalIntf )
      localIpStr = ipIntfConfig.addrWithMask.address if ipIntfConfig else '0.0.0.0'
      localIpAddr = Tac.Value( 'Arnet::IpGenAddr', localIpStr )
      if ( srcIpAddr and srcIpAddr != localIpStr or
           minRx and minRx != bfdConfigGlobal.sbfdReflectorMinRx ):
         return

      for sess in reflectorSess.itervalues():
         ipStr = socket.inet_ntoa( pack( "<I", sess.srcIp ) )
         if destIpAddr and destIpAddr != ipStr:
            continue
         peerStatus = getReflectorPeerStatus( sess, localIpAddr )
         if displayType == 'detail':
            printBfdNeighborDetailEntry( peerStatus, bfdNeighborsModel,
                                         None, destIpAddr,
                                         reflectorSess=reflectorSess )
            continue

         p = peerStatus.peer
         peerStatsModel = BfdModel.PeerStats()
         peerStatsModel.localDisc = peerStatus.localDisc
         peerStatsModel.remoteDisc = peerStatus.remoteDisc
         peerStatsModel.status = peerStatus.status
         peerStatsModel.sessType = peerStatus.type
         sess = reflectorSess.get( ( str( p.ip ), peerStatus.remoteDisc ) )
         peerStatsModel.destPort = sess.srcPort
         peerStatsModel.lastDiag = diagToEnum[ sess.sentDiag ]
         peerStatsModel.authType = authEnumTypeToText[ peerStatus.authType ]
         peerStatsModel.authProfileName = ''
         peerStatsModel.l3intf = ''
         if peerStatus.status == operState.up:
            peerStatsModel.lastUp = float( sess.txLastStateChgSec +
                                           sess.txLastStateChgNsec / 1000000000 )
            peerStatsModel.lastDown = 0.0
         else:
            peerStatsModel.lastUp = 0.0
            peerStatsModel.lastDown = float( sess.txLastStateChgSec +
                                             sess.txLastStateChgNsec / 1000000000 )
         updateNeighborsModel( p, peerStatus, bfdNeighborsModel,
                               peerStatsModel )

def showBfdNeighbor( mode, vrfName=None, intfs=None, destIpAddr=None, srcIpAddr=None,
                     minTx=None, minRx=None, multiplier=None, proto=None,
                     displayType=None, addrFamilies=None, bfdOnly=False, 
                     sbfdOnly=False, sbfdInitiatorOnly=False,
                     sbfdReflectorOnly=False ):

   if vrfName == ALL_VRF_NAME:
      vrfNames = set( getAllVrfNames( mode ) )
   elif vrfExists( vrfName ):
      vrfNames = [ vrfName ]
   else:
      mode.addError( "VRF %s not configured." % vrfName )
      if displayType == 'summary':
         return showBfdSummary( mode, [ vrfName ] )
      elif displayType == 'debug':
         return BfdModel.BfdDebug()
      elif displayType == 'history':
         return BfdModel.PeerHistory()
      elif displayType == 'detail':
         return BfdModel.BfdNeighbors( _detailed=True )
      else:
         # If request is not for bfd neighbor history|detail|summary|debug
         # then assumption is that it is for just bfd neighbor.
         return BfdModel.BfdNeighbors( _detailed=False )

   # Include member intfs if filtered by LAG intf
   intfSet = set()
   if intfs:
      for intf in intfs.intfNames():
         intfSet.add( intf )
         if intf.startswith( "Port-Channel" ) and \
            intf in configBfdLag.bfdLagConfig:
            bfdLagConfig = configBfdLag.bfdLagConfig[ intf ]
            for memberIntf in bfdLagConfig.bfdReqPort:
               intfSet.add( memberIntf )

   if destIpAddr is not None and not isinstance( destIpAddr, str ):
      destIpAddr = destIpAddr.stringValue
   if srcIpAddr is not None and not isinstance( srcIpAddr, str ):
      srcIpAddr = srcIpAddr.stringValue

   if addrFamilies is None:
      addrFamilies = [ 'ipv4', 'ipv6' ]
      if destIpAddr is not None:
         addrFamilies = [ 'ipv4' if '.' in destIpAddr else 'ipv6' ]
      elif srcIpAddr is not None:
         addrFamilies = [ 'ipv4' if '.' in srcIpAddr else 'ipv6' ]
   else:
      addrFamilies = [ addrFamilies ]

   # pylint: disable=too-many-function-args

   if ( bfdOnly or sbfdInitiatorOnly ):
      reflectorSess = {}
   else:
      _, reflectorSess = getReflectorStats()

   if displayType == 'summary':
      return showBfdSummary( mode, vrfNames, intfSet, destIpAddr, srcIpAddr,
                             minTx, minRx, multiplier, proto, af=addrFamilies,
                             bfdOnly=bfdOnly, sbfdOnly=sbfdOnly,
                             sbfdInitiatorOnly=sbfdInitiatorOnly,
                             sbfdReflectorOnly=sbfdReflectorOnly,
                             reflectorSess=reflectorSess )
   elif displayType == 'debug':
      return showBfdDebug( mode, vrfNames, intfSet, destIpAddr, srcIpAddr,
                           minTx, minRx, multiplier, proto, addrFamilies, 
                           bfdOnly, sbfdOnly )
   # pylint: disable=too-many-nested-blocks
   elif displayType == 'history':
      return showBfdNeighborHistory( vrfNames, intfSet, destIpAddr, srcIpAddr,
                                     minTx, minRx, multiplier, proto,
                                     af=addrFamilies, bfdOnly=bfdOnly, 
                                     sbfdOnly=sbfdOnly )
   else:

      if displayType == 'detail':
         bfdNeighborsModel = BfdModel.BfdNeighbors( _detailed=True )
      else:
         bfdNeighborsModel = BfdModel.BfdNeighbors( _detailed=False )

      showBfdNeighborHelper( bfdNeighborsModel,
                             displayType,
                             addrFamily=addrFamilies,
                             vrfNames=vrfNames,
                             intfSet=intfSet,
                             destIpAddr=destIpAddr,
                             srcIpAddr=srcIpAddr,
                             minTx=minTx, minRx=minRx,
                             multiplier=multiplier,
                             proto=proto,
                             bfdOnly=bfdOnly,
                             sbfdOnly=sbfdOnly,
                             sbfdInitiatorOnly=sbfdInitiatorOnly,
                             sbfdReflectorOnly=sbfdReflectorOnly,
                             reflectorSess=reflectorSess )
      
      return bfdNeighborsModel

helpDescBfdShow = 'Status for the BFD protocol'

def incrementCount( typeModel, sessType, peerStatus ):
   countByType = typeModel.types[ sessType ]
   countByState = countByType.states[ peerStatus.status ]
   countByState.async += 1
   if peerStatus.echoActive:
      countByState.echo += 1

def showBfdSummary( mode, vrfNames=None, intfSet=None, destIpAddr=None,
                    srcIpAddr=None, minTx=None, minRx=None, multiplier=None,
                    proto=None, af=None, bfdOnly=False, sbfdOnly=False,
                    sbfdInitiatorOnly=False, sbfdReflectorOnly=False,
                    reflectorSess=None ):
   bfdSummaryModel = BfdModel.BfdSummary()
   bfdSummaryModel.minTx = bfdConfigIntf.globalCfg.minTx
   bfdSummaryModel.minRx = bfdConfigIntf.globalCfg.minRx
   bfdSummaryModel.mult = bfdConfigIntf.globalCfg.mult
   bfdSummaryModel.mhMinTx = bfdConfigIntf.multihopGlobalCfg.minTx
   bfdSummaryModel.mhMinRx = bfdConfigIntf.multihopGlobalCfg.minRx
   bfdSummaryModel.mhMult = bfdConfigIntf.multihopGlobalCfg.mult
   bfdSummaryModel.slowTimer = bfdConfigGlobal.bfdSlowTimer
   bfdSummaryModel.adminDown = bfdConfigGlobal.bfdAdminDown
   if toggleBfdTelemetryEnabled():
      bfdSummaryModel.sessionStatsInterval = bfdConfigGlobal.sessionStatsInterval

   bfdSummaryModel.localIntf = bfdConfigGlobal.sbfdLocalIntf
   bfdSummaryModel.initiatorMinTx = bfdConfigGlobal.sbfdInitiatorConfig.minTx
   bfdSummaryModel.initiatorMult = bfdConfigGlobal.sbfdInitiatorConfig.mult
   bfdSummaryModel.reflectorLocalDisc = bfdConfigGlobal.sbfdReflectorLocalDisc
   bfdSummaryModel.reflectorMinRx = bfdConfigGlobal.sbfdReflectorMinRx
   # pylint: disable=protected-access
   bfdSummaryModel._initiatorOnly = sbfdInitiatorOnly
   bfdSummaryModel._reflectorOnly = sbfdReflectorOnly

   ipIntfConfig = ipConfig.ipIntfConfig.get( bfdSummaryModel.localIntf )
   if ipIntfConfig:
      bfdSummaryModel.localIntfIpAdd = ipIntfConfig.addrWithMask.address
   else:
      bfdSummaryModel.localIntfIpAdd = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )

   showBoth = not bfdOnly and not sbfdOnly
   
   types = set( operSessionType.attributes )
   sessions = []
   if bfdOnly:
      sessions = [ "All", "IPv4", "IPv6", "Tunnel", "L2" ]
      types.update( [ "All", "singlehopAll" ] )
   elif sbfdOnly:
      sessions = [ "All", "SR-TE Tunnel", "sbfdIPv4", "sbfdIPv6" ]
      types.update( [ "All" ] )
   else:
      sessions = [ "All", "IPv4", "IPv6", "Tunnel", "L2", "SR-TE Tunnel",
                   "sbfdIPv4", "sbfdIPv6" ]
      types.update( [ "All", "singlehopAll" ] )
      
   for session in sessions:
      bfdSummaryModel.sessions[ session ] = BfdModel.SessionCountByType()
      for sessType in types:
         model = BfdModel.SessionCountByState()
         bfdSummaryModel.sessions[ session ].types[ sessType ] = model
         for state in operState.attributes:
            model.states[ state ] = BfdModel.SessionCount()
            model.states[ state ].async = 0
            model.states[ state ].echo = 0

   allPeerStatus = filterPeerStatus( vrfNames=vrfNames, intfSet=intfSet,
                                     destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                     minTx=minTx, minRx=minRx, 
                                     multiplier=multiplier, proto=proto,
                                     addrFamily=af,
                                     bfdOnly=bfdOnly, sbfdOnly=sbfdOnly,
                                     sbfdInitiatorOnly=sbfdInitiatorOnly,
                                     sbfdReflectorOnly=sbfdReflectorOnly,
                                     reflectorSess=reflectorSess )
   if not allPeerStatus:
      return bfdSummaryModel

   if bfdOnly or showBoth:
      for peerStatus in allPeerStatus:
         typeModel = None
         sessType = peerStatus.type
         if ( sessType == operSessionType.sessionTypeSbfdInitiator or
              sessType == operSessionType.sessionTypeSbfdReflector ):
            continue
         if sessType == operSessionType.sessionTypeMicroLegacy or \
            sessType == operSessionType.sessionTypeMicroRfc7130:
            lagIntf = lagIntfIdForIntfId( peerStatus.peer.intf )
            if lagIntf is None:
               # unable to look up which port-channel this physical port is
               # configured in. The safe thing to do here to return without printing
               # anything.
               continue
            intfStatus = allIntfStatusDir.intfStatus[ lagIntf ]
            if intfStatus and intfStatus.forwardingModel != \
                                                      'intfForwardingModelRouted':
               typeModel = bfdSummaryModel.sessions[ 'L2' ]
         elif sessType == operSessionType.sessionTypeLagRfc7130:
            intfStatus = allIntfStatusDir.intfStatus[ peerStatus.peer.intf ]
            if intfStatus and intfStatus.forwardingModel != \
                                                      'intfForwardingModelRouted':
               typeModel = bfdSummaryModel.sessions[ 'L2' ]

         if not typeModel:
            if sessType == operSessionType.sessionTypeVxlanTunnel:
               typeModel = bfdSummaryModel.sessions[ 'Tunnel' ]
            elif '.' in peerStatus.peer.ip.stringValue:
               typeModel = bfdSummaryModel.sessions[ 'IPv4' ]
            else:
               typeModel = bfdSummaryModel.sessions[ 'IPv6' ]
            
         incrementCount( typeModel, sessType, peerStatus )
         if sessType != operSessionType.sessionTypeLagLegacy and \
            sessType != operSessionType.sessionTypeLagRfc7130:
            incrementCount( typeModel, "All", peerStatus )
            if sessType != operSessionType.sessionTypeMultihop:
               incrementCount( typeModel, "singlehopAll", peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "All" ], "All", peerStatus )
      
   if sbfdOnly or showBoth:
      for peerStatus in allPeerStatus:
         typeModel = None
         sessType = peerStatus.type
         if ( sessType == operSessionType.sessionTypeSbfdInitiator
              and not sbfdReflectorOnly ):
            if peerStatus.localIp.af == AddressFamily.ipv6:
               typeModel = bfdSummaryModel.sessions[ 'sbfdIPv6' ]
            else:
               typeModel = bfdSummaryModel.sessions[ 'sbfdIPv4' ]

            incrementCount( typeModel, sessType, peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "SR-TE Tunnel" ], "All",
                            peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "All" ], "All", peerStatus )

         if ( sessType == operSessionType.sessionTypeSbfdReflector
              and not sbfdInitiatorOnly ):
            typeModel = bfdSummaryModel.sessions[ 'sbfdIPv4' ]
            incrementCount( typeModel, sessType, peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "SR-TE Tunnel" ], "All",
                            peerStatus )
            incrementCount( bfdSummaryModel.sessions[ "All" ], "All", peerStatus )

   return bfdSummaryModel

def lagIntfIdForIntfId( intfId ):
   for key in configBfdLag.bfdLagConfig.iterkeys():
      bfdLagConfig = configBfdLag.bfdLagConfig[ key ]
      if bfdLagConfig.bfdReqPort.get( intfId, None ):
         return bfdLagConfig.intf
   return None

diagToEnum = { 0 : 'diagNone',
               1 : 'diagCtrlTimeout',
               2 : 'diagEchoFail',
               3 : 'diagNeighDown',
               4 : 'diagForwardingReset',
               5 : 'diagPathDown',
               6 : 'diagConcatPathDown',
               7 : 'diagAdminDown',
               8 : 'diagRevConcatPathDown'
             }

stateName = { 0 : 'adminDown',
              1 : 'down',
              2 : 'init',
              3 : 'up',
            }


def printNeighborHistory( pst ):
   peerSnapshotModel = BfdModel.PeerSnapshot()
   
   peerSnapshotModel.ip = pst.peer.ip
   if pst.peer.type == 'sbfdInitiator':
      peerSnapshotModel.intf = ''
   else:
      peerSnapshotModel.intf = pst.peer.intf
   peerSnapshotModel.vrf = pst.peer.vrf
   peerSnapshotModel.status = pst.status
   peerSnapshotModel.eventTime = pst.eventTime
   peerSnapshotModel.event = pst.event
   peerSnapshotModel.lastUp = pst.lastUp
   peerSnapshotModel.lastDown = pst.lastDown
   peerSnapshotModel.lastDiag = diagToEnum[ pst.lastDiag ]
   peerSnapshotModel.authType = authEnumNumToText[ pst.authType ]
   peerSnapshotModel.authProfileName = pst.profileName
   peerSnapshotModel.txInterval = pst.txInterval
   peerSnapshotModel.rxInterval = pst.rxInterval
   peerSnapshotModel.detectTime = pst.detectTime
   peerSnapshotModel.rxCount = pst.rxCount
   peerSnapshotModel.txCount = pst.txCount
   peerSnapshotModel.sentWithin1p = pst.sentWithin1p
   peerSnapshotModel.sentWithin2p = pst.sentWithin2p
   peerSnapshotModel.sentWithin3p = pst.sentWithin3p
   peerSnapshotModel.sentAfter3p = pst.sentAfter3p
   peerSnapshotModel.echoOn = pst.echoOn
   peerSnapshotModel.echoTxInterval = pst.echoTxInterval
   peerSnapshotModel.echoRxInterval = pst.echoRxInterval
   peerSnapshotModel.echoDetectTime = pst.echoDetectTime
   peerSnapshotModel.echoRxCount = pst.echoRxCount
   peerSnapshotModel.echoTxCount = pst.echoTxCount
   peerSnapshotModel.echoSentWithin1p = pst.echoSentWithin1p
   peerSnapshotModel.echoSentWithin2p = pst.echoSentWithin2p
   peerSnapshotModel.echoSentWithin3p = pst.echoSentWithin3p
   peerSnapshotModel.echoSentAfter3p = pst.echoSentAfter3p
   peerSnapshotModel.tunnelId = pst.peer.tunnelId

   return peerSnapshotModel

def intfMatchesFilter( intf, resource ):
   for resourceConfig in bfdHwResourceConfig.itervalues():
      if resourceConfig.hwResource.get( intf ) == resource:
         return True
   return False

def peerMatchesFilter( peer, addrFamily=None, vrfNames=None, intfSet=None,
                       destIpAddr=None, srcIpAddr=None,
                       minTx=None, minRx=None, multiplier=None, proto=None,
                       resource=None, bfdOnly=False, sbfdOnly=False ):
   if ( bfdOnly and peer.type == 'sbfdInitiator' ) or \
         ( sbfdOnly and peer.type != 'sbfdInitiator' ):
      return False
   if not vrfNames:
      vrfNames = [ DEFAULT_VRF ]
   peerStatus = bfdStatusPeer.peerStatus.get( peer )
   hwSessionStatus = None
   echoHwSessionStatus = None
   if peerStatus:
      hwSessionStatus = bfdHwStatus.sessionStatus.get( peerStatus.localDisc )
      if peerStatus.echoOn:
         echoHwSessionStatus = bfdHwStatus.sessionStatus.get( peerStatus.echoDisc )

   peerApps = getApp( peer )
   return ( ( not addrFamily or peer.ip.af in addrFamily )
            and peer.vrf in vrfNames
            and ( not destIpAddr or str( peer.ip ) == destIpAddr )
            and ( not srcIpAddr or peerStatus and
                                   str( peerStatus.localIp ) == srcIpAddr )
            and ( not intfSet or peer.intf in intfSet )
            and ( not minTx or peerStatus and
                               peerStatus.intervalConfig.minTx == minTx )
            and ( not minRx or peerStatus and
                               peerStatus.intervalConfig.minRx == minRx )
            and ( not multiplier or peerStatus and
                                    peerStatus.intervalConfig.mult == multiplier )
            and ( not resource or
               ( hwSessionStatus and hwSessionStatus.hwAccelIntf and
                 intfMatchesFilter( hwSessionStatus.hwAccelIntf, resource ) ) or
               ( echoHwSessionStatus and echoHwSessionStatus.hwAccelIntf and
                 intfMatchesFilter( echoHwSessionStatus.hwAccelIntf, resource ) ) )
            and ( not proto or
                  ( peerApps and
                    proto in set( mapAppName( a ) for a in peerApps ) ) )
   )

def filterPeerStatus( addrFamily=None, vrfNames=None, intfSet=None,
                      destIpAddr=None, srcIpAddr=None,
                      minTx=None, minRx=None, multiplier=None,
                      resource=None, proto=None, bfdOnly=False, sbfdOnly=False,
                      sbfdInitiatorOnly=False, sbfdReflectorOnly=False,
                      reflectorSess=None ):

   if not sbfdReflectorOnly:
      peers = [ p for p in bfdStatusPeer.peerStatus.itervalues()
                if ( peerMatchesFilter( p.peer, addrFamily=addrFamily,
                                        vrfNames=vrfNames, intfSet=intfSet,
                                        destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                        minTx=minTx, minRx=minRx,
                                        multiplier=multiplier, resource=resource,
                                        proto=proto, bfdOnly=bfdOnly,
                                        sbfdOnly=sbfdOnly ) ) ]
   else:
      peers = []

   if ( sbfdInitiatorOnly or bfdOnly or
        not addrFamily or AddressFamily.ipv4 not in addrFamily  or
        not vrfNames or DEFAULT_VRF not in vrfNames or not reflectorSess or
        minTx or multiplier or resource or proto ):
      return peers

   ipIntfConfig = ipConfig.ipIntfConfig.get( bfdConfigGlobal.sbfdLocalIntf )
   localIpStr = ipIntfConfig.addrWithMask.address if ipIntfConfig else '0.0.0.0'
   localIpAddr = Tac.Value( 'Arnet::IpGenAddr', localIpStr )
   if srcIpAddr and srcIpAddr != localIpStr or \
      minRx and minRx != bfdConfigGlobal.sbfdReflectorMinRx :
      return peers

   for sess in reflectorSess.itervalues():
      ipStr = socket.inet_ntoa( pack( "<I", sess.srcIp ) )
      if destIpAddr and destIpAddr != ipStr:
         continue
      peerStatus = getReflectorPeerStatus( sess, localIpAddr )
      peers.append( peerStatus )
   return peers

#------------------------------------------------------------
# check for rfc-7130 specific issues:
# rfc-7130 mac address not registred in kernel
# ipv6 is disabled at /proc/sys/net/ipv6/conf/ for ipv6 peer
# local route not installed for po interface
#------------------------------------------------------------
def checkRfc7130( allPeerStatus, bfdDebugModel ):
   routesByVrf = {}
   key1 = '--------- rfc-7130 mac address 01:00:5e:90:00:01 not registered ' \
          'in kernel---------'
   key2 = '--------- ipv6 disabled at /proc/sys/net/ipv6/conf/ ---------'
   for peerStatus in allPeerStatus:
      peer = peerStatus.peer
      if peerStatus.type == operSessionType.sessionTypeMicroRfc7130 and \
         peerStatus.localIp:
         if peer.vrf not in routesByVrf:
            routesByVrf[ peer.vrf ] = set()
         routesByVrf[ peer.vrf ].add( peerStatus.localIp.stringValue )

      elif peerStatus.type == operSessionType.sessionTypeLagRfc7130:
         # check whether rfc-7130 mac address is registered in kernel
         vrfCmd = 'ip netns exec ns-%s' % peer.vrf \
                  if peer.vrf != DEFAULT_VRF else ''
         devName = eosIntfToKernelIntf( peer.intf )

         cmd = '%s ip maddr show dev %s' % ( vrfCmd, devName )
         output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True,
                           stderr=Tac.CAPTURE, ignoreReturnCode=True )
         if '01:00:5e:90:00:01' not in output:
            bfdDebugModel.addIssue( key1, output )

         # for IPv6 peer check whether disable_ipv6 is 0
         if ':' in peer.ip.stringValue:
            cmd = 'cat /proc/sys/net/ipv6/conf/%s/disable_ipv6' % devName
            output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True,
                              stderr=Tac.CAPTURE, ignoreReturnCode=True )
            if output.strip() != '0':
               bfdDebugModel.addIssue( key2, peer.intf )

   key3 = '--------- missing rfc-7130 required routes ------------'
   for vrf, routes in routesByVrf.items():
      kernelRoutes = set()
      vrfCmd = 'ip netns exec ns-%s' % vrf if vrf != DEFAULT_VRF else ''
      for cmd in [ '%s ip route show table local' % vrfCmd,
                   '%s ip -6 route show table local' % vrfCmd ]:
         output = Tac.run( cmd.split(), stdout=Tac.CAPTURE, asRoot=True,
                           stderr=Tac.CAPTURE, ignoreReturnCode=True )
         lines = output.split( '\n' )
         for line in lines:
            tokens = line.split()
            if len( tokens ) > 2:
               kernelRoutes.add( tokens[ 1 ] )
      for route in routes:
         if route not in kernelRoutes:
            bfdDebugModel.addIssue( key3, vrf + ' ' + route )

#------------------------------------------------------------
# check peerStatus for following:
# peer with slow tx: tx delay > 3 *TxInt
# peer with minRx < 5ms or maxRx > detectTime
# number of sessions down and session diag respectively
#------------------------------------------------------------
def checkBfdPeerStatus( allPeerStatus, bfdDebugModel ):
   if not allPeerStatus:
      return
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )
   netlinkStub.sendGetStats( 0, False, False, False )
   netlinkStub.bfdRead()
   minrx_threshold = 5

   # sessions with slow tx and bursty/slow rx - search "show bfd neighbor detail"
   for peerStatus in allPeerStatus:
      try:
         peer = peerStatus.peer
         peerOperInfo = bfdOperInfoPeer.peerOperInfo[ peer ]
         operInfo = peerOperInfo.info
      except KeyError:
         # This peer has been removed during the show command, so skip it.
         continue

      # down session count
      if peerStatus.status != 'up':
         diag = diagEnumToReason[ diagToEnum[ operInfo.lastDiag ] ]
         if diag not in bfdDebugModel.downSessions:
            bfdDebugModel.downSessions[ diag ] = 0
         bfdDebugModel.downSessions[ diag ] += 1

      rxStats = netlinkStub.receivedRxStats.get( peerStatus.localDisc )
      if bfdHwStatus.isHwAccelerated( peerStatus.localDisc ):
         txStats = getSmashTxStats( peerStatus, echo=False )
      else:
         txStats = netlinkStub.receivedTxStats.get( peerStatus.localDisc )

      peerStr = ( peer.vrf + ' ' + peer.ip.stringValue + ' ' +
                  ( peer.intf if peer.intf else 'NA' ) + ' ' + peer.type )
      if rxStats:
         minPeriod = rxStats.bfdMsgRxStats.minPeriod / 1000
         maxPeriod = rxStats.bfdMsgRxStats.maxPeriod / 1000
         avgPeriod = rxStats.bfdMsgRxStats.avgPeriod / 1000
         if minPeriod < minrx_threshold or \
            maxPeriod > peerStatus.detectTimeInMicrosecond :
            bfdDebugModel.rxIntervals[ peerStr ] = str( minPeriod ) + '/' \
                                       + str( maxPeriod ) + '/' + str( avgPeriod )

      if txStats and txStats.bfdMsgTxStats.sentAfter3p:
         bfdDebugModel.slowTxs[ peerStr ] = txStats.bfdMsgTxStats.sentAfter3p

      # Get echo session stats
      if peerStatus.echoActive:
         rxStats = netlinkStub.receivedRxStats.get( peerStatus.echoDisc )
         if bfdHwStatus.isHwAccelerated( peerStatus.echoDisc ):
            txStats = getSmashTxStats( peerStatus, echo=True )
         else:
            txStats = netlinkStub.receivedTxStats.get( peerStatus.echoDisc )

         if rxStats and rxStats.bfdMsgRxStats.minPeriod:
            minPeriod = rxStats.bfdMsgRxStats.minPeriod / 1000
            maxPeriod = rxStats.bfdMsgRxStats.maxPeriod / 1000
            avgPeriod = rxStats.bfdMsgRxStats.avgPeriod / 1000
            if minPeriod < minrx_threshold \
               or maxPeriod > peerStatus.detectTimeInMicrosecond :
               bfdDebugModel.echoRxIntervals[ peerStr ] = str( minPeriod ) + '/' \
                                       + str( maxPeriod ) + '/' + str( avgPeriod )

         if txStats and txStats.bfdMsgTxStats.sentAfter3p:
            bfdDebugModel.slowEchoTxs[ peerStr ] = txStats.bfdMsgTxStats.sentAfter3p

#------------------------------------------------------------
# return the value after specified token, i.e. for 'rxbfd_nolagdev 3'
# return '3'
#------------------------------------------------------------
def findTokenValue( output, token ):
   match = re.search( r'(\s|\()%s\s\w*' % token, output )
   if match:
      return match.group().split()[ 1 ]
   return ''

#------------------------------------------------------------
# check in dma and record following issues:
# rxbfd_nolagdev > 0
# fdevtx_locktime > 1000us
#------------------------------------------------------------
def checkDma( bfdDebugModel ):
   tokens = [ 'rxbfd_nolagdev', 'fdevtx_locktime' ]
   key = '---------- show platform pkt -----------'
   locktime_threshold = 1000 # 1000us
   try: # pylint: disable-msg=R1702, too-many-nested-blocks
      output = Tac.run( [ '/usr/bin/fab', 'dump' ], stdout=Tac.CAPTURE, asRoot=True )
      for token in tokens:
         value = findTokenValue( output, token )
         if value:
            try:
               if token == 'rxbfd_nolagdev' and int( value ):
                  # rxbfd_nolagdev > 0
                  bfdDebugModel.addIssue( key, token + ' ' + value )
               if token == 'fdevtx_locktime':
                  # fdevtx_locktime > 1000us
                  if int( value.split( 'us' )[ 0 ] ) > locktime_threshold:
                     bfdDebugModel.addIssue( key, token + ' ' + value )
            except ( ValueError, IndexError ):
               pass
   except: # pylint: disable=bare-except
      pass

#------------------------------------------------------------
# check for intf counter discards
#------------------------------------------------------------
def checkIntfCounter( mode, bfdDebugModel ):
   with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) as out:
      mode.session_.runCmd( "show interfaces counters discards | nz", aaa=False )
   output = out.contents().strip( '\n' )
   if 'Totals' in output:
      key = "---------- interfaces counters discards -----------"
      bfdDebugModel.addIssue( key, output )

#------------------------------------------------------------
# check for cpu counters drop
#------------------------------------------------------------
def checkCpuCounter( mode, bfdDebugModel ):
   try:
      with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) as out:
         mode.session_.runCmd( "show cpu counters queue | nz", aaa=False )
      lines = out.contents().split( '\n' )
      key = "---------- cpu counters drop -----------"
      # print any queue that has drops
      for line in lines:
         tokens = line.split()
         if tokens:
            try:
               # the last value in line is drop counter
               if int( tokens[ -1 ] ):
                  # if drop counter is non-zero
                  bfdDebugModel.addIssue( key, line )
            except ValueError:
               pass
   except: # pylint: disable=bare-except
      # "show cpu counters queue" command failed on this platform
      pass

#------------------------------------------------------------
# check for kernel qdisc drop
#------------------------------------------------------------
def checkQdisc( bfdDebugModel ):
   key = '---------- kernel qdisc drop -----------------'
   output = Tac.run( [ 'tc', '-s', 'qdisc', 'show' ], stdout=Tac.CAPTURE,
                     asRoot=True, stderr=Tac.CAPTURE, ignoreReturnCode=True )
   lines = output.split( '\n' )
   while len( lines ) > 3:
      devName = findTokenValue( lines[ 0 ], 'dev' )
      # record qdisc drop on interfaces
      if devName:
         # output for each dev takes 3 lines
         dropped = findTokenValue( lines[ 1 ], 'dropped' )
         try:
            if int( dropped ):
               bfdDebugModel.addIssue( key, '\n'.join( lines[ :3 ] ) )
         except ( ValueError, IndexError ):
            pass
      lines = lines[ 3: ]

#------------------------------------------------------------
# check for iptables drop
#------------------------------------------------------------
def checkIptablesDrop( allPeerStatus, bfdDebugModel, vrfNames ):
   key = '---------- iptables drop -----------------'
   vrfs = set()
   for peerStatus in allPeerStatus:
      vrf = peerStatus.peer.vrf
      if vrf in vrfNames:
         vrfs.add( vrf )

   for vrf in vrfs:
      timeoutCmd = 'timeout -s SIGKILL 2'
      vrfCmd = 'ip netns exec ns-%s' % vrf if vrf != DEFAULT_VRF else ''
      cmd = '%s %s iptables -vnL' % ( timeoutCmd, vrfCmd )
      output = Tac.run( cmd.split(), stdout=Tac.CAPTURE,
                        asRoot=True, stderr=Tac.CAPTURE, ignoreReturnCode=True )
      lines = output.split( '\n' )
      for line in lines:
         if 'DROP' in line:
            tokens = line.split()
            try:
               if int( tokens[ 0 ] ) or int( tokens[ 1 ] ):
                  bfdDebugModel.addIssue( key, vrf + '\n' + line )
            except ( ValueError, IndexError ):
               pass

def showBfdDebug( mode, vrfNames, intfSet, destIpAddr, srcIpAddr,
                  minTx, minRx, multiplier, proto, addrFamily, 
                  bfdOnly=False, sbfdOnly=False ):
   bfdDebugModel = BfdModel.BfdDebug()
   allPeerStatus = filterPeerStatus( vrfNames=vrfNames, intfSet=intfSet,
                                     destIpAddr=destIpAddr, srcIpAddr=srcIpAddr,
                                     minTx=minTx, minRx=minRx,
                                     multiplier=multiplier, proto=proto,
                                     addrFamily=addrFamily,
                                     bfdOnly=bfdOnly, sbfdOnly=sbfdOnly )
   if not allPeerStatus:
      return bfdDebugModel
   checkRfc7130( allPeerStatus, bfdDebugModel )
   checkBfdPeerStatus( allPeerStatus, bfdDebugModel )

   checkCpuCounter( mode, bfdDebugModel )
   checkIntfCounter( mode, bfdDebugModel )
   checkIptablesDrop( allPeerStatus, bfdDebugModel, vrfNames )
   checkDma( bfdDebugModel )
   checkQdisc( bfdDebugModel )

   return bfdDebugModel

def getApp( peer ):
   peerEntry = bfdConfigPeer.peerConfig.get( peer )
   return sorted( peerEntry.registeredApp.keys() ) if peerEntry else None

def showBfdHwUtilization( mode, resourceFilter=None, vrfName=None, intfs=None,
      destIp=None, detail=False ):
   bfdHwResourceListModel = BfdModel.BfdHwResourceList(
         _detailed=detail )

   if resourceFilter:
      for resourceConfig in bfdHwResourceConfig.itervalues():
         if resourceFilter in resourceConfig.maxHwResourceSessions:
            break
      else:
         mode.addError( '"%s" is not a valid resource name' % resourceFilter )
         return None

   if detail and vrfName:
      if not vrfExists( vrfName ):
         mode.addError( "VRF %s not configured." % vrfName )
         return None
      vrfNames = [ vrfName ]
   else:
      vrfNames = set( getAllVrfNames( mode ) )

   # Include member intfs if filtered by LAG intf
   intfSet = set()
   if detail and intfs:
      for intf in list( intfs.intfNames() ):
         intfSet.add( intf )
         if intf.startswith( "Port-Channel" ) and \
            intf in configBfdLag.bfdLagConfig:
            bfdLagConfig = configBfdLag.bfdLagConfig[ intf ]
            for memberIntf in bfdLagConfig.bfdReqPort:
               intfSet.add( memberIntf )

   if destIp is not None and not isinstance( destIp, str ):
      destIp = destIp.stringValue

   for resourceConfig in bfdHwResourceConfig.itervalues():
      for resource, maxSessions in resourceConfig.maxHwResourceSessions.iteritems():
         if resourceFilter and resource != resourceFilter:
            continue
         resourceModel = BfdModel.BfdHwResource( maxSessions=maxSessions )
         bfdHwResourceListModel.resources[ resource ] = resourceModel

         allPeerStatus = filterPeerStatus( vrfNames=vrfNames,
                                           intfSet=intfSet,
                                           destIpAddr=destIp,
                                           resource=resource )

         if not detail:
            resourceModel.numSessions = 0
            resourceModel.sessions = None

         for peerStatus in allPeerStatus:
            if bfdHwStatus.isHwAccelerated( peerStatus.localDisc ):
               if detail:
                  p = peerStatus.peer
                  sessionModel = BfdModel.BfdHwResourceSession( ip=p.ip, intf=p.intf,
                        vrf=p.vrf, sessType=peerStatus.type,
                        localDisc=peerStatus.localDisc )
                  resourceModel.sessions.append( sessionModel )
               else:
                  resourceModel.numSessions += 1

            if ( peerStatus.echoOn and
                 bfdHwStatus.isHwAccelerated( peerStatus.echoDisc ) ):
               if detail:
                  p = peerStatus.peer
                  sessionModel = BfdModel.BfdHwResourceSession( ip=p.ip,
                        intf=p.intf, vrf=p.vrf, sessType=peerStatus.type + " echo",
                        localDisc=peerStatus.echoDisc )
                  resourceModel.sessions.append( sessionModel )
               else:
                  resourceModel.numSessions += 1

   return bfdHwResourceListModel

def showBfdHwUtilizationHandler( mode, args ):
   resourceFilter = args.get( "RESOURCE" )
   detail = ( "detail" in args )
   vrfName = args.get( 'VRF' )
   intfs = args.get( 'INTERFACE' )
   destIp = args.get( 'DESTIP' ) or args.get( 'DESTIP6' )

   return showBfdHwUtilization( mode,
                                resourceFilter=resourceFilter,
                                vrfName=vrfName,
                                intfs=intfs,
                                destIp=destIp,
                                detail=detail )

#--------------------------------------------------------------------------------
# show bfd ( ( ip access-list [ IPACL ] ) | ( ipv6 access-list [ IP6ACL ] ) )
#--------------------------------------------------------------------------------
def showAcl( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   name = args[ '<aclNameExpr>' ]
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 name,
                                 serviceName='bfd' )

class BfdIpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show bfd ( ( ip access-list [ IPACL ] ) | '
            '( ipv6 access-list [ IP6ACL ] ) )' )
   data = {
      'bfd': matcherBfd,
      'ip': ipMatcherForShow,
      'ipv6': ipv6MatcherForShow,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
      'IPACL': AclCli.ipAclNameExpression,
      'IP6ACL': AclCli.ip6AclNameExpression,
   }
   handler = showAcl
   cliModel = AllAclList

BasicCli.addShowCommandClass( BfdIpAccessListCmd )

#--------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] [ detail ]
#
# show bfd peers [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] 
#    [ protocol PROTO ] [ detail ]
# sbfd reflectors filter would be added in the future. 
#--------------------------------------------------------------------------------
def showBfdNeighborHandler( mode, args ):
   bfdOnly = 'bfdonly' in args
   sbfdOnly = 'sbfd' in args
   sbfdInitiatorOnly = 'initiators' in args
   sbfdReflectorOnly = 'reflectors' in args
   # As reflector stats is not supported yet, just blindly ignore arg 
   # 'initiators' and always print initiators stats when sbfd peers are
   # needed for show.
   vrfName = args.get( 'VRF', ALL_VRF_NAME )
   intfs = args[ 'INTERFACE' ] if 'interface' in args else None
   destIp = None
   srcIp = None
   if 'dest-ip' in args:
      if 'DESTIP' in args:
         destIp = args[ 'DESTIP' ]
      elif 'DESTIP6' in args:
         destIp = args[ 'DESTIP6' ]
   if 'src-ip' in args:
      if 'SRCIP' in args:
         srcIp = args[ 'SRCIP' ]
      elif 'SRCIP6' in args:
         srcIp = args[ 'SRCIP6' ]
   interval = args.get( 'INTERVAL' )
   minRx = args.get( 'MIN_RX' )
   mult = args.get( 'MULTIPLIER' )
   proto = args.get( 'PROTO' )
   af = args.get( 'AF' )
   displayType = None
   if 'summary' in args:
      displayType = 'summary'
   if 'debug' in args:
      displayType = 'debug'
   if 'history' in args:
      displayType = 'history'
   if 'detail' in args:
      displayType = 'detail'

   return showBfdNeighbor( mode, vrfName=vrfName, intfs=intfs,
                           destIpAddr=destIp, srcIpAddr=srcIp, minTx=interval,
                           minRx=minRx, multiplier=mult, proto=proto,
                           displayType=displayType, addrFamilies=af,
                           bfdOnly=bfdOnly, sbfdOnly=sbfdOnly,
                           sbfdInitiatorOnly=sbfdInitiatorOnly,
                           sbfdReflectorOnly=sbfdReflectorOnly )

def mapAppName( name, showDetail=False ):
   mapping = {
         'ospfv3_ipv4': 'ospfv3', # simplify ospfv3 tokens
         'ospfv3_ipv6': 'ospfv3',
         'pim6': 'pim', # simplify pim tokens
         'fhrp': 'vrrp', # fhrp encompasses vrrp and varp-- only vrrp cares about bfd
         'static': 'static-bfd', # clarity
         # make this a nicer token for cli
         'srtepolicy': 'sr-te policy' if showDetail else 'sr-te',
   }
   return mapping.get( name, name )

def getProtocolNames( mode ):
   # Vxlan peer config lives in a different dir in Sysdb
   apps = [  a for a in chain( appConfigDir, [ 'vxlan' ] ) if a != 'test' ]
   return { mapAppName( a ): '' for a in set( sorted( apps ) ) }

class BfdSbfdFilterExpression( CliCommand.CliExpression ):

   expression = 'bfdonly | ( sbfd [ initiators | reflectors ] )'

   data = { 
      'bfdonly': CliMatcher.KeywordMatcher( 'bfd',
                                 helpdesc='Status for BFD sessions only' ),
      'sbfd': matcherSbfd,
      'initiators': matcherInitiators,
      'reflectors': matcherReflectors,
   }

class BfdShowPeersCmdArgsExpression( CliCommand.CliExpression ):

   expression = '''{ AF
                   | VRF
                   | ( interface INTERFACE )
                   | ( dest-ip ( DESTIP | DESTIP6 ) )
                   | ( src-ip ( SRCIP | SRCIP6 ) )
                   | ( interval INTERVAL )
                   | ( ( min-rx | min_rx ) MIN_RX )
                   | ( multiplier MULTIPLIER )
                   | ( protocol PROTO ) }'''

   data = {
      'VRF': vrfExprFactory,
      'interface': singleNode( matcherInterface ),
      'INTERFACE': singleNode( IntfRangeMatcher() ),
      'dest-ip': singleNode( matcherDestIp ),
      'DESTIP': singleNode( IpAddrMatcher( helpdesc='IPv4 Neighbor address' ) ),
      'DESTIP6': singleNode( Ip6AddrMatcher( helpdesc='IPv6 Neighbor address' ) ),
      'src-ip': singleNode( matcherSrcIp ),
      'SRCIP': singleNode( IpAddrMatcher( helpdesc='IPv4 Source address' ) ),
      'SRCIP6': singleNode( Ip6AddrMatcher( helpdesc='IPv6 Source address' ) ),
      'interval': singleNode( matcherInterval ),
      'INTERVAL': singleNode( CliMatcher.IntegerMatcher( 50, 60000,
                     helpdesc='Rate in milliseconds between 50-60000' ) ),
      'min-rx': singleNode( matcherMinRx ),
      'min_rx': matcherMin_RxDeprecated,
      'MIN_RX': singleNode( CliMatcher.IntegerMatcher( 50, 60000,
                     helpdesc='Rate in milliseconds between 50-60000' ) ),
      'multiplier': singleNode( matcherMultiplier ),
      'MULTIPLIER': singleNode( CliMatcher.IntegerMatcher( 3, 50,
                     helpdesc='Range is 3-50' ) ) ,
      'protocol': singleNode( matcherProtocol ),
      'PROTO': singleNode( CliMatcher.DynamicKeywordMatcher( getProtocolNames ) ),
      'AF': singleNode( matcher=EnumMatcher( {
         'ipv4': 'Sessions over IPv4',
         'ipv6': 'Sessions over IPv6',
      } ) )
   }

class BfdPeersCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] [ detail ]' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'detail': matcherNbrDetail,
   }

   handler = showBfdNeighborHandler
   cliModel = BfdNeighbors

BasicCli.addShowCommandClass( BfdPeersCmd )

#--------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] debug
#
# show bfd peers [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ]
#    [ interface INTERFACE ] [ dest-ip ( DESTIP | DESTIP6 ) ]
#    [ src-ip ( SRCIP | SRCIP6 ) ] [ interval INTERVAL ] [ min-rx MIN_RX ]
#    [ multiplier MULTIPLIER ] [ protocol PROTO ] debug
#--------------------------------------------------------------------------------
class BfdPeersDebugCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] debug' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'debug': matcherDebug,
   }

   handler = showBfdNeighborHandler
   cliModel = BfdDebug

BasicCli.addShowCommandClass( BfdPeersDebugCmd )

#--------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] history
#
# show bfd peers [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] 
#    [ protocol PROTO ] history
# sbfd reflectors filter would be added in the future.
#--------------------------------------------------------------------------------
class BfdPeersHistoryCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] history' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'history': matcherHistory,
   }

   handler = showBfdNeighborHandler
   cliModel = PeerHistory

BasicCli.addShowCommandClass( BfdPeersHistoryCmd )

#--------------------------------------------------------------------------------
# Deprecated syntax:
# show bfd neighbors [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] summary
#
# show bfd peers [ bfd | ( sbfd [initiators | reflectors] ) ]
#    [ ipv4 | ipv6 ] [ vrf ( default | all | VRFNAME ) ] [ interface INTERFACE ]
#    [ dest-ip ( DESTIP | DESTIP6 ) ] [ src-ip ( SRCIP | SRCIP6 ) ]
#    [ interval INTERVAL ] [ min_rx MIN_RX ] [ multiplier MULTIPLIER ] 
#    [ protocol PROTO ] summary
# sbfd reflectors filter would be added in the future, but global state for 
# reflector is available now.
#--------------------------------------------------------------------------------
class BfdPeersSummaryCmd( ShowCommand.ShowCliCommandClass ):

   syntax = ( 'show bfd ( neighbors | peers ) [ BFD_SBFD_FILTER ] '
              '[ SHOW_CMD_EXPR ] summary' )
   data = {
      'bfd': matcherBfd,
      'neighbors': neighborsDeprecatedNode,
      'peers': matcherPeers,
      'BFD_SBFD_FILTER': BfdSbfdFilterExpression,
      'SHOW_CMD_EXPR': BfdShowPeersCmdArgsExpression,
      'summary': matcherSummary,
   }

   handler = showBfdNeighborHandler
   cliModel = BfdSummary

BasicCli.addShowCommandClass( BfdPeersSummaryCmd )

#--------------------------------------------------------------------------------
# show bfd rbfd-stats
#--------------------------------------------------------------------------------
def showRbfdStats( mode, args ):
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )
   # We get stats for all VRFs
   netlinkStub.sendGetStats( 0, False, False, True )
   netlinkStub.bfdRead()
   rbfdStatsModel = RbfdStats()
   rbfdStatsModel.rfcDrops = netlinkStub.rfcDrops
   rbfdStatsModel.rxPacketMissing = netlinkStub.rxPacketMissing
   rbfdStatsModel.numInvalidSessions = netlinkStub.numInvalidSessions
   rbfdStatsModel.numPassupDisc0 = netlinkStub.numPassupDisc0
   rbfdStatsModel.netlinkPacketChangeFail = netlinkStub.netlinkPacketChangeFail
   rbfdStatsModel.netlinkFailureFail = netlinkStub.netlinkFailureFail
   rbfdStatsModel.pskbExpandFail = netlinkStub.pskbExpandFail
   return rbfdStatsModel

class BfdRbfdStatsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bfd rbfd-stats'
   data = {
      'bfd': matcherBfd,
      'rbfd-stats': 'Show aggregated rbfd module statistics',
   }
   handler = showRbfdStats
   cliModel = RbfdStats
   hidden = True

BasicCli.addShowCommandClass( BfdRbfdStatsCmd )

#--------------------------------------------------------------------------------
# show bfd sbfd-stats reflectors
#--------------------------------------------------------------------------------
def showSbfdStats( mode, args ):
   globalStats, _ = getReflectorStats()
   stats = {} if globalStats == {} else globalStats._asdict()
   sbfdStatsModel = SbfdStats()
   for attr in sbfdReflectorGlobal._fields:
      setattr( sbfdStatsModel, attr, stats.get( attr, 0 ) )
   return sbfdStatsModel

class BfdSbfdStatsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bfd sbfd-stats reflectors'
   data = {
      'bfd': matcherBfd,
      'sbfd-stats': 'Show sbfd global statistics',
      'reflectors': 'Show sbfd reflectors global statistics',
   }
   handler = showSbfdStats
   cliModel = SbfdStats
   hidden = True

BasicCli.addShowCommandClass( BfdSbfdStatsCmd )

if toggleHwBfdEnabled():
   #------------------------------------------------------------------------------
   # show bfd hardware acceleration
   #------------------------------------------------------------------------------
   class ShowBfdHwAccelCmd( ShowCommand.ShowCliCommandClass ):
      syntax = '''show bfd hardware acceleration'''
      data = { 'bfd' : matcherBfd,
               'hardware': matcherHardware,
               'acceleration': 'BFD processing in hardware',
               }
      cliModel = BfdHwAccel

      @staticmethod
      def handler( mode, args ):
         reasons = []
         if bfdConfigGlobal.hwAccelerationConfig == 'disabledByCli':
            reasons.append( "user disabled" )
         if not ( bfdConfigPeer.hwEligiblePeers or bfdHwStatus.hwSessionsInUse ):
            reasons.append( "no eligible sessions" )
         if ( not bfdHwConfig.hwbfdSsoSupported and
              Cell.cellType() == 'supervisor' and
              redundancyConfig.protocol == 'sso' ):
            reasons.append( "not supported with SSO" )
         return BfdModel.BfdHwAccel(
               supported=bfdHwConfig.hwAccelerationSupported,
               running=bfdHwStatus.hwAccelerationEnabled,
               reasons=reasons )

   BasicCli.addShowCommandClass( ShowBfdHwAccelCmd )

   #--------------------------------------------------------------------------------
   # show bfd hardware utilization [ RESOURCE ] [ detail [ { ( vrf VRF ) |
   #    ( interface INTERFACE ) | ( dest-ip ( DESTIP | DESTIP6 ) ) } ] ]
   #--------------------------------------------------------------------------------
   class ShowBfdHwUtilizationCmd( ShowCommand.ShowCliCommandClass ):
      syntax = '''show bfd hardware utilization [ RESOURCE ]
                  [ detail [ { ( VRF )
                             | ( interface INTERFACE )
                             | ( dest-ip ( DESTIP | DESTIP6 ) ) } ] ]'''
      data = { 'bfd' : matcherBfd,
               'hardware' : matcherHardware,
               'utilization' : 'Resource utilization information',
               'RESOURCE' : matcherResource,
               'detail': 'Detailed list of sessions per resource',
               'VRF': vrfExprFactory,
               'interface': singleNode( matcherInterface ),
               'INTERFACE': singleNode( IntfRangeMatcher() ),
               'dest-ip': singleNode( matcherDestIp ),
               'DESTIP': singleNode(
                  IpAddrMatcher( helpdesc='IPv4 Neighbor address' ) ),
               'DESTIP6': singleNode(
                  Ip6AddrMatcher( helpdesc='IPv6 Neighbor address' ) ),
               }
      cliModel = BfdHwResourceList
      handler = showBfdHwUtilizationHandler

   BasicCli.addShowCommandClass( ShowBfdHwUtilizationCmd )

#-------------------------------------------------------------------------------
# BFD commands in "show tech-support"
#-------------------------------------------------------------------------------

def _showTechSupportCmds():
   cmds = [ "show bfd peers summary",
            "show bfd peers history",
            "show bfd peers detail",
            "show bfd rbfd-stats",
            "show bfd peers debug",
            "bash cat /proc/net/stat/arp_cache",
            "bash cat /proc/net/stat/ndisc_cache" ]
   if toggleHwBfdEnabled():
      cmds += [
            "show bfd hardware utilization detail",
          ]
   return cmds

# These commands were in the original "show tech" command list from the EOS
# package.  To avoid having them move around in the "show tech" output,
# we use a made up timestamp.
TechSupportCli.registerShowTechSupportCmdCallback(
   '2010-09-30 00:00:00', _showTechSupportCmds,
   summaryCmdCallback=lambda: [ 
      "show bfd peers summary",
   ] )


def Plugin( entityManager ):
   global aclStatus
   global aclCpConfig
   global aclCheckpoint
   global appConfigDir
   global bfdConfigGlobal
   global bfdHwStatus
   global bfdHwConfig
   global bfdStatusPeer
   global bfdConfigInfo
   global bfdConfigPeer
   global bfdOperInfoPeer
   global bfdStatsSmash
   global configBfdLag
   global allIntfStatusDir
   global bfdConfigIntf
   global bfdHwResourceConfig
   global redundancyConfig
   global ipConfig

   aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                'Acl::Status', 'r' )
   aclCpConfig = LazyMount.mount( entityManager, 'acl/cpconfig/cli',
                                    'Acl::Input::CpConfig', 'r' )
   aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                    'Acl::CheckpointStatus', 'r' )
   appConfigDir = LazyMount.mount( entityManager, 'bfd/config/app',
                                     'Tac::Dir', 'ri' )
   bfdConfigGlobal = LazyMount.mount( entityManager, 'bfd/config/global',
                                        'Bfd::ConfigGlobal', 'r' )
   bfdHwConfig = LazyMount.mount( entityManager, 'bfd/hwConfig',
                                  'Hardware::Bfd::Config', 'r' )
   bfdHwStatus = LazyMount.mount( entityManager, 'bfd/hwStatus',
                                  'Hardware::Bfd::Status', 'r' )
   bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                    'Bfd::StatusPeer', 'r')
   bfdConfigInfo = LazyMount.mount( entityManager, 'bfd/config/info',
                                    'Bfd::ConfigInfo', 'r')
   bfdConfigPeer = LazyMount.mount( entityManager, 'bfd/config/peer',
                                    'Bfd::ConfigPeer', 'r')
   bfdOperInfoPeer = LazyMount.mount( entityManager,
                                      'bfd/status/peerOper',
                                      'Bfd::OperInfoPeer', 'r')
   bfdStatsSmash = SmashLazyMount.mount( entityManager, 'bfd/hwTxStats',
                                         'Bfd::HwBfdSmashTxStats',
                                         Smash.mountInfo( 'reader' ) )
   configBfdLag = LazyMount.mount( entityManager, 'bfd/config/lag',
                                   'Bfd::ConfigBfdLag', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, 'interface/status/all',
                                       'Interface::AllIntfStatusDir', 'r' )
   bfdConfigIntf = LazyMount.mount( entityManager, 'bfd/config/intf',
                                      'Bfd::ConfigIntf', 'r')
   bfdHwResourceConfig = LazyMount.mount( entityManager, 'bfd/hwResourceConfig',
                                          'Tac::Dir', 'ri' )
   redundancyConfig = LazyMount.mount( entityManager, 'redundancy/config',
                                       'Redundancy::RedundancyConfig', 'r' )
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
