# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import time
from socket import AF_INET
from socket import AF_INET6
from collections import defaultdict
import TableOutput
from ArnetModel import IpGenericAddress
from IpLibConsts import DEFAULT_VRF
import Tac
from BfdLib import ( compare_peer, compareIp, compareHwSession, diagEnumToReason,
                     dispTime, operSessionEnumToType, authEnumNumToText )
from CliModel import Bool, Dict, Enum, Float, Int, List, Model, Str, Submodel
from CliPlugin.IntfModel import Interface
import CliPlugin.BfdCli as BfdCli
from TypeFuture import TacLazyType
from Toggles.BfdToggleLib import toggleHwBfdEnabled, toggleBfdTelemetryEnabled

operSessionType = Tac.Type( "Bfd::OperSessionType" )

OperSessionTypeEnum = Enum( values=[ operSessionType.sessionTypeNormal, 
                                     operSessionType.sessionTypeLagLegacy,
                                     operSessionType.sessionTypeLagRfc7130, 
                                     operSessionType.sessionTypeMicroLegacy, 
                                     operSessionType.sessionTypeMicroRfc7130,
                                     operSessionType.sessionTypeVxlanTunnel,
                                     operSessionType.sessionTypeMultihop,
                                     operSessionType.sessionTypeSbfdInitiator,
                                     operSessionType.sessionTypeSbfdReflector,
                                     operSessionType.sessionTypeNormal + " echo",
                                     operSessionType.sessionTypeLagLegacy + " echo",
                                     operSessionType.sessionTypeLagRfc7130 + " echo",
                                     operSessionType.sessionTypeMicroLegacy +
                                       " echo",
                                     operSessionType.sessionTypeMicroRfc7130 +
                                       " echo",
                                     operSessionType.sessionTypeVxlanTunnel +
                                       " echo",
                                     operSessionType.sessionTypeMultihop + " echo" ],
                            help="BFD session type" )

operSessionEnumToStr = {
   operSessionType.sessionTypeNormal :                  "normal",
   operSessionType.sessionTypeLagLegacy :               "LAG per-link *",
   operSessionType.sessionTypeLagRfc7130 :              "LAG RFC7130 *",
   operSessionType.sessionTypeMicroLegacy :             "micro per-link",
   operSessionType.sessionTypeMicroRfc7130 :            "micro RFC7130",
   operSessionType.sessionTypeVxlanTunnel :             "VXLAN",
   operSessionType.sessionTypeMultihop :                "multi-hop",
   operSessionType.sessionTypeSbfdInitiator:            "Initiator",
   operSessionType.sessionTypeSbfdReflector:            "Reflector",
   operSessionType.sessionTypeNormal + " echo" :        "normal echo",
   operSessionType.sessionTypeLagLegacy + " echo" :     "LAG per-link * echo",
   operSessionType.sessionTypeLagRfc7130 + " echo" :    "LAG RFC7130 * echo",
   operSessionType.sessionTypeMicroLegacy + " echo" :   "micro per-link echo",
   operSessionType.sessionTypeMicroRfc7130 + " echo" :  "micro RFC7130 echo",
   operSessionType.sessionTypeVxlanTunnel + " echo" :   "VXLAN echo",
   operSessionType.sessionTypeMultihop + " echo" :      "multi-hop echo",
   "singlehopAll":                                      "All",
   "All":                                               "All",
}

TunnelIdType = TacLazyType( "Tunnel::TunnelTable::TunnelId" )

def capitalize( x ):
   # x is a string. Capitalize first letter and leave the rest of the string
   # intact
   return x[0].upper() + x[1:]

def convertToMs( microseconds ):
   return microseconds / 1000

# timeElapsed can never be negative unless UTC wraps or the system is reset to an
# earlier time
def dispElapsedTime( timeElapsed, state ):
   if state != 'up' or timeElapsed < 0: 
      return "NA"

   secondsPerMinute = 60
   secondsPerHour   = secondsPerMinute*60
   secondsPerDay    = secondsPerHour*24

   rem = timeElapsed
   ( days, rem ) = divmod( rem, secondsPerDay )
   ( hrs, rem )  = divmod( rem, secondsPerHour )
   ( mins, rem ) = divmod( rem, secondsPerMinute )
   ( secs, rem ) = divmod( rem, 1 )
   # get first two digits
   centisecs = rem * 100

   if days:
      disp = "%d days, %02d:%02d:%02d.%02d" % ( days, hrs, mins, secs, centisecs )
   elif hrs:
      disp = "%02d:%02d:%02d.%02d" % ( hrs, mins, secs, centisecs )
   elif mins:
      disp = "%02d:%02d.%02d" % ( mins, secs, centisecs )
   else:
      disp = "%02d.%02d" % ( secs, centisecs )

   return disp

class PeerSnapshot( Model ):
   ip = IpGenericAddress( help="IP address of peer" )
   intf = Interface( help="Local egress interface" )
   vrf = Str( help="Vrf that peer is configured in" )
   status = Enum( values=[ "adminDown", "down", "init", "up" ], help="State of "
      "peer session" )
   eventTime = Float( help="Time of BFD session event in seconds" )
   event = Enum( values=[ "sessionDelete", "sessionDown", "sessionAdminDown", 
                          "sessionPeerAdminDown", "sessionNoEvent",
                          "sessionPeerDown" ], help="BFD session event" )
   lastUp = Float( help="Last time peer session went up in seconds, 0.0 if never "
      "up" )
   lastDown = Float( help="Last time peer session went down in seconds, 0.0 if "
      "never down" )
   lastDiag = Enum( values= [ 'diagNone', 'diagCtrlTimeout', 'diagEchoFail',
      'diagNeighDown', 'diagForwardingReset', 'diagPathDown', 'diagConcatPathDown',
      'diagAdminDown', 'diagRevConcatPathDown' ], help="Last diagnostic for reason "
      "of change to Down state" )
   authType = Enum( values=authEnumNumToText.values(), help="Authentication mode" )
   authProfileName = Str( help="Configured authentication profile" )
   txInterval = Int( help="Rate in microseconds at which BFD control packets are "
      "sent" )
   rxInterval = Int( help="Rate in microseconds at which BFD control packets are "
      "received" )
   detectTime = Int( help="Detection timeout in microseconds" )
   rxCount = Int( help="Number of packets received" )
   txCount = Int( help="Number of packets sent" )
   sentWithin1p = Int( help="Number of packets sent within configured period since "\
      "last packet" )
   sentWithin2p = Int( help="Number of packets sent later than one period but "
      "before two periods of last packet" )
   sentWithin3p = Int( help="Number of packets sent later than two periods but "
      "before three periods of last packet" )
   sentAfter3p = Int( help="Number of packets sent later than three periods of "
      "last packet" )
   echoOn = Bool( help="Echo turned on" )
   echoTxInterval = Int( help="Rate in microseconds at which BFD echo packets are "
      "sent" )
   echoRxInterval = Int( help="Rate in microseconds at which BFD echo packets are "
      "received" )
   echoDetectTime = Int( help="Detection timeout in microseconds for echo" )
   echoRxCount = Int( help="Number of echo packets received" )
   echoTxCount = Int( help="Number of echo packets sent" )
   echoSentWithin1p = Int( help="Number of echo packets sent within configured " \
      "period since last echo packets" )
   echoSentWithin2p = Int( help="Number of echo packets sent later than one " \
      "period but before two periods of last echo packet" )
   echoSentWithin3p = Int( help="Number of echo packets sent later than two " \
      "periods but before three periods of last echo packet" )
   echoSentAfter3p = Int( help="Number of echo packets sent later than three " \
      "periods of last echo packet" )
   tunnelId = Int( help="TunnelID of SBFD Initiator", optional=True )

class PeerHistory( Model ):
   peerSnapshots = List( valueType=PeerSnapshot, help="List of BFD peer history" )

   def render( self ):
      eventMessage = { 'sessionDown': "Processed BFD session down event at %s",
                       'sessionPeerDown': "Processed Peer BFD down event at %s",
                       'sessionAdminDown': "Processed BFD admin down event at %s",
                       'sessionPeerAdminDown': "Processed Peer BFD admin down event "
                                               "at %s",
                       'sessionDelete': "Processed BFD session delete request from "
                                        "the protocol at %s" }
      for pst in self.peerSnapshots:
         if pst.tunnelId:
            slId = TunnelIdType( pst.tunnelId ).extractIndexAndAf()
            col3 = "Tunnel ID %d(SR), Segment list ID %d" % ( pst.tunnelId, slId )
         else:
            col3 = "Intf %s" % pst.intf.stringValue
         print "Peer VRF %s, Addr %s, %s, State %s" % (
                pst.vrf, pst.ip, col3, capitalize( pst.status ) )
         if pst.event in eventMessage.keys():
            message = eventMessage[ pst.event ] % dispTime( pst.eventTime )
            if pst.tunnelId:
               message = message.replace( 'BFD', 'SBFD(Initiator)' )
            print message
         else:
            print "Processed UNKNOWN(%s) event at %s" % ( pst.event,
                  dispTime( pst.eventTime ) )
         print "Last Up %s, Last Down %s" \
             % ( dispTime( pst.lastUp ), dispTime( pst.lastDown ) )
         print "Last Diag: %s" % diagEnumToReason[ pst.lastDiag ]
         print "Authentication mode: %s" % pst.authType
         print "Shared-secret profile: %s" % ( 'None' if pst.authProfileName
                                               == '' else  pst.authProfileName )
         print "TxInt: %d, RxInt: %d, Detect Time: %d" \
             % ( pst.txInterval/1000, pst.rxInterval/1000, pst.detectTime/1000 )
         print "Rx Count: %d, Tx Count: %d" \
               % ( pst.rxCount, pst.txCount )
         print "SchedDelay: 1*TxInt: %d, 2*TxInt: %d, 3*TxInt: %d, GT 3*TxInt: %d"\
             % ( pst.sentWithin1p, pst.sentWithin2p, pst.sentWithin3p, 
                 pst.sentAfter3p )
         if pst.echoOn:
            print "Echo TxInt: %d, RxInt: %d, Detect Time: %d" \
               % ( pst.echoTxInterval / 1000, pst.echoRxInterval / 1000,
                   pst.echoDetectTime / 1000 )
            print "Echo Rx Count: %d, Tx Count: %d" \
               % ( pst.echoRxCount, pst.echoTxCount )
            print "Echo SchedDelay: 1*TxInt: %d, 2*TxInt: %d, 3*TxInt: %d, " \
               "GT 3*TxInt: %d" \
               % ( pst.echoSentWithin1p, pst.echoSentWithin2p, pst.echoSentWithin3p,
                   pst.echoSentAfter3p )
         # Adds newline since print adds one automatically at the end
         print ""

class PeerStatsDetail( Model ):
   apps = List( valueType=str, help="Registered protocols" )
   echoOn = Bool( help="Echo turned on" )
   echoActive = Bool( help="Active echo session with both tx and rx session" )
   echoDisc = Int( help="Discriminator of BFD Echo session" )
   operEchoTxRxInterval = Int( help="Operational rate in microseconds at which Bfd "
      "echo packets are sent and received" )
   operTxInterval = Int( help="Operational rate in microseconds at which BFD "
      "control packets are sent" )
   operRxInterval = Int( help="Operational rate in microseconds at which BFD "
      "control pckets are received" )
   detectMult = Int( help="Number of consecutive BFD packets missed before failure" )
   detectTime = Int( help="Detection timeout in microseconds" )
   remoteMinRxInterval = Int( help="Minimum rate in microseconds at which BFD "
      "packets can be received by the peer" )
   remoteDetectMult = Int( help="Receive multiplier of peer" )
   localIpAddr = IpGenericAddress( help="Local IP" )
   lastVersion = Int( help="Last received Peer BFD version" )
   lastDiag = Int( help="Last received Peer diagnostic code between values 0-8" )
   lastState = Enum( values=[ "adminDown", "down", "init", "up" ], help="Last "
      "received Peer BFD session state" )
   lastDemand = Bool( help="Last received Peer demand bit" )
   lastPoll = Bool( help="Last received Peer poll bit" )
   lastFinal = Bool( help="Last received Peer final bit" )
   lastDetectMult = Int( help="Last received Peer detection multiplier" )
   lastLength = Int( help="Last received packet length" )
   lastMyDiscU32 = Int( help="Last received Peer discriminator" )
   lastYourDiscU32 = Int( help="Last received local discriminator" )
   lastMinTxIntv = Int( help="Last received minimum Tx interval in microseconds" )
   lastMinRxIntv = Int( help="Last received minimum Rx interval in microseconds" )
   lastEchoRxIntv = Int( help="Last received Echo Rx interval in microseconds" )
   if toggleHwBfdEnabled():
      hwAcceleratedStates = Dict( keyType=str, valueType=bool,
            help="Mapping of session type to hw acceleration state" )
      _localSsoReady = Bool( help="Local system ready for SSO",
         default=False, optional=True )

class RxStats( Model ):
   numReceived = Int( help="Number of packets received" )
   minPeriod = Int( help="Minimum time between received packets in microseconds" )
   maxPeriod = Int( help="Maximum time between received packets in microseconds" )
   avgPeriod = Int( help="Average time between received packets in microseconds" )
   lastRxTime = Float( help="Time of last successfully authenticated and validated "
      "packet received in seconds" )
   lastRxStateChg = Float( help="Time of last Rx packet of state change "
                           "received in seconds", optional=True )
   lastRxState = Enum( values=[ "adminDown", "down", "init", "up" ],
                       help="Last Rx state of the peer", optional=True )

class TxStats( Model ):
   numSent = Int( help="Number of packets sent since last Tx config change" )
   minLateness = Int( help="Minimum number of microseconds a message sent after "
      "its configured period" )
   maxLateness = Int( help="Max number of microseconds a message sent after its " 
      "configured period" )
   avgLateness = Int( help ="Average number of microseconds messages sent after its "
      "configured period" )
   lastTxTime = Float( help="Time of last successfully transmitted packet in "
      "seconds" )
   sentWithin1p = Int( help="Number of packets sent within configured period since "
      "last packet" )
   sentWithin2p = Int( help="Number of packets sent later than one period but "
      "before two periods of last packet" )
   sentWithin3p = Int( help="Number of packets sent later than two periods but "
      "before three periods of last packet" )
   sentAfter3p = Int( help="Number of packets sent later than three periods of "
      "last packet" )

class PeerStats( Model ):
   status = Enum( values=[ "adminDown", "down", "init", "up" ], help="State of "
      "the peer" )
   sessType = OperSessionTypeEnum
   localDisc = Int( help="Local discriminator of BFD session" )
   remoteDisc = Int( help="Remote discriminator BFD session" )
   lastUp = Float( help="Last time peer session came up in seconds, 0.0 if never "
      "up" )
   lastDown = Float( help="Last time peer session went down in seconds, 0.0 if "
      "never down" )
   lastDiag = Enum( values= [ 'diagNone', 'diagCtrlTimeout', 'diagEchoFail',
      'diagNeighDown', 'diagForwardingReset', 'diagPathDown', 'diagConcatPathDown',
      'diagAdminDown', 'diagRevConcatPathDown' ], help="Last diagnostic for reason "
      "of change to Down state" )
   authType = Enum( values=authEnumNumToText.values(), help="Authentication mode" )
   authProfileName = Str( help="Configured authentication profile" )
   l3intf = Interface( help="Local egress interface" )
   kernelIfIndex = Int( help="Kernel interface", optional=True )
   peerStatsDetail = Submodel( valueType=PeerStatsDetail, help="Detailed peer "
      "stats", optional=True )
   rxStats = Submodel( valueType=RxStats, help="Rx statistics for peer", 
                       optional=True )
   txStats = Submodel( valueType=TxStats, help="Tx statistics for peer", 
                       optional=True )
   echoRxStats = Submodel( valueType=RxStats, help="Echo Rx statistics for peer",
                       optional=True )
   echoTxStats = Submodel( valueType=TxStats, help="Echo Tx statistics for peer",
                       optional=True )
   tunnelId = Int( help="TunnelID of SBFD Initiator", optional=True )
   destPort = Int( help="Destination Port of SBFD Reflector", optional=True )

class BfdNeighborSrcAddr( Model ):
   peerStats = Dict( keyType=IpGenericAddress, valueType=PeerStats,
      help="A mapping between source address and peer stats" )

class BfdNeighborIntfPeer( Model ):
   types = Dict( keyType=str, valueType=BfdNeighborSrcAddr,
      help="A mapping between peer type and source address" )

class BfdNeighborInterface( Model ):
   __revision__ = 3
   peers = Dict( keyType=Interface, valueType=BfdNeighborIntfPeer,
      help="A mapping between interface and peer type" )
   def degrade( self, dictRepr, revision ):
      dictRepr[ 'peerStats' ] = {}
      # pylint: disable=too-many-nested-blocks
      if dictRepr[ 'peers' ]:
         if revision == 1:
            for intf in dictRepr[ 'peers' ]:
               if dictRepr[ 'peers' ][ intf ][ 'types' ]:
                  for typeStr in dictRepr[ 'peers' ][ intf ][ 'types' ] :
                     neighSrcAddr = dictRepr[ 'peers' ][ intf ][ 'types' ][ typeStr ]
                     peerStats = neighSrcAddr[ 'peerStats' ]
                     # Revision 1 can only be used for single hop bfd, in which case
                     # there is only one source address and peer per interface. Thus,
                     # we can map each interface to the only existing PeerStats
                     # object
                     if peerStats:
                        peerStats = peerStats.values()[ 0 ]
                     else:
                        # It is also possible for CliModelRevisionTest to generate an
                        # empty peerStats dictionary, in which case we can map the
                        # interface to a dummy PeerStats dictionary to satisfy the
                        # test
                        peerStats = PeerStats( peerStatsDetail=PeerStatsDetail(),
                                               rxStats=RxStats(),
                                               txStats=TxStats(),
                                               echoRxStats=RxStats(),
                                               echoTxStats=TxStats() ).__dict__
                        for attr, val in peerStats.items():
                           if issubclass( type( val ), Model ):
                              peerStats[ attr ] = val.__dict__
                     dictRepr[ 'peerStats' ][ intf ] = peerStats
         elif revision == 2:
            # in revision 2, peerStats is keyed by peer intf
            # peerStats = Dict( keyType=Interface, valueType=BfdNeighborSrcAddr,
            # help = "A mapping between interface and source address" )
            for intf in dictRepr[ 'peers' ]:
               for _ in dictRepr[ 'peers' ][ intf ]:
                  if dictRepr[ 'peers' ][ intf ][ 'types' ]:
                     for typeStr in dictRepr[ 'peers' ][ intf ][ 'types' ]:
                        neighSrcAddr = dictRepr[ 'peers' ][ intf ][ 'types' ] \
                                       [ typeStr ]
                        dictRepr[ 'peerStats' ][ intf ] = neighSrcAddr

      if 'peers' in dictRepr:
         del dictRepr[ 'peers' ]
      return dictRepr

class SbfdInitiatorTunnel( Model ):
   peerStats = Dict( keyType=long, valueType=PeerStats,
      help="A mapping between tunnel ID and peer stats" )

class SbfdReflector( Model ):
   # keyType is long to cover the whole range of remoteDisc (0x0-0xFFFFFFFF)
   peerStats = Dict( keyType=long, valueType=PeerStats,
      help="A mapping between remote discriminator and peer stats" )

class BfdNeighborVrf( Model ):
   ipv4Neighbors = Dict( keyType=IpGenericAddress, valueType=BfdNeighborInterface,
      help="A mapping between IPv4 peer address and interface" )
   ipv6Neighbors = Dict( keyType=IpGenericAddress, valueType=BfdNeighborInterface,
      help="A mapping between IPv6 peer address and interface" )
   ipv4InitiatorNeighbors = Dict( keyType=IpGenericAddress, 
                        valueType=SbfdInitiatorTunnel,
                        help="A mapping between IPv4 peer address and tunnel ID" )
   ipv6InitiatorNeighbors = Dict( keyType=IpGenericAddress, 
                        valueType=SbfdInitiatorTunnel,
                        help="A mapping between IPv6 peer address and tunnel ID" )
   ipv4ReflectorNeighbors = Dict( keyType=IpGenericAddress, valueType=SbfdReflector,
            help="A mapping between IPv4 peer address and remote discriminators" )
   ipv6ReflectorNeighbors = Dict( keyType=IpGenericAddress, valueType=SbfdReflector,
            help="A mapping between IPv6 peer address and remote discriminators" )

def calculateMsAgo( sec ):
   if sec == 0.0:
      return "never"
   return "%d ms ago" % ( ( time.time() - sec ) * 1000 )

def printRxStats( rxStats, statsType=None ):
   linePrefix = "Echo " if statsType == "echo" else ""
   if not rxStats:
      print linePrefix + \
         "Rx Count: 0, Rx Interval (ms) min/max/avg: 0/0/0 last: never"
   else:
      print linePrefix + \
         "Rx Count: %d, Rx Interval (ms) min/max/avg: %d/%d/%d last: %s" \
          % ( rxStats.numReceived,
              rxStats.minPeriod / 1000,
              rxStats.maxPeriod / 1000,
              rxStats.avgPeriod / 1000,
              calculateMsAgo( rxStats.lastRxTime ) )

def printTxStats( txStats, peerStatsDetail, statsType=None ):
   linePrefix = "Echo " if statsType == "echo" else ""

   if not txStats:
      print linePrefix + \
         "Tx Count: 0, Tx Interval (ms) min/max/avg: 0/0/0 last: never"
      if statsType != "echo":
         print linePrefix + \
               "Detect Time: %s" % ( convertToMs( peerStatsDetail.detectTime ) )
      print linePrefix + \
         "Sched Delay: 1*TxInt: 0, 2*TxInt: 0, 3*TxInt: 0, GT 3*TxInt: 0"
   else:
      print linePrefix + \
         "Tx Count: %d, Tx Interval (ms) min/max/avg: %d/%d/%d last: %s"\
            % ( txStats.numSent,
                txStats.minLateness / 1000,
                txStats.maxLateness / 1000,
                txStats.avgLateness / 1000,
                calculateMsAgo( txStats.lastTxTime ) )
      if statsType != "echo":
         print "Detect Time: %s" % \
               ( convertToMs( peerStatsDetail.detectTime ) )
      print linePrefix + \
         "Sched Delay: 1*TxInt: %d, 2*TxInt: %d, 3*TxInt: %d, GT 3*TxInt: %d" \
          % ( txStats.sentWithin1p,
              txStats.sentWithin2p,
              txStats.sentWithin3p,
              txStats.sentAfter3p )

def printPeerStatsDetail( peerStats, ip, intf, vrf ):
   peerStatsDetail = peerStats.peerStatsDetail
      
   if peerStats.sessType==operSessionType.sessionTypeSbfdInitiator:
      slId = TunnelIdType( peerStats.tunnelId ).extractIndexAndAf()
      print ( "Peer Addr %s, Tunnel ID %d(SR), Segment list ID %d, "
              "Type SBFD(initiator), State %s" ) \
            % ( str( ip ), peerStats.tunnelId, slId, capitalize( peerStats.status ) )
   elif peerStats.sessType == operSessionType.sessionTypeSbfdReflector:
      print "Peer Addr %s, Dest Port %d, Type SBFD(reflector), State %s" \
         % ( ip, peerStats.destPort, capitalize( peerStats.status ) )
   else:
      intf = intf or "NA"
      print "Peer Addr %s, Intf %s, Type %s, State %s" \
         % ( str( ip ), intf, operSessionEnumToType[ peerStats.sessType ], 
            capitalize( peerStats.status ) )
   print "VRF %s, LAddr %s, LD/RD %s/%s" % ( vrf, peerStatsDetail.localIpAddr,
      peerStats.localDisc, peerStats.remoteDisc )
   sessState = "Session state is %s" % capitalize( peerStats.status )
   echoActive = peerStatsDetail.echoActive
   echoSuffix = ''
   sessState += ' and %susing echo function' \
                  % ( '' if echoActive else 'not ' )
   if peerStats.sessType in [ operSessionType.sessionTypeLagLegacy,
      operSessionType.sessionTypeLagRfc7130 ]:
      print sessState

      # pylint: disable-msg=protected-access
      if peerStatsDetail._localSsoReady:
         print "Local system ready for SSO"

      if peerStatsDetail.echoOn:
         if not echoActive:
            print "BFD echo over per-link still inactive on this port-channel. "
            print "Peer system MAY NOT be ready for SSO"
         else:
            print "BFD echo over per-link has converged on this port-channel. "
            print "Peer system ready for SSO if other BFD per-link echo have "\
               "also converged"
      print "Last Diag: %s" % diagEnumToReason[ peerStats.lastDiag ]
      print "Registered protocols: %s" % ", ".join( peerStatsDetail.apps )  
      print "Parent session, please check port channel config for member info\n"  
      return 

   if echoActive:
      echoSuffix = ' with %s ms interval. LD: %s' % \
          ( convertToMs( peerStatsDetail.operEchoTxRxInterval ),
            peerStatsDetail.echoDisc )
   sessState += '%s' % echoSuffix

   if peerStats.sessType == operSessionType.sessionTypeSbfdReflector:
      print "Session state is %s" % capitalize( peerStats.status )
      print "Last Up %s" % dispTime( peerStats.lastUp )
      print "Last Down %s" % dispTime( peerStats.lastDown )
      print "Last Diag: %s" % diagEnumToReason[ peerStats.lastDiag ]
      print "Last Rx state change: %s, old Rx state is %s" \
         % ( dispTime( peerStats.rxStats.lastRxStateChg ),
             capitalize( peerStats.rxStats.lastRxState ) )
      print "RxInt: %s" % peerStatsDetail.operRxInterval
      print "Received TxInt: %s Received RxInt: %s, Received Multiplier: %s" \
         % ( convertToMs( peerStatsDetail.lastMinTxIntv ),
             convertToMs( peerStatsDetail.lastMinRxIntv ),
             peerStatsDetail.lastDetectMult )
      rxStats = peerStats.rxStats
      print "Rx Count: %d, Rx Interval (ms) min/max/avg: %d/%d/%d last: %s" \
         % ( rxStats.numReceived, rxStats.minPeriod / 1000,
             rxStats.maxPeriod / 1000, rxStats.avgPeriod / 1000,
             calculateMsAgo( rxStats.lastRxTime ) )
      txStats = peerStats.txStats
      print "Tx Count: %d, last: %s" \
         % ( peerStats.txStats.numSent, calculateMsAgo( txStats.lastTxTime ) )
   else:
      print sessState
      if toggleHwBfdEnabled():
         print "Hardware Acceleration: " + ", ".join(
            "{} {}".format( k, "On" if v else "Off" )
            for ( k, v ) in peerStatsDetail.hwAcceleratedStates.items() )
      print "Last Up %s" % dispTime( peerStats.lastUp )
      print "Last Down %s" % dispTime( peerStats.lastDown )
      print "Last Diag: %s" % diagEnumToReason[ peerStats.lastDiag ]
      print "Authentication mode: %s" % peerStats.authType
      print "Shared-secret profile: %s" % ( 'None' if peerStats.authProfileName
                                            == '' else  peerStats.authProfileName )
      print "TxInt: %s, RxInt: %s, Multiplier: %s" \
         % ( convertToMs( peerStatsDetail.operTxInterval ),
             convertToMs( peerStatsDetail.operRxInterval ),
             peerStatsDetail.detectMult )
      print "Received RxInt: %s, Received Multiplier: %s" \
         % ( convertToMs( peerStatsDetail.remoteMinRxInterval ),
             peerStatsDetail.remoteDetectMult )

      printRxStats( peerStats.rxStats )
      printTxStats( peerStats.txStats, peerStatsDetail )
      if echoActive:
         printRxStats( peerStats.echoRxStats, statsType="echo" )
         printTxStats( peerStats.echoTxStats, peerStatsDetail, statsType="echo" )

      print "Registered protocols: %s" % ", ".join( peerStatsDetail.apps )
      print "Uptime: %s" % dispElapsedTime( Tac.utcNow() - peerStats.lastUp,
                                            peerStats.status )

   table = TableOutput.TableFormatter( tableWidth = 74 )
   f = TableOutput.Format( justify = 'left' )
   f.padLimitIs( True )
   f.noPadLeftIs( True )
   table.formatColumns( f, f, f )
   lastPacket = "Last packet: "
   space = " "*len( lastPacket )
   table.newRow( lastPacket,
                 "Version: %s" % peerStatsDetail.lastVersion,
                 "- Diagnostic: %s" % peerStatsDetail.lastDiag )
   table.newRow( space, 
                 "State bit: %s" % capitalize( peerStatsDetail.lastState ), 
                 "- Demand bit: %s" % int( peerStatsDetail.lastDemand ) )
   table.newRow( space, 
                 "Poll bit: %s" % int( peerStatsDetail.lastPoll ), 
                 "- Final bit: %s" % int( peerStatsDetail.lastFinal ) ) 
   table.newRow( space,
                 "Multiplier: %d" % peerStatsDetail.lastDetectMult, 
                 "- Length: %s" % peerStatsDetail.lastLength )
   table.newRow( space,
                 "My Discr.: %s" % peerStatsDetail.lastMyDiscU32,
                 "- Your Discr.: %s" % peerStatsDetail.lastYourDiscU32 ) 
   table.newRow( space, "Min tx interval: %d" % \
                 convertToMs( peerStatsDetail.lastMinTxIntv ),
                 "- Min rx interval: %d" % \
                 convertToMs( peerStatsDetail.lastMinRxIntv ) )
   table.newRow( space, "Min Echo interval: %d" % \
                 convertToMs( peerStatsDetail.lastEchoRxIntv ), "" )
   if peerStats.sessType in [ operSessionType.sessionTypeMicroLegacy, 
      operSessionType.sessionTypeMicroRfc7130 ]:
      print table.output(),
      print "Member session under parent interface %s" % \
         peerStats.l3intf.stringValue
   else:
      print table.output().rstrip( '\n' )

   addrFamily = AF_INET if '.' in str( ip ) else AF_INET6 

   if addrFamily == AF_INET:
      routingConfigured = BfdCli.routingConfig.routing 
      version = ''
   else:
      routingConfigured = BfdCli.routing6Config.routing 
      version = '6'

   if peerStatsDetail.echoOn and not peerStatsDetail.echoActive:
      print "WARNING: Echo packets are not being echoed back to the peer. \
            Perhaps Ip%s routing is not enabled on the peer." % version
   
   if peerStats.sessType == operSessionType.sessionTypeVxlanTunnel and \
      str( peerStats.status ) == 'down' and not routingConfigured:
      print "WARNING: Ip%s routing is not enabled" % version
   print
   
'''
Capi Model Hierarchy for "show bfd peers [detail]" CLI command
__revision__ 2
Summary: bfd -> vrf -> ipv4/ipv6 -> peerIp -> intf -> srcIp -> peerStats
Models: BfdNeighbors -> BfdNeighborVrf -> BfdNeighborInterface -> 
   BfdNeighborSrcAddr -> PeerStats
Json Object:
{
   'vrfs' : {
      'default' : {
         'ipv4Neighbors' : {
            '1.1.1.1' : {
               'peerStats' : {
                  'Ethernet1' : {
                     'peerStats' : {
                        '1.1.1.0' : {
                           'status' : 'up',
                           'sessType' : 'normal',
                           ...
                        }
                     }
                  }
                  
               }
            }
         }
         'ipv6Neighbors' : {
            '1::1' {
               'peerStats' : {
                  'Ethernet1' : {
                     'peerStats' : {
                        '1::0' : {
                           'status' : 'up',
                           'sessType' : 'normal',
                           ...
                        }
                     }
                  }
               }
            }
         }
      }
   }
}

__revision__ 3
Summary: bfd -> vrf -> ipv4/ipv6 -> peerIp -> intf -> peerType -> srcIp -> peerStats
         bfd -> vrf -> ipv4Initiator/ipv6Initiator -> peerIp -> tunnelId -> peerStats
         bfd -> vrf -> ipv4Reflector/ipv6Reflector -> peerIp -> remoteDisc
             -> peerStats
Models: BfdNeighbors -> BfdNeighborVrf -> BfdNeighborInterface -> bfdNeighborIntfPeer
   -> BfdNeighborSrcAddr -> PeerStats
        BfdNeighbors -> BfdNeighborVrf -> SbfdInitiatorTunnel -> PeerStats
        BfdNeighbors -> BfdNeighborVrf -> SbfdReflector -> PeerStats
Json Object:
{
   'vrfs' : {
      'default' : {
         'ipv4Neighbors' : {
            '1.1.1.1' : {
               'peers' : {
                  'Ethernet1' : {
                     'types' : {
                        'normal' : {
                           'peerStats' : {
                              '1.1.1.0' : {
                                 'status' : 'up',
                                 'sessType' : 'normal',
                                 ...
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
         'ipv6Neighbors' : {
            '1::1' {
               'peers' : {
                  'Ethernet1' : {
                     'types' : {
                        'normal' : {
                           'peerStats' : {
                              '1::0' : {
                                 'status' : 'up',
                                 'sessType' : 'normal',
                                 ...
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
         'ipv4Initiator' : {
            '1.1.1.1' {
               'peers' : {
                  '1' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'initiator',
                        ...
                     }
                  }
               }
            }
         }
         'ipv4Reflector' : {
            '2.2.2.2' {
               'peers' : {
                  '100' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'reflector',
                        ...
                     }
                  }
               }
            }
         }
         'ipv6Initiator' : {
            '1::1' {
               'peers' : {
                  '1' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'initiator',
                        ...
                     }
                  }
               }
            }
         }
         'ipv6Reflector' : {
            '2::2' {
               'peers' : {
                  '1' : {
                     'peerStats' : {
                        'status' : 'up',
                        'sessType' : 'reflector',
                        ...
                     }
                  }
               }
            }
         }
      }
   }
}
'''

class BfdNeighbors( Model ):
   __revision__ = 3
   vrfs = Dict( valueType=BfdNeighborVrf, help="A mapping between vrf and peer "
      "stats" )
   _detailed = Bool( help="Detailed show bfd peers command" )

   def render( self ):
      firstVrf = True
     
      if DEFAULT_VRF in self.vrfs:
         print "VRF name:", DEFAULT_VRF 
         print "-----------------"
         self.printBfd( DEFAULT_VRF, 
                        self.vrfs[ DEFAULT_VRF ].ipv4Neighbors, 
                        self.vrfs[ DEFAULT_VRF ].ipv4InitiatorNeighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv4ReflectorNeighbors )
         self.printBfd( DEFAULT_VRF, 
                        self.vrfs[ DEFAULT_VRF ].ipv6Neighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv6InitiatorNeighbors,
                        self.vrfs[ DEFAULT_VRF ].ipv6ReflectorNeighbors )
         firstVrf = False
      for vrf, vrfModel in sorted( self.vrfs.items() ):   
         if vrf == DEFAULT_VRF:
            continue
         if not firstVrf:
            print 
         print "VRF name:", vrf 
         print "-----------------"
         self.printBfd( vrf, vrfModel.ipv4Neighbors, 
                        vrfModel.ipv4InitiatorNeighbors,
                        vrfModel.ipv4ReflectorNeighbors )
         self.printBfd( vrf, vrfModel.ipv6Neighbors, 
                        vrfModel.ipv6InitiatorNeighbors,
                        vrfModel.ipv6ReflectorNeighbors )
         firstVrf = False

   def sortByIntf( self, bfdNeighborInterface ):
      bfdNeighborIntfPeerList = sorted( bfdNeighborInterface.peers.items(),
                                       cmp=compare_peer )
      # convert bfdNeighborIntfPeerList into dict keyed by l3Intf to list of
      # peer stats
      intfBfdNeighborMap = defaultdict( list )
      for intf, intfPeer in bfdNeighborIntfPeerList:
         # First append normal session
         if 'normal' in intfPeer.types:
            bfdNeighborSrcAddr = intfPeer.types[ 'normal' ]
            intfBfdNeighborMap[ bfdNeighborSrcAddr.peerStats.values()[ 0 ].\
                  l3intf.stringValue ].append( ( intf, bfdNeighborSrcAddr ) )
         # append other session types
         for typeStr, bfdNeighborSrcAddr in intfPeer.types.iteritems() :
            if typeStr != "normal":
               intfBfdNeighborMap[ bfdNeighborSrcAddr.peerStats.values()[ 0 ].\
                     l3intf.stringValue ].append( ( intf, bfdNeighborSrcAddr ) )
      sortedBfdNeighborSrcAddrList = []
      for intf, intfBfdNeighborList in sorted( intfBfdNeighborMap.items() ):
         sortedBfdNeighborSrcAddrList.extend( intfBfdNeighborList )
      return sortedBfdNeighborSrcAddrList

   def printBfd( self, vrf, bfdNeighbors, sbfdInitiators, sbfdReflectors ):
      if not bfdNeighbors and not sbfdInitiators and not sbfdReflectors:
         return
      
      if not self._detailed:
         headings = ( "DstAddr", "MyDisc", "YourDisc", "Interface/Transport", "Type",
                         "LastUp", "LastDown", "LastDiag", "State" )
         formatLeft = TableOutput.Format( justify="left" )
         formatLeft.noPadLeftIs( True )
         formatRight = TableOutput.Format( justify="right" )
         table = TableOutput.createTable( headings )
         table.formatColumns( formatLeft, formatRight, formatRight, formatRight, 
            formatRight, formatRight, formatRight, formatRight, formatRight )
         
         if bfdNeighbors:
            for ip, bfdNeighborInterface in sorted( bfdNeighbors.items(),
               cmp=compareIp ):
               for intf, bfdNeighborSrcAddr in self.sortByIntf( 
                                                            bfdNeighborInterface ):
                  for _, peerStats in sorted( bfdNeighborSrcAddr.peerStats.items(),
                     cmp=compareIp ):
                     interface = intf + ( "(%d)" % peerStats.kernelIfIndex ) \
                                 if intf else "NA" 
                     table.newRow( str( ip ), peerStats.localDisc, 
                           peerStats.remoteDisc,
                           interface,
                           operSessionEnumToType[ peerStats.sessType ],
                           dispTime( peerStats.lastUp, shortDisp=True ),
                           dispTime( peerStats.lastDown, shortDisp=True ), 
                           diagEnumToReason[ peerStats.lastDiag ],
                           capitalize( str( peerStats.status ) ) )
         if sbfdInitiators:
            for ip, sbfdInitiatorTunnel in sorted( sbfdInitiators.items(),
               cmp=compareIp ):
               for _, peerStats in sorted(
                     sbfdInitiatorTunnel.peerStats.items() ):
                  slId = TunnelIdType( peerStats.tunnelId ).extractIndexAndAf()
                  transport = "SR-Tunnel" + ( "(%d)" % slId )
                  table.newRow( str( ip ), peerStats.localDisc,
                        peerStats.remoteDisc,
                        transport,
                        operSessionEnumToType[ peerStats.sessType ],
                        dispTime( peerStats.lastUp, shortDisp=True ),
                        dispTime( peerStats.lastDown, shortDisp=True ),
                        diagEnumToReason[ peerStats.lastDiag ],
                        capitalize( str( peerStats.status ) ) )
         if sbfdReflectors:
            for ip, sbfdReflector in sorted( sbfdReflectors.items(), cmp=compareIp ):
               for _, peerStats in sorted( sbfdReflector.peerStats.items() ):
                  table.newRow( str( ip ), peerStats.localDisc,
                                peerStats.remoteDisc,
                                "NA",
                                operSessionEnumToType[ peerStats.sessType ],
                                dispTime( peerStats.lastUp, shortDisp=True ),
                                dispTime( peerStats.lastDown, shortDisp=True ),
                                diagEnumToReason[ peerStats.lastDiag ],
                                capitalize( str( peerStats.status ) ) )
         print table.output()
      else:
         if bfdNeighbors:
            for ip, bfdNeighborInterface in sorted( bfdNeighbors.items(), 
               cmp=compareIp ):
               for intf, bfdNeighborSrcAddr in self.sortByIntf( 
                                                            bfdNeighborInterface ):
                  for _, peerStats in sorted( bfdNeighborSrcAddr.peerStats.items(), 
                     cmp=compareIp ):
                     printPeerStatsDetail( peerStats, ip, intf, vrf )
         if sbfdInitiators:
            for ip, sbfdInitiatorTunnel in sorted( sbfdInitiators.items(),
                  cmp=compareIp ):
               for _, peerStats in sorted(
                     sbfdInitiatorTunnel.peerStats.items() ):
                  printPeerStatsDetail( peerStats, ip, '', vrf )

         if sbfdReflectors:
            for ip, sbfdReflector in sorted( sbfdReflectors.items(), cmp=compareIp ):
               for _, peerStats in sorted( sbfdReflector.peerStats.items() ):
                  printPeerStatsDetail( peerStats, ip, '', vrf )

class RbfdStats( Model ):
   rfcDrops = Int( help="Number of packets dropped on receive before processing" )
   rxPacketMissing = Int( help="Number of times packet missing upon Rx trigger" )
   numInvalidSessions = Int( help="Number of packets without session in Rx" )
   numPassupDisc0 = Int( help="Number of packets passed up with discriminator zero" )
   netlinkPacketChangeFail = Int( help="Number of times packet change sent but "
                                       "broadcast failed" )
   netlinkFailureFail = Int( help="Number of times session failure sent but " 
                                  "broadcast failed" )
   pskbExpandFail = Int( help="Number of times pskb needed to be expanded but "
                              "failed for packets without a session" )

   def render( self ):
      print 'rfcDrops: %d' % self.rfcDrops
      print 'rxPacketMissing: %d' % self.rxPacketMissing
      print 'numInvalidSessions: %d' % self.numInvalidSessions
      print 'numPassupDisc0: %d' % self.numPassupDisc0
      print 'netlinkPacketChangeFail: %d' % self.netlinkPacketChangeFail
      print 'netlinkFailureFail: %d' % self.netlinkFailureFail
      print 'pskbExpandFail: %d' % self.pskbExpandFail

class SbfdStats( Model ):
   rxValid = Int( help="Number of valid packets received" )
   rxDrop = Int( help="Number of packets dropped on receive" )
   rxTruncated = Int( help="Number of packets truncated on receive" )
   rxInvalidVer = Int( help="Number of packets with invalid version on receive" )
   rxInvalidLen = Int( help="Number of packets with invalid length on receive" )
   rxInvalidSrcPort = Int(
      help="Number of packets with invalid source port on receive" )
   rxDemandUnset = Int( help="Number of packets with demand mode unset on receive" )
   rxInvalidMulti = Int(
      help="Number of packets with invalid multiple detect on receive" )
   rxInvalidMyDisc = Int(
      help="Number of packets with invalid my discriminator on receive" )
   rxYourDiscUnmatch = Int(
      help="Number of packets with unmatched your discriminator on receive" )
   rxInvalidBfdState = Int(
      help="Number of packets with invalid bfd state on receive" )
   rxUnsupported = Int( help="Number of packets unsupported on receive" )
   rxInvalidPktFormat = Int(
      help="Number of packets with invalid packet format on receive" )
   txSend = Int( help="Number of packets sent" )
   txSendToFail = Int( help="Number of packets failed to send" )
   txInvalidLen = Int( help="Number of packets with invalid length" )
   rxHashMiss = Int( help="Number of session cache lookup miss" )
   rxHashHit = Int( help="Number of session cache lookup hit" )
   rxHashTrim = Int( help="Number of session cache lookup trim" )

   def render( self ):
      print 'rxValid: %d' % self.rxValid
      print 'rxDrop: %d' % self.rxDrop
      print 'rxTruncated: %d' % self.rxTruncated
      print 'rxInvalidVer: %d' % self.rxInvalidVer
      print 'rxInvalidLen: %d' % self.rxInvalidLen
      print 'rxInvalidSrcPort: %d' % self.rxInvalidSrcPort
      print 'rxDemandUnset: %d' % self.rxDemandUnset
      print 'rxInvalidMulti: %d' % self.rxInvalidMulti
      print 'rxInvalidMyDisc: %d' % self.rxInvalidMyDisc
      print 'rxYourDiscUnmatch: %d' % self.rxYourDiscUnmatch
      print 'rxInvalidBfdState: %d' % self.rxInvalidBfdState
      print 'rxUnsupported: %d' % self.rxUnsupported
      print 'rxInvalidPktFormat: %d' % self.rxInvalidPktFormat
      print 'txSend: %d' % self.txSend
      print 'txSendToFail: %d' % self.txSendToFail
      print 'txInvalidLen: %d' % self.txInvalidLen

class SessionCount( Model ):
   async = Int( help="Number of async sessions" )
   echo = Int( help="Number of echo sessions" )

class SessionCountByState( Model ):
   states = Dict( keyType=str, valueType=SessionCount,
                  help="Mapping of session state to number of sessions" )

class SessionCountByType( Model ):
   types = Dict( keyType=str, valueType=SessionCountByState,
                 help="Mapping of session type to session count by state" )

def countByState( model, sessionType, state, sbfd=False ):
   stateModel = model.states[ state ]
   if sbfd:
      count = '%s' % str( stateModel.async )
   else:
      count = '%s [%s]' % ( str( stateModel.async ), str( stateModel.echo ) )
   return count

def printRow( table, sessionModel, session, sessionType, sbfd=False ):
   typeModel = sessionModel.types[ sessionType ]
   operState = Tac.Type( "Bfd::OperState" )
   table.newRow( session, operSessionEnumToStr[ sessionType ],
                 countByState( typeModel, sessionType, operState.up, sbfd ),
                 countByState( typeModel, sessionType, operState.init, sbfd ),
                 countByState( typeModel, sessionType, operState.down, sbfd ),
                 countByState( typeModel, sessionType, operState.adminDown, sbfd ),
               )
   
class BfdSummary( Model ):
   sessions = Dict( keyType=str, valueType=SessionCountByType,
                    help="Mapping of addressing to session count by type" )
   minTx = Int( help='Operational rate in microseconds at which BFD '
                     'control packets are sent' )
   minRx = Int( help='Operational rate in microseconds at which BFD '
                     'control pckets are received' )
   mult = Int( help='Number of consecutive BFD packets missed before failure' )
   mhMinTx = Int( help='Operational rate in microseconds at which BFD '
                       'control packets are sent on multihop BFD session' )
   mhMinRx = Int( help='Operational rate in microseconds at which BFD '
                       'control pckets are received on multihop BFD session' )
   mhMult = Int( help='Number of consecutive BFD packets missed before failure on'
                      'multihop BFD session' )
   slowTimer = Int( help='Operation rate in microseconds at which BFD control '
                         'packets are received on async sessions when echo is '
                         'enabled.' )
   adminDown = Bool( help='BFD has been shut down administratively' )
   if toggleBfdTelemetryEnabled():
      sessionStatsInterval = Int( help='Interval in seconds between unsolicited '
                                    'per-session statistics from BFD acclerators' )

   # SBFD related attributes.
   _initiatorOnly = Bool( help='Display initiators only' )
   _reflectorOnly = Bool( help='Display reflectors only' )
   localIntf = Interface( help="SBFD local interface", optional=True )
   localIntfIpAdd = IpGenericAddress( help="Local interface IP address" )
   initiatorMinTx = Int( help='Operational rate in microseconds at which SBFD '
                              'control packets are sent on initiator' )
   initiatorMult = Int( help='Number of consecutive SBFD packets missed '
                             'before failure' )
   reflectorLocalDisc = Int( help="Local discriminator of SBFD session" )
   reflectorMinRx = Int( help='Operational rate in microseconds at which SBFD '
                              'control pckets are received on reflector' )
   # IPv6 support would be added later

   def render( self ):
      print "Global administrative shutdown: ", "Yes" if self.adminDown else "No"
      if toggleBfdTelemetryEnabled():
         print ( "Configured session stats snapshot interval: %ds" %
                  self.sessionStatsInterval )

      print "BFD:"
      print ( "Configured global single hop interval %dms min_rx %dms multiplier %d"
            % ( self.minTx, self.minRx, self.mult ) )
      print ( 
            "Configured global multiple hop interval %dms min_rx %dms multiplier %d"
            % ( self.mhMinTx, self.mhMinRx, self.mhMult ) )
      if self.mhMinRx * self.mhMult < self.minRx * self.mult:
         print "WARNING: Multi-hop detect time should be larger than" \
               " single hop detect time"
      print "Slow timer: %dms" % self.slowTimer
      
      print "SBFD:"
      if self.adminDown:
         print "IPv4 operational state: globally disabled (adminDown state)"
      elif not self.localIntf:
         print ( "IPv4 operational state: globally disabled "
                 "(Local interface is not configured)" )
      elif self.localIntfIpAdd.isAddrZero:     # pylint: disable=E1101
         print ( "IPv4 operational state: globally disabled "
                 "(Local interface IP address not configured)" )
      elif self.reflectorLocalDisc == 0:
         print ( "IPv4 operational state: initiator enabled, reflector disabled "
                 "(Reflector discriminator not configured)" )
      else:
         print "IPv4 operational state: enabled"
      print ( "Configured global initiator tx interval %dms multiplier %d" 
            % ( self.initiatorMinTx, self.initiatorMult ) )
      print "Configured reflector rx interval %dms" % self.reflectorMinRx

      print "\nLegend:"
      print "*: pseudo LAG session (not counted in total sessions)"
      print "<N>[<M>]: Number of sessions [ Number of sessions with echo enabled ]\n"

      headings = ( "Addressing", "Type", "Up", "Init", "Down", "AdminDown" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatLeft, formatRight, formatRight,
         formatRight, formatRight )

      printRow( table, self.sessions[ "All" ], "All", "All" )
      singlehopTypes = [
                   "singlehopAll",
                   operSessionType.sessionTypeNormal,
                   operSessionType.sessionTypeLagRfc7130,
                   operSessionType.sessionTypeMicroRfc7130,
                   operSessionType.sessionTypeLagLegacy,
                   operSessionType.sessionTypeMicroLegacy,
                 ]
      if "IPv4" in self.sessions.keys():
         sessionModel = self.sessions[ "IPv4" ]
         printRow( table, sessionModel, "IPv4", sessionType="All" )
         for sessionType in singlehopTypes:
            printRow( table, sessionModel, "    single hop" if \
                      sessionType == "singlehopAll" else "", sessionType )
         printRow( table, sessionModel, "    multi-hop", 
                   operSessionType.sessionTypeMultihop )
         sessionModel = self.sessions[ "IPv6" ]
         printRow( table, sessionModel, "IPv6", sessionType="All" )
         for sessionType in singlehopTypes:
            printRow( table, sessionModel, "    single hop" if \
                      sessionType == "singlehopAll" else "", sessionType )
         printRow( table, sessionModel, "    multi-hop", 
                   operSessionType.sessionTypeMultihop )
         printRow( table, self.sessions[ "Tunnel" ], "Tunnel",
                   operSessionType.sessionTypeVxlanTunnel )
         printRow( table, self.sessions[ "L2" ], "L2", 
                   operSessionType.sessionTypeLagRfc7130 )
         printRow( table, self.sessions[ "L2" ], "", 
                   operSessionType.sessionTypeMicroRfc7130 )

      if "SR-TE Tunnel" in self.sessions.keys():
         printRow( table, self.sessions[ "SR-TE Tunnel" ], "SR-TE Tunnel",
                   "All", sbfd=True )
         sessionModel = self.sessions[ "sbfdIPv4" ]
         if self._initiatorOnly:
            printRow( table, sessionModel, "    IPv4",
                      operSessionType.sessionTypeSbfdInitiator, sbfd=True )
         elif self._reflectorOnly:
            printRow( table, sessionModel, "    IPv4",
                   operSessionType.sessionTypeSbfdReflector, sbfd=True )
         else:
            printRow( table, sessionModel, "    IPv4",
                      operSessionType.sessionTypeSbfdInitiator, sbfd=True )
            printRow( table, sessionModel, "",
                      operSessionType.sessionTypeSbfdReflector, sbfd=True )
         sessionModel = self.sessions[ "sbfdIPv6" ]
         if self._initiatorOnly:
            printRow( table, sessionModel, "    IPv6",
                   operSessionType.sessionTypeSbfdInitiator, sbfd=True )
         elif self._reflectorOnly:
            printRow( table, sessionModel, "    IPv6",
                   operSessionType.sessionTypeSbfdReflector, sbfd=True )
         else:
            printRow( table, sessionModel, "    IPv6",
                      operSessionType.sessionTypeSbfdInitiator, sbfd=True )
            printRow( table, sessionModel, "",
                      operSessionType.sessionTypeSbfdReflector, sbfd=True )

      print table.output()

class BfdDebug( Model ):
   downSessions = Dict( keyType=str, valueType=int,
                        help="Mapping of diag to number of down sessions" )
   class Issue( Model ):
      issueList = List( help="List of issues", valueType=str )
   issues = Dict( keyType=str, valueType=Issue,
                  help="Mapping of issue type to issue detail" )
   slowTxs = Dict( keyType=str, valueType=int,
                   help="Mapping of peer to number of slow tx" )
   rxIntervals = Dict( keyType=str, valueType=str,
                       help="Mapping of peer to rx interval" )
   slowEchoTxs = Dict( keyType=str, valueType=int,
                       help="Mapping of peer to number of slow echo tx" )
   echoRxIntervals = Dict( keyType=str, valueType=str,
                           help="Mapping of peer to echo rx interval" )

   def addIssue( self, key, issue ):
      if key in self.issues:
         self.issues[ key ].issueList.append( issue )
      else:
         issueModel = BfdDebug.Issue()
         issueModel.issueList.append( issue )
         self.issues[ key ] = issueModel

   def render( self ):
      if self.downSessions:
         print "------- Number of sessions down --------"
         for diag, numSessions in self.downSessions.items():
            print diag, ': ', numSessions
         print '\n'

      for key, issueModel in self.issues.items():
         if issueModel:
            print key
            for issue in sorted( issueModel.issueList ) :
               print issue

      header = [ "VRF", "DstAddr", "Interface", "Type" ]
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )

      def printTable( headings, rows ):
         table = TableOutput.createTable( headings )
         table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                              formatLeft )
         for peer, number in sorted( rows.items() ):
            tokens = peer.split()
            table.newRow( tokens[ 0 ], tokens[ 1 ], tokens[ 2 ], tokens[ 3 ],
                          number )
         print table.output()

      if self.slowTxs:
         printTable( tuple( header + [ "GT 3*TxInt" ] ),
                     self.slowTxs )

      if self.rxIntervals:
         printTable( tuple( header + [ "Rx interval (ms) min/max/avg" ] ),
                     self.rxIntervals )

      if self.slowEchoTxs:
         printTable( tuple( header + [ "GT 3*TxInt" ] ),
                     self.slowEchoTxs )

      if self.echoRxIntervals:
         printTable( tuple( header + [ "Echo rx interval (ms) min/max/avg" ] ),
                     self.echoRxIntervals )

class BfdHwAccel( Model ):
   supported = Bool( help="Hardware acceleration is supported by platform" )
   running = Bool( help="Hardware acceleration is running" )
   reasons = List( help="List of reasons why hardware acceleration is not running",
                   valueType=str )

   def render( self ):
      if not self.supported:
         print "Hardware acceleration is not supported"
      elif self.running:
         print "Hardware acceleration is running"
      else:
         print "Hardware acceleration is not running{}".format(
               ": " + ", ".join( self.reasons ) if self.reasons else "" )

class BfdHwResourceSession( Model ):
   ip = IpGenericAddress( help="IP address of peer" )
   localDisc = Int( help="Local discriminator of BFD session" )
   intf = Interface( help="Local egress interface" )
   vrf = Str( help="VRF that peer is configured in" )
   sessType = OperSessionTypeEnum

class BfdHwResource( Model ):
   maxSessions = Int( help="Maximum number of sessions configured of the chip" )
   numSessions = Int( help="Number of sessions configured on a chip", optional=True )
   sessions = List( valueType=BfdHwResourceSession,
         help="List of sessions configured on the chip", optional=True )

class BfdHwResourceList( Model ):
   resources = Dict( valueType=BfdHwResource,
         help="Hardware resource information keyed by the resource's name" )
   _detailed = Bool( help="Detailed show bfd hardware utilization command" )

   def render( self ):
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatRight = TableOutput.Format( justify="right" )
      if not self._detailed:
         header = ( "Chip Name", "Number Of HW Sessions*",
               "Maximum Number Of HW Sessions*" )

         table = TableOutput.createTable( header )
         table.formatColumns( formatLeft, formatRight, formatRight )
         for name, resource in sorted( self.resources.iteritems() ):
            table.newRow( name, resource.numSessions, resource.maxSessions )
         print table.output().rstrip( "\n" )
         print '* A BFD session with echo enabled typically consumes two hardware ' \
               'sessions'
         print ''

      else:
         header = ( "Dst Addr", "My Disc", "Interface", "VRF", "Type" )

         for name, resource in sorted( self.resources.iteritems() ):
            table = TableOutput.createTable( header )
            table.formatColumns( formatLeft, formatRight, formatLeft, formatLeft,
                  formatLeft )
            for session in sorted( resource.sessions, cmp=compareHwSession ):
               table.newRow( session.ip, session.localDisc, session.intf.stringValue,
                     session.vrf, operSessionEnumToStr[ session.sessType ] )
            print 'Chip:', name
            print table.output()
