# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import LazyMount
import CliCommand
import CliMatcher
import CliPlugin.IntfCli as IntfCli
from CliPlugin.BfdCli import ( matcherBfd,
                               matcherAuthProfile,
                               matcherInterval,
                               matcherMinRxDeprecated,
                               matcherMinRx,
                               matcherMultiplier,
                               BfdIpIntfConfigModelet,
                               getAuthModeFromArgs, )
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import CliPlugin.IraVrfCli as IraVrfCli
from IpLibConsts import DEFAULT_VRF
import Tac
import ConfigMount

l3IntfConfigDir = None
bfdConfigIntf = None
bfdConfigGlobal = None
staticAppConfigDir = None
authSharedSecretConfig = None

authType = Tac.Type( 'Bfd::BfdAuthType' )
matcherTxRxIntervalMs = CliMatcher.IntegerMatcher( 50, 60000,
      helpdesc='Rate in milliseconds between 50-60000' )

class BfdIpIntf ( IntfCli.IntfDependentBase ):
   def __init__( self, intf, sysdbRoot, createIfMissing=True ):
      IntfCli.IntfDependentBase.__init__( self, intf, sysdbRoot )

   def setDefault( self ):
      intf = IntfId( self.intf_.name )
      vrfName = getIntfVrf( intf )
      if intf in bfdConfigIntf.intfConfig:
         del bfdConfigIntf.intfConfig[ intf ]
      if intf in bfdConfigIntf.echoOn:
         del bfdConfigIntf.echoOn[ intf ]
      if intf in bfdConfigIntf.authType:
         del bfdConfigIntf.authType[ intf ]
      if vrfName in staticAppConfigDir:
         staticConfig = staticAppConfigDir[ vrfName ][ 'static' ]
         for peer in staticConfig.peerConfigVer:
            if peer.intf == self.intf_.name:
               del staticConfig.peerConfigVer[ peer ]

#-------------------------------------------------------------------------------
# Get the vrf configured for an interface ( looks up the L3::Intf::ConfigDir )
#-------------------------------------------------------------------------------
def getIntfVrf( intfName ):
   interfaceConfig = l3IntfConfigDir.intfConfig.get( intfName, None )
   if interfaceConfig:
      return interfaceConfig.vrf
   return DEFAULT_VRF

routedPortWarning = {}
existingSessionWarning = {}
perlinkOnWarning = {}
rfc7130SupportedWarning = {}
rfc7130BfdNeighborWarning = {}

IntfCli.IntfConfigMode.addModelet( BfdIpIntfConfigModelet )
modelet = BfdIpIntfConfigModelet

# Interface mode config commands
#------------------------------------------------------------------------------
# "[no|default] bfd interval <50-60000> min-rx <50-60000> multiplier <3-50>"

# "[no|default] bfd interval <50-60000> min_rx <50-60000> multiplier <3-50>"
# command in config-if mode (hidden command)
# "[no|default] bfd interval <50-60000> min_rx <50-60000> multiplier <3-50>
#  default" in config mode (hidden command)
#------------------------------------------------------------------------------

def IntfId( intfName ):
   return Tac.Value( 'Arnet::IntfId', intfName )

def BfdIntervalConfig( minTx, minRx, mult ):
   return Tac.Value( 'Bfd::BfdIntervalConfig', minTx, minRx, mult )

def intfConfig( mode ):
   intf = IntfId( mode.intf.name )
   return bfdConfigIntf.intfConfig.newMember( intf )

def setIntfBfdParams( mode, args ):
   doSetIntfBfdParams( mode,
                       args[ 'INTERVAL' ],
                       args[ 'MIN_RX' ],
                       args[ 'MULTIPLIER' ],
                       legacy=( 'min_rx' in args ) )

def doSetIntfBfdParams( mode, bfdTxInt, bfdRxInt, bfdMult, legacy=False ):
   # If a non-legacy command was used to configure this attribute, flip legacyConfig
   # in Bfd::ConfigGlobal to False.
   bfdConfigGlobal.legacyConfig = legacy
   intf = IntfId( mode.intf.name )
   bfdConfigIntf.intfConfig[ intf ] = (
      BfdIntervalConfig( bfdTxInt, bfdRxInt, bfdMult ) )

def resetIntfBfdParams( mode, args ):
   intf = IntfId( mode.intf.name )
   del bfdConfigIntf.intfConfig[ intf ]

class BfdIntfParamsCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd interval INTERVAL ( min-rx | min_rx ) MIN_RX multiplier MULTIPLIER'
   noOrDefaultSyntax = 'bfd interval ...'
   data = {
      'bfd': matcherBfd,
      'interval': matcherInterval,
      'INTERVAL': matcherTxRxIntervalMs,
      'min-rx': matcherMinRx,
      'min_rx': CliCommand.Node( matcherMinRxDeprecated, hidden=True ),
      'MIN_RX': matcherTxRxIntervalMs,
      'multiplier': matcherMultiplier,
      'MULTIPLIER': CliMatcher.IntegerMatcher( 3, 50, helpdesc='Range is 3-50' ),
   }
   handler = setIntfBfdParams
   noOrDefaultHandler = resetIntfBfdParams

modelet.addCommandClass( BfdIntfParamsCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd echo"
# command in config-if mode
#------------------------------------------------------------------------------
def echoFunctionCmd( mode, no=False ):
   intf = IntfId( mode.intf.name )
   if no:
      if intf in bfdConfigIntf.echoOn:
         del bfdConfigIntf.echoOn[ intf ]
   else:
      bfdConfigIntf.echoOn[ intf ] = True

class BfdIntfEchoCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd echo'
   noOrDefaultSyntax = syntax
   data = {
      'bfd': matcherBfd,
      'echo': 'Echo function'
   }

   @staticmethod
   def handler( mode, args ):
      echoFunctionCmd( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      echoFunctionCmd( mode, no=True )

modelet.addCommandClass( BfdIntfEchoCmd )

#------------------------------------------------------------------------------
# "[no|default] bfd static neighbor" command in config-if mode
#------------------------------------------------------------------------------
def getBfdStaticAppVrfAppConfig( staticAppDir, vrfName ):
   staticAppVrfDir = staticAppDir.newEntity( 'Tac::Dir', vrfName )
   staticConfig = staticAppVrfDir.newEntity( 'Bfd::AppConfig', 'static' )
   staticConfig.appPid = 1
   return staticConfig

def delStaticPeer( peer, staticConfig, staticAppDir ):
   if peer in staticConfig.peerConfigVer:
      del staticConfig.peerConfigVer[ peer ]

   # If the last entry was deleted, remove the parent entity
   if not staticConfig.peerConfigVer:
      staticAppDir.deleteEntity( peer.vrf )

def addStaticPeer( peer, staticConfig ):
   if peer not in staticConfig.peerConfigVer:
      staticConfig.newPeerConfigVer( peer, 1 )

def staticNeighborFunctionCmd( mode, peer, no ):
   vrfName = getIntfVrf( mode.intf.name )

   staticConfigDir = ConfigMount.force( staticAppConfigDir )
   staticConfig = getBfdStaticAppVrfAppConfig( staticConfigDir, vrfName )
   if staticConfig is None:
      return

   bfdPeer = Tac.Value( 'Bfd::Peer', peer, vrfName )
   bfdPeer.intf = Tac.Value( 'Arnet::IntfId', mode.intf.name )

   if no:
      delStaticPeer( bfdPeer, staticConfig, staticConfigDir )
   else:
      addStaticPeer( bfdPeer, staticConfig )

# Ira hook that handles intf vrf changes
def changeVrfHook( intfId, oldVrf, newVrf, vrfDelete ):
   staticConfigDir = ConfigMount.force( staticAppConfigDir )
   oldStaticConfig = getBfdStaticAppVrfAppConfig( staticConfigDir, oldVrf )
   newStaticConfig = getBfdStaticAppVrfAppConfig( staticConfigDir, newVrf )

   if oldStaticConfig is None or newStaticConfig is None:
      return( False, None )

   # Get all peers associated with the old vrf that match the passed intfId
   peers = [ p for p in oldStaticConfig.peerConfigVer if p.intf == intfId ]

   for peer in peers:
      # Delete all peers for the old vrf that correspond with this intfId
      delStaticPeer( peer, oldStaticConfig, staticConfigDir )

      # Create a new peer entry with the new vrf
      newPeer = Tac.Value( 'Bfd::Peer', peer.ip, newVrf )
      newPeer.intf = Tac.Value( 'Arnet::IntfId', intfId )
      addStaticPeer( newPeer, newStaticConfig )

   return ( True, None )

class BfdIntfStaticNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd static neighbor PEERADDR'
   noOrDefaultSyntax = syntax
   data = {
      'bfd': matcherBfd,
      'static': 'Add static BFD configuration',
      'neighbor': 'Add a single-hop static BFD neighbor',
      'PEERADDR': IpGenAddrMatcher( helpdesc='IPv4/IPv6 neighbor address' ),
   }

   @staticmethod
   def handler( mode, args ):
      staticNeighborFunctionCmd( mode, args[ 'PEERADDR' ], False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      staticNeighborFunctionCmd( mode, args[ 'PEERADDR' ], True )

modelet.addCommandClass( BfdIntfStaticNeighborCmd )

#------------------------------------------------------------------------------
# Authentication command in config-if mode:
# [ no | default ] authentication mode ( disabled |
#                                        simple shared-secret profile PROFILE |
#                                        md5 shared-secret profile PROFILE
#                                            [ meticulous ] |
#                                        sha1 shared-secret profile PROFILE
#                                            [ meticulous ] )
#------------------------------------------------------------------------------
class BfdIntfAuthModeCmd( CliCommand.CliCommandClass ):
   # Having a kw arg passed to the CliParser named "mode" is problematic, since that
   # name is used all over the code as an argument.  To get around this, we have to
   # name the arg for this command "<mode>" in the data instead of simply "mode".
   syntax = ( 'bfd authentication mode ( disabled |'
                                     '( simple shared-secret profile PROFILE ) |'
                                     '( ( sha1 | md5 ) shared-secret profile PROFILE'
                                       '[ meticulous ] ) )' )
   noOrDefaultSyntax = 'bfd authentication ...'
   data = {
            'bfd': matcherBfd,
            'authentication': 'Configure BFD authentication',
            'mode': 'Specify BFD authentication mode',
            'disabled': 'Authentication disabled',
            'simple': 'Simple password authentication',
            'md5': 'Keyed MD5 authentication',
            'sha1': 'Keyed Sha1 authentication',
            'shared-secret': 'Specify a shared-secret',
            'profile': 'Shared-secret profile',
            'PROFILE': matcherAuthProfile,
            'meticulous': CliCommand.Node( CliMatcher.KeywordMatcher( 'meticulous',
                              helpdesc='Meticulous mode' ), hidden=True ),
          }

   @staticmethod
   def handler( mode, args ):
      intf = IntfId( mode.intf.name )
      bfdConfigIntf.authType[ intf ] = getAuthModeFromArgs( args )

      # If the mode configured is disabled, then there's no need to configure a
      # secret
      bfdConfigIntf.secretProfileName[ intf ] = args.get( 'PROFILE', '' )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = IntfId( mode.intf.name )
      bfdConfigIntf.authType[ intf ] = authType.authInherit
      bfdConfigIntf.secretProfileName[ intf ] = ''

modelet.addCommandClass( BfdIntfAuthModeCmd )

def Plugin( entityManager ):
   global l3IntfConfigDir
   global bfdConfigIntf
   global bfdConfigGlobal
   global staticAppConfigDir
   global authSharedSecretConfig

   l3IntfConfigDir = LazyMount.mount( entityManager, 'l3/intf/config',
                                      'L3::Intf::ConfigDir', 'r' )
   bfdConfigIntf = ConfigMount.mount( entityManager, 'bfd/config/intf',
                                      'Bfd::ConfigIntf', 'w')
   bfdConfigGlobal = ConfigMount.mount( entityManager, 'bfd/config/global',
                                        'Bfd::ConfigGlobal', 'w' )
   staticAppConfigDir = ConfigMount.mount( entityManager,
                                           'bfd/config/app/static',
                                           'Tac::Dir', 'wic' )
   IraVrfCli.canSetVrfHook.addExtension( changeVrfHook )
   authSharedSecretConfig = LazyMount.mount( entityManager,
                                   'mgmt/security/sh-sec-prof/config',
                                   'Mgmt::Security::SharedSecretProfile::Config',
                                   'r' )
   IntfCli.Intf.registerDependentClass( BfdIpIntf  )
