#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForClear
from CliToken.Ipv6 import ipv6MatcherForClear
import ConfigMount
import LazyMount
import SmashLazyMount
import Smash
import Tac

bfdHwStatus = None
bfdStatusPeer = None
hwStatusCli = None
hwConfigCli = None
vrfIdMap = None
bfdConfigInfo = None
aclStatus = None
aclCheckpoint = None

matcherBfd = CliMatcher.KeywordMatcher( 'bfd',
      helpdesc='Clear BFD counters and statistics' )
matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Clear BFD counters and statistics in all sessions' )

#--------------------------------------------------------------------------------
# clear bfd counters [ dest-ip ( DESTIP | DESTIP6 ) ]
#--------------------------------------------------------------------------------
def clearBfdCounters( mode, args ):
   netlinkStub = Tac.newInstance( "BfdPyUtils::BfdNetlinkControlStub" )
   netlinkStub.createFd( False )

   hwCommand = None

   # Extract the destination ip address from the passed args
   destIpAddr = None
   if 'dest-ip' in args:
      if 'DESTIP' in args:
         destIpAddr = args[ 'DESTIP' ]
      elif 'DESTIP6' in args:
         destIpAddr = args[ 'DESTIP6' ]

   if destIpAddr is None:
      # We clear stats for all peers.
      discriminator = 0
      module = True
      if bfdHwStatus.hwAccelerationEnabled:
         hwCommand = Tac.Value( "Bfd::BfdSessionId" )

   else:
      # Only clear for the indicated peer
      module = False
      ip = Tac.Value( 'Arnet::IpGenAddr', destIpAddr )

      peers = [ bfdStatusPeer.peerStatus[ peer ] for peer in bfdStatusPeer.peerStatus
                if peer.ip == ip ]

      if not peers:
         print ( "%s is an invalid BFD neighbor" % destIpAddr )
         return
      elif len( peers ) > 1:
         print ( "%s is ambiguous" % destIpAddr )
         return
      else:
         discriminator = peers[ 0 ].localDisc

      if bfdHwStatus.isHwAccelerated( discriminator ):
         for vrfId, entry in vrfIdMap.vrfIdToName.iteritems():
            if peers[ 0 ].peer.vrf == entry.vrfName:
               hwCommand = Tac.newInstance( "Bfd::BfdSessionId",
                                            vrfId,
                                            discriminator )
               break

   netlinkStub.sendGetStats( discriminator, True, True, module )

   # Cleanup previous commands that were processed by PlatformBfd
   if bfdHwStatus.hwAccelerationEnabled:
      statusId = hwStatusCli.clearStatsCommandId
      for commandId in hwConfigCli.clearStatsCommand:
         if 0 < ( statusId - commandId ) & 0xffffffff < 0x7fffffff:
            del hwConfigCli.clearStatsCommand[ commandId ]

   if hwCommand is not None:
      currentId = hwConfigCli.clearStatsCommandId
      hwConfigCli.clearStatsCommand[ currentId ] = hwCommand
      hwConfigCli.clearStatsCommandId = ( currentId + 1 ) & 0xffffffff

class ClearBfdCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bfd counters [ dest-ip ( DESTIP | DESTIP6 ) ]'
   data = {
      'clear': clearKwNode,
      'bfd': matcherBfd,
      'counters': matcherCounters,
      'dest-ip': 'BFD neighbor IP address',
      'DESTIP': IpAddrMatcher( helpdesc='IPv4 Neighbor address' ),
      'DESTIP6': Ip6AddrMatcher( helpdesc='IPv6 Neighbor address' ),
   }
   handler = clearBfdCounters

BasicCliModes.EnableMode.addCommandClass( ClearBfdCountersCmd )

#--------------------------------------------------------------------------------
# clear bfd history
#--------------------------------------------------------------------------------
def clearBfdHistory( mode, args ):
   if bfdStatusPeer.peerHistory:
      bfdConfigInfo.historyGeneration = bfdConfigInfo.historyGeneration + 1

class ClearBfdHistoryCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bfd history'
   data = {
      'clear': clearKwNode,
      'bfd': matcherBfd,
      'history': 'Clear BFD history',
   }
   handler = clearBfdHistory

BasicCliModes.EnableMode.addCommandClass( ClearBfdHistoryCmd )

#--------------------------------------------------------------------------------
# clear bfd ip/ipv6 access-list counters
#--------------------------------------------------------------------------------
def clearAclCounters( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

class ClearBfdAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bfd ( ip | ipv6 ) access-list counters'
   data = {
      'clear': clearKwNode,
      'bfd': matcherBfd,
      'ip': ipMatcherForClear,
      'ipv6': ipv6MatcherForClear,
      'access-list': 'Named access-list',
      'counters': matcherCounters,
   }
   handler = clearAclCounters

BasicCliModes.EnableMode.addCommandClass( ClearBfdAccessListCountersCmd )

def Plugin( entityManager ):
   global bfdHwStatus
   global bfdStatusPeer
   global hwStatusCli
   global hwConfigCli
   global vrfIdMap
   global bfdConfigInfo
   global aclStatus
   global aclCheckpoint

   bfdHwStatus = LazyMount.mount( entityManager, 'bfd/hwStatus',
                                  'Hardware::Bfd::Status', 'r' )
   bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                    'Bfd::StatusPeer', 'r')
   hwStatusCli = LazyMount.mount( entityManager, "bfd/hwStatusCli",
                                  'Bfd::HwStatusCli', 'r' )
   hwConfigCli = ConfigMount.mount( entityManager, 'bfd/hwConfigCli',
                                    'Bfd::HwConfigCli', 'w' )
   vrfIdMap = SmashLazyMount.mount( entityManager, 'vrf/vrfIdMapStatus',
                                    'Vrf::VrfIdMap::Status',
                                    Smash.mountInfo( 'reader' ) )
   bfdConfigInfo = LazyMount.mount( entityManager, 'bfd/config/info',
                                    'Bfd::ConfigInfo', 'w')
   aclStatus = LazyMount.mount( entityManager, 'acl/status/all',
                                'Acl::Status', 'r' )
   aclCheckpoint = LazyMount.mount( entityManager, 'acl/checkpoint',
                                    'Acl::CheckpointStatus', 'w' )
