# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from CliModel import Model, Dict, Bool, Int
from IntfModels import Interface
import Arnet
from Arnet.NsLib import DEFAULT_NS, runMaybeInNetNs
import os
import re
import Tac

bessProcDirName = "BessProc"

class OutgoingInterfaceGate( Model ):
   outgoingInterface = Interface( help="Outgoing interface for the route" )
   outgoingInterfaceGate = \
         Int( help="Software networking gate for outgoing interface" )

   def render( self ):
      print "    %s,oifGate=%s" % \
            ( self.outgoingInterface.stringValue, self.outgoingInterfaceGate )

class MfibMroute( Model ):
   incomingInterface = Interface( help="Incoming interface for the route",
                                  optional=True )
   incomingInterfaceFrr = \
         Interface( help="Incoming interface for fast failover route",
                    optional=True )
   notifyInterface = Interface( help="Notify interface for the route",
                                optional=True )
   staticFastdropInterface = \
         Interface( help="Interface with static fastdrop installed",
                    optional=True )
   toCpu = Bool( help="Route has local listeners", default=False )
   iifRegister = Bool( help="Route is iif register", default=False )
   oifRegister = Bool( help="Route is oif register", default=False )
   outgoingInterfaceGates = \
         Dict( help="A mapping of outgoing interface to gate information",
               keyType=Interface, valueType=OutgoingInterfaceGate )

   def render( self ):
      # pylint: disable=no-member
      if self.incomingInterface:
         print "    %s (iif)" % self.incomingInterface.stringValue
      if self.incomingInterfaceFrr:
         print "    %s (iifFrr)" % self.incomingInterfaceFrr.stringValue
      if self.notifyInterface:
         print "    %s (notify)" % self.notifyInterface.stringValue
      if self.staticFastdropInterface:
         print "    %s (staticFastdrop)" % self.staticFastdropInterface.stringValue
      # pylint: enable=no-member
      if self.toCpu:
         print "    Cpu"
      if self.iifRegister:
         print "    IifRegister"
      if self.oifRegister:
         print "    OifRegister"
      for oif in Arnet.sortIntf( self.outgoingInterfaceGates ):
         self.outgoingInterfaceGates[ oif ].render()

class MfibMrouteGroup( Model ):
   mroutes = Dict( help="A mapping of source address to MFIB mroutes",
                   keyType=IpGenericAddress, valueType=MfibMroute )

   def renderMroutes( self, group ):
      for source, mrouteModel in self.mroutes.iteritems():
         print "%s %s" % ( group, source )
         mrouteModel.render()

class MfibMrouteGroups( Model ):
   groups = Dict( help="A mapping of group address to MFIB mroute groups",
                  keyType=IpGenericAddress, valueType=MfibMrouteGroup )

   def fromBessProc( self, af, ns ):
      fileName = os.path.join( os.sep, bessProcDirName,
                               "Mroute%s_%s" % ( af[ -1 ], ns ) )

      if ns == DEFAULT_NS:
         with open( fileName ) as f:
            lines = f.readlines()
      else:
         output = runMaybeInNetNs( ns, [ 'cat', fileName ],
            asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
         lines = output.rstrip().split( '\n' )

      # Extract BessProc file output to generate more human-friendly CLI output
      sgRegex = "Source: (.*) Group: (.*)"
      iifRegex = "Iif: (.*)"
      frrRegex = "IifFrr: (.*)"
      notifyRegex = "Notify: (.*)"
      fastdropRegex = "Fastdrop: (.*)"
      toCpuRegex = "ToCpu: (.*)"
      iifRegisterRegex = "IifRegister: (.*)"
      oifRegisterRegex = "OifRegister: (.*)"
      oifsRegex = "Oifs: (.*)"
      oifGatesRegex = "OifGates: (.*)"

      currMrouteModel = None
      oifList = []
      for line in lines:
         sgMatch = re.match( sgRegex, line )
         iifMatch = re.match( iifRegex, line )
         frrMatch = re.match( frrRegex, line )
         notifyMatch = re.match( notifyRegex, line )
         fastdropMatch = re.match( fastdropRegex, line )
         toCpuMatch = re.match( toCpuRegex, line )
         iifRegisterMatch = re.match( iifRegisterRegex, line )
         oifRegisterMatch = re.match( oifRegisterRegex, line )
         oifsMatch = re.match( oifsRegex, line )
         oifGatesMatch = re.match( oifGatesRegex, line )

         if sgMatch:
            if sgMatch.group( 2 ) not in self.groups:
               self.groups[ sgMatch.group( 2 ) ] = MfibMrouteGroup()
            mfibMrouteGroup = self.groups[ sgMatch.group( 2 ) ]

            mfibMrouteGroup.mroutes[ sgMatch.group( 1 ) ] = MfibMroute()
            currMrouteModel = mfibMrouteGroup.mroutes[ sgMatch.group( 1 ) ]
         elif iifMatch:
            currMrouteModel.incomingInterface = iifMatch.group( 1 )
         elif frrMatch:
            currMrouteModel.incomingInterfaceFrr = frrMatch.group( 1 )
         elif notifyMatch:
            currMrouteModel.notifyInterface = notifyMatch.group( 1 )
         elif fastdropMatch:
            currMrouteModel.staticFastdropInterface = fastdropMatch.group( 1 )
         elif toCpuMatch and int( toCpuMatch.group( 1 ) ):
            currMrouteModel.toCpu = True
         elif iifRegisterMatch and int( iifRegisterMatch.group( 1 ) ):
            currMrouteModel.iifRegister = True
         elif oifRegisterMatch and int( oifRegisterMatch.group( 1 ) ):
            currMrouteModel.oifRegister = True
         elif oifsMatch:
            oifList = oifsMatch.group( 1 ).split( " " )
         elif oifGatesMatch and oifGatesMatch.group( 1 ):
            for i, oifGate in enumerate( oifGatesMatch.group( 1 ).split( " " ) ):
               oifGateModel = OutgoingInterfaceGate()
               oifGateModel.outgoingInterface = oifList[ i ]
               oifGateModel.outgoingInterfaceGate = int( oifGate )
               currMrouteModel.outgoingInterfaceGates[ oifList[ i ] ] = oifGateModel

   def render( self ):
      for group, groupModel in self.groups.iteritems():
         groupModel.renderMroutes( group )
