#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import ConfigMount
import RouterMulticastCliLib
from McastCommonCliLib import AddressFamily
import BasicCli
from Arnet.NsLib import DEFAULT_NS
import LazyMount
import Cell
import BessMgrModel
import ShowCommand
import CliCommand
import CliMatcher
import CliToken.Platform
from CliPlugin.SfeCliLib import nodeSfeEos
from CliPlugin.VrfCli import DEFAULT_VRF, VrfExprFactory

traceHandle = Tracing.Handle( "BessMgrCli" )
t0 = traceHandle.trace0

bessAgentStatus = None
allVrfStatusLocal = None

class ForwardingType( CliCommand.CliCommandClass ):
   syntax = 'software-forwarding FORWARDING_TYPE'
   data = {
      'software-forwarding': 'Multicast packet software forwarding configuration',
      'FORWARDING_TYPE': CliMatcher.EnumMatcher( {
         'kernel': 'Software forwarding by the Linux kernel',
         'sfe': 'Software forwarding by SFE (Software Forwarding Engine)'
      } )
   }

   @staticmethod
   def handler( mode, args ):
      forwardingType = args[ 'FORWARDING_TYPE' ]
      af = RouterMulticastCliLib.getAddressFamilyFromMode( mode )
      bessEnabled = forwardingType.title() == 'Sfe'
      if af == AddressFamily.ipv6:
         bessAgentStatus.v6Enabled = bessEnabled
      else:
         bessAgentStatus.v4Enabled = bessEnabled

RouterMulticastCliLib.RouterMulticastNonVrfAfModelet.addCommandClass(
   ForwardingType )

def nsFromVrf( vrfName=DEFAULT_VRF ):
   ns = DEFAULT_NS
   if vrfName == DEFAULT_VRF:
      return ns
   elif vrfName in allVrfStatusLocal.vrf:
      ns = allVrfStatusLocal.vrf[ vrfName ].networkNamespace
      return ns
   else:
      return None

class ShowMroute( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform sfe mroute AF [ VRF ]'

   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'sfe' : nodeSfeEos,
      'mroute' : 'multicast route information',
      'AF' : CliMatcher.EnumMatcher( { "ipv4" : "IPV4",
                                       "ipv6" : "IPV6" } ),
      'VRF' : VrfExprFactory( helpdesc='Use a specific VRF' ),
   }
   cliModel = BessMgrModel.MfibMrouteGroups

   @staticmethod
   def handler( mode, args ):
      vrfName = args.get( 'VRF', DEFAULT_VRF )
      af = args[ 'AF' ]

      model = BessMgrModel.MfibMrouteGroups()

      ns = nsFromVrf( vrfName )
      # Namespace required for BessProc file name
      if not ns:
         mode.addWarning( "VRF name %s is not defined." % vrfName )
         return model

      try:
         model.fromBessProc( af, ns )
      except ( IOError, Tac.SystemCommandError ):
         mode.addWarning(
            "BESS mroute data for VRF %s in IPv%s is not available." %
            ( vrfName, af[ -1 ] ) )
         return BessMgrModel.MfibMrouteGroups()

      return model

BasicCli.addShowCommandClass( ShowMroute )

def Plugin( entityManager ):
   global bessAgentStatus, allVrfStatusLocal
   bessAgentStatus = ConfigMount.mount( entityManager,
      'bess/agentstatus', 'McastCommon::BessAgentStatus', 'w' )
   allVrfStatusLocal = LazyMount.mount( entityManager,
      Cell.path( "ip/vrf/status/local" ), "Ip::AllVrfStatusLocal", "r" )
