# Copyright (c) 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, CliParser, LazyMount, Tac
import CliCommand
import CliPlugin.FruCli as FruCli
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import CommonGuards
import Intf.IntfRange
import LedCommon
import ShowCommand

verbose = False

ledConfig = None
hwLedConfig = None
# set to either "Fixed" or "Modular"
systemType = None

plutoStatus = Tac.Type( "Led::PlutoStatus" )
beaconSetName = "beacon"

def _registerSystemTypeFixed():
   global systemType
   systemType = "Fixed"
FruCli.registerFixedSystemCallback( _registerSystemTypeFixed )

def _registerSystemTypeModular():
   global systemType
   systemType = "Modular"
FruCli.registerModularSystemCallback( _registerSystemTypeModular )

# -----------------------------------------------------------------------------
# [no] locator-led { module { Supervisor <#> | Linecard <#> | Fabric <#> |
#                             Switchcard | <#> } |
#                    interface <interface range> |
#                    powersupply <powersupply number> |
#                    fan <fan number> |
#                    chassis |
#                    multifan }
# -----------------------------------------------------------------------------

def _setLed( led, disable ):
   assert hwLedConfig != None
   assert ledConfig != None

   ledExists, led = LedCommon.checkLedExists( led, hwLedConfig )
   if not ledExists:
      print "This LED cannot be used as locator-led target"
      return

   beacon = LedCommon.getLedSet( beaconSetName, ledConfig )
   ledShowName = LedCommon.ledToShowName( led, systemType )

   if disable:
      if led in beacon.led:
         print "Disabling locator led for", ledShowName
         del beacon.led[ led ]
      else:
         print "Locator led for", ledShowName, "already disabled"
   else:
      if led not in beacon.led:
         print "Enabling locator led for", ledShowName

         LedColor = Tac.Type( "Led::Color" )
         hwLed = hwLedConfig.leds[ led ]

         ledSetting = Tac.Value( "Led::LightSetting" )
         ledSetting.rampRate = 0.0104
         ledSetting.maxBright = 100
         ledSetting.flashRate = 1
         ledSetting.ledFlashRate = 1
         ledSetting.plutoStatus = plutoStatus.plutoStatusBeacon
         # Check if Led supports blue beacon, if so, set to blue.  Otherwise,
         # beacons for interfaces are flashing yellow, while for modules
         # we use flashing red.
         if hwLed.ledAttribute and hwLed.ledAttribute.blue:
            ledSetting.blue = True
            ledSetting.color = LedColor.BLUE
         elif led.startswith( 'Ethernet' ):
            ledSetting.yellow = True
            ledSetting.color = LedColor.YELLOW
         else:
            ledSetting.red = True
            ledSetting.color = LedColor.RED
         
         beacon.led[ led ] = ledSetting
      else:
         print "Locator led for", ledShowName, "already enabled"

def _targetToLed( target ):
   if verbose:
      print( "Target tag: ", target.tag )
      print( "Target label: ", target.label )

   for prefix in ( 'Supervisor', 'Linecard', 'Fabric', 'Switchcard', 'PowerSupply',
                   'Fan' ):
      if target.tag.startswith( prefix ):
         if prefix == 'Supervisor':
            prefix = "Status"
         return "%s%s" % ( prefix, target.label )

   return None

def doLocatorLedTarget( mode, noOption=None, target=None ):
   assert target != None
   if verbose:
      print "NoOption: ", noOption
      print "Target: ", target
      print "Name :", target.name

   _setLed( _targetToLed( target ), noOption )


def guardMultiFanBeacon( mode, token ):
   hwLed = hwLedConfig.leds.get( "MultiFan1" )
   if hwLed:
      return None
   return CliParser.guardNotThisPlatform

nodeLocatorLed = CliCommand.guardedKeyword( 'locator-led',
      helpdesc='Set a specific LED to flash',
      guard=CommonGuards.ssoStandbyGuard )

#--------------------------------------------------------------------------------
# [ no ] locator-led ( ( fantray FAN-TRAY ) | ( powersupply POWER_SUPPLY ) )
#--------------------------------------------------------------------------------
class LocatorLedCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led ( ( fantray FAN_TRAY ) | ( powersupply POWER_SUPPLY ) )'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'fantray' : 'Fan tray LED',
      'FAN_TRAY' : FruCli.FanTrayMatcher(),
      'powersupply' : 'Power supply LED',
      'POWER_SUPPLY' : FruCli.PowerSupplyMatcher(),
   }

   @staticmethod
   def handler( mode, args ):
      target = args.get( 'FAN_TRAY', args.get( 'POWER_SUPPLY' ) )
      doLocatorLedTarget( mode, CliCommand.isNoCmd( args ), target=target )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedCmd )

#--------------------------------------------------------------------------------
# [ no ] locator-led multifan
#--------------------------------------------------------------------------------
class LocatorLedMultifanCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led multifan'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'multifan' : CliCommand.guardedKeyword( 'multifan', helpdesc='Multi fan LED',
         guard=guardMultiFanBeacon ),
   }

   @staticmethod
   def handler( mode, args ):
      _setLed( 'MultiFan1', CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedMultifanCmd )
                                 
#--------------------------------------------------------------------------------
# [ no ] locator-led interface INTF
#--------------------------------------------------------------------------------
class LocatorLedInterfaceIntf1Cmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led interface INTF'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'interface' : 'Interface LED',
      'INTF' : Intf.IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
   }
   
   @staticmethod
   def handler( mode, args ):
      for led in args[ 'INTF' ].intfNames():
         _setLed( led, CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedInterfaceIntf1Cmd )

#--------------------------------------------------------------------------------
# [ no ] locator-led module SLOT
#--------------------------------------------------------------------------------
class LocatorLedModuleCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led module SLOT'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'module' : CliCommand.guardedKeyword( 'module', helpdesc='Module LED',
         guard=FruCli.modularSystemGuard ),
      'SLOT' : FruCli.SlotExpressionFactory()
   }

   @staticmethod
   def handler( mode, args ):
      target = args[ 'SLOT' ].slot
      doLocatorLedTarget( mode, CliCommand.isNoCmd( args ), target=target )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedModuleCmd )

# -----------------------------------------------------------------------------
# [ no ] locator-led chassis
# 'locator-led chassis' command is only available on FixedSystems that support
# a blue status Led.  The guard for this command just looks up the capability
# of the 'Status1' led and see if it supports blue.  Older platform (Napa)
# are not supported.
# -----------------------------------------------------------------------------
def guardChassisBeacon( mode, token ):
   if FruCli.fixedSystemGuard( mode, token ):
      return CliParser.guardNotThisPlatform
   
   for hwLed in hwLedConfig.leds.itervalues():
      if hwLed.name.startswith( 'Status' ):
         if hwLed.ledAttribute.blue:
            return None
   return CliParser.guardNotThisPlatform

class LocatorLedChassisCmd( CliCommand.CliCommandClass ):
   syntax = 'locator-led chassis'
   noSyntax = syntax
   data = {
      'locator-led' : nodeLocatorLed,
      'chassis': CliCommand.guardedKeyword( 'chassis', 'Chassis beacon LED',
         guard=guardChassisBeacon )
   }
   @staticmethod
   def handler( mode, args ):
      for led in hwLedConfig.leds.keys():
         if led.startswith( 'Status' ):
            _setLed( led, CliCommand.isNoCmd( args ) )

   noHandler = handler

BasicCli.EnableMode.addCommandClass( LocatorLedChassisCmd )

# -----------------------------------------------------------------------------
# show locator-led
#
# Shows the list of currently enabled locator-led (aka beacon led)
# -----------------------------------------------------------------------------
class ShowLocatorLedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show locator-led'
   data = {
      'locator-led' : CliCommand.guardedKeyword( 'locator-led',
         helpdesc='Show specific LED to flash',
         guard=CommonGuards.ssoStandbyGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      beacon = LedCommon.getLedSet( beaconSetName, ledConfig )

      if beacon.led:
         print "Locator LED currently enabled"
         print "-----------------------------"
         # Print chassis only once for FixedSystems
         chassisLeds = [ led for led in beacon.led if led.startswith( 'Status' ) ]
         if chassisLeds:
            if systemType == "Fixed":
               print LedCommon.ledToShowName( chassisLeds[ 0 ], systemType )
            else:
               for led in sorted( chassisLeds ):
                  print LedCommon.ledToShowName( led, systemType )
         otherLeds = [ led for led in beacon.led if led not in chassisLeds ]
         for led in sorted( otherLeds ):
            print LedCommon.ledToShowName( led, systemType )
      else:
         print "There are no locator LED enabled"

BasicCli.addShowCommandClass( ShowLocatorLedCmd )

# -----------------------------------------------------------------------------
# Cli Plugin initialization
# -----------------------------------------------------------------------------

def Plugin( entityManager ):
   global hwLedConfig, ledConfig

   hwLedConfig = LazyMount.mount( entityManager, "hardware/led/configInit",
                                  "Hardware::Led::LedSystemConfigDir", "r" )
   ledConfig = LazyMount.mount( entityManager, "led/config", "Tac::Dir", "w" )
