#!/usr/bin/env python
# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements interface-specific configuration.
#
# [no|default] switchport backup interface <intf> prefer vlan <vlans>
#-------------------------------------------------------------------------------
import BasicCli, IntfCli, Tracing, VlanCli, Ethernet
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
from MacAddr import macAddrMatcher
from StpCliUtil import ( stpConfigIs,
                         stpInputConfig,
                         stpInputConfigIs,
                         stpStatus,
                         stpStatusIs )
import Vlan

__defaultTraceHandle__ = Tracing.Handle( 'StpCli' )

modelet = VlanCli.SwitchportModelet

ethIntfStatusDir = None

deprecatedInterfaceMatcher = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='interface' )
backupLinkMatcher = CliMatcher.KeywordMatcher( 'backup-link',
      helpdesc='Specify a backup interface' )
preferMatcher = CliMatcher.KeywordMatcher( 'prefer',
      helpdesc='Specify VLANs to carry on the backup interface' )
backupMatcher = CliMatcher.KeywordMatcher( 'backup',
      helpdesc='Specify a backup interface' )
vlanMatcher = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Carry on the backup interface' )
preemptionDelayMatcher = CliMatcher.KeywordMatcher( 'preemption-delay',
      helpdesc='Specify preemption delay in milliseconds' )
macMoveBurstMatcher = CliMatcher.KeywordMatcher( 'mac-move-burst', 
      helpdesc='Specify size of MAC move bursts' )
interMacMoveBurstDelayMatcher = CliMatcher.KeywordMatcher( 'mac-move-burst-interval',
      helpdesc='Specify MAC move burst interval in milliseconds' )
initialMacMoveDelayMatcher = CliMatcher.KeywordMatcher( 'initial-mac-move-delay',
      helpdesc='Specify initial MAC move delay in milliseconds' )
dstMacAddressMatcher = CliMatcher.KeywordMatcher( 'dest-macaddr',
      helpdesc='Specify dest MAC address for MAC move updates' )

millisecondRange = CliMatcher.IntegerMatcher( 0, 65535,
      helpdesc='Milliseconds' )

#-------------------------------------------------------------------------------
# [no|default] switchport backup-link <intf>
#
# legacy:
# [no|default] switchport backup interface <intf>
#-------------------------------------------------------------------------------
def setBackupIntf( mode, args ): 
   intf = args[ 'INTF' ]
   vlanSet = args.get( 'VLAN_SET' )
   backupIntf = intf
   if mode.intf.name == backupIntf.name:
      mode.addError( "Backup interface configuration error: the interface cannot "
                     "back up itself" )
      return
   stpConfig = stpInputConfig()
   if mode.intf.name in stpConfig.backupToBackupIntfConfig:
      mode.addError( "Backup interface configuration error: the primary interface "
                     "is already configured as a backup" )
      return
   if backupIntf.name in stpConfig.backupIntfConfig:
      mode.addError( "Backup interface configuration error: the backup interface "
                     "is already configured as a primary" )
      return
   bic = stpConfig.backupToBackupIntfConfig.get( backupIntf.name )
   if bic and (bic.name != mode.intf.name):
      mode.addError( "Backup interface configuration error: the backup interface "
                     "is already configured as a backup for another interface" )
      return
   bic = None
   oldConfig = None
   if mode.intf.name in stpConfig.backupIntfConfig:
      oldConfig = stpConfig.backupIntfConfig[ mode.intf.name ]
      del stpConfig.backupIntfConfig[ mode.intf.name ]
   bic = stpConfig.backupIntfConfig.newMember( mode.intf.name, backupIntf.name )
   stpConfig.backupToBackupIntfConfig[ backupIntf.name ] = bic
   if oldConfig:
      bic.preemptionDelay = oldConfig.preemptionDelay
      bic.macMoveBurst = oldConfig.macMoveBurst
      bic.interMacMoveBurstDelay = oldConfig.interMacMoveBurstDelay
      bic.initialMacMoveDelay = oldConfig.initialMacMoveDelay
      bic.dstMacAddress = oldConfig.dstMacAddress

   if vlanSet is not None:
      for vlanId in vlanSet.ids:
         bic.backupPreferredVlans[ int( vlanId ) ] = True

def noBackupIntf( mode, args ):
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      return
   del stpConfig.backupIntfConfig[ mode.intf.name ]
   if bic.backupIntfName in stpConfig.backupToBackupIntfConfig:
      del stpConfig.backupToBackupIntfConfig[ bic.backupIntfName ]
   return

class BackupLinkIntf( CliCommand.CliCommandClass ):
   syntax = 'switchport backup-link INTF [ prefer vlan VLAN_SET ]'
   noOrDefaultSyntax = 'switchport backup-link [ INTF prefer vlan VLAN_SET ]'
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup-link': backupLinkMatcher,
         'INTF': IntfCli.Intf.matcherWOSubIntf,
         'prefer': preferMatcher,
         'vlan' : vlanMatcher,
         'VLAN_SET': VlanCli.vlanSetMatcher 
   }
   
   handler = setBackupIntf
   noOrDefaultHandler = noBackupIntf

modelet.addCommandClass( BackupLinkIntf )

class BackupIntf( CliCommand.CliCommandClass ):
   syntax = 'switchport backup interface INTF'
   noOrDefaultSyntax = syntax
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup': backupMatcher,
         'interface': CliCommand.Node(
                      matcher=deprecatedInterfaceMatcher,
                      deprecatedByCmd='switchport backup-link INTF' ),
         'INTF': IntfCli.Intf.matcherWOSubIntf
         }

   handler = setBackupIntf 
   noOrDefaultHandler = noBackupIntf

modelet.addCommandClass( BackupIntf )

def setBackupIntfPreemptionDelay( mode, args ):
   preemptionDelay = args[ 'RANGE' ]
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      mode.addError( "The backup interface needs to be configured first" )
      return
   bic.preemptionDelay = preemptionDelay

def noBackupIntfPreemptionDelay( mode, args ):
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      mode.addError( "The backup interface needs to be configured first" )
      return
   bic.preemptionDelay = bic.defaultPreemptionDelay


class BackupIntfPreemptionDelay( CliCommand.CliCommandClass ):
   syntax = 'switchport backup preemption-delay RANGE'
   noOrDefaultSyntax = 'switchport backup preemption-delay'
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup': backupMatcher,
         'preemption-delay': preemptionDelayMatcher,
         'RANGE': millisecondRange 
         }

   handler = setBackupIntfPreemptionDelay
   noOrDefaultHandler = noBackupIntfPreemptionDelay

modelet.addCommandClass( BackupIntfPreemptionDelay )

def configBackupIntfMacMoveBurstRange( mode, args ):
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      mode.addError( "The backup interface needs to be configured first" )
      return
   bic.macMoveBurst = args.get( 'MAC_MOVE_BURST', bic.defaultMacMoveBurst )

class BackupIntfMacMoveBurst( CliCommand.CliCommandClass ):
   syntax = 'switchport backup mac-move-burst MAC_MOVE_BURST'
   noOrDefaultSyntax = 'switchport backup mac-move-burst ...'
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup': backupMatcher,
         'mac-move-burst': macMoveBurstMatcher,
         'MAC_MOVE_BURST': millisecondRange 
   }
  
   handler = configBackupIntfMacMoveBurstRange
   noOrDefaultHandler = configBackupIntfMacMoveBurstRange

modelet.addCommandClass( BackupIntfMacMoveBurst )

def configInterMacMoveBurstDelay( mode, args ):
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      mode.addError( "The backup interface needs to be configured first" )
      return
   bic.interMacMoveBurstDelay = args.get( 'MAC_BURST_DELAY',
                                          bic.defaultInterMacMoveBurstDelay )

class BackupIntfMacMoveBurstInterval( CliCommand.CliCommandClass ):
   syntax = 'switchport backup mac-move-burst-interval MAC_BURST_DELAY'
   noOrDefaultSyntax = 'switchport backup mac-move-burst-interval ...'
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup': backupMatcher,
         'mac-move-burst-interval': interMacMoveBurstDelayMatcher,
         'MAC_BURST_DELAY': millisecondRange 
   }
   
   handler = configInterMacMoveBurstDelay
   noOrDefaultHandler = configInterMacMoveBurstDelay

modelet.addCommandClass( BackupIntfMacMoveBurstInterval )

def configInitialMacMoveDelay( mode, args ):
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      mode.addError( "The backup interface needs to be configured first" )
      return
   bic.initialMacMoveDelay = args.get( 'MAC_MOVE_DELAY',
                                       bic.defaultInitialMacMoveDelay )

class BackupIntfInitialMacMoveDelay( CliCommand.CliCommandClass ):
   syntax = 'switchport backup initial-mac-move-delay MAC_MOVE_DELAY'
   noOrDefaultSyntax = 'switchport backup initial-mac-move-delay ...'
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup': backupMatcher,
         'initial-mac-move-delay': initialMacMoveDelayMatcher,
         'MAC_MOVE_DELAY': millisecondRange
   }
   
   handler = configInitialMacMoveDelay
   noOrDefaultHandler = configInitialMacMoveDelay

modelet.addCommandClass( BackupIntfInitialMacMoveDelay )

def configDstMacAddress( mode, args ):
   stpConfig = stpInputConfig()
   bic = stpConfig.backupIntfConfig.get( mode.intf.name, None )
   if not bic:
      mode.addError( "The backup interface needs to be configured first" )
      return
   macAddr = args.get( 'MAC_ADDR' )
   if ( macAddr and
        not ( Ethernet.isMulticast( macAddr ) or Ethernet.isBroadcast( macAddr ) ) ):
      mode.addError( "The destination mac address should be multicast "
                        "or broadcast" )
   bic.dstMacAddress = args.get( 'MAC_ADDR', bic.defaultDstMacAddress )

class BackupIntfDstMacAddress( CliCommand.CliCommandClass ):
   syntax = 'switchport backup dest-macaddr MAC_ADDR'
   noOrDefaultSyntax = 'switchport backup dest-macaddr ...'
   data = {
         'switchport': VlanCli.switchportMatcher,
         'backup': backupMatcher,
         'dest-macaddr': dstMacAddressMatcher,
         'MAC_ADDR': macAddrMatcher
   }
   
   handler = configDstMacAddress
   noOrDefaultHandler = configDstMacAddress

modelet.addCommandClass( BackupIntfDstMacAddress )

#-------------------------------------------------------------------------------
# show interfaces <intf> switchport backup-link
#
# legacy:
# show interfaces <intf> switchport backup
#-------------------------------------------------------------------------------
def backupIntfState( mode, backupStatus, which ):
   if backupStatus is None:
      return "Inactive Configuration"
   if which == 'primary':
      intfName = backupStatus.name
   else:
      intfName = backupStatus.backupIntfName

   intf = ethIntfStatusDir.get( intfName )

   if (intf is not None) and \
      (intf.forwardingModel == 'intfForwardingModelBridged') and \
      (intf.linkStatus == 'linkUp'):
      return 'Up'
   else:
      return 'Down'

def printActiveVlans( backupStatus, which ):
   if which == 'primary':
      intfName = backupStatus.name
      vlans = dict( backupStatus.primaryActiveVlan.items() )
   else:
      intfName = backupStatus.backupIntfName
      vlans = dict( backupStatus.backupActiveVlan.items() )
   
   leadIn = "%s forwarding vlans: " % intfName
   print leadIn,
   
   canonicalStr = Vlan.vlanSetToCanonicalString( vlans )
   canonicalParts = canonicalStr.split( ',' )
   if canonicalParts == []:
      return
   
   prefix = ' ' * len( leadIn )
   firstLine = True
   line = canonicalParts[ 0 ]
   for part in canonicalParts[ 1 : ]:
      if len( line ) + len( ',' ) + len( part ) > 39:
         if not firstLine:
            print prefix,
         print line
         firstLine = False
         line = ''
      else:
         line += ','
      line += part
   if not firstLine:
      print prefix,
   print line
      
def doShowSwitchportBackup( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   if intfs is None:
      return

   config = stpInputConfig()
   status = stpStatus()
   backupConfigs = [ x for x in config.backupIntfConfig.itervalues() ]

   for i in intfs:
      for backupConfig in backupConfigs:
         if (i.name != backupConfig.name) and \
            (i.name != backupConfig.backupIntfName):
            continue
         backupStatus = status.backupIntfStatus.get( backupConfig.name )

         primaryState = backupIntfState( mode, backupStatus, 'primary' )
         backupState = backupIntfState( mode, backupStatus, 'backup' )

         print "Switch backup interface pair: %s, %s" % \
               (backupConfig.name, backupConfig.backupIntfName )
         print "%-18s %-16s %-7s %s" % ('Primary Interface:', backupConfig.name,
                                     'State:', primaryState )
         print "%-18s %-16s %-7s %s" % ('Backup Interface:',
                                     backupConfig.backupIntfName, 'State:',
                                     backupState )
         if backupStatus is not None:
            printActiveVlans( backupStatus, 'primary' )
            printActiveVlans( backupStatus, 'backup' )
         print "%s: %d milliseconds" % ( 'Preemption delay', 
                                        backupConfig.preemptionDelay )
         print "%s: %d" % ( 'Mac move burst size', 
                                        backupConfig.macMoveBurst )
         print "%s: %d milliseconds" % ( 'Mac move burst interval',
                                         backupConfig.interMacMoveBurstDelay )
         print "%s: %s" % ( 'Mac move destination',
                                       backupConfig.dstMacAddress )
         backupConfigs.remove( backupConfig )
         print

backupShowDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'backup',
                           helpdesc='Show interface switchport backup information' ),
   deprecatedByCmd='show interfaces <intf> switchport backup-link' )

class ShowIntfSwitchportBackup( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces switchport backup-link | backup'
   data = { 'switchport' : 'Details on switchports',
            'backup-link' : 'Show interface switchport backup information',
            'backup' : backupShowDeprecated }
   handler = doShowSwitchportBackup
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfSwitchportBackup )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   config = LazyMount.mount( entityManager, "stp/config", "Stp::Config", "r" )
   stpConfigIs( config )
   cliConfig = ConfigMount.mount( entityManager, "stp/input/config/cli",
                                  "Stp::Input::Config", "w" )
   stpInputConfigIs( cliConfig )
   status = LazyMount.mount( entityManager, "stp/status", "Stp::Status", "r" )
   stpStatusIs( status )
   global ethIntfStatusDir
   ethIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/intf",
                                       "Interface::EthIntfStatusDir", "r" )


