# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import BasicCliUtil
import subprocess
import AsuUtil

from ProductAttributes import productAttributes
from CliPlugin.ReloadCli import reloadKwNode
from CliPlugin.AsuReloadCli import ( _asuData, _asuReloadCliHelper,
                                     generateProcOutput, rebootHitlessGuard )

def isReloadNow( mode, args ):
   return "now" in args

# Silly pylint, don't complain about space after = in decorators:
# pylint: disable-msg=C0322
@BasicCliUtil.EapiIncompatible( overrideFn=isReloadNow,
                            additionalMsg="To reload the machine over the API, "
                            "please use 'reload fast-boot now' instead." )
def doFastReboot( mode, args ):
   now = "now" in args
   cmdInvoked = _asuData.AsuMode.fastboot
   if _asuData.asuHwStatus.asuModeSupported == _asuData.AsuMode.hitless:
      doRebootHitlessCommon( mode, None, None, now, cmdInvoked )
   elif _asuData.asuHwStatus.asuModeSupported == _asuData.AsuMode.fastboot:
      _asuReloadCliHelper.doFastBootCommon( mode, False, now, cmdInvoked )
   else:
      mode.addError( "Unsupported ASU mode" )

def fastRebootGuard( mode, token ):
   if _asuData.asuHwStatus.asuModeSupported == _asuData.AsuMode.fastboot:
      return None
   # If Asu fastboot is not supported then check for Asu hitless support
   return rebootHitlessGuard( mode, token )

def doRebootHitlessSim( mode ):
   '''
   Simulate a minimal reboot hitless, by running the AsuPStore save,
   and creating a commandline sim file to simulate hitless boot stages.
   A ptest can then issue a "service ProcMgr stop" "service ProcMgr start"
   to simulate the hitless startup of the agents.

   The "relaod hitless sim cleanup" can be used by the ptest to remove the sim
   files.
   '''
   AsuUtil.doSaveAsuState( mode.entityManager )

   fs = lambda: None
   setattr( fs, 'realFilename_', "simFile" )
   procOutput = generateProcOutput( fs, hitless=True )
   with open( "/var/tmp/asuCmdlineSim", "w" ) as f:
      f.write( " ".join( procOutput ) )

def doRebootHitlessSimCleanup( mode ):
   subprocess.call( [ "rm", "-f", "/var/tmp/asuCmdlineSim" ] )
   subprocess.call( [ "rm", "-f", "/mnt/flash/asu-persistent-store" ] )
   subprocess.call( [ "rm", "-f", "/mnt/flash/asu-persistent-store.json" ] )

# Silly pylint, don't complain about space after = in decorators:
# pylint: disable-msg=C0322
@BasicCliUtil.EapiIncompatible( overrideFn=isReloadNow,
                            additionalMsg="To reload the machine over the API, "
                            "please use 'reload fast-boot now' instead." )
def doRebootHitless( mode, args ):
   simOpt = 'sim' in args
   cleanOpt = 'cleanup' in args
   now = "now" in args
   doRebootHitlessCommon( mode, simOpt, cleanOpt, now, _asuData.AsuMode.hitless )

def doRebootHitlessCommon( mode, simOpt, cleanOpt, now, cmdInvoked ):
   # Set reboot mode to hitless, regardless of cmd invoked
   # hitless is treated as ASU2 in product code, see go/asuprimer for more info
   _asuData.asuCliConfig.asuRebootMode = _asuData.AsuMode.hitless

   if simOpt:
      if cleanOpt:
         doRebootHitlessSimCleanup( mode )
      else:
         doRebootHitlessSim( mode )
   else:
      _asuReloadCliHelper.doFastBootCommon( mode, True, now, cmdInvoked )

#--------------------------------------------------------------------------------
# reload fast-boot [ now ]
#--------------------------------------------------------------------------------
class DoFastRebootCmd( CliCommand.CliCommandClass ):
   syntax = 'reload fast-boot [ now ]'
   data = {
      'reload' : reloadKwNode,
      'fast-boot' : CliCommand.guardedKeyword( "fast-boot",
                                               helpdesc="Perform hitless "
                                                        "reload with image",
                                               guard=fastRebootGuard ),
      'now' : "Perform action immediately without prompting"
   }
   hidden = productAttributes().bootAttributes.asuHitlessHidden
   syncAcct = True
   handler = doFastReboot

BasicCli.EnableMode.addCommandClass( DoFastRebootCmd )

#--------------------------------------------------------------------------------
# reload hitless [ sim ] [ cleanup ] [ now ]
#--------------------------------------------------------------------------------
class DoRebootHitlessCmd( CliCommand.CliCommandClass ):
   syntax = 'reload hitless [ sim ] [ cleanup ] [ now ]'
   data = {
      'reload' : reloadKwNode,
      'hitless' : CliCommand.Node( CliMatcher.KeywordMatcher( "hitless",
                                                 helpdesc="Perform hitless "
                                                          "reload with image" ),
                                   guard=rebootHitlessGuard,
                                   hidden=True ),
      'sim' : CliCommand.Node( CliMatcher.KeywordMatcher( "sim",
                                             helpdesc="Perform hitless "
                                                      "reload simulation" ),
                               hidden=True ),
      'cleanup' : CliCommand.Node( CliMatcher.KeywordMatcher( "cleanup",
                                             helpdesc="Remove hitless "
                                                      "reload simulation files" ),
                                   hidden=True ),
      'now' : "Perform action immediately without prompting"
   }
   syncAcct = True
   handler = doRebootHitless

BasicCli.EnableMode.addCommandClass( DoRebootHitlessCmd )
