# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library SecureBoot

import Tac
import CliParser
import LazyMount, Cell
import TacSigint
import BasicCliUtil
import Tracing
import re, os, sys, time
import Url, subprocess, SimpleConfigFile
import FileUrl
import AsuUtil
import imp
import EosVersion
import PersistentRestartLogUtils
import FpgaUtil
import datetime
import EventHistoryUtil
import HwEpochPolicy
import pickle
import ReloadPolicy
import AsuPatchBase

from CliPlugin.IntfCli import Intf
from CliPlugin.ReloadConfigSaveCli import doConfigSaveBeforeReload
from CliPlugin.AsuPStoreModel import ReloadHitlessDisruption
from CliPlugin.AsuPStoreModel import ReloadFastbootDisruption
from StageMgr import defaultStageInstance
from ProductAttributes import productAttributes
from CliPlugin.ReloadCli import answerMatches
from CEosHelper import isCeos

__defaultTraceHandle__ = Tracing.Handle( "AsuReloadCli" )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

class AsuReloadCliGlobals( object ):
   def __init__( self ):
      self.abootSbStatus = None
      self.asuHwStatus = None
      self.asuCliConfig = None
      self.asuShutDownEnabledConfig = None
      self.asuShutDownStatus = None
      self.fatalError = None
      self.asuDmaCancelStatus = None
      self.asuDebugConfig = None
      self.eventHistoryPersistentDir = None
      self.dotBootReloadSwi = "/mnt/flash/.boot-reload.swi"
      self.AsuMode = Tac.Type( "Asu::AsuMode" )
      # setup AsuPatch persistent message support
      self.ehAsuPatchDir = None
      self.ehAsuPatchWriter = None

_asuData = AsuReloadCliGlobals()
_extractPath = '/tmp/'
_asuPatchSwiData = [ 'AsuPatchDb.py', 'AsuPatchBase.py', 'AsuPatchPkgs.tar.gz' ]
_asuPatchFsData = [ 'AsuPatchPkgs', 'AsuPatchDb.pyc', 'AsuPatchBase.pyc' ]

def cleanupAsuPatchData():
   cmd = [ 'rm', '-rf' ] + [ _extractPath + name for name in _asuPatchSwiData ]
   if not _asuData.asuDebugConfig.testAutoPatch:
      cmd = cmd + [ _extractPath + name for name in _asuPatchFsData ]
   Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
   TacSigint.check()

def extractAsuPatchFromSwi( swiFilename ):
   def doCopy( fromFile, toFile ):
      cpCmd = 'cp %s %s' % ( fromFile, toFile )
      cmd = os.environ.get( 'TEMP_IMAGE_CMD', cpCmd )
      Tac.run( cmd.split( ' ' ), asRoot=True )

   cmd = os.environ.get( 'UNZIP_SQUASHFS_CMD' )
   if cmd == None:
      cmd = [ 'unzip', '-o', swiFilename ] + _asuPatchSwiData + [ '-d',
              _extractPath ]
   else:
      cmd = cmd.split( ' ' )
   Tac.run( cmd, stdout=Tac.DISCARD, stderr=Tac.CAPTURE, asRoot=True )
   TacSigint.check()

   # move the AsuPatchBase to usr/lib/python
   _pythonPath = '/usr/lib/python2.7/site-packages/'
   doCopy( _extractPath + 'AsuPatchBase.py', _pythonPath + 'AsuPatchBase.py' )

   # extract the tar file
   tarFile = _extractPath + 'AsuPatchPkgs.tar.gz'
   untarCmd = 'tar -xzf %s -C %s' % ( tarFile, _extractPath )
   Tac.run( untarCmd.split( ' ' ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
            asRoot=True )
   TacSigint.check()

class ReturnCode:
   SUCCESS = 0
   FAILURE = 1

class CleanupStages:
   NoCleanup = 0
   FpgaCleanup = 1
   FailedCmdCleanup = 2

class AsuReloadContext( object ):
   def __init__( self, mode, now, url, fs, hitless, asuReloadCliGlobals ):
      self.mode = mode
      self.now = now
      self.url = url
      self.fs = fs
      self.hitless = hitless
      self.asuReloadCliGlobals = asuReloadCliGlobals

def logFuncAsuPatch( message ):
   if _asuData.ehAsuPatchWriter:
      _asuData.ehAsuPatchWriter.eventIs( message )

def asuReloadCliCheckSwi():
   _asuReloadCliHelper.logAsuPatch( 'Calling AsuReloadCli:checkSwi()' )
   return _asuReloadCliHelper.checkSwi()

class AsuReloadCliHelper( AsuPatchBase.AsuPatchBase ):
   def __init__( self, initName ):
      self.qualifiedFpgas = []
      self.skipFpgaUpgrade = False
      self.doDelayedSwiCopyBack = False
      self.url = None
      self.fs = lambda:None
      self.mode = None
      self.hitless = True
      self.now = False
      self.cmdInvoked = _asuData.AsuMode.fastboot
      self.cleanupStage = CleanupStages.NoCleanup
      AsuPatchBase.AsuPatchBase.__init__( self, initName )

   def cleanup( self ):
      if self.cleanupStage == CleanupStages.FpgaCleanup:
         self.fpgaRecovery()
      if self.cleanupStage != CleanupStages.NoCleanup:
         doFailedCmdCleanup()

   def logAsuPatch( self, message, lvl=None ):
      if lvl:
         if lvl == 'Msg':
            self.mode.addMessage( message )
         elif lvl == 'Warn':
            self.mode.addWarning( message )
         elif lvl == 'Err':
            self.mode.addError( message )
      self.log( message )
      logFuncAsuPatch( message )

   # Install the required patches
   def doPatch( self ):
      self.cleanupStage = CleanupStages.FailedCmdCleanup
      self.cleanup()

      try:
         # pylint: disable=maybe-no-member
         extractAsuPatchFromSwi( self.fs.realFilename_ )
         currVersion = EosVersion.VersionInfo(
               sysdbRoot=self.mode.entityManager.root() )
         if _asuData.ehAsuPatchDir and not _asuData.ehAsuPatchWriter:
            ehAsuPatchWConfig = Tac.newInstance( 'EventHistory::WriterConfig' )
            ehAsuPatchWConfig.size = 50
            _asuData.ehAsuPatchWriter = _asuData.ehAsuPatchDir.newWriter(
                  'AsuPatch-%d' % Cell.cellId(), ehAsuPatchWConfig )
         AsuPatchDb = imp.load_source( 'AsuPatchDb', '/tmp/AsuPatchDb.py' )
         # pylint: disable=protected-access
         if ( not hasattr( AsuPatchDb, '_asuPatchData' ) or
              AsuPatchDb._asuPatchData.version < 2 ):
            self.logAsuPatch( 'Destination AsuPatch version is less than 2.' )
            # Going to a version 1 means we need to explicitly run checkSwi()
            rc = self.checkSwi()
            if rc != ReturnCode.SUCCESS:
               return rc
         fromVersion = 'test' if _asuData.asuDebugConfig.testAutoPatch else \
               currVersion.internalVersion()
         self.logAsuPatch( 'Running AsuPatchDb:doPatch( version=%s, model=%s )' %
                           ( fromVersion, getPlatformModel() ), 'Msg' )
         asuReloadContext = AsuReloadContext( mode=self.mode, now=self.now,
                                url=self.url, fs=self.fs, hitless=self.hitless,
                                asuReloadCliGlobals=_asuData )
         asuPatchContext = AsuPatchBase.AsuPatchData()
         asuPatchContext.logFunc = logFuncAsuPatch
         asuPatchContext.checkSwi = asuReloadCliCheckSwi
         err = AsuPatchDb.doPatch( version=fromVersion,
                                   model=getPlatformModel(),
                                   asuReloadData=asuReloadContext,
                                   asuPatchData=asuPatchContext )
         if err != 0:
            self.logAsuPatch( 'AsuPatch failed error=%s' % err, 'Err' )
            return ReturnCode.FAILURE
      except Tac.SystemCommandError as e:
         self.logAsuPatch( 'AsuPatch failed swi file missing data %s' %
                          e.message, 'Err' )
         # To allow upgrade to a release that does not support AsuPatch
         return self.checkSwi()
      except KeyboardInterrupt:
         self.logAsuPatch( 'AsuPatch failed. Command aborted by user.', 'Warn' )
         return ReturnCode.FAILURE
      except SyntaxError:
         self.logAsuPatch( 'AsuPatch failed. Corrupted file(s) in swi.', 'Warn' )
         return ReturnCode.FAILURE
      except IOError:
         self.logAsuPatch( 'AsuPatch failed. Missing file(s) in swi.', 'Warn' )
         return ReturnCode.FAILURE

      self.cleanup()
      self.cleanupStage = CleanupStages.NoCleanup
      self.logAsuPatch( 'Done AsuPatchDb:doPatch' )
      return ReturnCode.SUCCESS

   def evalDelayedSwiCopyBack( self, url ):
      doDelayedSwiCopyBack = False
      if "FASTBOOT_SIM" in os.environ:
         if 'DELAYED_COPY_BACK' in os.environ:
            doDelayedSwiCopyBack = True
      else:
         minFlashPartitionSize = 2 * 1024 * 1024 * 1024  # 2GB, aka Cloverdale
         flashPartitionSize, _ = url.fs.stat()
         doDelayedSwiCopyBack = ( flashPartitionSize <= minFlashPartitionSize )
      return doDelayedSwiCopyBack

   def initArgs( self, *args, **kwargs ):
      if 'asuReloadData' not in kwargs:
         self.logAsuPatch( 'AsuPatch failed missing args %s' % kwargs, 'Warn' )
         return ReturnCode.FAILURE
      for argName, argVal in kwargs.iteritems():
         if argName == 'asuReloadData':
            asuReloadContext = argVal
            self.mode = asuReloadContext.mode
            self.now = asuReloadContext.now
            self.url = asuReloadContext.url
            self.fs = asuReloadContext.fs
            self.hitless = asuReloadContext.hitless
            self.doDelayedSwiCopyBack = self.evalDelayedSwiCopyBack( self.url )
            global _asuData
            _asuData = asuReloadContext.asuReloadCliGlobals
      return ReturnCode.SUCCESS

   # We only check that none of the args were modified after initArgs (as other
   # stages may not see the change). If this is desired we can store the data in
   # persistent store and recover it in initArgs.
   def saveArgs( self, *args, **kwargs ):
      assert 'asuReloadData' in kwargs
      for argName, argVal in kwargs.iteritems():
         if argName == 'asuReloadData':
            asuReloadContext = argVal
            assert self.mode == asuReloadContext.mode
            assert self.now == asuReloadContext.now
            assert self.url == asuReloadContext.url
            assert self.fs == asuReloadContext.fs
            assert self.hitless == asuReloadContext.hitless

   # We open the new swi and do any validation from the swi
   def checkSwi( self ):
      blockReload = doAsuReloadPolicyChecks( self.mode, self.fs.realFilename_,
                                             self.hitless )
      if blockReload:
         self.logAsuPatch( 'ReloadPolicyChecks failed', 'Err' )
         return ReturnCode.FAILURE

      if "FASTBOOT_SIM" in os.environ:
         self.logAsuPatch( 'Delayed Copy Back = %s' % self.doDelayedSwiCopyBack,
                           'Msg' )
      else:
         if not self.doDelayedSwiCopyBack:
            spaceOk = doSpaceCheck( self.url, self.fs, self.mode )
            if not spaceOk:
               self.logAsuPatch( 'Space check failed', 'Err' )
               return ReturnCode.FAILURE
         allowed, errorStr = HwEpochPolicy.checkEOSImageCompatible(
               self.fs.realFilename_ )
         if not allowed:
            self.logAsuPatch( errorStr, 'Err' )
            return ReturnCode.FAILURE

      return ReturnCode.SUCCESS

   # Get the boot image FS object
   def doGetBootImageFs( self ):
      '''Get the boot image FS. Return True if the boot image exists, otherwise
      return False and add error message.
      '''
      validFs = True
      if "FASTBOOT_SIM" in os.environ:
         setattr( self.fs, 'realFilename_', os.environ[ "FASTBOOT_SIM" ] )
         self.doDelayedSwiCopyBack = self.evalDelayedSwiCopyBack( self.url )
         return validFs

      self.url = FileUrl.localBootConfig( self.mode.entityManager,
                                          self.mode.session_.disableAaa_ )
      bootConfig = SimpleConfigFile.SimpleConfigFileDict( self.url.realFilename_,
            createIfMissing=True, autoSync = True )
      self.fs = Url.parseUrl( bootConfig[ 'SWI' ],
                  Url.Context( *Url.urlArgsFromMode( self.mode ) ),
                  lambda fs: fs.fsType == 'flash' )

      if not self.fs.exists():
         self.mode.addError( "Boot image %s does not exist."
                             % self.fs.realFilename_ )
         validFs = False
      else:
         self.doDelayedSwiCopyBack = self.evalDelayedSwiCopyBack( self.url )
      return validFs

   # Note: Any code within this function would NOT allow the AsuPatch to extend
   # it in future. Consider adding your code within checkSwi() instead.
   def doFastBootCommon( self, mode, hitless, now, cmdInvoked ):
      self.mode = mode
      self.hitless = hitless
      self.now = now
      self.cmdInvoked = cmdInvoked

      Tac.run( [ "rm", "-f", "/mnt/flash/asu-persistent-store.json" ], asRoot=True )

      if not self.doGetBootImageFs():
         return

      if "FASTBOOT_SIM" in os.environ:
         ret = self.checkSwi()
      else:
         # We call checkSwi from doPatch to validate the swi
         ret = self.doPatch()

      if ( ret == ReturnCode.SUCCESS and self.check() == ReturnCode.SUCCESS ):
         self.reboot()
      self.cleanup()

   def check( self ):
      # check if config is ok with 'reload hitless'
      if self.hitless:
         ( warningList, blockingList ) = \
               AsuUtil.doCheckHitlessReloadSupported( self.mode.entityManager )
         if self.cmdInvoked == _asuData.AsuMode.fastboot:
            ReloadDisruption = ReloadFastbootDisruption
         else:
            ReloadDisruption = ReloadHitlessDisruption

         ReloadDisruption( blockingList=blockingList, 
                           warningList=warningList ).render()
         sys.stdout.flush()
         if len( blockingList ):
            log( "features blocking asu boot" )
            return ReturnCode.FAILURE

      # Check if there is any unsaved config
      if not doConfigSaveBeforeReload( self.mode, power=False, now=self.now,
            noMeansFailure=True ):
         self.mode.addError( "Cannot reload without saving the configuration. "
                        "Aborting the command." )
         return ReturnCode.FAILURE

      self.cleanupStage = CleanupStages.FailedCmdCleanup
      try:
         # Extract kernel files from the SWI image
         doExtractBoot0FromSwi( self.fs.realFilename_ )
         log( "Kernel Files %s extracted from SWI" % self.fs.realFilename_ )

         # Save prefdl data for use on way back up
         prefdl = savePrefdlData()

         # Extract FPGA image here to avoid the .boot-reboot.swi and extracted
         # file system to be present at same time
         self.skipFpgaUpgrade = doCheckSkipFpgaUpgrade( self.mode )
         if not self.skipFpgaUpgrade:
            self.qualifiedFpgas = doPrepareFpgaUpgrade( self.mode, prefdl, self.fs )
         # Generate the commandline to be passed to the kernel
         procOutput = generateProcOutput( self.fs, hitless=self.hitless,
               doDelayedSwiCopyBack=self.doDelayedSwiCopyBack )
         log( "ProcOutput passed to Kernel %s" % procOutput )

         setupBootScript( self.fs, procOutput )

         TacSigint.check()

         doMaybeCopyDotBootSWI( self.fs, self.doDelayedSwiCopyBack )

         TacSigint.check()
      except ( Tac.SystemCommandError, FpgaUtil.UpdateFailed,
               FpgaUtil.UnprogrammedFpga ) as e:
         self.mode.addError( e.message )
         log( "FPGA upgrade failed" )
         return ReturnCode.FAILURE
      except KeyboardInterrupt:
         self.mode.addWarning( "Command aborted by user" )
         return ReturnCode.FAILURE
      except IOError as e:
         print 'unexpected IO error: ' + str( e )
         return ReturnCode.FAILURE

      if self.hitless:
         # issue a warning if the recommendend amount of free disk space is not
         # available to persist debug info in case of a fatal error. we do this check
         # after completing the file processing above, as these steps may change the
         # available disk space on persistent store.
         doFatalErrorSpaceCheck( self.mode )

      # Get confirmation from user
      if not self.now:
         answer = BasicCliUtil.getSingleByte( "Proceed with reload? [confirm]" )
         print answer
         sys.stdout.flush()
         if answer.lower() not in 'y\r\n':
            return ReturnCode.FAILURE

      self.mode.addMessage( 'Proceeding with reload' )
      sys.stdout.flush()

      # Reload Check passed
      self.cleanupStage = CleanupStages.NoCleanup
      return ReturnCode.SUCCESS

   def fpgaRecovery( self ):
      # Restore original FPGA image files
      for fpga in self.qualifiedFpgas:
         self.mode.addMessage( "Recovering image in flash for %s FPGA"
                               % fpga.name() )
         if os.path.isfile( fpga.imageFilePath() + '.old' ):
            doMove( fpga.imageFilePath() + '.old', fpga.imageFilePath() )
         if os.path.isfile( fpga.spiImageFilePath() + '.old' ):
            doMove( fpga.spiImageFilePath() + '.old', fpga.spiImageFilePath() )
         try:
            fpga.spiProgram()
         except FpgaUtil.UpdateFailed:
            self.mode.addError( "Recovering for %s FPGA failed" % fpga.name() )
         else:
            self.mode.addMessage( "Recovering for %s FPGA succeeded"
                                  % fpga.name() )

   def reboot( self ):
      self.cleanupStage = CleanupStages.FpgaCleanup
      # Mount the EventHistory path now so that there is no delay while saving it
      # after Dma is shutdown
      # Since C++ expects a fully mounted directory, mounts must be forced.
      _asuData.eventHistoryPersistentDir.force()

      # Programming the FPGA flash for upgrade
      try:
         if not self.skipFpgaUpgrade:
            if len( self.qualifiedFpgas ) > 0:
               self.mode.addMessage( 'Upgrading FPGAs' )
               programFpgaFlash( self.mode, self.qualifiedFpgas )
            else:
               self.mode.addMessage( 'No qualified FPGAs to upgrade' )
      except ( Tac.SystemCommandError, FpgaUtil.UnprogrammedFpga,
               FpgaUtil.UpdateFailed, KeyboardInterrupt ) as e:
         if e is KeyboardInterrupt:
            self.mode.addWarning( "Command aborted by user" )
         else:
            self.mode.addError( 'FPGA upgrade failed: ' + e.message )
         return ReturnCode.FAILURE

      procMgrCmd = os.environ.get( "PROCMGR_CMD", "service ProcMgr stoppm" )
      Tac.run( procMgrCmd.split( " " ), asRoot=True )

      shutdownWatchdog( self.mode )

      # Do preprocessing work for ASU
      # E.g. send protocol packets for protocols that support ASU.
      sys.stdout.flush()
      if not "FASTBOOT_SIM" in os.environ:
         if not doASUPreProcessing( self.mode, self.hitless ):
            return ReturnCode.FAILURE

      self.mode.addMessage( 'Shutting down packet drivers' )
      killForwardingAgents( self.mode, self.hitless )

      if not "FASTBOOT_SIM" in os.environ:
         dmaCancelSuccess = cancelDma()
         if not dmaCancelSuccess:
            fatalErrorResetMode = Tac.Type( "Stage::FatalError::ResetMode" )
            self.mode.addError(
               'DMA did not shut down properly. Forcing full reboot.' )
            _asuData.fatalError.rebootReason = 'DMA cancellation failure'
            _asuData.fatalError.requestReboot = fatalErrorResetMode.resetLocal
            return ReturnCode.FAILURE

         if isStrata(): #BUG240574
            log( "Bringing fab down" )
            Tac.run( [ "fab", "devstate", "1", "down" ], asRoot=True )

            log( "Bringing fifo down" )
            Tac.run( [ "fifo", "l2mod", "devstate", "1", "down" ], asRoot=True )
         elif isSand():
            # These three agents can be using sand-dma, so kill them before
            # removing sand-dma. Mpls agent runs only when Mpls routing is enabled.
            cmd = "killall -w -q Sand Acl Mpls"
            Tac.run( cmd.split( " " ), asRoot=True, ignoreReturnCode=True )
            Tac.run( [ "rmmod", "sand-dma" ], asRoot=True )

      EventHistoryUtil.doSaveEventHistory( _asuData.eventHistoryPersistentDir )

      Tac.waitFor( lambda:( not _asuData.asuDebugConfig.holdHitlessShutdown ),
                   description="debug hold shutdown release",
                   maxDelay=0.1, sleep=True, timeout=180.0 )

      self.mode.addMessage( 'reloading %s' % self.fs.realFilename_ )
      sys.stdout.flush()

      if not "FASTBOOT_SIM" in os.environ:
         # Shut all the mgmt interface
         shutMgmtIntfs( self.mode )

      # Move the EOS swi to .boot-reload.swi. This delayed copy is done for
      # cloverdale duts. In case a fatalError is triggered before this call, we don't
      # need to take care of moving the .boot-reload.swi back to EOS.swi
      doMaybeMoveDotBootSWI( self.fs, self.doDelayedSwiCopyBack )
      # Don't put any code inbetween these two lines.
      # Move should happen immediately before kexec
      runBootScriptCmd = os.environ.get( "BOOTSCRIPT_CMD",
                                         "bash -c /tmp/boot-script" )
      Tac.run( runBootScriptCmd.split( " " ), stdin=sys.stdin, asRoot=True )

      self.cleanupStage = CleanupStages.NoCleanup
      return ReturnCode.SUCCESS

_asuReloadCliHelper = AsuReloadCliHelper( "AsuReloadCli" )

def execute( stageVal, *args, **kwargs ):
   return _asuReloadCliHelper.execute( stageVal, *args, **kwargs )

# A simple script that sets up environment to run boot0

#  For ASU+/ASU2, most of kernel args are inherited from previous
#  boot/aboot ( from /proc/cmdline ), functions used by boot0 in aboot
#  context are defined as dummy. kernel args per EOS image are setup by
#  boot0 which is invoked by Aboot or ASU boot.
# 
bootScriptTemplate = """
#!/bin/sh
parseconfig() {
   return
}

ifget() {
   return
}

writebootconfig() {
   return
}

export arch=i386
export swipath=%s

. /tmp/boot0
"""

def log( msg ):
   date = datetime.datetime.now()
   print date, msg

def printWatchdogValue( mode ):
   watchDogValueCmd = os.environ.get( "WATCHDOGVALUE_CMD", 
      "pciread32 4:0 0 0x120" )
   nowStr = time.strftime("%d %b %H:%M:%S")
   output = Tac.run( watchDogValueCmd.split( " " ), asRoot=True,
      stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
   mode.addMessage( '%s : Watchdog output %s' % ( nowStr, output ) )

# format the give number in appropriate unit
def sizeFmt(num):
   for x in [ 'bytes', 'KB', 'MB', 'GB', 'TB' ]:
      if num < 1024.0:
         return "%3.1f %s" % (num, x)
      num /= 1024.0

# Return a list of kernel interfaces names for all management ports on the dut
def getMgmtIntfList( mode ):
   mgmtIntf = []
   # List all interfaces
   intfs = Intf.getAll( mode )

   # Go over all interfaces and find matching ones
   for intf in intfs:
      m = re.match( r"Management(\d+)(?:/(\d+))?$", intf.name )
      if m:
         # Convert it to kernel interface name
         kernelIntfName = "ma%s" % m.group( 1 )
         if m.group( 2 ) is not None:
            kernelIntfName += "_%s" % m.group( 2 )
         mgmtIntf.append( kernelIntfName )
   return mgmtIntf

# Shut all mgmt interfaces
def shutMgmtIntfs( mode ):
   if not "FASTBOOT_SIM" in os.environ:
      print "Shutting down management interface(s)"
      sys.stdout.flush()
      mgmtIntfs = getMgmtIntfList( mode )
      for mgmtIntf in mgmtIntfs:
         intfShutCmd = os.environ.get( "INTF_SHUT_CMD",
               "ip link set %s down" % mgmtIntf )
         Tac.run( intfShutCmd.split( " " ), asRoot=True,
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )

# Validate available storage space
def doSpaceCheck( url, fs, mode ):
   blockSize = 4096
   def _blocks( sz ):
      return (sz / blockSize)

   _, flashFree = url.fs.stat()
   freeBlocks = _blocks( flashFree )
   urlBlocks = _blocks( fs.size() ) + 1 # round up to be conservative
   blocksNeeded = urlBlocks
   blocksNeeded *= 1.05 # Add 5% slop factor to be conservative

   # This needs to be more elaborate to make sure we have space to copy
   # the .boot-image.swi to EOS.swi. Basically the space after removing
   # the current free space + current .boot-image.swi > fs.realFilename_
   if freeBlocks < blocksNeeded:
      mode.addError( "Insufficient free space on the internal flash disk to copy "
                     "the boot image." )
      mode.addError( "Free up at least %s and try again." % (
                                         sizeFmt( blocksNeeded * blockSize ) ) )
      return False
   return True

# Issue warning if there the recommended amount of free space is not available to
# persist debug info in case of a fatal error during 'reload hitless'
def doFatalErrorSpaceCheck( mode ):
   feConstants = Tac.Type( "Stage::FatalErrorConstants" )

   # determine the persistent storage device that will be used for persisting logs
   # and cores.
   def _getFatalErrorFilesystem( mode ):
      # check if ssd filesystem is present.
      fs = feConstants.primaryPersistentFs
      if not os.path.exists( fs ):
         # assume no SSDs present; debug info will be persisted at /mnt/flash
         fs = feConstants.secondaryPersistentFs
         if not os.path.exists( fs ):
            # Failing to do the space check should not prevent someone from
            # performing hitless reload. Just print a warning. 
            mode.addWarning( "No filesystem is available for storing debug "
                  "information in the event a problem occurs during system reload." )
            return None
      return fs 

   fs = _getFatalErrorFilesystem( mode )
   if not fs:
      return

   fsStat = os.statvfs( fs )
   freeBlocks = fsStat.f_bfree
   fsBlkSize = fsStat.f_bsize
   blocksNeeded = feConstants.persistedInfoBytesNeeded / fsBlkSize
   if freeBlocks < blocksNeeded:
      mode.addWarning( "It is recommended that at least "
            + sizeFmt( blocksNeeded * fsBlkSize ) + " of free space is available on "
            + fs + " to store debug information in the event that a problem arises"
            + " during system reload. Free up at least "
            + sizeFmt( ( blocksNeeded - freeBlocks ) * fsBlkSize )
            + " before continuing." )

def doExtractBoot0FromSwi( swiFilename ):
   unzipCmd = "unzip -o %s %s %s %s -d /tmp" % ( swiFilename, 
          "linux-i386", "initrd-i386", "boot0" )
   cmd = os.environ.get( "UNZIP_SQUASHFS_CMD", unzipCmd )
   Tac.run( cmd.split( " " ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
         asRoot=True )
   TacSigint.check()

# Generate the command line we will pass to the kexec
def generateProcOutput( fs, hitless=False, doDelayedSwiCopyBack=False ):
   regexMatch = [ re.compile( r"SWI=[\S]* " ),
                  re.compile( "kexec_jump_back_entry=0x[0-9A-Fa-f]*"),
                  re.compile( "arista.asu_reboot"),
                  re.compile( "arista.asu_hitless"),
                  re.compile( r"arista.doDelayedSwiCopyBack"),
                  re.compile( r"NETIP=[\S]*"),
                  re.compile( r"NETGW=[\S]*"),
                  re.compile( r"dmamem=[\w-]+" ),
                  re.compile( r"SKIP_CMP=[\S]*" ) ]

   proc = subprocess.Popen( [ 'cat', '/proc/cmdline' ],
         stdout=subprocess.PIPE )
   procOutput = proc.communicate()[ 0 ]
   procOutput = procOutput.strip( '\n' )
   for regex in regexMatch:
      procOutput = re.sub( regex, '', procOutput )

   # remove duplicates, later setting wins and preserve non-duplicates
   # order
   args = procOutput.split()
   procOutDict = {}
   procOutList = []
   for arg in args:
      n, v = arg, arg
      if '=' in arg:
         n = arg.split( '=' )[ 0 ]
      if n in procOutDict.keys():
         # use later arg if v is different, else skip adding to list
         if v != procOutDict[ n ]:
            procOutList.remove( procOutDict[ n ] )
            procOutList.append( v )
      else:
         procOutList.append( v )
      procOutDict[ n ] = v

   procOutList.append( "SWI=%s" % fs.realFilename_ )
   if hitless:
      procOutList.append( "arista.asu_hitless" )
   else:
      procOutList.append( "arista.asu_reboot" )

   # Is the partition less than 2GB in size? We need to do some work in
   # boot1 script to accomodate that
   if doDelayedSwiCopyBack:
      procOutList.append( "arista.doDelayedSwiCopyBack" )

   return procOutList

# Save prefdl data to be used on the way back up
def savePrefdlData():

   def getPreFdl():
      path = '/etc/prefdl'
      if os.path.isfile( path ):
         with open( path, 'r' ) as f:
            return f.read()
      else:
         return Tac.run( [ "/bin/sh", "-c", "/usr/bin/genprefdl" ],
                         stdout=Tac.CAPTURE,
                         stderr=Tac.DISCARD,
                         asRoot=True )

   prefdlInEnvVariables =  "PREFDL" in os.environ
   prefdl = os.environ[ "PREFDL" ] if prefdlInEnvVariables else getPreFdl()

   # Save identifyCell and prefdl output to use on boot up
   # and save time (only if not a test)
   if not prefdlInEnvVariables:

      asuConstants = Tac.Type( "Asu::AsuConstants" )
      Tac.run( [ "mkdir", "-p", asuConstants.asuReloadCacheDir ] )
      # Need to run as root because of ham file mapping permission issue
      Tac.run( [ "/usr/bin/identifyCell", "-d", 
         asuConstants.asuReloadCacheDir ],
         stdout=Tac.DISCARD, stderr=Tac.DISCARD, asRoot=True )

      prefdlPath = "%s/prefdl.txt" % asuConstants.asuReloadCacheDir
      with open( prefdlPath, 'w+' ) as f:
         f.write( prefdl )
   return prefdl

def doAsuReloadPolicyChecks( mode, swiFile, hitless ):
   '''Run the ASU ReloadPolicy checks. If the checks produced any errors or warnings,
   then output those to the CLI. Returns a tuple of booleans indicating if the checks
   succeeded and if the feature-keys check was run.
   '''
   blockReload = False

   # Run the checks and determine if feature-keys check ran
   category = [ 'ASU' ] if hitless else [ 'ASU+' ]
   policyResult = ReloadPolicy.doCheck( swiFile, category=category, mode=mode,
                                        abootSbStatus=_asuData.abootSbStatus )
   ranFeatureKeysCheck = 'AsuFeatureKeysCheck' in policyResult.policySuccess

   # Output errors/warnings to CLI
   for errStr in policyResult.errors:
      mode.addError( errStr )
   for warnStr in policyResult.warnings:
      mode.addWarning( warnStr )

   # Errors block reload
   if policyResult.errors:
      blockReload = True
      if ( 'AsuDowngradeCheck' in policyResult.policySuccess and
           not policyResult.policySuccess[ 'AsuDowngradeCheck' ] ):
         mode.addError( "ASU hitless upgrade failed downgrade check." )
      if ( ranFeatureKeysCheck and
           not policyResult.policySuccess[ 'AsuFeatureKeysCheck' ] ):
         mode.addError( "ASU hitless upgrade failed feature-keys check." )

   return blockReload
         
# Create the .boot.swi file so that after reboot copy does not happen
def doMaybeCopyDotBootSWI( fs, doDelayedSwiCopyBack ):
   if not doDelayedSwiCopyBack:
      cpCmd = "cp %s %s" % ( fs.realFilename_, _asuData.dotBootReloadSwi )
      cmd = os.environ.get( "TEMP_IMAGE_CMD", cpCmd )
      Tac.run( cmd.split( " " ), asRoot=True )

def doMaybeMoveDotBootSWI( fs, doDelayedSwiCopyBack ):
   # Call this function for cd* duts just before Kexec.
   # This will prevent the system from booting with a missing EOS.swi
   # in case a fatalError is triggerd before kexec
   if doDelayedSwiCopyBack:
      # If the partition is small, we just rename the EOS.swi instead of copying here
      # We will remember this, on the way back up, once the old .boot-image.swi has
      # been removed, we will make a copy of the .boot-reload.swi to EOS.swi
      mvCmd = "mv %s %s" % ( fs.realFilename_, _asuData.dotBootReloadSwi )
      cmd = os.environ.get( "TEMP_IMAGE_CMD", mvCmd )
      Tac.run( cmd.split( " " ), asRoot=True )
   
def requestFatalError( mode, reason ):
   mode.addError( "%s. Forcing cold reboot" % reason.capitalize() )
   fatalErrorResetMode = Tac.Type( "Stage::FatalError::ResetMode" )
   _asuData.fatalError.rebootReason = reason
   _asuData.fatalError.requestReboot = fatalErrorResetMode.resetLocal

def debugTimeoutValue():
   # increase watchdog timeout to 30 min for hold shutdown test
   timeout = None
   if _asuData.asuDebugConfig.holdHitlessShutdown:
      timeout = 1800
   return timeout

# Do all preprocessing work ( support for lacp, arp, pim etc )
def doASUPreProcessing( mode, hitless ):
   # Save a reload cause file under /mnt/flash/debug
   if not "FASTBOOT_SIM" in os.environ:
      reloadCauseConstants = Tac.Value( "ReloadCause::ReloadCauseConstants" )
      if hitless:
         reloadCauseConstants.writeLocalReloadCause(
                         reloadCauseConstants.hitlessReloadDesc, "", True )
      else:
         reloadCauseConstants.writeLocalReloadCause(
                         reloadCauseConstants.fastRebootDesc, "", True )

   _asuData.asuShutDownEnabledConfig.enabled[ defaultStageInstance ] = True
   newTimeout = debugTimeoutValue() or 150
   try:
      instStatus = _asuData.asuShutDownStatus.instStatus
      Tac.waitFor( lambda: ( defaultStageInstance in instStatus ) and \
                      instStatus[ defaultStageInstance ].complete,
                   description="platform processing",
                   maxDelay=0.1, sleep=True, timeout=newTimeout )
   except Tac.Timeout:
      requestFatalError( mode, "platform processing timed out" )
      return False
   for agent, l in PersistentRestartLogUtils.getPersistentLogs().iteritems():
      mode.addWarning( 'Agent ' + agent + ' reported a warning: ' + l )
   return True

def getPlatformKillRe( mode ):
   agentsToKill = []
   for agent, valid in _asuData.asuHwStatus.forwardingAgent.iteritems():
      # Filter out any invalid or "null" agents.
      if valid and ( agent != "" ):
         try:
            if not os.environ.get( "PLATFORM_CMD" ):
               # Use signal 0 to check whether the agent is running
               cmd = "killall -q --signal 0 %s" % agent
               Tac.run( cmd.split(), asRoot=True )
            agentsToKill.append( agent )
         except Tac.SystemCommandError:
            # Agent not running
            mode.addMessage( "Skipping %s (not running)" % agent )

   return " ".join( agentsToKill )

def killForwardingAgents( mode, hitless ):
   if hitless:
      AsuUtil.doSaveAsuState( mode.entityManager )

   agentsToKill = getPlatformKillRe( mode )
   if agentsToKill:
      platformAgentCmd = os.environ.get( "PLATFORM_CMD",
                                         "killall -w -q %s" % agentsToKill )
      try:
         Tac.run( platformAgentCmd.split( " " ) , asRoot=True)
      except Tac.SystemCommandError:
         checkAgents = getPlatformKillRe( mode )
         # If there are no forwarding agents on recheck one or more shut themselves
         # down gracefully. Ignore the error
         for agent in checkAgents.split():
            mode.addWarning( "Unable to shutdown " + agent )

def getPlatformModel(): #BUG240574
   platform = _asuData.asuHwStatus.platform
   assert platform
   return platform

def isStrata(): #BUG240574
   return 'Strata' == getPlatformModel()

def isSand():
   return 'Sand' == getPlatformModel()

def shutdownWatchdog( mode ):
   try:
      watchdogCmd = os.environ.get( "WATCHDOGKILL_CMD", "watchdog -k" )
      Tac.run( watchdogCmd.split( " " ), asRoot=True )

      newTimeout = debugTimeoutValue() or 240
      watchdogTimeoutCmd = os.environ.get( "WATCHDOGTIMEOUT_CMD", 
            "watchdog -o %d" % newTimeout )
      Tac.run( watchdogTimeoutCmd.split( " " ), asRoot=True )

   except Tac.SystemCommandError as e:
      mode.addWarning(
         'unexpected error: ' + e.message  + ' proceeding with normal reload' )
      watchdogCmd = os.environ.get( "WATCHDOGKILL_CMD", "watchdog -k" )
      #Kill watchdog again in case
      Tac.run( watchdogCmd.split( " " ), asRoot=True )
      shutdownCmd = os.environ.get( "SHUTDOWN_CMD", "shutdown -r now" )
      Tac.run( shutdownCmd.split( " " ), asRoot=True )

def doFailedCmdCleanup():
   cleanupAsuPatchData()
   cmdDefault = "rm -rf %s" % _asuData.dotBootReloadSwi
   cmd = os.environ.get( "CLEANUP_TEMP_IMAGE", cmdDefault )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_KERNEL",
                         "rm -rf /tmp/initrd-i386 /tmp/linux-i386 /tmp/boot0" )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_HITLESS_MANIFEST", 
                         "rm -rf /tmp/asu-manifest /tmp/asu-manifestc" )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_HITLESS_POLICY", 
                         "rm -rf /tmp/asu-upgrade-policy /tmp/asu-upgrade-policyc" )
   Tac.run( cmd.split( " " ), asRoot=True )

   if os.path.exists( "/tmp/etc-bootconfig-created" ):
      bootconfigCleanup =  os.environ.get( "CLEANUP_ASUBOOTCONFIG",
                                           "rm -rf /etc/boot-config" )
      Tac.run( bootconfigCleanup.split( " " ), asRoot=True )
      Tac.run( "rm -rf /tmp/etc-bootconfig-created".split( " " ), asRoot=True )

   if os.path.exists( "/tmp/etc-cmdline-created" ):
      cmdlineCleanup = os.environ.get( "CLEANUP_ASUCMDLINE", "rm -rf /etc/cmdline" )
      Tac.run( cmdlineCleanup.split( " " ), asRoot=True )
      Tac.run( "rm -rf /tmp/etc-cmdline-created".split( " " ), asRoot=True )

   bootScriptCleanup = os.environ.get( "CLEANUP_BOOTSCRIPT",
                                       "rm -rf /tmp/boot-script" )
   Tac.run( bootScriptCleanup.split( " " ), asRoot=True )

   cmd = os.environ.get( "CLEANUP_FILE_SYSTEM",
                         "rm -rf /mnt/flash/rootfs-i386.sqsh" )
   Tac.run( cmd.split( " " ), asRoot=True )
   cmd = os.environ.get( "CLEANUP_FPGA_IMAGE", "rm -rf /tmp/squashfs-root" )
   Tac.run( cmd.split( " " ), asRoot=True )

def setupBootScript( fs, procOutList ):
   bootconfigFile = os.environ.get( "ASUBOOTCONFIG_FILE", "/etc/boot-config" )
   if not os.path.exists( bootconfigFile ):
      bootConfigCmd = "touch %s" % bootconfigFile
      Tac.run( bootConfigCmd.split( " " ), asRoot=True )
      Tac.run( "touch /tmp/etc-bootconfig-created".split( " " ), asRoot=True )

   cmdlineFile = os.environ.get( "ASUCMDLINE_FILE", "/etc/cmdline" )
   Tac.run( [ 'touch', cmdlineFile ], asRoot=True )
   Tac.run( [ 'chmod', '777', cmdlineFile ], asRoot=True )
   procOutStr = '\n'.join( procOutList ) + '\n'
   with open( cmdlineFile, "w" ) as f:
      f.write( procOutStr )
   Tac.run( [ 'chmod', '777', cmdlineFile ], asRoot=True )
   Tac.run( "touch /tmp/etc-cmdline-created".split( " " ), asRoot=True )

   bootscriptFile = os.environ.get( "BOOTSCRIPT_FILE", "/tmp/boot-script" )
   with open( bootscriptFile, "w" ) as f:
      f.write( bootScriptTemplate % _asuData.dotBootReloadSwi ) 
   cmd = "chmod 777 %s" % bootscriptFile
   Tac.run( cmd.split( " " ), asRoot=True )

def cancelDma():
   _asuData.asuCliConfig.dmaCancelInitiated = True
   newTimeout = debugTimeoutValue() or 60
   try:
      Tac.waitFor( lambda:_asuData.asuDmaCancelStatus.asuDmaCancelComplete,
                   description="dma quiesce",
                   maxDelay=0.1, sleep=True, timeout=newTimeout )
   except Tac.Timeout:
      return False

   return _asuData.asuDmaCancelStatus.asuDmaCancelSuccess

def doCheckSkipFpgaUpgrade( mode ):
   with open( '/proc/cmdline' ) as f:
      cmdlineStr = f.read()
   if "AristaDiagnosticsMode=NoFPGA" in cmdlineStr:
      mode.addMessage( 'Skipping fpga upgrade due to kernel '
                       'AristaDiagnosticsMode parameter.' )
      return True
   if os.path.exists( "/mnt/flash/skipFpgaUpgrade" ):
      mode.addMessage( 'Skipping fpga upgrade due to configuration in flash.' )
      return True
   return False

def doCheckFpgaVersionManifest( swiFilename, fpgaVersionManifestFile ):
   fpgaImageManifestFound = False
   # Try extracting the manifest from swi, ignore error on unzip (if any)
   unzipCmd = ( "unzip -q -o %s %s -d /tmp"
                % ( swiFilename, fpgaVersionManifestFile ) )
   cmd = os.environ.get( "UNZIP_FPGA_MANIFEST_CMD", unzipCmd )
   Tac.run( cmd.split( " " ), stdout=Tac.DISCARD, stderr=Tac.DISCARD,
            asRoot=True, ignoreReturnCode=True )
   if os.path.exists( "/tmp/" + fpgaVersionManifestFile ):
      fpgaImageManifestFound = True
   return fpgaImageManifestFound

def doPrepareFpgaUpgrade( mode, prefdl, fs ):
   flavor = FpgaUtil.translateSwiFlavor()
   fpgas = FpgaUtil.systemFpgas( prefdl, flavor ).fpgaList()
   qualifiedFpgas = [ fpga for fpga in fpgas
                      if fpga.isHitlessResetSupported() and fpga.spiEnabled() ]
   # Update the qualifedFpgas list based on the version found in the manifest file
   upgradeNeededFpgas = []
   if qualifiedFpgas:
      fpgaVersionManifestFile = "fpga-image-versions"
      fpgaImageManifestFound = doCheckFpgaVersionManifest( fs.realFilename_,
                                                           fpgaVersionManifestFile )
      # Perform a FPGA version fast check only when the manifest file is found
      # Otherwise, the list of qualifiedFpgas will not be updated and unsquashing
      # the file system will be needed
      if fpgaImageManifestFound:
         # Check against the manifest file rather than unsquashing the file system
         fpgaManifestVersions = {}
         with open ( "/tmp/" + fpgaVersionManifestFile, 'r' ) as f:
            fpgaManifestVersions = pickle.load( f )
         for fpga in qualifiedFpgas:
            fpgaName = fpga.name()
            if fpgaName in fpgaManifestVersions:
               currentFpgaVersion = fpga.hardwareVersion()
               newFpgaVersion = fpgaManifestVersions[ fpgaName ]
               mode.addMessage( 'Checking Fpga %s image version: '
                                'Current version is %s, New version is %s' %
                                ( fpgaName, currentFpgaVersion.major,
                                  newFpgaVersion.major ) )
               # Logic here matches needsUpgrade mathod
               if ( currentFpgaVersion.major != newFpgaVersion.major or
                    currentFpgaVersion.minor < newFpgaVersion.minor ):
                  upgradeNeededFpgas.append( fpga )
            else:
               upgradeNeededFpgas.append( fpga )
         # update the qualifiedFpgas list to be all Fpgas that need update
         qualifiedFpgas = upgradeNeededFpgas

   # Extract file system only when there are qualified FPGAs in switch
   # Important for cloverdale duts to reload hitless without upgrading FPGA
   # Peach FPGA in cloverdale should not be qualified
   if qualifiedFpgas:
      mode.addMessage( 'Extracting file system for upgrading FPGAs' )
      mode.addMessage( 'This process may take a few minutes' )
      doExtractFileSystemFromSwi( fs.realFilename_ )
      skipFpgaUpgrade = doUnsquashFpgaImageFiles( mode, qualifiedFpgas )
      if skipFpgaUpgrade:
         qualifiedFpgas = []
   cleanupCmd = os.environ.get( "CLEANUP_FILE_SYSTEM",
                                "rm -rf /mnt/flash/rootfs-i386.sqsh" )
   Tac.run( cleanupCmd.split( " " ), asRoot=True )
   return qualifiedFpgas

def doExtractFileSystemFromSwi( swiFilename ):
   unzipCmd = "unzip -o %s %s -d /mnt/flash" % ( swiFilename, "rootfs-i386.sqsh" )
   cmd = os.environ.get( "UNZIP_SQUASHFS_CMD", unzipCmd )
   try:
      Tac.run( cmd.split( " " ), stdout=Tac.DISCARD, stderr=Tac.CAPTURE,
               asRoot=True )
   except ( Tac.SystemCommandError, Tac.Timeout ) :
      raise FpgaUtil.UpdateFailed( "Can not extract file system from SWI image. "
                                   "Please check that the flash has enough free "
                                   "space." )
   TacSigint.check()

def doMove( fromPath, toPath ):
   mvCmd = "mv %s %s" % ( fromPath, toPath )
   cmd = os.environ.get( "TEMP_IMAGE_CMD", mvCmd )
   Tac.run( cmd.split( " " ), asRoot=True )

def programFpgaFlash( mode, fpgas ):
   # Assume all of the fpgas are programmed, and working currently.
   # Assume all FPGAs passed in are qualified and have images in /tmp
   for fpga in fpgas:
      # Need to copy because spiProgram use the path /usr/share/fpga/
      doCopyFpgaImageFiles( fpga )
      hwVersion = fpga.hardwareVersion()
      ( upgradeNeeded, _ ) = fpga.needsUpgrade( hwVersion )
      if upgradeNeeded:
         mode.addMessage( "Programming flash for %s FPGA" % fpga.name() )
         fpga.spiProgram()
      else:
         mode.addMessage( "No need to reprogram %s FPGA" % fpga.name() )
   TacSigint.check()

# Assume the number of qualified FPGAs is greater than 0
def doUnsquashFpgaImageFiles( mode, qualifiedFpgas ):
   filesToUnsquash = ""
   for fpga in qualifiedFpgas:
      filesToUnsquash += fpga.imageFilePath() + " "
      filesToUnsquash += fpga.spiImageFilePath() + " "
   unsqshCmd = "unsquashfs -f -d %s %s %s" % ( "/tmp/squashfs-root",
                                               "/mnt/flash/rootfs-i386.sqsh",
                                               filesToUnsquash )
   cmd = os.environ.get( "UNSQSH_CMD", unsqshCmd )
   Tac.run( cmd.split( " " ), asRoot=True )

   skipFpgaUpgrade = False
   for fpga in qualifiedFpgas:
      if not ( os.path.isfile( "/tmp/squashfs-root" + fpga.imageFilePath() ) and
               os.path.isfile( "/tmp/squashfs-root" + fpga.spiImageFilePath() ) ):
         mode.addWarning( "Can not extract %s FPGA image from SWI" % fpga.name() )
         prompt = "Are you downgrading to an older version of EOS? "
         answer = BasicCliUtil.getChoice( mode, prompt, [ 'yes', 'no' ], 'no' )
         if answerMatches( answer, "yes" ):
            prompt = ( "WARNING: Downgrading with reload hitless is not supported. "
                       "You may experience unexpected behavior until you perform a "
                       "cold reboot. Proceed anyway? " )
            confirm = BasicCliUtil.getChoice( mode, prompt, [ 'yes', 'no' ], 'no' )
         if answerMatches( answer, "yes" ) and answerMatches( confirm, "yes" ):
            mode.addMessage( "Skipping FPGA upgrade" )
            skipFpgaUpgrade = True
         else:
            raise FpgaUtil.UpdateFailed( "Please contact technical support" )
   return skipFpgaUpgrade

def doCopyFpgaImageFiles( fpga ):
   def doCopy( fromFile, toFile ):
      cpCmd = "cp %s %s" % ( fromFile, toFile )
      cmd = os.environ.get( "TEMP_IMAGE_CMD", cpCmd )
      Tac.run( cmd.split( " " ), asRoot=True )

   newImagePath = "/tmp/squashfs-root" + fpga.imageFilePath()
   currentImagePath = fpga.imageFilePath()
   doCopy( currentImagePath, currentImagePath + '.old' )
   doCopy( newImagePath, currentImagePath )
   newSpiImagePath = "/tmp/squashfs-root" + fpga.spiImageFilePath()
   currentSpiImagePath = fpga.spiImageFilePath()
   doCopy( currentSpiImagePath, currentSpiImagePath + '.old' )
   doCopy( newSpiImagePath, currentSpiImagePath )

def rebootHitlessGuard( mode, token ):
   # BUG240982 : Need to remove asuHitlessHidden from here
   if productAttributes().bootAttributes.asuHitlessHidden or \
      ( ( _asuData.AsuMode.hitless == _asuData.asuHwStatus.asuModeSupported ) and
        not isCeos() ):
      return None
   return CliParser.guardNotThisPlatform

def Plugin( entityManager ):
   abootSbStatusPath = Cell.path( "aboot/sb/status" )
   asuShutdownEnabledConfigPath = Cell.path( "stageEnable/shutdown" )
   asuShutdownStatusPath = Cell.path( "stage/shutdown/status" )
   _asuData.abootSbStatus = LazyMount.mount( entityManager, abootSbStatusPath,
                                             "Aboot::Secureboot::Status", "r" )
   _asuData.asuHwStatus = LazyMount.mount( entityManager, "asu/hardware/status",
                                           "Asu::AsuStatus", "r" )
   _asuData.asuCliConfig = LazyMount.mount( entityManager, "asu/cli/config",
                                            "Asu::CliConfig", "w" )
   _asuData.asuShutDownEnabledConfig = LazyMount.mount( entityManager,
                                               asuShutdownEnabledConfigPath,
                                               "Stage::EnabledConfig", "w" )
   _asuData.asuShutDownStatus = LazyMount.mount( entityManager,
                                                 asuShutdownStatusPath,
                                                 "Stage::Status", "r" )
   _asuData.fatalError = LazyMount.mount( entityManager,
                               Cell.path( "fatalError/config/request/Asu-Cli" ),
                               "Stage::FatalError", "fcw" )
   _asuData.asuDmaCancelStatus = LazyMount.mount( entityManager,
                                                  "asu/status/dmaCancel",
                                                  "Asu::DmaCancelStatus", "r" )
   _asuData.asuDebugConfig = LazyMount.mount( entityManager, "asu/debug/config",
                                              "Asu::DebugConfig", "r" )
   eventHistoryPersistentPath = Cell.path( "eventhistory/persistent" )
   _asuData.eventHistoryPersistentDir = LazyMount.mount( entityManager,
         eventHistoryPersistentPath, "Tac::Dir", "ri" )
   ehAsuPatchPath = Cell.path( "eventhistory/persistent/AsuPatch" )
   _asuData.ehAsuPatchDir = LazyMount.mount( entityManager, ehAsuPatchPath,
         "EventHistory::WriterDir", "wcf" )
