#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import DeferredModel
from CliModel import Str
from CliModel import Dict
class Neighbor( Model ):
   """Base class for IPv4 ARP and IPv6 ND neighbor models."""

   hwAddress = Str(
      help="Hardware (layer 2) address learnt for the corresponding network "
      "(layer 3) address" )
   age = Int(
      help="How many minutes ago the neighbor entry was learned",
      optional=True )
   interface = Str( help="Interface name the neighbor was learned on" )
   extra = Str( help="Extra metadata about this neighbor's discovery",
                optional=True )


class IpV4Neighbor( Neighbor ):
   """Describes information about one IPv4 neighbor learnt via ARP."""

   address = Str( help="IPv4 address or hostname" )


class IpV6Neighbor( Neighbor ):
   """Describes information about a single IPv6 neighbor learned via ND."""

   address = Str( help="IPv6 address or hostname" )
   state = Str( help="Neighbor state (e.g., REACHable)" )


class NeighborCollection( DeferredModel ):
   """Abstract information about local IPv4 or IPv6 neighbors."""

   totalEntries = Int( default=0,
      help="Total number of addresses in ARP table" )
   staticEntries = Int( default=0,
      help="Number of statically configured addresses in ARP table" )
   dynamicEntries = Int( default=0,
      help="Number of dynamically learnt addresses in ARP table" )
   notLearnedEntries = Int( default=0,
      help="Number of addresses in ARP table not due to learning" )


class IpV4Neighbors( NeighborCollection ):
   """A sequence of ARP discovered IPv4 neighbors."""

   ipV4Neighbors = List(
      valueType=IpV4Neighbor,
      help="IPv4 neighbors discovered by the Address Resolution Protocol" )


class IpV6Neighbors( NeighborCollection ):
   """A sequence of ND discovered IPv6 neighbors."""

   ipV6Neighbors = List(
      valueType=IpV6Neighbor,
      help="IPv6 neighbors discovered by neighbor discovery" )

class VrfIpV4Neighbors( DeferredModel ):
   """Lists all ARP discovered IPv4 neighbors within each VRF."""

   vrfs = Dict( keyType=str,
      valueType=IpV4Neighbors,
      help="A mapping of vrf name to ipv4 neighbor addresses" )

class VrfIpV6Neighbors( DeferredModel ):
   """Lists all ND discovered IPv6 neighbors within each VRF."""

   vrfs = Dict( keyType=str,
      valueType=IpV6Neighbors,
      help="A mapping of vrf name to ipv6 neighbor addresses" )
