# Copyright (c) 2005-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP configuration.
#-------------------------------------------------------------------------------
"""Configuration commands supported for IP"""

import Arnet
from Arnet.Verify import verify
import ArpLogMsgs
import ArpCommon
from ArpCommon import sendArpRefreshReqMsg
from ArpCommon import defaultEthernetTimeout
import LazyMount
import Logging
import Tac
import UtmpDump
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import getAllVrfNames, vrfExists, ALL_VRF_NAME, DEFAULT_VRF
import Cell
import signal

arpConfig = None
ipStatus = None
allVrfStatusLocal = None
noArpTableForVrfMsg = "ARP table for VRF %s does not exist."
ipv4Af = Tac.Type( 'Arnet::AddressFamily' ).ipv4
arp = ArpCommon.Arp()

def addStaticArp( mode, args ):
   ipAddr = args[ 'IPADDR' ]
   macAddr = args[ 'MACADDR' ]
   vrfName = args.get( 'VRF', DEFAULT_VRF )

   if not vrfExists( vrfName ):
      mode.addError( noArpTableForVrfMsg % vrfName )
      return
   arp.arpInputVrfConfig( vrfName ).ipv4.addMember(
         Tac.Value( "Arp::ArpEntryInput", ipAddr=ipAddr, ethAddr=macAddr ) )

def delStaticArp( mode, args ):
   ipAddr = args[ 'IPADDR' ]
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   # XXX The 'no arp <ip-address>' command deletes all ARP entries for the given IP
   # address.  It's not possible to remove an ARP entry just for one interface.  This
   # is pretty lame, but it's the industry-standard.
   if not vrfExists( vrfName ):
      mode.addError( noArpTableForVrfMsg % vrfName )
      return
   del arp.arpInputVrfConfig( vrfName ).ipv4[ ipAddr ]

def setMonitorMac( mode, no=False ):
   if no == 'default':
      # default is off
      no = True
   arpConfig.globalMonitorMac = not no

def enableArpPersistent( mode, args ):
   arpConfig.restoreOnReboot = True

def disableArpPersistent( mode, args ):
   arpConfig.restoreOnReboot = False

def enableIpv4Persistent( mode, args ):
   refreshDelay = args.get( 'SECONDS', arpConfig.refreshDelayDefault )
   arpConfig.ip4PersistanceEnabled = True
   arpConfig.refreshDelay = refreshDelay

def disableIpv4Persistent( mode, args ):
   arpConfig.ip4PersistanceEnabled = False
   arpConfig.refreshDelay = arpConfig.refreshDelayDefault

def setProxyMaxDelay( mode, args ):
   arpConfig.proxyMaxDelay = args[ 'MILLISECONDS' ]

def unsetProxyMaxDelay( mode, args ):
   arpConfig.proxyMaxDelay = arpConfig.proxyMaxDelayDefault

def runClearArpCmd( cmd, vrfName=DEFAULT_VRF ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   try:
      if vrfName != DEFAULT_VRF:
         vrf = allVrfStatusLocal.vrf.get( vrfName )
         if vrf and vrf.state == 'active':
            Arnet.NsLib.runMaybeInNetNs( vrf.networkNamespace,
                                         cmd,
                                         stdout=Tac.DISCARD,
                                         stderr=Tac.DISCARD,
                                         asRoot=True )
            return True
         else:
            return False
      else:
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         return True
   except Tac.SystemCommandError as e:
      if e.error != 255:
         if ( e.error == -signal.SIGINT or
              e.error == -signal.SIGQUIT ):
            return True
         raise e

def clearArpCache( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   uInfo = UtmpDump.getUserInfo()
   # pylint:disable-msg=E1101
   cmd = [ "ip", "-4", "neigh", "flush", "to", "0.0.0.0/0" ]
   # No state needs to be saved, and we never run Cli remotely, so
   # we'll just do this:
   if vrfName == ALL_VRF_NAME:
      vrfList = getAllVrfNames( mode )
      for vrf in vrfList:
         runClearArpCmd( cmd, vrf )
      Logging.log( ArpLogMsgs.ETH_ARP_CACHE_CLEAR,
            vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )
   else:
      if runClearArpCmd( cmd, vrfName ):
         Logging.log( ArpLogMsgs.ETH_ARP_CACHE_CLEAR,
               vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )


def clearArpInterface( mode, args ):
   intf = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   if not intf.lookup():
      mode.addError( "Interface does not exist" )
      return
   ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name_ )
   if not ipIntfStatus:
      mode.addError( "%s is not a layer 3 interface" % intf.name_ )
      return
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName != ipIntfStatus.vrf:
      mode.addError( "Interface vrf does not match" )
      return
   cmd = [ "ip", "-4", "neigh", "flush", "dev", intf.status().deviceName ]
   if runClearArpCmd( cmd, vrfName ):
      uInfo = UtmpDump.getUserInfo()
      # pylint:disable-msg=E1101
      Logging.log( ArpLogMsgs.ETH_ARP_CACHE_INTERFACE_CLEAR, vrfName, intf.name_,
                   uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )

def clearIpArp( mode, args ):
   arpIpAddr = args[ 'IPADDR' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   uInfo = UtmpDump.getUserInfo()
   # pylint:disable-msg=E1101
   cmd = [ "ip", "-4", "neigh", "flush", arpIpAddr ]
   if runClearArpCmd( cmd, vrfName ):
      Logging.log( ArpLogMsgs.ETH_ARP_CACHE_IP_CLEAR,
            arpIpAddr, vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )

def refreshArpByVrf( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   sendArpRefreshReqMsg( mode, ipv4Af, vrfName, None, None )

def refreshArpByIntf( mode, args ):
   intf = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name )
   if not ipIntfStatus:
      return
   if vrfName != ipIntfStatus.vrf:
      mode.addError( "Interface vrf does not match" )
      return
   sendArpRefreshReqMsg( mode, ipv4Af, vrfName, intf.name, None )

def refreshIpArp( mode, args ):
   arpIpAddr = args[ 'IPADDR' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   sendArpRefreshReqMsg( mode, ipv4Af, vrfName, None, arpIpAddr )

def verifyArp( mode, args ):
   actors = [ 'Arp' ]
   if 'all' in args:
      objects = [ ( 'ALL', 'ALL' ) ]
   else:
      ipAddr = args[ 'IPADDR' ]
      vrfName = args.get( 'VRF' )
      objects = [ ( str( ipAddr ), vrfName ) ]

   verify( 'Arp', objects, actors, myEntManager )

def setGlobalArpTimeOut( mode, args ):
   arpConfig.globalArpTimeout = args[ 'SECONDS' ]

def noSetGlobalArpTimeOut( mode, args ):
   arpConfig.globalArpTimeout = defaultEthernetTimeout

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
myEntManager = None
def Plugin( entityManager ):
   arp.plugin( entityManager )
   global allVrfStatusLocal
   global ipStatus
   global arpConfig
   global myEntManager

   myEntManager = entityManager

   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   arpConfig = LazyMount.mount( entityManager, "arp/config", "Arp::Config", "w" )
