# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements non-interface-specific IP6 configuration.
#-------------------------------------------------------------------------------
'''Configuration commands supported for IP'''

import Logging
import UtmpDump
import ArpCommon
from ArpCommon import sendArpRefreshReqMsg
import LazyMount
import ArpLogMsgs
from IpLibConsts import DEFAULT_VRF
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import getAllVrfNames, vrfExists, ALL_VRF_NAME
import Tac
import Cell
import Arnet
from Arnet.Verify import verify
import signal

arp = ArpCommon.Arp()
arpConfig = None
ip6Status = None
allVrfStatusLocal = None
routingHardwareStatus = None
noNeighborTableForVrfMsg = "Neighbor table for VRF %s does not exist."
ipv6Af = Tac.Type( 'Arnet::AddressFamily' ).ipv6

def noipv6Neighbor( mode, args ):
   ipv6Addr = args[ 'IP6ADDR' ]
   ipv6IntfName = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   assert vrfName and vrfName != ''
   if not vrfExists( vrfName ):
      mode.addError( noNeighborTableForVrfMsg % vrfName )
      return

   key = Tac.Value( "Arp::NeighborEntryKey", ip6Addr=ipv6Addr,
                    intfId=ipv6IntfName.strWithLongTag() )
   if not arp.arpInputVrfConfig( vrfName ).ipv6.has_key ( key ):
      mode.addError( "No matching neighbor to delete" )
   else:
      del arp.arpInputVrfConfig( vrfName ).ipv6[ key ]

def doipv6Neighbor( mode, args ):
   ipv6Addr = args[ 'IP6ADDR' ]
   ipv6IntfName = args[ 'INTF' ]
   macAddr = args[ 'MACADDR' ]
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   assert vrfName and vrfName != ''
   if not vrfExists( vrfName ):
      mode.addError( noNeighborTableForVrfMsg % vrfName )
      return

   key = Tac.Value( "Arp::NeighborEntryKey", ip6Addr=ipv6Addr,
                    intfId=ipv6IntfName.strWithLongTag() )
   arp.arpInputVrfConfig( vrfName ).ipv6.addMember(
      Tac.Value( "Arp::NeighborEntryConfigInput", key=key, ethAddr=macAddr ) )

def enableNCPersistent( mode, args ):
   arpConfig.ip6RestoreOnReboot = True

def disableNCPersistent( mode, args ):
   arpConfig.ip6RestoreOnReboot = False

def enableIpv6Persistency( mode, args ):
   arpConfig.ip6PersistanceEnabled = True
   arpConfig.refreshDelay = args.get( 'SECONDS', arpConfig.refreshDelayDefault )

def disableIpv6Persistency( mode, args ):
   arpConfig.ip6PersistanceEnabled = False
   arpConfig.refreshDelay = arpConfig.refreshDelayDefault

def runClearIp6NeighborsCmd( cmd, vrfName=DEFAULT_VRF ):
   if not vrfName:
      vrfName = DEFAULT_VRF
   assert vrfName != ''
   try:
      if vrfName != DEFAULT_VRF:
         vrf = allVrfStatusLocal.vrf.get( vrfName )
         if vrf and vrf.state == 'active':
            Arnet.NsLib.runMaybeInNetNs( vrf.networkNamespace,
                                         cmd,
                                         stdout=Tac.DISCARD,
                                         stderr=Tac.DISCARD,
                                         asRoot=True )
            return True
         else:
            return False
      else:
         Tac.run( cmd, asRoot=True, stdout=Tac.DISCARD, stderr=Tac.DISCARD )
         return True
   except Tac.SystemCommandError as e:
      if e.error != 255:
         if ( e.error == -signal.SIGINT or
              e.error == -signal.SIGQUIT ):
            return True
         raise e

def clearAllIpv6Neighbors( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   uInfo = UtmpDump.getUserInfo()
   cmd =  [ "ip", "-6", "neigh", "flush", "to", "0::/0" ]
   # pylint:disable-msg=E1101
   # No state needs to be saved, and we never run Cli remotely, so
   # we'll just do this:
   if vrfName == ALL_VRF_NAME:
      Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_CLEAR,
            vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )
      vrfList = getAllVrfNames( mode )
      for vrf in vrfList:
         vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrf )
         runClearIp6NeighborsCmd( cmd, vrfName )
   else:
      if runClearIp6NeighborsCmd( cmd, vrfName ):
         Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_CLEAR,
               vrfName, uInfo[ 'user' ], uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )

def clearIntfIpv6Neighbors( mode, args ):
   ipv6IntfName = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   ipv6Addr = args.get( 'IPV6ADDR' )
   if not ipv6IntfName.lookup():
      mode.addError( "Interface does not exist" )
      return
   ip6IntfStatus = ip6Status.ipIntfStatus( ipv6IntfName.name_ )
   if not ip6IntfStatus:
      mode.addError( "%s is not a layer 3 interface" % ipv6IntfName.name_ )
      return
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if vrfName != ip6IntfStatus.vrf:
      mode.addError( "Interface vrf does not match" )
      return

   # Build the minimum command to execute
   tacCmd = [ "ip", "-6", "neigh", "flush", "dev", ipv6IntfName.status().deviceName ]

   # if the optional ipaddress is specified append it to the CLI
   if ipv6Addr:
      tacCmd.append( ipv6Addr.stringValue )

   if runClearIp6NeighborsCmd( tacCmd, vrfName ):
      uInfo = UtmpDump.getUserInfo()
      # pylint:disable-msg=E1101
      if ipv6Addr:
         Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_INTERFACE_IP_CLEAR,
               ipv6Addr, vrfName, ipv6IntfName.name_, uInfo[ 'user' ],
               uInfo[ 'tty' ], uInfo[ 'ipAddr' ] )
      else:
         Logging.log( ArpLogMsgs.ETH_IPV6_NEIGHBOR_INTERFACE_CLEAR,
               vrfName, ipv6IntfName.name_, uInfo[ 'user' ], uInfo[ 'tty' ],
               uInfo[ 'ipAddr' ] )

def refreshIpv6NeighborsByVrf( mode, args ):
   vrfName = args.get( 'VRF' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   sendArpRefreshReqMsg( mode, ipv6Af, vrfName, None, None )

def refreshIpv6NeighborsByIntfOrAddr( mode, args ):
   ipv6IntfName = args[ 'INTF' ]
   vrfName = args.get( 'VRF' )
   ipv6Addr = args.get( 'IPV6ADDR' )
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not ipv6IntfName.lookup():
      mode.addError( "Interface does not exist" )
      return
   sendArpRefreshReqMsg( mode, ipv6Af, vrfName, ipv6IntfName.name,
                         ipv6Addr.stringValue if ipv6Addr else None )

def verifyNeighbor( mode, args ):
   actors = [ 'Arp' ]
   if 'all' in args:
      objects = [ ( 'ALL', 'ALL' ) ]
   else:
      ip6Addr = args[ 'IPV6ADDR' ]
      vrfName = args.get( 'VRF' )
      objects = [ ( str( ip6Addr ), vrfName ) ]

   verify( 'Neighbor', objects, actors, myEntManager )

myEntManager = None
#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global arpConfig
   global ip6Status
   global allVrfStatusLocal
   global routingHardwareStatus
   global myEntManager

   myEntManager = entityManager

   arp.plugin( entityManager )
   arpConfig = LazyMount.mount( entityManager, "arp/config", "Arp::Config", "w" )

   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   allVrfStatusLocal = LazyMount.mount( entityManager,
                                        Cell.path( "ip/vrf/status/local" ),
                                        "Ip::AllVrfStatusLocal", "r" )
   routingHardwareStatus = LazyMount.mount( entityManager,
                                            "routing/hardware/status",
                                            "Routing::Hardware::Status", "r" )
