#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.ArpIpIntfCli as ArpIpIntfCli
from CliPlugin.ArpCommon import (
   arpCacheCapacity,
)
from CliPlugin.ArpIpIntfCli import setMonitorMac
from CliPlugin.ArpIpIntfCli import noMonitorMac
from CliPlugin.ArpIpIntfCli import defaultMonitorMac
from CliPlugin.IraIp6IntfCli import ndMatcher
from CliPlugin.Ip6AddrMatcher import Ip6PrefixMatcher
from CliPlugin.VlanIntfCli import VlanIntfModelet
from CliPlugin.ArpCommon import minArpTimeout, maxArpTimeout
from CliToken.Ipv6 import ipv6MatcherForConfigIf
from Toggles.ArpToggleLib import (
   toggleArpCacheCapacityEnabled,
)

#--------------------------------------------------------------------------------
# common tokens
#--------------------------------------------------------------------------------
matcherArp = CliMatcher.KeywordMatcher(
   'arp', helpdesc='ARP config commands' )
matcherCacheCapacity = CliMatcher.IntegerMatcher(
   arpCacheCapacity.min(),
   arpCacheCapacity.max(),
   helpdesc='Capacity of dynamic cache entries' )
matcherProxy = CliMatcher.KeywordMatcher(
   'proxy', helpdesc='proxy' )
matcherSeconds = CliMatcher.IntegerMatcher(
   minArpTimeout, maxArpTimeout, helpdesc='Seconds' )
matcherTimeout = CliMatcher.KeywordMatcher(
   'timeout', helpdesc='Set timeout for dynamic ARP entries' )

def guardNdProxySupported( mode, args ):
   return ArpIpIntfCli.ndProxySupportedGuard( mode, args )

#--------------------------------------------------------------------------------
# arp aging timeout SECONDS
# ( no | default ) arp aging timeout
#--------------------------------------------------------------------------------
class ArpAgingTimeoutArptimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'arp aging timeout SECONDS'
   noOrDefaultSyntax = 'arp aging timeout  ...'
   data = {
      'arp': matcherArp,
      'aging': 'Set timeout for dynamic ARP entries',
      'timeout': "timeout",
      'SECONDS': matcherSeconds,
   }
   handler = ArpIpIntfCli.setArpTimeout
   noOrDefaultHandler = ArpIpIntfCli.noArpTimeout

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpAgingTimeoutArptimeoutCmd )

#--------------------------------------------------------------------------------
# arp timeout SECONDS
# ( no | default ) arp timeout
#--------------------------------------------------------------------------------
class ArpTimeoutArptimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'arp timeout SECONDS'
   noOrDefaultSyntax = 'arp timeout ...'
   data = {
      'arp': matcherArp,
      'timeout': CliCommand.Node( matcher=matcherTimeout,
                                  deprecatedByCmd='arp aging timeout' ),
      'SECONDS': matcherSeconds,
   }
   handler = ArpIpIntfCli.setArpTimeout
   noOrDefaultHandler = ArpIpIntfCli.noArpTimeout
   hidden = True

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpTimeoutArptimeoutCmd )

#--------------------------------------------------------------------------------
# arp cache dynamic capacity NUMBER
# ( no | default ) arp cache dynamic capacity
#--------------------------------------------------------------------------------
class ArpCacheCapacityCmd( CliCommand.CliCommandClass ):
   syntax = 'arp cache dynamic capacity CAPACITY'
   noOrDefaultSyntax = 'arp cache dynamic capacity  ...'
   data = {
      'arp': matcherArp,
      'cache': 'Manage the ARP cache',
      'dynamic': 'Cache parameters for dynamic neighbor resolutions',
      'capacity': 'Capacity of the ARP cache',
      'CAPACITY': matcherCacheCapacity,
   }
   handler = ArpIpIntfCli.setArpCacheCapacity
   noOrDefaultHandler = ArpIpIntfCli.noArpCacheCapacity

if toggleArpCacheCapacityEnabled():
   ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( ArpCacheCapacityCmd )

#--------------------------------------------------------------------------------
# ipv6 nd cache expire SECONDS
# ( no | default ) ipv6 nd cache expire
#--------------------------------------------------------------------------------
class Ipv6NdCacheExpireNdtimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd cache expire SECONDS'
   noOrDefaultSyntax = 'ipv6 nd cache expire ...'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'cache': 'Manage the IPv6 neighbor cache',
      'expire': 'Expiry time for ND entries',
      'SECONDS': matcherSeconds,
   }
   handler = ArpIpIntfCli.setNeighborCacheExpire
   noOrDefaultHandler = ArpIpIntfCli.noNeighborCacheExpire

ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( Ipv6NdCacheExpireNdtimeoutCmd )

#--------------------------------------------------------------------------------
# ipv6 nd cache maximum NUMBER
# ( no | default ) ipv6 nd cache maximum
#--------------------------------------------------------------------------------
class Ipv6NdCacheCapacityCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd cache dynamic capacity CAPACITY'
   noOrDefaultSyntax = 'ipv6 nd cache dynamic capacity ...'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'cache': 'Manage the IPv6 neighbor cache',
      'dynamic': 'Cache parameters for dynamic neighbor resolutions',
      'capacity': 'Capacity of the IPv6 neighbor cache',
      'CAPACITY': matcherCacheCapacity,
   }
   handler = ArpIpIntfCli.setNeighborCacheCapacity
   noOrDefaultHandler = ArpIpIntfCli.noNeighborCacheCapacity

if toggleArpCacheCapacityEnabled():
   ArpIpIntfCli.ArpIpIntfConfigModelet.addCommandClass( Ipv6NdCacheCapacityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] arp monitor mac-address
#--------------------------------------------------------------------------------
class ArpMonitorMacAddressCmd( CliCommand.CliCommandClass ):
   syntax = 'arp monitor mac-address'
   noOrDefaultSyntax = 'arp monitor mac-address'
   data = {
      'arp': matcherArp,
      'monitor': "monitor ARP events",
      'mac-address' : "monitor ARP's MAC address"
   }
   handler = setMonitorMac
   noHandler = noMonitorMac
   defaultHandler = defaultMonitorMac

VlanIntfModelet.addCommandClass( ArpMonitorMacAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd proxy prefix ( connected | IPV6PREFIX )
#--------------------------------------------------------------------------------
class NdProxyCmdForPrefix( CliCommand.CliCommandClass ):
   syntax = 'ipv6 nd proxy prefix ( connected | IPV6PREFIX )'
   noOrDefaultSyntax = 'ipv6 nd proxy prefix ( connected | IPV6PREFIX )'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'proxy': CliCommand.Node( matcherProxy,
                                guard=guardNdProxySupported ),
      'prefix' : "prefix",
      'connected' : "connected",
      'IPV6PREFIX': Ip6PrefixMatcher( helpdesc="ipv6 prefix for proxies" )
   }
   handler = ArpIpIntfCli.setNdProxyForPrefix
   noOrDefaultHandler = ArpIpIntfCli.noNdProxyForPrefix

VlanIntfModelet.addCommandClass( NdProxyCmdForPrefix )

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 nd proxy
#--------------------------------------------------------------------------------
class NdProxyCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ipv6 nd proxy'
   data = {
      'ipv6': ipv6MatcherForConfigIf,
      'nd': ndMatcher,
      'proxy': CliCommand.Node( matcherProxy,
                                guard=guardNdProxySupported ),
   }
   noOrDefaultHandler = ArpIpIntfCli.noNdProxy

VlanIntfModelet.addCommandClass( NdProxyCmd )
