#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool, Dict, Enum, Float, Int, List, Model, Str, Submodel
from ArnetModel import Ip4Address, MacAddress
from IntfModel import Interface
import TableOutput

formatLeft = TableOutput.Format( justify="left" )
formatLeft.noPadLeftIs( True )
formatRight = TableOutput.Format( justify="right" )

def utcTimestampToStr( timestamp, relative=True, now=None ):
   """ Convert a UTC timestamp to a string. If relative is True, then describe
   the timestamp in terms of time past since now. Based off
   Ark.timestampToStr."""
   import Tac, datetime
   if now is None:
      now = Tac.utcNow()
   if not timestamp:
      return 'never'
   elif relative:
      td = datetime.timedelta( seconds=int( now - timestamp ) )
      return str( td ) + ' ago'
   else:
      td = datetime.datetime.fromtimestamp( timestamp )
      return td.strftime( '%Y-%m-%d %H:%M:%S' )

class VlanArpInspectionStatus( Model ):
   configuration = Bool( help="Arp Inspection enabled on this VLAN" )
   operationState = Bool( help="Arp Inspection operational state on this VLAN" )

class ArpInspectionEnabled( Model ):
   vlanStatus = Dict( keyType=int, valueType=VlanArpInspectionStatus,
                      help="VLANs with Arp inspection enabled on the switch" )

   def render( self ):
      for vlan, status in self.vlanStatus.iteritems():
         configEnabled = 'Enabled' if status.configuration else 'Disabled'
         operational = 'Active' if status.operationState else 'Inactive'
         print 'VLAN %d' % vlan
         print '----------'
         print 'Configuration : %s' % configEnabled
         print 'Operation State : %s' % operational

class ArpInspectionInterfaceConfig( Model ):
   interfaceTrusted = Bool( help="Interface trusted with Arp traffic" )
   errdisableRateLimit = Int( help="Maximum number of Arp packets per second "
                              "before the interface is disabled" )
   errdisableBurstInterval = Int( help="Consecutive interval in seconds over "
                                  "which the interface is monitored for "
                                  "disabling" )
   loggingRateLimit = Int( help="Maximum number of Arp packets per second "
                           "before a log message is emitted" )
   loggingBurstInterval = Int( help="Consecutive interval in seconds over "
                               "which the interface is monitored for logging" )

class ArpInspectionInterfaces( Model ):
   interfaceConfigs = Dict( keyType=Interface,
                            valueType=ArpInspectionInterfaceConfig,
                            help="Interfaces with non-default configurations" )

   def render( self ):
      # XXX_darylwang Try to find a way to abbreviate these headings
      headings = ( "Interface", "Trust State", "Errdisable Rate (pps)",
                   "Errdisable Burst Interval", "Logging Rate (pps)",
                   "Logging Burst Interval" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatRight, formatRight, formatRight,
                           formatRight, formatRight )
      for intf, intfConfig in self.interfaceConfigs.iteritems():
         # Figure out how to represent an unlimited rate in the CAPI model
         # For now, use values of 0 for rate limit and burst interval
         if intfConfig.errdisableRateLimit:
            errdisableRateLimit = intfConfig.errdisableRateLimit
            errdisableBurstInterval = intfConfig.errdisableBurstInterval
         else:
            errdisableRateLimit = "None"
            errdisableBurstInterval = "N/A"
         if intfConfig.loggingRateLimit:
            loggingRateLimit = intfConfig.loggingRateLimit
            loggingBurstInterval = intfConfig.loggingBurstInterval
         else:
            loggingRateLimit = "None"
            loggingBurstInterval = "N/A"
         trustState = "Trusted" if intfConfig.interfaceTrusted else "Untrusted"
         table.newRow( intf, trustState, errdisableRateLimit,
                       errdisableBurstInterval, loggingRateLimit,
                       loggingBurstInterval )
      print table.output()

class IpMacBindingEntry( Model ):
   macAddr = MacAddress( help="MAC address" )
   # XXX_darylwang Should this be IPGeneric now to make future IPv6 support
   # easier?
   ipAddr = Ip4Address( help="IPv4 address" )
   interface = Interface( help="Interface name" )
   vlanId = Int( help="Vlan this binding applies to" )
   entryType = Enum( values=( "static", "dynamic" ),
                     help="Binding type" )
   # Currently always infinite. Will set the lease time in phase 2 of
   # Arp-Inspection
   leaseTime = Int( help="Lease time in seconds" )

class IpMacBindingTable( Model ):
   # XXX_darylwang Thought about using a Dict here, but couldn't come up with a
   # good key type. Also want to keep this sorted by IP address if at all
   # possible.
   bindingEntries = List( valueType=IpMacBindingEntry,
                          help="List of valid IP-MAC bindings" )

   # TODO darylwang: Entries should be sorted by IP addr
   def render( self ):
      headings = ( "MacAddress", "IpAddress", "Lease(sec)", "Type", "VLAN",
                   "Interface" )
      table = TableOutput.createTable( headings )
      table.formatColumns( formatLeft, formatRight, formatRight, formatRight,
                           formatRight, formatRight, formatRight )

      for bindingEntry in self.bindingEntries:
         # Lease time is always infinite for now. Will set in phase 2.
         leaseTime = "infinite"
         # Interface seems to add single quotes around interface name. Remove
         # the single quotes.
         intfName = str( bindingEntry.interface ).strip( "\"'" )
         displayMacAddr = bindingEntry.macAddr.displayString
         table.newRow( displayMacAddr, bindingEntry.ipAddr, leaseTime,
                       bindingEntry.entryType, bindingEntry.vlanId, intfName )
      print table.output()

class ArpPacketBrief( Model ):
   receivedInterfaceId = Interface( help="Interface Arp packet was received on" )
   receivedVlanId = Int( help="Vlan Arp packet was received on" )
   timestamp = Float( help="Time packet was received" )
   sourceMac = MacAddress( help="Source Mac address" )
   destinationMac = MacAddress( help="Destination Mac address" )
   arpOpcode = Enum( values=( "invalid", "request", "response" ),
                     help="Arp operation type" )
   senderMac = MacAddress( help="Mac address of the sender" )
   senderIp = Ip4Address( help="IPv4 address of the sender" )
   dropReason = Enum( values=( "rateLimitExceeded", "invalidBinding" ),
                      help="Drop reason of the Arp packet", optional=True )

class ArpInspectionStatistics( Model ):
   requestsForwarded = Int( help="Number of Arp requests forwarded" )
   responsesForwarded = Int( help="Number of Arp responses forwarded" )
   requestsDropped = Int( help="Number of Arp requests dropped" )
   responsesDropped = Int( help="Number of Arp responses dropped" )

   lastDroppedPacket = Submodel( valueType=ArpPacketBrief,
                                 help="The last Arp packet dropped",
                                 optional=True )

class VlanArpInspectionStatistics( ArpInspectionStatistics ):
   vlanId = Int( help="Vlan id" )
   # TODO darylwang: Need to make this optional; might make this default to
   # empty string or might use Submodel instead
   name = Str( help="Vlan name" )

class IntfArpInspectionStatistics( ArpInspectionStatistics ):
   intfId = Interface( help="Interface name" )

class ArpInspectionStatisticsCollection( Model ):
   vlanStatistics = Dict( keyType=int, valueType=VlanArpInspectionStatistics,
                          help="Arp inspection statistics for vlans" )
   intfStatistics = Dict( keyType=Interface, valueType=ArpInspectionStatistics,
                          help="Arp inspection statistics for interfaces" )

   def printPacketBrief( self, arpPacketBrief, vlan=False ):
      timeStr = utcTimestampToStr( arpPacketBrief.timestamp, relative=False )
      timeAgo = utcTimestampToStr( arpPacketBrief.timestamp, relative=True )
      receivedLocation = ( arpPacketBrief.receivedInterfaceId if vlan else
                           arpPacketBrief.receivedVlanId )
      print "Time: %s (%s)" % ( timeStr, timeAgo )
      if arpPacketBrief.dropReason == "rateLimitExceeded":
         print "Reason: Rate limit exceeded"
      else:
         print "Reason: Invalid source binding"
      print "Received on %s" % receivedLocation
      print "Packet:"
      print "  Source MAC: %s" % arpPacketBrief.sourceMac.displayString
      print "  Destination MAC: %s" % arpPacketBrief.destinationMac.displayString
      print "  ARP Type: %s" % arpPacketBrief.arpOpcode
      print "  ARP Sender MAC: %s" % arpPacketBrief.senderMac.displayString
      print "  ARP Sender IP: %s" % arpPacketBrief.senderIp

   def printStatSummmary( self, arpInspStats ):
      print "ARP Req Forwarded = %d" % arpInspStats.requestsForwarded
      print "ARP Res Forwarded = %d" % arpInspStats.responsesForwarded
      print "ARP Req Dropped = %d" % arpInspStats.requestsDropped
      print "ARP Res Dropped = %d" % arpInspStats.responsesDropped

   def render( self ):
      for vlanStats in self.vlanStatistics.itervalues():
         print "Vlan : %d" % vlanStats.vlanId
         print "------------"
         self.printStatSummmary( vlanStats )
         print ""
         if vlanStats.lastDroppedPacket:
            print "Last invalid ARP:"
            self.printPacketBrief( vlanStats.lastDroppedPacket, vlan=True )
            print ""
      for intfStats in self.intfStatistics.itervalues():
         print "Interface : %s" % intfStats.intfId
         print "------------"
         self.printStatSummmary( intfStats )
         print ""
         if intfStats.lastDroppedPacket:
            print "Last invalid ARP:"
            self.printPacketBrief( intfStats.lastDroppedPacket, vlan=False )
            print ""
