#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliPlugin.ArpInspectionModels import ArpInspectionEnabled, \
    VlanArpInspectionStatus, ArpInspectionInterfaceConfig, ArpInspectionInterfaces, \
    IpMacBindingEntry, IpMacBindingTable, ArpPacketBrief, \
    VlanArpInspectionStatistics, IntfArpInspectionStatistics, \
    ArpInspectionStatisticsCollection
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.MacAddr as MacAddr
from CliPlugin.VirtualIntfRule import IntfMatcher
import CliPlugin.VlanCli as VlanCli
import CliToken.Clear
import CliToken.Ip
import ConfigMount
import LazyMount
import ShowCommand
import Tac
import Tracing
from TypeFuture import TacLazyType

# Bogus import to generate dependency on HostSecurity-lib
# pkgdeps: rpm HostSecurity-lib

t0 = Tracing.trace0

# Arp Inspection Sysdb state; written by Plugin function at end of this file
arpInspectionConfig = None
arpInspectionStatus = None
arpInspectionCheckpoint = None
arpInspectionHwStatusDir = None
ipMacBindingTable = None
entryTypes = TacLazyType( "IpMacBinding::EntryType" )

def arpInspectionSupportedGuard( mode, token ):
   for hwStatus in arpInspectionHwStatusDir.itervalues():
      if hwStatus.arpInspectionSupported:
         return None
   return CliParser.guardNotThisPlatform

# XXX_darylwang Might move this into CliToken/Arp.py, if this isn't defined
# already
arpKwMatcherConfig = CliMatcher.KeywordMatcher( 'arp', helpdesc='ARP configuration' )
arpKwMatcherShow = CliMatcher.KeywordMatcher( 'arp', helpdesc='ARP information' )
arpKwMatcherClear = CliMatcher.KeywordMatcher( 'arp',
      helpdesc='Clear dynamic ARP entry by IP' )
arpKwMatcherClearDeprecated = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'arp',
         helpdesc='Clear dynamic ARP entry by IP' ),
      deprecatedByCmd='clear arp' )

inspectionKwMatcherConfig = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'inspection',
         helpdesc='ARP Inspection configuration' ),
      guard=arpInspectionSupportedGuard )
inspectKwMatcherShow = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'inspection',
         helpdesc='ARP Inspection information' ),
      guard=arpInspectionSupportedGuard )
inspectionKwMatcherClear = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'inspection',
         helpdesc='Clear ARP Inspection information' ),
      guard=arpInspectionSupportedGuard )

# XXX_darylwang This isn't already defined in some CliToken file?
vlanKwMatcher = CliMatcher.KeywordMatcher( 'vlan', helpdesc='VLAN Keyword' )

ethOrLagMatcher = IntfMatcher()
ethOrLagMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
ethOrLagMatcher |= LagIntfCli.EthLagIntf.matcher

#-------------------------------------------------------------------------------
# Helper functions for setting interface specific configuration and
# instantiating ArpInspIntfConfig objects if none exist.
#-------------------------------------------------------------------------------
def maybeCreateArpInspIntfConfig( intfName ):
   if arpInspectionConfig.arpInspIntfConfig.has_key( intfName ):
      arpInspIntfConfig = arpInspectionConfig.arpInspIntfConfig[ intfName ]
   else:
      arpInspIntfConfig = arpInspectionConfig.newArpInspIntfConfig( intfName )
   return arpInspIntfConfig

def setRate( mode, intfName, limitType, rate=0, burstInterval=None,
             default=False ):
   arpInspIntfConfig = maybeCreateArpInspIntfConfig( intfName )
   if default:
      if arpInspIntfConfig.trust:
         rate = 0
      else:
         if limitType == 'limit':
            rate = arpInspIntfConfig.rateLimitErrdisableDefault
         elif limitType == 'logging':
            rate = arpInspIntfConfig.rateLimitLoggingDefault
         else:
            mode.addError( "Need to specify logging or limit" )
      burstInterval = arpInspIntfConfig.burstIntervalDefault
   if limitType == 'logging':
      arpInspIntfConfig.rateLimitLogging = rate
      if burstInterval:
         arpInspIntfConfig.burstIntervalLogging = burstInterval
   elif limitType == 'limit':
      arpInspIntfConfig.rateLimitErrdisable = rate
      if burstInterval:
         arpInspIntfConfig.burstIntervalErrdisable = burstInterval
   else:
      mode.addError( "Need to specify logging or limit" )

#-------------------------------------------------------------------------------
# Helper functions for checking whether an interface has the same configuration
# as the default values and for deleting ArpInspIntfConfig with only default
# values.
#-------------------------------------------------------------------------------
def hasDefaultRateConfig( intfName, limitType='limit', trusted=False ):
   if arpInspectionConfig.arpInspIntfConfig.has_key( intfName ):
      arpInspIntfConfig = arpInspectionConfig.arpInspIntfConfig[ intfName ]
      if trusted:
         defaultLimitRate = 0
      else:
         if limitType == 'limit':
            defaultLimitRate = arpInspIntfConfig.rateLimitErrdisableDefault
         elif limitType == 'logging':
            defaultLimitRate = arpInspIntfConfig.rateLimitLoggingDefault
         else:
            assert False, "Need to specify 'limit' or 'logging' as limitType"
      defaultBurstInterval = arpInspIntfConfig.burstIntervalDefault
      if limitType == 'limit':
         return ( arpInspIntfConfig.rateLimitErrdisable == defaultLimitRate and
            arpInspIntfConfig.burstIntervalErrdisable == defaultBurstInterval )
      elif limitType == 'logging':
         return ( arpInspIntfConfig.rateLimitLogging == defaultLimitRate and
                  arpInspIntfConfig.burstIntervalLogging == defaultBurstInterval )
      else:
         assert False, "Need to specify 'limit' or 'logging' as limitType"
   else:
      return True

def hasDefaultIntfConfig( intfName ):
   if arpInspectionConfig.arpInspIntfConfig.has_key( intfName ):
      arpInspIntfConfig = arpInspectionConfig.arpInspIntfConfig[ intfName ]
      currentTrust = arpInspIntfConfig.trust
      defaultTrust = False
      return ( hasDefaultRateConfig( intfName, 'limit', currentTrust ) and
               hasDefaultRateConfig( intfName, 'logging', currentTrust ) and
               currentTrust == defaultTrust )
   else:
      return True

def deleteConfigIfDefault( intfName ):
   if ( arpInspectionConfig.arpInspIntfConfig.has_key( intfName ) and
        hasDefaultIntfConfig( intfName ) ):
      del arpInspectionConfig.arpInspIntfConfig[ intfName ]

#-------------------------------------------------------------------------------
# Helper for calculating counter values
#-------------------------------------------------------------------------------
def getGlobalVlanCheckpoint( vlanId ):
   if vlanId in arpInspectionCheckpoint.vlanStatus:
      globalVlanCkpt = arpInspectionCheckpoint.vlanStatus[ vlanId ]
   else:
      globalVlanCkpt = arpInspectionCheckpoint.newVlanStatus( vlanId )
      globalVlanCkpt.stats = ()
   return globalVlanCkpt

def getLatestVlanCheckpoint( vlanId ):
   """ Returns either the global or private checkpoint for the given vlan,
   whichever was last updated. """
   globalVlanCkpt = getGlobalVlanCheckpoint( vlanId )
   return globalVlanCkpt

def getGlobalIntfCheckpoint( intfId ):
   if intfId in arpInspectionCheckpoint.intfStatus:
      globalIntfCkpt = arpInspectionCheckpoint.intfStatus[ intfId ]
   else:
      globalIntfCkpt = arpInspectionCheckpoint.newIntfStatus( intfId )
      globalIntfCkpt.stats = ()
   return globalIntfCkpt

def getLatestIntfCheckpoint( intfId ):
   """ Returns either the global or private checkpoint for the given intf,
   whichever was last updated. """
   globalIntfCkpt = getGlobalIntfCheckpoint( intfId )
   return globalIntfCkpt

def counterDifference( globalStatus, checkpointStatus ):
   """ Takes in the global interface/vlan status and the latest checkpoint
   interface/vlan status and returns the difference in an ArpInspStats
   entity. """
   globalStats = globalStatus.stats
   checkpointStats = checkpointStatus.stats
   counterStats = Tac.newInstance( 'ArpInsp::ArpInspStats' )
   counterStats.reqFwd = globalStats.reqFwd - checkpointStats.reqFwd
   counterStats.resFwd = globalStats.resFwd - checkpointStats.resFwd
   counterStats.reqDrop = globalStats.reqDrop - checkpointStats.reqDrop
   counterStats.resDrop = globalStats.resDrop - checkpointStats.resDrop
   return counterStats

def vlanCounterValues( vlanId ):
   """ Return the counter values for the given vlan. Calculate these values by
   taking the difference between the agent counters and the latest checkpoint
   counters."""
   checkpoint = getLatestVlanCheckpoint( vlanId )
   vlanStatus = arpInspectionStatus.arpInspVlanStatus[ vlanId ]
   return counterDifference( vlanStatus, checkpoint )

def intfCounterValues( intfId ):
   """ Return the counter values for the given intf. Calculate these values by
   taking the difference between the agent counter and the latest checkpoint
   counters. """
   checkpoint = getLatestIntfCheckpoint( intfId )
   intfStatus = arpInspectionStatus.arpInspIntfStatus[ intfId ]
   return counterDifference( intfStatus, checkpoint )

def lastVlanDroppedPacket( vlanId ):
   checkpoint = getLatestVlanCheckpoint( vlanId )
   vlanStatus = arpInspectionStatus.arpInspVlanStatus[ vlanId ]
   if checkpoint.lastDrop == vlanStatus.lastDrop:
      return None
   else:
      return vlanStatus.lastDrop

def lastIntfDroppedPacket( intfId ):
   checkpoint = getLatestIntfCheckpoint( intfId )
   intfStatus = arpInspectionStatus.arpInspIntfStatus[ intfId ]
   if checkpoint.lastDrop == intfStatus.lastDrop:
      return None
   else:
      return intfStatus.lastDrop

def clearCheckpoint( globalStatus, checkpointStatus ):
   globalStats = globalStatus.stats
   checkpointStats = checkpointStatus.stats
   t0( "checkpointStats reqFwd %d" % checkpointStats.reqFwd )
   t0( "checkpointStats resFwd %d" % checkpointStats.resFwd )
   t0( "checkpointStats reqDrop %d" % checkpointStats.reqDrop )
   t0( "checkpointStats resDrop %d" % checkpointStats.resDrop )

   t0( "globalStats reqFwd %d" % globalStats.reqFwd )
   t0( "globalStats resFwd %d" % globalStats.resFwd )
   t0( "globalStats reqDrop %d" % globalStats.reqDrop )
   t0( "globalStats resDrop %d" % globalStats.resDrop )
   checkpointStats.reqFwd = globalStats.reqFwd
   checkpointStats.resFwd = globalStats.resFwd
   checkpointStats.reqDrop = globalStats.reqDrop
   checkpointStats.resDrop = globalStats.resDrop
   checkpointStatus.timestamp = Tac.now()
   checkpointStatus.lastDrop = globalStatus.lastDrop

#-------------------------------------------------------------------------------
# TODO darylwang: Currently unused functions to delete checkpoints. I don't
# think the ArpInsp agent ever resets the counters, but if it does then we'll
# need to use these functions to destroy the now stale checkpoints. Will remove
# if nobody in code review thinks they'll be needed.
#-------------------------------------------------------------------------------
def delIntfCheckpoint( intfId ):
   if intfId in arpInspectionCheckpoint.intfStatus:
      del arpInspectionCheckpoint.intfStatus[ intfId ]

def delVlanCheckpoint( vlanId ):
   if vlanId in arpInspectionCheckpoint.vlanStatus:
      del arpInspectionCheckpoint.vlanStatus[ vlanId ]

#-------------------------------------------------------------------------------
# Enable/disable ARP inspection for the given VLANs
# "[no|default] ip arp inspection vlan VLAN_SET"
#-------------------------------------------------------------------------------
class ConfigArpInspectionCmd( CliCommand.CliCommandClass ):
   syntax = 'ip arp inspection vlan VLAN_SET'
   noOrDefaultSyntax = 'ip arp inspection vlan [ VLAN_SET ]'
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'arp': arpKwMatcherConfig,
            'inspection': inspectionKwMatcherConfig,
            'vlan': vlanKwMatcher,
            'VLAN_SET': VlanCli.vlanSetMatcher
          }

   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      for vlanId in vlanSet.ids:
         arpInspectionConfig.arpInspEnabled[ vlanId ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanSet = args.get( 'VLAN_SET' )
      if vlanSet:
         for vlanId in vlanSet.ids:
            if arpInspectionConfig.arpInspEnabled.has_key( vlanId ):
               del arpInspectionConfig.arpInspEnabled[ vlanId ]
               delVlanCheckpoint( vlanId )
      else:
         for vlan in arpInspectionConfig.arpInspEnabled.keys():
            del arpInspectionConfig.arpInspEnabled[ vlan ]
            delVlanCheckpoint( vlan )

BasicCli.GlobalConfigMode.addCommandClass( ConfigArpInspectionCmd )

#-------------------------------------------------------------------------------
# Show ARP inspection configuration for the given VLANs
# "show ip arp inspection vlan [VLAN_SET]"
#-------------------------------------------------------------------------------
class ShowArpInspectionConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip arp inspection vlan [ VLAN_SET ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'arp': arpKwMatcherShow,
            'inspection': inspectKwMatcherShow,
            'vlan': vlanKwMatcher,
            'VLAN_SET': VlanCli.vlanSetMatcher
            }
   cliModel = ArpInspectionEnabled

   @staticmethod
   def handler( mode, args ):
      vlanStatus = {}
      vlanSet = args.get( 'VLAN_SET' )
      if vlanSet:
         enabledVlans = [ vlanId for vlanId in vlanSet.ids if vlanId in
                          arpInspectionConfig.arpInspEnabled ]
      else:
         enabledVlans = arpInspectionConfig.arpInspEnabled.keys()
      for enabledVlan in enabledVlans:
         operationalStatus = False
         for hwStatus in arpInspectionHwStatusDir.itervalues():
            if hwStatus.arpInspectVlan.has_key( enabledVlan ):
               operationalStatus = ( hwStatus.arpInspectVlan[ enabledVlan ] and
                                     hwStatus.hwArpInspectionEnabled )
               break
         vlanStatus[ enabledVlan ] = \
             VlanArpInspectionStatus( configuration=True,
                                      operationState=operationalStatus )

      return ArpInspectionEnabled( vlanStatus=vlanStatus )

BasicCli.addShowCommandClass( ShowArpInspectionConfigCmd )

#-------------------------------------------------------------------------------
# Set trusted status on an interface for ARP inspection
# "[no|default] ip arp inspection trust"
#-------------------------------------------------------------------------------
class ConfigArpInspectionTrustCmd( CliCommand.CliCommandClass ):
   syntax = 'ip arp inspection trust'
   noOrDefaultSyntax = 'ip arp inspection trust'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'arp': arpKwMatcherConfig,
            'inspection': inspectionKwMatcherConfig,
            'trust': 'Configure trusted status for ARP inspection'
            }

   @staticmethod
   def _configArpInspectionTrust( mode, trusted=True ):
      intfName = mode.intf.name
      arpInspIntfConfig = maybeCreateArpInspIntfConfig( intfName )
      oldTrustState = arpInspIntfConfig.trust
      arpInspIntfConfig.trust = trusted
      # Check if the current rate limits are set to the default values for the
      # old trust state. If so, then set the rate limits to the default values
      # for the new trust state.
      for limitType in [ 'limit', 'logging' ]:
         if hasDefaultRateConfig( intfName, limitType, oldTrustState ):
            setRate( mode, intfName, limitType, default=True )
      deleteConfigIfDefault( intfName )

   @staticmethod
   def handler( mode, args ):
      ConfigArpInspectionTrustCmd._configArpInspectionTrust( mode, trusted=True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ConfigArpInspectionTrustCmd._configArpInspectionTrust( mode, trusted=False )

VlanCli.SwitchportModelet.addCommandClass( ConfigArpInspectionTrustCmd )

#-------------------------------------------------------------------------------
# Set incoming ARP rate and burst interval for logging and errdisabling the
# interface
# "[no|default] ip arp inspection { logging | limit } { rate RATE
# [burst-interval INTERVAL] | none }
#-------------------------------------------------------------------------------
# TODO darylwang: Refactor this code into two separate commands for logging and
# limit. The commands should be parsed the same way as now and should not have
# any user visible differences, but it might make the code easier to read.
class ArpInspectionRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip arp inspection LIMIT_TYPE '
              '( ( rate RATE [ burst interval INTERVAL ] ) | none )' )
   noOrDefaultSyntax = 'ip arp inspection LIMIT_TYPE ...'
   data = { 'ip': CliToken.Ip.ipMatcherForConfigIf,
            'arp': arpKwMatcherConfig,
            'inspection': inspectionKwMatcherConfig,
            'LIMIT_TYPE': CliMatcher.EnumMatcher( { 
               'logging': 'Configure logging limits',
               'limit': 'Configure errdisable limits' } ), 
            'rate': 'Configure maximum incoming ARP rate',
            'RATE': CliMatcher.IntegerMatcher( 1, 2048,
                                 helpdesc='Maximum incoming ARP rate in '
                                          'packets per second' ),
            'burst': 'Configure Burst parameters for ARP packets',
            'interval': 'Number of seconds to check the rate',
            'INTERVAL': CliMatcher.IntegerMatcher( 1, 15,
                                 helpdesc='The consecutive length of time over '
                                          'which to monitor the interface' ),
            'none': 'Allow unlimited incoming ARPs',
            }

   @staticmethod
   def handler( mode, args ):
      rate = 0 if 'none' in args else args[ 'RATE' ]
      setRate( mode, mode.intf.name, args[ 'LIMIT_TYPE' ], rate,
               args.get( 'INTERVAL' ) )
      deleteConfigIfDefault( mode.intf.name )

   @staticmethod
   def noHandler( mode, args ):
      setRate( mode, mode.intf.name, args[ 'LIMIT_TYPE' ], rate=0,
               burstInterval=None, default=False )
      deleteConfigIfDefault( mode.intf.name )

   @staticmethod
   def defaultHandler( mode, args ):
      setRate( mode, mode.intf.name, args[ 'LIMIT_TYPE' ], rate=None,
               burstInterval=None, default=True )
      deleteConfigIfDefault( mode.intf.name )

VlanCli.SwitchportModelet.addCommandClass( ArpInspectionRateCmd )

#-------------------------------------------------------------------------------
# Show arp inspection configuration for the given interface
# show ip arp inspection interface [intf]
#-------------------------------------------------------------------------------
class ShowIntfArpInspectionConfigCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip arp inspection interface [ INTF ]'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'arp': arpKwMatcherShow,
            'inspection': inspectKwMatcherShow,
            'interface': 'Interface',
            'INTF': ethOrLagMatcher
            }
   cliModel = ArpInspectionInterfaces

   @staticmethod
   def handler( mode, args ):
      intfConfigs = {}
      arpInspIntfConfig = arpInspectionConfig.arpInspIntfConfig
      specificIntf = args.get( 'INTF' )
      if specificIntf:
         intfName = specificIntf.name
         if intfName in arpInspIntfConfig:
            intfs = [ intfName ]
         else:
            # Return nothing if given an unconfigured interface
            intfs = []
      else:
         intfs = arpInspIntfConfig.keys()
      for intf in intfs:
         intfConfig = arpInspIntfConfig[ intf ]
         intfConfigs[ intf ] = \
             ArpInspectionInterfaceConfig( interfaceTrusted=intfConfig.trust,
                          errdisableRateLimit=intfConfig.rateLimitErrdisable,
                      errdisableBurstInterval=intfConfig.burstIntervalErrdisable,
                          loggingRateLimit=intfConfig.rateLimitLogging,
                          loggingBurstInterval=intfConfig.burstIntervalLogging )
      return ArpInspectionInterfaces( interfaceConfigs=intfConfigs )

BasicCli.addShowCommandClass( ShowIntfArpInspectionConfigCmd )

#-------------------------------------------------------------------------------
# Add IP-MAC bindings to the IpMacBinding table
# [no|default] ip source binding IP_ADDR MAC_ADDR vlan VLAN_ID (interface
# ethernet <ethernet-interface> | port-channel <channel-no>)
#-------------------------------------------------------------------------------
class ConfigArpEntryCmd( CliCommand.CliCommandClass ):
   syntax = 'ip source binding IP_ADDR MAC_ADDR vlan VLAN_ID interface INTF'
   noOrDefaultSyntax = ( 'ip source binding IP_ADDR MAC_ADDR vlan VLAN_ID '
                         'interface INTF' )
   # XXX_darylwang Pick some better helpdesc for these keywords
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'source': 'Source',
            'binding': 'IP-MAC binding configuration',
            'IP_ADDR': IpAddrMatcher.ipAddrMatcher,
            'MAC_ADDR': MacAddr.macAddrMatcher,
            'vlan': vlanKwMatcher,
            'VLAN_ID': VlanCli.vlanIdMatcher,
            'interface': 'Interface',
            'INTF': ethOrLagMatcher,
            }

   @staticmethod
   def _validBinding( mode, args ):
      validBinding = True
      ipAddr = Tac.Value( "Arnet::IpAddr", stringValue=args.get( 'IP_ADDR' ) )
      macAddr = Tac.Value( "Arnet::EthAddr",
                           stringValue=args.get( 'MAC_ADDR' ) )
      # XXX_darylwang Check style on capitalization in error messages
      if ipAddr.isMulticast:
         mode.addError( "Ip address %s is not a unicast address." % ipAddr )
         validBinding = False
      if macAddr.isMulticast:
         mode.addError( "Mac address %s is not a unicast address." % macAddr )
         validBinding = False
      return validBinding

   @staticmethod
   def _createIpMacBindingKey( mode, args ):
      ipAddr = Tac.Value( "Arnet::IpAddr", stringValue=args.get( 'IP_ADDR' ) )
      macAddr = Tac.Value( "Arnet::EthAddr",
                           stringValue=args.get( 'MAC_ADDR' ) )
      vlanId = Tac.Value( "Bridging::VlanId",
                          value=args.get( 'VLAN_ID' ).id )
      intfId = Tac.Value( 'Arnet::IntfId', stringValue=args.get( 'INTF' ).name )
      key = Tac.Value( 'IpMacBinding::Key', macAddr, ipAddr, vlanId )
      return intfId, key

   @staticmethod
   def _createIpMacBindingEntry( mode, intfId, key ):
      bindingList = ipMacBindingTable.ipMacBinding.get( intfId )
      if bindingList is None:
         bindingList = ipMacBindingTable.newIpMacBinding( intfId )
      entry = bindingList.bindingEntry.get( key )
      if entry is None:
         entry = bindingList.newBindingEntry( key )
      entry.type = entryTypes.staticEntry

   @staticmethod
   def handler( mode, args ):
      if ConfigArpEntryCmd._validBinding( mode, args ):
         intfId, key = ConfigArpEntryCmd._createIpMacBindingKey( mode, args )
         ConfigArpEntryCmd._createIpMacBindingEntry( mode, intfId, key )
      else:
         mode.addError( "Failed to add binding; invalid arguments" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfId, key = ConfigArpEntryCmd._createIpMacBindingKey( mode, args )
      bindingList = ipMacBindingTable.ipMacBinding.get( intfId )
      if bindingList is not None:
         del bindingList.bindingEntry[ key ]
         if not bindingList.bindingEntry:
            del ipMacBindingTable.ipMacBinding[ intfId ]

BasicCli.GlobalConfigMode.addCommandClass( ConfigArpEntryCmd )

#-------------------------------------------------------------------------------
# Remove all matching IP-MAC bindings from the ARP table
# no|default ip source binding vlan VLAN_ID
#-------------------------------------------------------------------------------
class RemoveArpEntriesCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip source binding vlan VLAN_ID'

   # XXX_darylwang Pick some better helpdesc for these keywords
   data = { 'ip': CliToken.Ip.ipMatcherForConfig,
            'source': "Source",
            'binding': "IP-MAC binding configuration",
            'vlan': vlanKwMatcher,
            'VLAN_ID': VlanCli.vlanIdMatcher,
            }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for intf, bindingList in ipMacBindingTable.ipMacBinding.iteritems():
         for key in bindingList.bindingEntry.keys():
            if key.vlan == args.get( 'VLAN_ID' ).id:
               del bindingList.bindingEntry[ key ]
         if not bindingList.bindingEntry:
            del ipMacBindingTable.ipMacBinding[ intf ]

BasicCli.GlobalConfigMode.addCommandClass( RemoveArpEntriesCmd )

#-------------------------------------------------------------------------------
# Show currently configured IP-MAC bindings
# show ip source binding [ip-addr] [mac-addr] static [vlan VLAN_ID] [interface
# INTF]
#-------------------------------------------------------------------------------
class ShowArpEntryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show ip source binding [IP_ADDR] [MAC_ADDR] static
               [vlan VLAN_ID] [interface INTF]"""
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'source': "Source",
            'binding': "IP-MAC binding configuration",
            'IP_ADDR': IpAddrMatcher.ipAddrMatcher,
            'MAC_ADDR': MacAddr.macAddrMatcher,
            'static': "Statically configured binding",
            'vlan': vlanKwMatcher,
            'VLAN_ID': VlanCli.vlanIdMatcher,
            'interface': "Interface",
            'INTF': ethOrLagMatcher
            }
   cliModel = IpMacBindingTable

   @staticmethod
   def _filterEntries( ipAddr=None, macAddr=None, vlanId=None, intf=None ):
      # Currently lease times are always infinite. Will implement finite lease
      # times with dynamic ArpInsp.
      defaultLeaseTime = 0
      bindings = []
      # XXX_darylwang Find a better way to get addresses into canonical form
      if ipAddr:
         canonicalIp = str( Tac.Value( 'Arnet::IpAddr', stringValue=ipAddr ) )
      if macAddr:
         canonicalMac = str( Tac.Value( 'Arnet::EthAddr', stringValue=macAddr ) )
      for intfId, bindingList in ipMacBindingTable.ipMacBinding.iteritems():
         if intf and intfId != intf.name:
            continue
         for key, entry in bindingList.bindingEntry.iteritems():
            if ipAddr and key.ip != canonicalIp:
               continue
            if macAddr and key.mac != canonicalMac:
               continue
            if vlanId and key.vlan != vlanId.id:
               continue
            # Display static/dynamic instead of staticEntry/dynamicEntry for
            # entryType
            cliEntryType = entry.type.rstrip( 'Entry' )
            ipMacBindingEntry = IpMacBindingEntry( macAddr=key.mac,
                                                   ipAddr=key.ip,
                                                   interface=intfId,
                                                   vlanId=key.vlan,
                                                   entryType=cliEntryType,
                                                   leaseTime=defaultLeaseTime )
            bindings.append( ipMacBindingEntry )
      return bindings

   @staticmethod
   def _sortBindings( bindings ):
      bindingKey = lambda binding: ( binding.interface, binding.vlanId,
                                     binding.ipAddr, binding.macAddr )
      return sorted( bindings, key=bindingKey )

   @staticmethod
   def handler( mode, args ):
      ipAddr = args.get( 'IP_ADDR' )
      macAddr = args.get( 'MAC_ADDR' )
      vlanId = args.get( 'VLAN_ID' )
      intf = args.get( 'INTF' )
      bindingEntries = \
            ShowArpEntryCmd._filterEntries( ipAddr, macAddr, vlanId, intf )
      sortedBindings = ShowArpEntryCmd._sortBindings( bindingEntries )
      return IpMacBindingTable( bindingEntries=sortedBindings )

BasicCli.addShowCommandClass( ShowArpEntryCmd )

#-------------------------------------------------------------------------------
# Show statistics on the number of ARP packets dropped or forwarded
# show ip arp inspection statistics {vlan [VLAN_ID] | interface [INTF]}
#-------------------------------------------------------------------------------
class ShowArpInspectionStatisticsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip arp inspection statistics (vlan [VLAN_ID]) | (interface [INTF])'
   data = { 'ip': CliToken.Ip.ipMatcherForShow,
            'arp': arpKwMatcherShow,
            'inspection': inspectKwMatcherShow,
            'statistics': 'Show Arp Inspection counters',
            'vlan': vlanKwMatcher,
            'VLAN_ID': VlanCli.vlanIdMatcher,
            'interface': 'Interface',
            'INTF': ethOrLagMatcher
            }
   cliModel = ArpInspectionStatisticsCollection

   @staticmethod
   def _getArpPacketBriefModel( lastDrop ):
      # Convert timestamp to UTC
      utcTimestamp = lastDrop.timestamp + Tac.utcNow() - Tac.now()
      return ArpPacketBrief( receivedInterfaceId=lastDrop.rxIntfId,
                             receivedVlanId=lastDrop.rxVlanId,
                             timestamp=utcTimestamp,
                             sourceMac=lastDrop.srcMac,
                             destinationMac=lastDrop.dstMac,
                             arpOpcode=lastDrop.arpOpcode,
                             senderMac=lastDrop.senderMac,
                             senderIp=lastDrop.senderIp,
                             dropReason=lastDrop.dropReason )

   @staticmethod
   def _getVlanStatisticsModel( vlanId ):
      stats = vlanCounterValues( vlanId )
      lastDrop = lastVlanDroppedPacket( vlanId )
      vlanModel = VlanArpInspectionStatistics( requestsForwarded=stats.reqFwd,
                                               responsesForwarded=stats.resFwd,
                                               requestsDropped=stats.reqDrop,
                                               responsesDropped=stats.resDrop,
                                               vlanId=vlanId,
                                               name=''
                                               )
      if lastDrop:
         vlanModel.lastDroppedPacket = \
               ShowArpInspectionStatisticsCmd._getArpPacketBriefModel( lastDrop )
      return vlanModel

   @staticmethod
   def _getIntfStatisticsModel( intfId ):
      stats = intfCounterValues( intfId )
      lastDrop = lastIntfDroppedPacket( intfId )
      intfModel = IntfArpInspectionStatistics( requestsForwarded=stats.reqFwd,
                                               responsesForwarded=stats.resFwd,
                                               requestsDropped=stats.reqDrop,
                                               responsesDropped=stats.resDrop,
                                               intfId=intfId )
      if lastDrop:
         intfModel.lastDroppedPacket = \
               ShowArpInspectionStatisticsCmd._getArpPacketBriefModel( lastDrop )
      return intfModel

   @staticmethod
   def handler( mode, args ):
      if args.get( 'vlan' ):
         vlanStatistics = {}
         vlanIdArg = args.get( 'VLAN_ID' ).id if args.get( 'VLAN_ID' ) else None
         if vlanIdArg and vlanIdArg in arpInspectionStatus.arpInspVlanStatus:
            vlanStatistics[ vlanIdArg ] = \
                  ShowArpInspectionStatisticsCmd._getVlanStatisticsModel( vlanIdArg )
         else:
            for vlanId in arpInspectionStatus.arpInspVlanStatus:
               vlanStatistics[ vlanId ] = \
                     ShowArpInspectionStatisticsCmd._getVlanStatisticsModel( vlanId )
         return ArpInspectionStatisticsCollection( vlanStatistics=vlanStatistics )
      elif args.get( 'interface' ):
         intfStatistics = {}
         intfArg = args.get( 'INTF' ).name if args.get( 'INTF' ) else None
         if intfArg and intfArg in arpInspectionStatus.arpInspIntfStatus:
            intfStatistics[ intfArg ] = \
                  ShowArpInspectionStatisticsCmd._getIntfStatisticsModel( intfArg )
         else:
            for intfId in arpInspectionStatus.arpInspIntfStatus:
               intfStatistics[ intfId ] = \
                     ShowArpInspectionStatisticsCmd._getIntfStatisticsModel( intfId )
         return ArpInspectionStatisticsCollection( intfStatistics=intfStatistics )

BasicCli.addShowCommandClass( ShowArpInspectionStatisticsCmd )

#-------------------------------------------------------------------------------
# Clear Arp inspection counters
# clear arp inspection statistics
#
# legacy:
# clear ip arp inspection statistics
#-------------------------------------------------------------------------------
def clearIpArp( mode, args ):
   for vlanId, vlanStatus in arpInspectionStatus.arpInspVlanStatus.iteritems():
      vlanCkpt = getGlobalVlanCheckpoint( vlanId )
      clearCheckpoint( vlanStatus, vlanCkpt )
   for intfId, intfStatus in arpInspectionStatus.arpInspIntfStatus.iteritems():
      intfCkpt = getGlobalIntfCheckpoint( intfId )
      clearCheckpoint( intfStatus, intfCkpt )

class ClearArpInspectionStatsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp inspection statistics'
   data = {
            'clear': CliToken.Clear.clearKwNode,
            'arp': arpKwMatcherClear,
            'inspection': inspectionKwMatcherClear,
            'statistics': 'Clear ARP inspection statistics'
          }
   handler = clearIpArp

class ClearArpInspectionStatsDeprecatedCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip arp inspection statistics'
   data = {
            'clear': CliToken.Clear.clearKwNode,
            'ip': CliToken.Ip.ipMatcherForClear,
            'arp': arpKwMatcherClearDeprecated,
            'inspection': inspectionKwMatcherClear,
            'statistics': 'Clear ARP inspection statistics',
          }
   handler = clearIpArp

BasicCli.EnableMode.addCommandClass( ClearArpInspectionStatsCmd )
BasicCli.EnableMode.addCommandClass( ClearArpInspectionStatsDeprecatedCmd )

#-------------------------------------------------------------------------------
# Cleanup interface specific configuration when interface is defaulted
#-------------------------------------------------------------------------------
class ArpInspIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del arpInspectionConfig.arpInspIntfConfig[ self.intf_.name ]

def Plugin( entityManager ):
   global arpInspectionConfig, arpInspectionStatus, arpInspectionHwStatusDir
   global ipMacBindingTable, arpInspectionCheckpoint

   arpInspectionConfig = ConfigMount.mount( entityManager,
                                            "security/arpInspection/config",
                                            "ArpInsp::Config", "w" )
   arpInspectionStatus = LazyMount.mount( entityManager,
                                          "security/arpInspection/status",
                                          "ArpInsp::Status", "r" )
   arpInspectionCheckpoint = LazyMount.mount( entityManager,
                                              "security/arpInspection/"
                                              "checkpointStatus",
                                              "ArpInsp::CheckpointStatus","w" )
   arpInspectionHwStatusDir = \
       LazyMount.mount( entityManager,
                        "security/arpInspection/hardware/status",
                        "Tac::Dir", "ri" )
   ipMacBindingTable = ConfigMount.mount( entityManager,
                                          "security/ipMacBinding",
                                          "IpMacBinding::IpMacBindingDir", "w" )
   IntfCli.Intf.registerDependentClass( ArpInspIntf )
