#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
from CliPlugin.ArpInputModel import ArpInputEntry
from CliPlugin.ArpInputModel import ArpInputEntryList
from CliPlugin.ArpInputModel import ArpInputEntriesModel
import CliPlugin.VrfCli as VrfCli
from CliPlugin.VrfCli import getAllVrfNames, vrfExists, ALL_VRF_NAME

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
arpInputConfigDir = None
arpInputStatusDir = None
ipStatus = None
ip6Status = None

tacArpTypeId = Tac.Type( 'Arp::ArpType::ArpTypeId' )
tacArpType = Tac.Type( 'Arp::ArpType' )
permanent = tacArpType.idToValue( tacArpTypeId.permanent )
dynamic = tacArpType.idToValue( tacArpTypeId.dynamic )

arpSourceId = Tac.Type( 'Arp::ArpSource::ArpSourceId' )
arpSource = Tac.Type( 'Arp::ArpSource' )
cliSource = arpSource.idToValue( arpSourceId.cli )
evpnVxlanSource = arpSource.idToValue( arpSourceId.evpnVxlan )
vxlanSource = arpSource.idToValue( arpSourceId.vxlan )
arpSuppressionSource = arpSource.idToValue( arpSourceId.arpSuppression )

class ArpInputSources( object ):
   def __init__( self ):
      self._commandToken = []
      self._prettyName = []
      self._sourceDirName = []
      self._sourceName = []
      self._mapping = {}
      self._sourceNameMapping = {}

   def entryIs( self, commandToken, prettyName, sourceDirName, sourceName ):
      '''set entry for a source. '''
      if commandToken:
         self._commandToken.append( commandToken )
         self._mapping.update( { commandToken : [ prettyName, sourceDirName ] } )
      self._prettyName.append( prettyName )
      self._sourceDirName.append( sourceDirName )
      self._sourceName.append( sourceName )
      self._sourceNameMapping.update( { sourceName : prettyName } )

   def commandToken( self ):
      '''returns command token of all sources '''
      return self._commandToken

   def prettyName( self, token ):
      '''returns display name associated with command tokens'''
      info = self._mapping.get( token )
      return info[ 0 ]

   def sourceDirName( self, token ):
      '''returns display name associated with command tokens'''
      info = self._mapping.get( token )
      return info[ 1 ]

   def prettyNameFromSource( self, name ):
      '''returns display name associated with command tokens'''
      return self._sourceNameMapping.get( name )

# Populate lookup tables
arpInputSources = ArpInputSources()
arpInputSources.entryIs( 'cli', 'CLI', 'cli', cliSource )
arpInputSources.entryIs( 'evpn-vxlan', 'EvpnVxlan', 'evpnVxlan', evpnVxlanSource )
arpInputSources.entryIs( 'vxlan', 'Vxlan', 'vxlan', vxlanSource )
arpInputSources.entryIs( 'arp-suppression', 'ArpSuppression',
                         'arpSuppression', arpSuppressionSource )

# maps entry type to refresh time and gen id.
def entryTypeToInfo( entryInfo ):
   if entryInfo.type == permanent:
      return "static", "NA", "NA"
   else:
      return "dynamic", str( int( entryInfo.refreshTime ) ), str( entryInfo.genId )

def getArpInputConfigEntries ( version, source, model, vrfList ):
   if vrfList is None or source is None:
      return

   # function to get all the entry information belonging to a
   # particular source in a VRF
   def getEntryList( vrfName, version, dirName, prettyName ):
      if dirName not in arpInputConfigDir:
         return None
      if vrfName in arpInputConfigDir[ dirName ].vrf:
         entriesPerVrf = arpInputConfigDir[ dirName ].vrf[ vrfName ]
         versionEntries = ArpInputEntryList()
         ethAddr = Tac.newInstance( "Arnet::EthAddr" )
         if version == 'ipv4':
            for i in entriesPerVrf.ipv4:
               info = entriesPerVrf.ipv4[ i ]
               entryType, refreshTime, genId = entryTypeToInfo( info )
               ethAddr.stringValue = info.ethAddr
               entry = ArpInputEntry( ipAddr=info.ipAddr, intfId=info.intfId,
                                      ethAddr=ethAddr.displayString,
                                      entryType=entryType, refreshTime=refreshTime,
                                      genId=genId, source=prettyName )
               versionEntries.entrys.append( entry )
         else:
            for i in entriesPerVrf.ipv6:
               info = entriesPerVrf.ipv6[ i ]
               entryType, refreshTime, genId = entryTypeToInfo( info )
               ethAddr.stringValue = info.ethAddr
               entry = ArpInputEntry( ipAddr=info.ip6Addr.stringValue,
                                      intfId=info.intfId,
                                      ethAddr=ethAddr.displayString,
                                      entryType=entryType, refreshTime=refreshTime,
                                      genId=genId, source=prettyName )
               versionEntries.entrys.append( entry )
         return versionEntries
      return None

   model.entryMap = { vrf : ArpInputEntryList() for vrf in vrfList }
   if source == 'all':
      # get information from every ARP source
      for tok in arpInputSources.commandToken():
         dirName = arpInputSources.sourceDirName( tok )
         prettyName = arpInputSources.prettyName( tok )
         for vrfName in vrfList:
            entryList = getEntryList( vrfName, version, dirName, prettyName )
            if entryList is not None and entryList.entrys:
               model.entryMap[ vrfName ].entrys.extend( entryList.entrys )
   else:
      for vrfName in vrfList:
         dirName = arpInputSources.sourceDirName( source )
         prettyName = arpInputSources.prettyName( source )
         entryList = getEntryList( vrfName, version, dirName, prettyName )
         if entryList is not None and entryList.entrys:
            model.entryMap[ vrfName ] = entryList
   return

def showArpInputConfigEntries( mode, args ):
   if 'ipv6' in args:
      filterAf = 'ipv6'
   else:
      filterAf = 'ipv4'

   vrfName = args.get( 'VRF' )
   assert vrfName != ''
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   source = args.get( 'SOURCE' )
   if not source:
      source = "all"

   if vrfName == ALL_VRF_NAME:
      vrfList = getAllVrfNames( mode )
   elif vrfExists( vrfName ):
      vrfList = [ vrfName ]
   else:
      mode.addError( "Vrf %s does not exist" % vrfName )
      return None

   model = ArpInputEntriesModel()
   getArpInputConfigEntries( filterAf, source, model, vrfList )
   return model

def getArpInputStatusEntries ( version, intf, model, vrfList ):
   if vrfList is None:
      return

   # function to get all the entry information belonging to a
   # VRF and interface
   def getEntryList( vrfName, version, intf ):
      if vrfName in arpInputStatusDir.vrf:
         entriesPerVrf = arpInputStatusDir.vrf[ vrfName ]
         versionEntries = ArpInputEntryList()
         ethAddr = Tac.newInstance( "Arnet::EthAddr" )
         if version == 'ipv4':
            for intfId in entriesPerVrf.ipv4:
               # filter based on interface if given
               if intf is None or intfId == intf.name:
                  for i in entriesPerVrf.ipv4[ intfId ].arpEntry:
                     info = entriesPerVrf.ipv4[ intfId ].arpEntry[ i ]
                     entryType, refreshTime, genId = entryTypeToInfo( info )
                     prettyName = arpInputSources.prettyNameFromSource(
                                                 info.source )
                     ethAddr.stringValue = info.ethAddr
                     entry = ArpInputEntry( ipAddr=info.ipAddr, intfId=info.intfId,
                                            ethAddr=ethAddr.displayString,
                                            entryType=entryType,
                                            refreshTime=refreshTime,
                                            genId=genId, source=prettyName )
                     versionEntries.entrys.append( entry )
         else:
            for intfId in entriesPerVrf.ipv6:
               # filter based on interface if given
               if intf is None or intfId == intf.name:
                  for i in entriesPerVrf.ipv6[ intfId ].neighborEntry:
                     info = entriesPerVrf.ipv6[ intfId ].neighborEntry[ i ]
                     entryType, refreshTime, genId = entryTypeToInfo( info )
                     prettyName = arpInputSources.prettyNameFromSource(
                                                 info.source )
                     ethAddr.stringValue = info.ethAddr
                     entry = ArpInputEntry( ipAddr=info.ip6Addr.stringValue,
                                            intfId=info.intfId,
                                            ethAddr=ethAddr.displayString,
                                            entryType=entryType,
                                            refreshTime=refreshTime,
                                            genId=genId, source=prettyName )
                     versionEntries.entrys.append( entry )
         return versionEntries
      return None

   model.entryMap = { vrf : ArpInputEntryList() for vrf in vrfList }
   for vrfName in vrfList:
      if vrfName in arpInputStatusDir.vrf:
         entryList = getEntryList( vrfName, version, intf )
         if entryList is not None and entryList.entrys:
            model.entryMap[ vrfName ] = entryList
   return

def showArpInputStatusEntries( mode, args ):
   if 'ipv6' in args:
      filterAf = 'ipv6'
   else:
      filterAf = 'ipv4'
   intf = args.get( 'INTF' )
   vrfName = args.get( 'VRF' )
   assert vrfName != ''
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if intf is not None:
      if not intf.lookup():
         mode.addError( "Interface does not exist" )
         return None
      if filterAf == 'ipv4':
         ipIntfStatus = ipStatus.ipIntfStatus.get( intf.name_ )
      else:
         ipIntfStatus = ip6Status.ipIntfStatus( intf.name_ )
      if not ipIntfStatus:
         mode.addError( "%s is not a layer 3 interface" % intf.name_ )
         return None
      if vrfName != ALL_VRF_NAME and vrfName != ipIntfStatus.vrf:
         mode.addError( "Interface VRF does not match" )
         return None

   if vrfName == ALL_VRF_NAME:
      vrfList = getAllVrfNames( mode )
   elif vrfExists( vrfName ):
      vrfList = [ vrfName ]
   else:
      mode.addError( "VRF %s does not exist" % vrfName )
      return None

   model = ArpInputEntriesModel()
   getArpInputStatusEntries( filterAf, intf, model, vrfList )
   return model

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global arpInputConfigDir
   global arpInputStatusDir
   global ipStatus
   global ip6Status

   ipStatus = LazyMount.mount( entityManager, "ip/status", "Ip::Status", "r" )
   ip6Status = LazyMount.mount( entityManager, "ip6/status", "Ip6::Status", "r" )
   arpInputConfigDir = LazyMount.mount( entityManager, "arp/input/config",
                                        "Tac::Dir", "ri" )
   arpInputStatusDir = LazyMount.mount( entityManager, "arp/input/status",
                                        "Arp::ArpInputStatus", "ri" )
