#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.ArpIp6Cli as ArpIp6Cli
import CliPlugin.ArpIpCli as ArpIpCli
import CliPlugin.VrfCli as VrfCli
from CliPlugin.IntfCli import Intf
from CliPlugin.Ip6AddrMatcher import Ip6AddrMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
from CliToken.Clear import clearKwNode
from CliToken.Ip import ipMatcherForRefresh, ipMatcherForClear
from CliToken.Ipv6 import ipv6MatcherForRefresh, ipv6MatcherForClear
from CliToken.Refresh import refreshMatcherForExec

matcherArp = CliMatcher.KeywordMatcher(
   'arp', helpdesc='Clear dynamic ARP entry by IP' )
matcherArpCacheClear = CliMatcher.KeywordMatcher(
   'arp-cache', helpdesc='Clear dynamic entries from the ARP table' )
matcherArpCacheRefresh = CliMatcher.KeywordMatcher(
   'arp-cache', helpdesc='Refresh dynamic entries in the ARP table' )
matcherNeighborsRefresh = CliMatcher.KeywordMatcher(
   'neighbors', helpdesc='Neighbor entries to refresh' )
matcherNeighborsClear = CliMatcher.KeywordMatcher(
   'neighbors', helpdesc='Neighbor entries to clear' )
vrfExprFactoryClearArp = VrfCli.VrfExprFactory(
   helpdesc='Clear dynamic ARP entries in a VRF' )
vrfExprFactoryClearNbr = VrfCli.VrfExprFactory(
   helpdesc='Clear dynamic Neighbor entries in a VRF' )
vrfExprFactoryRefreshArp = VrfCli.VrfExprFactory(
   helpdesc='Refresh dynamic ARP entries in a VRF' )
vrfExprFactoryRefreshNbr = VrfCli.VrfExprFactory(
   helpdesc='Refresh dynamic Neighbor entries in a VRF' )

#--------------------------------------------------------------------------------
# clear arp-cache [ VRF ]
#--------------------------------------------------------------------------------
class ClearArpCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp-cache [ VRF ]'
   data = {
      'clear': clearKwNode,
      'arp-cache': matcherArpCacheClear,
      'VRF': vrfExprFactoryClearArp,
   }
   handler = ArpIpCli.clearArpCache

BasicCliModes.EnableMode.addCommandClass( ClearArpCacheCmd )

#--------------------------------------------------------------------------------
# clear arp-cache [ VRF ] interface INTF
#--------------------------------------------------------------------------------
class ClearArpCacheInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp-cache [ VRF ] interface INTF'
   data = {
      'clear': clearKwNode,
      'arp-cache': matcherArpCacheClear,
      'VRF': vrfExprFactoryClearArp,
      'interface': 'Clear dynamic entries from the ARP table by interface',
      'INTF': Intf.matcher,
   }
   handler = ArpIpCli.clearArpInterface

BasicCliModes.EnableMode.addCommandClass( ClearArpCacheInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# clear arp [ VRF ] IPADDR
#--------------------------------------------------------------------------------
class ClearArpCmd( CliCommand.CliCommandClass ):
   syntax = 'clear arp [ VRF ] IPADDR'
   data = {
      'clear': clearKwNode,
      'arp': matcherArp,
      'VRF': vrfExprFactoryClearArp,
      'IPADDR': IpAddrMatcher( helpdesc='Address to be cleared from ARP table' ),
   }
   handler = ArpIpCli.clearIpArp

BasicCliModes.EnableMode.addCommandClass( ClearArpCmd )

#--------------------------------------------------------------------------------
# clear ip arp [ VRF ] IPADDR
#--------------------------------------------------------------------------------
class ClearIpArpCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ip arp [ VRF ] IPADDR'
   data = {
      'clear': clearKwNode,
      'ip': ipMatcherForClear,
      'arp': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'arp',
         helpdesc='Clear dynamic ARP entry by IP' ),
                              deprecatedByCmd='clear arp' ),
      'VRF': vrfExprFactoryClearArp,
      'IPADDR': IpAddrMatcher( helpdesc='Address to be cleared from ARP table' ),
   }
   handler = ArpIpCli.clearIpArp

BasicCliModes.EnableMode.addCommandClass( ClearIpArpCmd )

#--------------------------------------------------------------------------------
# clear ipv6 neighbors [ VRF ]
#--------------------------------------------------------------------------------
class ClearIpv6NeighborsCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 neighbors [ VRF ]'
   data = {
      'clear': clearKwNode,
      'ipv6': ipv6MatcherForClear,
      'neighbors': matcherNeighborsClear,
      'VRF': vrfExprFactoryClearNbr,
   }
   handler = ArpIp6Cli.clearAllIpv6Neighbors

BasicCliModes.EnableMode.addCommandClass( ClearIpv6NeighborsCmd )

#--------------------------------------------------------------------------------
# clear ipv6 neighbors [ VRF ] INTF [ INTF ]
#--------------------------------------------------------------------------------
class ClearIpv6NeighborsIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ipv6 neighbors [ VRF ] INTF [ IPV6ADDR ]'
   data = {
      'clear': clearKwNode,
      'ipv6': ipv6MatcherForClear,
      'neighbors': matcherNeighborsClear,
      'VRF': vrfExprFactoryClearNbr,
      'INTF': Intf.matcher,
      'IPV6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address of neighbor' ),
   }
   handler = ArpIp6Cli.clearIntfIpv6Neighbors

BasicCliModes.EnableMode.addCommandClass( ClearIpv6NeighborsIntfCmd )

#--------------------------------------------------------------------------------
# refresh arp-cache [ VRF ]
#--------------------------------------------------------------------------------
class RefreshArpCacheCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh arp-cache [ VRF ]'
   data = {
      'refresh': refreshMatcherForExec,
      'arp-cache': matcherArpCacheRefresh,
      'VRF': vrfExprFactoryRefreshArp,
   }
   handler = ArpIpCli.refreshArpByVrf

BasicCliModes.EnableMode.addCommandClass( RefreshArpCacheCmd )

#--------------------------------------------------------------------------------
# refresh arp-cache [ VRF ] interface INTF
#--------------------------------------------------------------------------------
class RefreshArpCacheInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh arp-cache [ VRF ] interface INTF'
   data = {
      'refresh': refreshMatcherForExec,
      'arp-cache': matcherArpCacheRefresh,
      'VRF': vrfExprFactoryRefreshArp,
      'interface': 'Refresh dynamic entries from the ARP table by interface',
      'INTF': Intf.matcher,
   }
   handler = ArpIpCli.refreshArpByIntf

BasicCliModes.EnableMode.addCommandClass( RefreshArpCacheInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# refresh ip arp [ VRF ] IPADDR
#--------------------------------------------------------------------------------
class RefreshIpArpCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ip arp [ VRF ] IPADDR'
   data = {
      'refresh': refreshMatcherForExec,
      'ip': ipMatcherForRefresh,
      'arp': 'Refresh dynamic ARP entry by IP',
      'VRF': vrfExprFactoryRefreshArp,
      'IPADDR': IpAddrMatcher( helpdesc='Address to be refreshed in ARP table' ),
   }
   handler = ArpIpCli.refreshIpArp

BasicCliModes.EnableMode.addCommandClass( RefreshIpArpCmd )

#--------------------------------------------------------------------------------
# verify ip arp ( all | ( IPADDR VRF ) )
#--------------------------------------------------------------------------------
class VerifyArpCmd( CliCommand.CliCommandClass ):
   syntax = 'verify ip arp ( all | ( IPADDR VRF ) )'
   data = {
      'verify': "Verify the state of an object in the system",
      'ip': 'Verify the state of an IP object',
      'arp': 'Verify the state of an arp entry',
      'all': 'Verify the state for all arps in the system',
      'IPADDR': IpAddrMatcher( helpdesc='Address to be verified in ARP table' ),
      'VRF': VrfCli.vrfMatcher,
   }
   hidden = True
   handler = ArpIpCli.verifyArp

BasicCliModes.EnableMode.addCommandClass( VerifyArpCmd )

#--------------------------------------------------------------------------------
# refresh ipv6 neighbors [ VRF ]
#--------------------------------------------------------------------------------
class RefreshIpv6NeighborsCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ipv6 neighbors [ VRF ]'
   data = {
      'refresh': refreshMatcherForExec,
      'ipv6': ipv6MatcherForRefresh,
      'neighbors': matcherNeighborsRefresh,
      'VRF': vrfExprFactoryRefreshNbr,
   }
   handler = ArpIp6Cli.refreshIpv6NeighborsByVrf

BasicCliModes.EnableMode.addCommandClass( RefreshIpv6NeighborsCmd )

#--------------------------------------------------------------------------------
# refresh ipv6 neighbors [ VRF ] INTF [ IPV6ADDR ]
#--------------------------------------------------------------------------------
class RefreshIpv6NeighborsIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'refresh ipv6 neighbors [ VRF ] INTF [ IPV6ADDR ]'
   data = {
      'refresh': refreshMatcherForExec,
      'ipv6': ipv6MatcherForRefresh,
      'neighbors': matcherNeighborsRefresh,
      'VRF': vrfExprFactoryRefreshNbr,
      'INTF': Intf.matcher,
      'IPV6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address of neighbor' ),
   }
   handler = ArpIp6Cli.refreshIpv6NeighborsByIntfOrAddr

BasicCliModes.EnableMode.addCommandClass( RefreshIpv6NeighborsIntfCmd )

#--------------------------------------------------------------------------------
# verify ipv6 neighbor ( all | ( IPV6ADDR VRF ) )
#--------------------------------------------------------------------------------
class VerifyNeighborCmd( CliCommand.CliCommandClass ):
   syntax = 'verify ipv6 neighbor  ( all | ( IPV6ADDR VRF ) )'
   data = {
      'verify': 'Verify the state of an object in the system',
      'ipv6': 'Verify the state of an IPv6 object',
      'neighbor': 'Verify the state of an neighbor entry',
      'all': 'Verify the state for all neighbors in the system',
      'IPV6ADDR': Ip6AddrMatcher( helpdesc='IPv6 address of neighbor' ),
      'VRF': VrfCli.vrfMatcher,
   }
   hidden = True
   handler = ArpIp6Cli.verifyNeighbor

BasicCliModes.EnableMode.addCommandClass( VerifyNeighborCmd )
