# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import SpaceMgmtLib.Utils
from CliModel import (
   Model,
   Bool,
   Int,
   Str,
)

class ArchiveModel( Model ):
   """
   The file archive model used to be used as a json object with Capi.
   """

   __revision__ = 2

   # General information
   destName = Str( help='Nickname of the archival destination.' )
   destPath = Str( help='Absolute path of the archival destination.' )
   shutdown = Bool( help='True if file archival is disabled.' )
   exists = Bool( help='The archive symlink and directory tree exist'
                       ' and have the right permissions.' )
   fileArchivalEnabled = Bool( "EOS file archival is enabled" )
   # Destination device information:
   devName = Str( help='Name of the device containing the archival destination.' )
   devPresent = Bool( help='The device is present on the switch.' )
   devPhysical = Bool( help='The device is physical.' )
   devMounted = Bool( help='The installed device is mounted.' )
   devForceDisabled = Bool( help='Using the device is not allowed.'
                                 ' This is only relevant of the ssd drive.' )
   # Filesystem information:
   fsMntPt = Str( help='Mountpoint of the filesystem.' )
   fsSize = Int( help='Size of the filesystem in Kibibytes.' )
   fsUsedKiB = Int( help='Filesystem used space in Kibibytes.' )
   fsUsedPct = Int( help='Filesystem used space as a percentage.' )
   fsQuotaMount = Bool( help='The filesystem is mounted with quotas.' )
   # Quota information:
   quotaOn = Bool( help='Quota are enabled for the filesystem.' )
   quotaKiB = Int( help='Quota limit in Kibibytes.' )
   quotaPct = Int( help='Quota limit as a percentage.' )
   # Archive space usage information
   usedKiB = Int( help='Archive space usage in Kibibytes.' )
   usedPct = Int( help='Archive space usage as a percentage.' )

   def degrade( self, dictRepr, revision ):
      # Previous model attributes:
      #  ssdPhysical = Bool( help='Whether an SSD device is physical or virtual' )
      #  ssdDisabled = Bool( help='Whether the device has been forced disabled' )
      #  ssdPresent = Bool( help='Whether an SSD device is installed on switch' )
      #  ssdMounted = Bool( help='Whether the SSD installed device is mounted' )
      #  inUsePct = Int( help='Percentage of total SSD space in use' )
      #  aristaPct = Int( help='Percentage of SSD space in use by EOS (quotapct)' )
      #  resvPct = Int( help='Maximum Percentage of SSD space reserved for EOS use' )
      #  devSize = Int( help='Total SSD device size, sized in KB units' )
      #  inUse = Int( help='Total SSD device space in use in KB units' )
      #  aristaUse = Int( help='EOS used SSD device space in KB units' )
      #  resvUse = Int( help='EOS reserved SSD device space in KB units' )
      #  shutdown = Bool( help='Whether file archival is shutdown or not (active)' )

      if revision == 1:
         oldModel = {
            'ssdPhysical' : dictRepr[ 'devPhysical' ],
            'ssdDisabled' : dictRepr[ 'devForceDisabled' ],
            'ssdPresent' : dictRepr[ 'devPresent' ],
            'ssdMounted' : dictRepr[ 'devMounted' ],
            'inUsePct' : dictRepr[ 'fsUsedPct' ],
            'aristaPct' : dictRepr[ 'usedPct' ],
            'resvPct' : dictRepr[ 'quotaPct' ],
            'devSize' : dictRepr[ 'fsSize' ],
            'inUse' : dictRepr[ 'fsUsedKiB' ],
            'aristaUse' : dictRepr[ 'usedKiB' ],
            'resvUse' : dictRepr[ 'quotaKiB' ],
            'shutdown' : dictRepr[ 'shutdown' ],
         }

         # old model was only for the ssd
         if dictRepr[ 'destPath' ] != '/mnt/drive':
            d = dict.fromkeys( oldModel, None )
            d[ 'ssdMounted' ] = False
            d[ 'shutdown' ] = dictRepr[ 'shutdown' ]
         return oldModel
      else:
         return dictRepr

   def render( self ):

      outputFmt = (
         "File archival status: {statusStr}\n"
         "Archive destination: {destName}\n"
         "Archive destination path: {destPath}\n"
         "Archive exists: {exists}\n"
         "Device: {devName}\n"
         "Device status: {devStatusStr}\n"
         "Filesystem size: {fsSizeStr}\n"
         "Total space usage: {fsUsedStr} ({fsUsedPct}%)\n"
         "EOS space usage: {usedStr} ({usedPct}%)\n"
         "Quota status: {quotaStatusStr}\n"
         "Reserved quota: {quotaKiBStr}{quotaPctStr}"
      )

      def devStatusStr():
         if not self.devPhysical:
            return 'Virtual device'
         installed = '%sInstalled' % ( 'Not ' if not self.devPresent else '' )
         if self.devMounted:
            quotaMountPrefix = 'with' if self.fsQuotaMount else 'without'
            mounted = ( 'Mounted on %s %s quota mount options'
                        % ( self.fsMntPt, quotaMountPrefix ) )
         else:
            mounted = 'Not Mounted'
         statusStr = '%s, %s' % ( installed, mounted )
         if self.devForceDisabled:
            statusStr += ', Force disabled'
         return statusStr

      quotaKiBStr = None
      if self.quotaKiB is not None:
         quotaKiBStr = SpaceMgmtLib.Utils.scaleSizeFromKiB( self.quotaKiB )

      quotaPctStr = ''
      if self.quotaPct is not None:
         quotaPctStr = ' (%s%%)' % self.quotaPct

      fmtInfo = {
         'statusStr' : "Enabled" if self.fileArchivalEnabled else "Disabled",
         'destName' : self.destName,
         'destPath' : '/archive -> %s' % self.destPath,
         'exists' : self.exists,
         'devName' : self.devName,
         'devStatusStr' : devStatusStr(),
         'fsSizeStr' : SpaceMgmtLib.Utils.scaleSizeFromKiB( self.fsSize ),
         'fsUsedStr' : SpaceMgmtLib.Utils.scaleSizeFromKiB( self.fsUsedKiB ),
         'fsUsedPct' : self.fsUsedPct,
         'usedStr' : SpaceMgmtLib.Utils.scaleSizeFromKiB( self.usedKiB ),
         'usedPct' : self.usedPct,
         'quotaStatusStr' : 'Enabled' if self.quotaOn else 'Disabled',
         'quotaKiBStr' : quotaKiBStr,
         'quotaPctStr' : quotaPctStr
      }

      print outputFmt.format( **fmtInfo )
