# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ArTogglesPyAgent as tg
import BasicCli
import CliMatcher
import CliModel
import CliPlugin.TechSupportCli
import ShowCommand
import TableOutput

class ArToggleEntry( CliModel.Model ):
   default = CliModel.Int( help="Is this toggle enabled by default" )
   enabled = CliModel.Int( help="Is this toggle enabled" )

class ArToggleInfo( CliModel.Model ):
   toggleInfo = CliModel.Dict( help="Collection of feature toggles" , 
                               keyType=str, valueType=ArToggleEntry )

   def render( self ):
      table = TableOutput.createTable( [ "Toggle Name", "Default", "Enabled" ], 
                                       tableWidth=130 )
      formats = []
      justify = [ "left", "right", "right" ]
      for col in xrange( 0, 3 ):
         f = TableOutput.Format( justify=justify[ col ], minWidth=5 )
         f.noPadLeftIs( True )
         f.padLimitIs( True )
         f.borderStartPosIs( 3 )
         formats.append( f )
      table.formatColumns( *formats )

      for toggleName, toggle in sorted( self.toggleInfo.iteritems() ):
         table.newRow( toggleName, toggle[ 'default' ], toggle[ 'enabled' ] ) 

      print table.output()

class ArToggleOverrideStatus( CliModel.Model ):
   name = CliModel.Str( help="Describes this status attribute" )
   status = CliModel.Enum( help="Status of toggle override file parse", \
                           values=( "OK", "NA", "ERROR" ) )

   def render( self ):
      frmt = "%-30s %s"
      print frmt % ( "Toggle override status:", self.status )

matcherFeatureToggle = CliMatcher.KeywordMatcher( 'feature-toggle',
      helpdesc='Display feature toggle information' )

#--------------------------------------------------------------------------------
# show feature-toggle
#--------------------------------------------------------------------------------
class FeatureToggleCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show feature-toggle'
   data = {
      'feature-toggle' : matcherFeatureToggle,
   }
   cliModel = ArToggleInfo
   hidden = True
   privileged = True

   @staticmethod
   def handler( mode, args ):
      toggles = tg.pyGetAllToggles()
      toggleInfoObj = ArToggleInfo()
      toggleInfoObj.toggleInfo = {}

      for toggle in toggles:
         toggleEntry = ArToggleEntry()
         toggleEntry.enabled = toggle[ 'enabled' ]
         toggleEntry.default = toggle[ 'default' ]
         toggleInfoObj.toggleInfo[ toggle[ 'name' ] ] = toggleEntry

      return toggleInfoObj

BasicCli.addShowCommandClass( FeatureToggleCmd )

#--------------------------------------------------------------------------------
# show feature-toggle override status
#--------------------------------------------------------------------------------
class FeatureToggleOverrideStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show feature-toggle override status'
   data = {
      'feature-toggle' : matcherFeatureToggle,
      'override' : 'Display feature toggle override parse status',
      'status' : 'Display feature toggle override parse status',
   }
   cliModel = ArToggleOverrideStatus
   hidden = True
   privileged = True

   @staticmethod
   def handler( mode, args ):
      override_status = tg.pyGetOverrideStatus()
      overrideInfoObj = ArToggleOverrideStatus()

      overrideInfoObj.name = "ArToggles override file status"
      if override_status == tg.pyToggleStatusOK():
         overrideInfoObj.status = "OK"
      elif override_status == tg.pyToggleStatusERROR():
         overrideInfoObj.status = "ERROR"
      elif override_status == tg.pyToggleStatusNA():
         overrideInfoObj.status = "NA"
      else:
         assert False, "Internal override_status should always be: {OK, ERROR, NA}"

      return overrideInfoObj

BasicCli.addShowCommandClass( FeatureToggleOverrideStatusCmd )

#includes "show feature-toggle override status" into "show tech" commands
def _showTechCmds():
   cmds = [ 'show feature-toggle override status',
            'show feature-toggle' ]
   return cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( 
      '2016-09-30 15:03:06', _showTechCmds )

