# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.


# CliPlugin module for BGP configuration commands

# pylint: disable-msg=W0611
import AgentCommandRequest
import ArBgpCliCommon
import Arnet
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.RcfCliLib as RcfCliLib
import CliPlugin.VrfCli as VrfCli
from CliPlugin import ShowTaskSchedulerModel
import DynamicPrefixListModel
import Ip6AddrMatcher
import IpAddrMatcher
import LazyMount
import ShowCommand
import StringIO
import Tac
import TechSupportCli
import Toggles.BgpCommonToggleLib
import Toggles.ArBgpToggleLib
import Toggles.RoutingLibToggleLib
import Tracing
import Url
import difflib
import itertools
import simplejson as json

from AgentSnapshotCore import snapshotCoreCallBack
from ArBgpAgent import BgpAgentName
from ArnetLib import bgpFormatAsn, formatRd
from BgpCliModels import (
   ShowBgpConvergenceModel,
   ShowBgpInstanceModel,
   ShowNeighborsHistoryModel,
)
from ArBgpCliModels import (
   ShowBgpRcfModel,
   ShowRtMembershipFloodlistModel,
   ShowRtMembershipNeighborFilterModel,
)
from BgpLib import (
        PeerConfigKey,
        bgpConfigAttrAfList,
        createKey,
        doLogClearIpBgp,
        isValidV6PeerAddr,
)
from BgpMaintenanceModels import BgpMaintenanceStatus, BgpPeerMaintenanceStatus
from BgpMaintenanceModels import BgpPeerMaintenanceInfo, BgpMaintenanceStatusPerPeer
from BgpMaintenanceShowCli import BgpMaintenanceStatusVrf, BgpPeerMaintenanceInfoVrf
from BgpMaintenanceShowCli import initiatorRM, getBgpGroupProfileInfo, getVrf
from CliModel import cliPrinted
from DynamicPrefixListCli import routeMapShowCmdDict
from IpLibConsts import ALL_VRF_NAME, DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from MaintenanceModels import MaintenanceInterfaceBgpPeerStatus
from MaintenanceModels import MaintenanceInterfaceBgpStatus
from RouteMapLib import matchPermitEnum, printRouteMapEntryAttributes
from RoutingBgpCli import peergroupNameRegex
from IraServiceCli import getEffectiveProtocolModel

import CliToken
from CliToken.Ip import ipMatcherForShow
from CliToken import RoutingBgpShowCliTokens
from CliToken.RoutingBgpShowCliTokens import (
   active,
   aspath,
   aspathRegexMatcher,
   bgpAfterShow,
   commRegexp,
   detailConfig,
   regexp,
   scheduler,
   schedulerVerbose,
   showCommunity,
   unsupported,
)

from CliSavePlugin.RoutingBgpCliSave import (
   saveAfConfig,
   saveAfConfigCallbackDict,
   showBgpConfigInstanceCallbacks,
   saveBgpConfig,
   saveBgpVpnAfConfig,
   saveBgpVpnConfig,
   saveUcmpConfig,
   getLsImportCmds,
)

from RouteMapCli import (
   CommonTokens as RouteMapCommonTokens,
   RtSooExtCommCliMatcher,
   policyEvaluationLogsContainerModel,
   routeMapContainerModel,
)

from RouteMapCliModels import (
   PolicyEvalMultiPeersNlriResult,
   PolicyEvalMultiSourceNlriResult,
)

from RoutingBgpShowCli import (
        ArBgpShowOutput,
        allVrfExprFactory,
        arBgpShowCmdDict,
        doShowBgpOutputNotSupported,
        getCommunityValuesScalarList,
        getNlriAfForPfxAddrRule,
        getVrfsFunc,
        peerGroupVrfModel,
        peerListVrfModel,
        queuedWithdrawalsVrfModel,
        routeSummaryVrfModel,
        showBgpCombinedRegexpHelper,
        statisticsVrfModel,
        summaryVrfModel,
        updateGroupVrfModel,
        updateErrorsVrfModel,
)

traceHandle = Tracing.Handle( 'ArBgpCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

notSupported = "Not supported"
routingHwStatus = None
allVrfConfig = None
config = None
vrfConfigDir = None
mapConfig = None
asnConfig = None
arBgpMapConfig = None
arBgpConfig = None
arBgpVrfConfigDir = None
defaultIntiatorRmDir = None
allIntfStatusDir = None
defaultInitiatorRmDir = None
l3Config = None
securityConfig = None
vrfNameStatus = None
dynPfxListConfigDir = None
dynPfxListStatusDir = None
allVrfRmStatusSet = None
ucmpConfig = None
ucmpVrfConfigDir = None
arUcmpConfig = None
arUcmpVrfConfigDir = None
aclCpConfig = None
rdConfigDir = None
lsImportConfig = None

# By default show commands execute in a forked process. This allows
# the BgpCore task scheduler to continue normal execution while the
# show command is in progress.
bgpShowCmdUseFork = True
# pylint: disable-msg=E0602

class ArBgpShowForkToggleCmd( CliCommand.CliCommandClass ):
   syntax = 'bgp-show-fork ( enable | disable )'
   data = {
      'bgp-show-fork' :
         'Control whether ArBgp show commands execute in a forked child process',
      'enable' : 'Execute show command in a forked child',
      'disable' : 'Execute show command in the main ArBgp process',
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      global bgpShowCmdUseFork
      bgpShowCmdUseFork = 'enable' in args

BasicCli.EnableMode.addCommandClass( ArBgpShowForkToggleCmd )

def peerGroupNames( mode ):
   return ( x.group for x in config.neighborConfig if x.type == 'peerGroup' )

class CommonTokens( object ):
   '''
   This class is intended to keep common tokens that may be shared across several
   commands with the new CLI parser.
   '''

   showBgp = CliMatcher.KeywordMatcher( 'bgp', helpdesc='BGP information' )
   neighbors = RoutingBgpShowCliTokens.neighbors
   neighborAddr = IpAddrMatcher.IpAddrMatcher( 'Neighbor address' )
   neighborAddr6 = Ip6AddrMatcher.Ip6AddrMatcher( 'Neighbor address' )
   prefixExpr = IpAddrMatcher.ipPrefixExpr(
      'IPv4 address', 'Subnet mask', 'IPv4 prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   prefix6Expr = Ip6AddrMatcher.ip6PrefixExpr(
      'IPv6 address', 'Subnet mask', 'IPv6 prefix',
      overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )
   neighborPrefix = IpAddrMatcher.IpPrefixMatcher(
      'Neighbors covered by prefix' )
   neighborPrefix6 = Ip6AddrMatcher.Ip6PrefixMatcher(
      'Neighbors covered by prefix' )
   peerGroup = CliMatcher.KeywordMatcher( 'peer-group',
                                          helpdesc='Filter by peer group name' )
   peerGroupName = CliMatcher.DynamicNameMatcher( peerGroupNames,
                                                  'Name of the peer group',
                                                  helpname='WORD',
                                                  pattern=peergroupNameRegex )
   vrf = VrfCli.VrfExprFactory(
      helpdesc='Virtual routing and forwarding instance',
      inclDefaultVrf=True, inclAllVrf=True )
   rtMembership = CliMatcher.KeywordMatcher(
      'rt-membership', helpdesc='Route Target Membership address family' )

class ArBgpCliCommand( object ):
   # Note: the following demilimiter fields must be kept in sync with
   #       BgpCommandParamsBase::commandDelimiter() and
   #       BgpCommandParamsBase::delimiterReplace().
   # Warning: do not use the Tac type to extract these fields as it will lead
   #       to the dlopen('libArBgp') which might take up to 9 seconds and also
   #       unnecessarily take more memory inside ConfigAgent.
   # Todo: BgpCommandParamsBase should be moved into its own lib (or with more
   #       types that are only relevant to the CLI rendering).
   #       It was very tempting to move it into libArBgpCommunity.so but that
   #       would end up generating more confusion.
   Delimiter = '&'
   DelimiterReplace = '\t'

   def __init__( self, mode, command, vrfName=None, nlriAfiSafi=None,
                 extraNlriAfiSafis=None, nlriType=None, transportAfi=None,
                 prefix=None, peerAddr=None, routerId=None,
                 disableFork=False, revision=1 ):
      self.delimiter = ArBgpCliCommand.Delimiter
      self.replace = ArBgpCliCommand.DelimiterReplace
      self.entityManager = mode.entityManager
      self.command = command
      self.params = []
      self.addParam( 'fork', bgpShowCmdUseFork and not disableFork )
      # Process common parameters
      if vrfName:
         self.addParam( 'vrf', vrfName )
      self.addNlriAfiSafi( nlriAfiSafi )
      if extraNlriAfiSafis:
         for afiSafi in extraNlriAfiSafis:
            self.addParam( 'extraNlriAfiSafis', afiSafi )
      if nlriType:
         self.addParam( 'nlriType', nlriType )
      if transportAfi:
         if transportAfi == 'ip':
            transportAfi = 'ipv4'
         self.addParam( 'transportAfi', transportAfi )
      if prefix:
         self.addParam( 'addr', prefix )
      if peerAddr:
         self.addParam( 'peerAddr', peerAddr )
      if routerId:
         self.addParam( 'routerId', routerId )

   def addNlriAfiSafi( self, nlriAfiSafi ):
      if not nlriAfiSafi:
         return
      if nlriAfiSafi in ( 'ip', 'ipv4' ):
         nlriAfiSafi = 'ipv4Unicast'
      elif nlriAfiSafi == 'ipv6':
         nlriAfiSafi = 'ipv6Unicast'
      self.addParam( 'nlriAfiSafi', nlriAfiSafi )

   def addParam( self, keyword, argument=None ):
      if argument is None:
         self.params.append( '%s' % ( keyword ) )
      else:
         self.params.append( '%s=%s' % ( keyword, argument ) )

   def updateParam( self, keyword, oldValue, newValue ):
      if oldValue == newValue:
         # Param is already set to intended value
         return
      param = '%s=%s' % ( keyword, oldValue )
      self.params.remove( param )
      # All the core params (e.g. 'fork') should be in the beginning
      param = '%s=%s' % ( keyword, newValue )
      self.params.insert( 0, param )

   def paramString( self ):
      ps = [ str( p ).replace( self.delimiter, self.replace ) \
             for p in self.params ]
      return str( self.delimiter ).join( ps )

   def run( self, **kwArgs ):
      t0( "Running {!r} {!r}".format( self.command, self.params ) )
      AgentCommandRequest.runSocketCommand( self.entityManager, BgpAgentName,
                                            self.command,
                                            self.paramString(), **kwArgs )

# Support commands that use TaskScheduler and yield after some time quantum or
# if the output buffer is full. This command class will disable 'fork' and
# enable 'keepalive' so the Bgp agent can keep the socket alive between task runs.
class ArBgpAsyncCliCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      # Pass common parameters on specifically.
      vrfName = kwargs.pop( 'vrfName', None )
      nlriAfiSafi = kwargs.pop( 'nlriAfiSafi', None )
      extraNlriAfiSafis = kwargs.pop( 'extraNlriAfiSafis', None )
      nlriType = kwargs.pop( 'nlriType', None )
      transportAfi = kwargs.pop( 'transportAfi', None )
      prefix = kwargs.pop( 'prefix', None )
      peerAddr = kwargs.pop( 'peerAddr', None )
      routerId = kwargs.pop( 'routerId', None )
      super( ArBgpAsyncCliCommand, self ).__init__(
         mode, command, vrfName=vrfName,
         nlriAfiSafi=nlriAfiSafi,
         extraNlriAfiSafis=extraNlriAfiSafis,
         nlriType=nlriType,
         transportAfi=transportAfi,
         prefix=prefix,
         peerAddr=peerAddr,
         routerId=routerId,
         disableFork=True )
      self.mode = mode

      # For whichever parameters are left, add to our list.
      for k, v in kwargs.iteritems():
         self.addParam( k, v )

   def run( self, **kwArgs ):
      connErrMsg = 'Bgp agent is inactive'
      AgentCommandRequest.runCliPrintSocketCommand( self.entityManager, BgpAgentName,
                                                    self.command, self.paramString(),
                                                    self.mode, keepalive=True,
                                                    connErrMsg=connErrMsg,
                                                    **kwArgs )

#-------------------------------------------------------------------------------
# "show bgp instance [vrf <vrfName>]
#     (hidden options) [internal [dump-duplicates]]
#-------------------------------------------------------------------------------
@VrfCli.VrfExecCmdDec( getVrfsFunc=getVrfsFunc )
def doShowBgpInstanceInternal( mode, internal, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show instance-internal', vrfName=vrfName )
   if 'dump-duplicates' in internal:
      cmd.addParam( 'dump-duplicates' )
   elif 'brib' in internal and 'count' in internal:
      cmd.addParam( 'brib-count' )
      if 'detail' in internal:
         cmd.addParam( 'brib-detail' )
      elif 'stash' in internal:
         cmd.addParam( 'stash' )
   elif 'iar' in internal:
      assert 'events' in internal
      cmd.addParam( 'iar-history' )
   elif 'events' in internal:
      assert 'tracking' in internal
      if 'start' in internal:
         cmd.addParam( 'events-start' )
         if 'advertise-threshold' in internal:
            cmd.addParam( 'advertise-threshold' )
            cmd.addParam( internal.get( 'numRibRoutes' ) )
            cmd.addParam( internal.get( 'numRibOutRoutes' ) )
      elif 'stop' in internal:
         cmd.addParam( 'events-stop' )
      elif 'summary' in internal:
         cmd.addParam( 'events-summary' )
      elif 'detail' in internal:
         cmd.addParam( 'events-detail' )
   cmd.run()

def doShowBgpInstance( mode, internal=None, vrfName=None ):
   if internal:
      return doShowBgpInstanceInternal( mode, internal, vrfName=vrfName )

   # We can use the same command string ('show instance') here because
   # the new async command callback will register using
   # 'agentCommandCallbackWithFormatIs' instead of 'agentCommandCallbackIs'.
   # This helps with adding JSON format to the command.
   cmd = ArBgpAsyncCliCommand( mode, 'show instance', vrfName=vrfName, revision=2 )
   # The callback is shared with 'show bgp convergence'
   cmd.addParam( 'instance' )
   cmd.run()
   return cliPrinted( ShowBgpInstanceModel )

arBgpShowCmdDict[ 'doShowBgpInstance' ] = doShowBgpInstance

#-------------------------------------------------------------------------------
# "show bgp update-group [index | A.B.C.D | A:B:C:D:E:F:G:H] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def doShowBgpUpdateGroup( mode, opt=None, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show update-group', vrfName=vrfName )
   if opt is not None:
      if isinstance( opt, ( int, long ) ):
         cmd.addParam( 'index', opt )
      else:
         peerAddr = ""
         llIntf = ""
         if opt.type == 'peerIpv4':
            peerAddr = opt.v4Addr
         else:
            assert opt.type in [ 'peerIpv6', 'peerIpv6Ll' ]
            peerAddr = opt.v6Addr.stringValue
            llIntf = Arnet.IntfId( opt.llIntf ).stringValue
         cmd.addParam( 'peerAddr', peerAddr )
         cmd.addParam( 'llIntf', llIntf )
   cmd.run()
   return cliPrinted( updateGroupVrfModel )

arBgpShowCmdDict[ 'doShowBgpUpdateGroup' ] = doShowBgpUpdateGroup

#-------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]
#-------------------------------------------------------------------------------
def doShowBgpStatistics( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show statistics', vrfName=vrfName )
   cmd.run()
   return cliPrinted( statisticsVrfModel )

arBgpShowCmdDict[ 'doShowBgpStatistics' ] = doShowBgpStatistics

#-------------------------------------------------------------------------------
# "show bgp scheduler [verbose] [internal] [history]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpScheduler', arBgpModeOnly=True )
def doShowBgpScheduler( mode, history=None, internal=None, verbose=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show scheduler' )
   # This command does not use the ArBgp ACR framework, remove any
   # arguments it may have added
   cmd.params = []
   for option in [ arg for arg in ( history, internal, verbose )
                   if arg is not None ]:
      cmd.addParam( option )
   stringBuff = None
   if not mode.session_.outputFormatIsJson():
      stringBuff = StringIO.StringIO()
   cmd.run( stringBuff=stringBuff )
   if mode.session_.outputFormatIsJson():
      # If we're using json, just return the output from the
      # agent directly to the user.  Use cliPrinted() to let
      # the infrastructure know that we've done this.
      return cliPrinted( ShowTaskSchedulerModel.Overall )
   jsonTxt = stringBuff.getvalue()
   stringBuff.close()
   from CliPlugin.ShowTaskScheduler import showTaskSchedulerJsonToModel
   ret = showTaskSchedulerJsonToModel( jsonTxt, paramDebug=internal is not None )
   if ret is None:
      mode.addError( "Received invalid response from agent Bgp\n%s" % ( jsonTxt ) )
      return None
   return ret

schedulerHistory = CliMatcher.KeywordMatcher( 'history',
                                              'Display scheduler event history' )
schedulerInternal = CliMatcher.KeywordMatcher( 'internal',
                                               'Display memory addresses of tasks' )

#--------------------------------------------------------------------------------
# Deprecate 'show bgp scheduler reset'.  It was a
# lazy implementation and doesn't fit the new one.
# --------------------------------------------------------------------------------
class ShowBgpSchedulerReset( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp scheduler reset'
   data = {
         "bgp" : bgpAfterShow,
         "scheduler" : scheduler,
         "reset" : CliCommand.singleNode(
            CliMatcher.KeywordMatcher( 'reset',
                                       helpdesc='Reset scheduler statistics' ),
            deprecatedByCmd='clear agent bgp task scheduler' ),
      }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      # Since it's deprecated, this should never get called, but
      # if it is, let's make sure that the user knows that it didn't
      # work.
      mode.addError( 'Use "clear agent bgp task scheduler"' )

BasicCli.addShowCommandClass( ShowBgpSchedulerReset )

class ShowBgpScheduler( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp scheduler [ { history | internal | verbose } ]'
   data = {
         "bgp" : bgpAfterShow,
         "scheduler" : scheduler,
         "history" : CliCommand.Node( matcher=schedulerHistory, maxMatches=1,
                                     hidden=True ),
         "internal" : CliCommand.Node( matcher=schedulerInternal, maxMatches=1,
                                     hidden=True ),
         "verbose" : CliCommand.Node( matcher=schedulerVerbose, maxMatches=1 ),
      }
   privileged = True

   cliModel = ShowTaskSchedulerModel.Overall

   @staticmethod
   def handler( mode, args ):
      return doShowBgpScheduler( mode,
                                 history=args.get( "history" ),
                                 internal=args.get( "internal" ),
                                 verbose=args.get( "verbose" ) )

BasicCli.addShowCommandClass( ShowBgpScheduler )

#-------------------------------------------------------------------------------
# show bgp debug policy
#  { inbound neighbor { <address> | all } | outbound neighbor <address>
#    | redistribute }
#      <afi> <safi> [ vrf <vrf name> ] [ route-map <route-map> ]  <prefix>
#-------------------------------------------------------------------------------
safiOptions = { 'unicast' : 'Unicast subsequent address family' }

def modifyResultsWithMissingPolicy( results, vrf, application ):
   # check the results for peers that didn't evaluate a route map and determine
   # if it was due to a non existent route map at the point of application or
   # no route map being applied.

   vrfConfig = config
   if vrf != 'default':
      vrfConfig = vrfConfigDir.vrfConfig[ vrf ]

   if application in [ 'inbound', 'outbound' ]:
      for peer in results:
         if not peer[ 'log' ]:
            # The log was empty, it's of no use now.
            peer.pop( 'log' )
            # the route map was not found for the peer
            peerAddr = peer[ 'peerAddr' ]
            key = createKey( peerAddr )
            peerConfig = vrfConfig.neighborConfig[ key ]
            if application == "inbound":
               if peerConfig.routeMapInPresent:
                  peer[ 'missingPolicy' ] = peerConfig.routeMapIn
            elif application == "outbound":
               if peerConfig.routeMapOutPresent:
                  peer[ 'missingPolicy' ] = peerConfig.routeMapOut
            else:
               assert 0, "application not supported"
   elif application == 'redistribute':
      for source in results:
         if not source[ 'log' ]:
            # The log was empty, it's of no use now.
            source.pop( 'log' )
            # the route map was not found for the source
            proto = source.pop( 'redistProto' )
            redistConfig = vrfConfig.redistributeConfig[ proto ]
            if redistConfig.routeMap:
               source[ 'missingPolicy' ] = redistConfig.routeMap
   else:
      assert 0, "application not supported"
   return results

def removeUnsupportedMatchers( log, rmName, unsupportedMatchers ):
   for seqLog in log[ 'seqs' ].values():
      newMatchers = []
      for matcher in seqLog[ 'passedMatchers' ]:
         if matcher not in unsupportedMatchers:
            newMatchers.append( matcher )
      seqLog[ 'passedMatchers' ] = newMatchers
      if 'subRouteMap' in seqLog:
         subRmName = seqLog[ 'subRouteMap' ][ 'rmname' ]
         removeUnsupportedMatchers( seqLog[ 'subRouteMap' ],
               subRmName, unsupportedMatchers )

def removePartiallySupportedMatchers( results, application ):
   """Remove matchers that are not supported across all applications.
   In such case we may find matcher present in passedMatchers list. so
   we need to remove them from the list.
   """
   # matchInterface and match ip resolved-next-hop is supported only in Redistribute
   # policy case and not in Inbound and Outbound policy application
   # Storing unsupported matchers as set for faster searching
   if application == 'inbound' or application == 'outbound':
      unsupportedMatchers = { 'matchInterface', 'matchResolvedNextHopPrefixList',
                              'matchIpv6ResolvedNextHopPrefixList' }
   else:
      return

   for peer in results:
      if not peer.get( 'log' ):
         continue
      rmName = peer[ 'log' ][ 'rmname' ]
      removeUnsupportedMatchers( peer[ 'log' ], rmName, unsupportedMatchers )
   return

@staticmethod
@ArBgpShowOutput( 'doShowBgpDebugPolicy', arBgpModeOnly=True )
def policyDebugHandler( mode, args ):
   peerAddr = args.get( 'ADDR6', args.get( 'ADDR' ) )
   afi = args.get( 'ipv6', args.get( 'ipv4' ) )
   # The nlri value should correspond to the enum name in BgpNlriType.tac
   nlri = afi + args[ 'SAFI' ].capitalize()
   # Get the routing context vrf if none was specified
   if 'VRF' not in args:
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
   else:
      vrfName = args[ 'VRF' ]

   if vrfName not in VrfCli.getAllVrfNames():
      if vrfName.lower() == 'all':
         mode.addError( "Policy debug for all VRFs is not supported" )
      else:
         mode.addError( "Failed to find VRF {}".format( vrfName ) )
      return None

   prefix = args[ 'PREFIX6' ] if args.get( 'ipv6' ) else args[ 'PREFIX' ]
   if args.get( 'ipv6' ):
      prefix = prefix.stringValue() if prefix else ""

   cmd = ArBgpAsyncCliCommand( mode, 'show debug policy',
                               vrfName=vrfName,
                               nlriType=nlri,
                               prefix=prefix,
                               peerAddr=peerAddr )

   application = ""
   if 'inbound' in args:
      application = "inbound"
   elif 'outbound' in args:
      application = "outbound"
   elif 'redistribute' in args:
      application = "redistribute"
   else:
      mode.addError( "Not supported" )
      return None
   cmd.addParam( 'application', application )

   if 'RMAP_NAME' in args:
      cmd.addParam( 'rmName', args[ 'RMAP_NAME' ] )

   buff = StringIO.StringIO()
   cmd.run( stringBuff=buff, forceOutputFormat="json" )

   if not buff.getvalue():
      # The command returned nothing which means it is likely that BGP crashed.
      mode.addError( "" )
      return None

   resultJson = buff.getvalue()
   results = json.loads( resultJson )

   if isinstance( results, str ):
      mode.addError( results )
   elif "error" in results:
      mode.addError( results[ "error" ] )
   else:
      results = modifyResultsWithMissingPolicy( results[ "results" ], vrfName,
                                                application )
      removePartiallySupportedMatchers( results, application )
      m = policyEvaluationLogsContainerModel( results, prefix, application )
      return m

   return None

vrfExprFactory = VrfCli.VrfExprFactory(
      helpdesc='Specify the VRF',
      inclDefaultVrf=True )

class ShowBgpDebugPolicyInbound( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp debug policy ' \
            'inbound neighbor ( ADDR | ADDR6 | all ) ' \
            '( ipv4 SAFI [ VRF ] [ route-map RMAP_NAME ] PREFIX ) |' \
            '( ipv6 SAFI [ VRF ] [ route-map RMAP_NAME ] PREFIX6 )'
   data = {
      'bgp' : CommonTokens.showBgp,
      'debug' : 'Debug BGP',
      'policy' : 'Debug BGP policy application',
      'inbound' : 'Debug inbound policy application',
      'neighbor' : 'Select BGP neighbor',
      'all' : 'Debug policy application to all neighbors',
      'ADDR' : CommonTokens.neighborAddr,
      'ADDR6' : CommonTokens.neighborAddr6,
      'ipv4' : 'IPv4 related',
      'ipv6' : 'IPv6 related',
      'SAFI' : CliMatcher.EnumMatcher( safiOptions ),
      'VRF' : vrfExprFactory,
      'route-map' : 'Route map to apply',
      'RMAP_NAME' : RouteMapCommonTokens.routeMapName,
      'PREFIX' : CommonTokens.prefixExpr,
      'PREFIX6' : CommonTokens.prefix6Expr,
   }

   cliModel = PolicyEvalMultiPeersNlriResult
   handler = policyDebugHandler

BasicCli.addShowCommandClass( ShowBgpDebugPolicyInbound )

class ShowBgpDebugPolicyOutbound( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp debug policy ' \
            'outbound neighbor ( ADDR | ADDR6 ) ' \
            '( ipv4 SAFI [ VRF ] [ route-map RMAP_NAME ] PREFIX ) |' \
            '( ipv6 SAFI [ VRF ] [ route-map RMAP_NAME ] PREFIX6 )'
   data = {
      'bgp' : CommonTokens.showBgp,
      'debug' : 'Debug BGP',
      'policy' : 'Debug BGP policy application',
      'outbound' : 'Debug outbound policy application',
      'neighbor' : 'Select BGP neighbor',
      'ADDR' : CommonTokens.neighborAddr,
      'ADDR6' : CommonTokens.neighborAddr6,
      'ipv4' : 'IPv4 related',
      'ipv6' : 'IPv6 related',
      'SAFI' : CliMatcher.EnumMatcher( safiOptions ),
      'VRF' : vrfExprFactory,
      'route-map' : 'Route map to apply',
      'RMAP_NAME' : RouteMapCommonTokens.routeMapName,
      'PREFIX' : CommonTokens.prefixExpr,
      'PREFIX6' : CommonTokens.prefix6Expr,
   }

   cliModel = PolicyEvalMultiPeersNlriResult
   handler = policyDebugHandler

BasicCli.addShowCommandClass( ShowBgpDebugPolicyOutbound )

class ShowBgpDebugPolicyRedistribute( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp debug policy redistribute ' \
            '( ipv4 SAFI [ VRF ] [ route-map RMAP_NAME ] PREFIX ) |' \
            '( ipv6 SAFI [ VRF ] [ route-map RMAP_NAME ] PREFIX6 )'
   data = {
      'bgp' : CommonTokens.showBgp,
      'debug' : 'Debug BGP',
      'policy' : 'Debug BGP policy application',
      'redistribute' : 'Debug redistribute policy application',
      'ipv4' : 'IPv4 related',
      'ipv6' : 'IPv6 related',
      'SAFI' : CliMatcher.EnumMatcher( safiOptions ),
      'VRF' : vrfExprFactory,
      'route-map' : 'Route map to apply',
      'RMAP_NAME' : RouteMapCommonTokens.routeMapName,
      'PREFIX' : CommonTokens.prefixExpr,
      'PREFIX6' : CommonTokens.prefix6Expr,
   }

   cliModel = PolicyEvalMultiSourceNlriResult
   handler = policyDebugHandler

BasicCli.addShowCommandClass( ShowBgpDebugPolicyRedistribute )

#-------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]
#-------------------------------------------------------------------------------
def doShowBgpConvergence( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show instance', vrfName=vrfName )
   cmd.addParam( 'convergence' )
   cmd.run()
   return cliPrinted( ShowBgpConvergenceModel )

arBgpShowCmdDict[ 'doShowBgpConvergence' ] = doShowBgpConvergence

#-------------------------------------------------------------------------------
# "show (ip | ipv6) bgp [labeled-unicast] [<ip> | <prefix>] [longer-prefixes]
#  [detail] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def doShowAfBgpInternal( mode, **kwargs ):
   vrfName = kwargs.pop( "vrfName", None )
   prefix = kwargs.pop( "addr", None )
   afi = kwargs.pop( 'afi', '' )
   safi = kwargs.pop( 'safi', '' )
   extraNlriAfiSafis = []
   if safi == 'mplslabel':
      # Just do labeled-unicast
      nlriAfiSafi = afi + 'Lu'
   elif safi == 'multicast':
      # Just do multicast
      nlriAfiSafi = afi + 'Multicast'
   else:
      # Do both unicast and labeled-unicast
      nlriAfiSafi = afi + 'Unicast'
      extraNlriAfiSafis = [ afi + 'Lu' ]
   ArBgpAsyncCliCommand( mode, "show bgp <af> internal",
                         vrfName=vrfName,
                         prefix=prefix,
                         nlriAfiSafi=nlriAfiSafi,
                         extraNlriAfiSafis=extraNlriAfiSafis ).run()
   return cliPrinted( routeSummaryVrfModel )

arBgpShowCmdDict[ 'doShowAfBgpInternal' ] = doShowAfBgpInternal

def mplsLabelSafiArg( kwargs ):
   labUni = kwargs.pop( "labUni", None )
   result = ""
   if labUni == "labeled-unicast":
      result = "mplslabel"
      kwargs[ 'safi' ] = result
   return result

def getAfiSafiFromAddressFamilyAndType( kwargs ):
   addressFamilyAndType = kwargs.pop( "addressFamilyAndType", None )
   if addressFamilyAndType is None:
      addressFamilyAndType = kwargs.pop( "routeFamilyAndType", None )
   afi = None
   safi = None
   if addressFamilyAndType:
      afi = addressFamilyAndType.get( 'addressFamily' )
      if afi == 'ip':
         afi = 'ipv4'
      if addressFamilyAndType.get( 'multicast' ):
         safi = 'multicast'
      elif addressFamilyAndType.get( 'mplslabel' ) or \
            addressFamilyAndType.get( 'labeled-unicast' ):
         safi = 'mplslabel'
      elif addressFamilyAndType.get( 'unicast' ):
         safi = 'unicast'
   return afi, safi

def flattenShowBgpNlriAf( kwargs ):
   afi, safi = getAfiSafiFromAddressFamilyAndType( kwargs )
   # if we have contradicting afis here we remove it from
   # kwargs so show command will be empty in case like:
   # show ip bgp neighbors ipv6 unicast received-routes
   # ... except when we have "peerAddr" then we leave it e.g.:
   # show ip bgp neighbors 10.20.30.40 ipv6 unicast received-routes
   # should return ipv6 path from this peer even though we have
   # "show ip ..." here
   if afi is not None:
      if kwargs.get( "peerAddr" ) is None \
            and kwargs.get( "afi" ) is not None \
            and kwargs.get( "afi" ) != afi:
         kwargs.pop( "afi", None )
      else:
         kwargs[ "afi" ] = afi
         kwargs[ "safi" ] = safi
   elif "pfxAddr" in kwargs:
      kwargs[ "afi" ] = getNlriAfForPfxAddrRule( kwargs[ "pfxAddr" ] )
   elif kwargs.get( "peerAddr" ) is not None:
      # "show (ip|ipv6) bgp <peer_addr> routes" case: we dont' have address family
      # in the arguments so we should add both ipv4 and ipv6
      kwargs[ "afi" ] = "ipv4"
      kwargs[ "extraNlriAfiSafis" ] = [ "ipv6" ]

   if not "vrfName" in kwargs:
      kwargs[ "vrfName" ] = None

   communityValuesAndExact = kwargs.pop( "communityValuesAndExact", None )
   if communityValuesAndExact:
      commValues = communityValuesAndExact.get( 'communityValues' )
      exact = communityValuesAndExact.get( 'exact' )
      if commValues:
         kwargs[ "commValues" ] = commValues
      if exact:
         kwargs[ "exact" ] = exact
   return kwargs

#-------------------------------------------------------------------------------
# "show ip bgp community regex <comm-regex> as-path regex <asp-regex>
#     [vrf <vrfName>]"
# "show ip bgp [as-path] regex <asp-regex> community regex <comm-regex>
#     [vrf <vrfName>]"
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowIpBgpAndRegexp', arBgpModeOnly=True, vrfLookup=True )
def doShowIpBgpAndRegexp( mode, aspRegex=None, commRegex=None, vrfName=None ):
   result = showBgpCombinedRegexpHelper( mode, aspRegex, commRegex, vrfName=vrfName )
   return result

class ShowIpBgpAndRegexp( ShowCommand.ShowCliCommandClass ):
   syntax = """show ip bgp
               ( ( community commregexp COMMREGEX aspath aspregexp ASPREGEX ) |
                 ( [ aspath ] aspregexp ASPREGEX community commregexp COMMREGEX ) )
               [ VRF ]"""
   data = {
         "ip" : ipMatcherForShow,
         "bgp" : bgpAfterShow,
         "community" : showCommunity,
         "commregexp" : commRegexp,
         "COMMREGEX" : RouteMapCommonTokens.commListRegexMatcher,
         "aspath" : aspath,
         "aspregexp" : regexp,
         "ASPREGEX" : aspathRegexMatcher,
         "VRF" : allVrfExprFactory,
      }
   cliModel = routeSummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      return doShowIpBgpAndRegexp( mode, aspRegex=args[ "ASPREGEX" ],
                                   commRegex=args[ "COMMREGEX" ],
                                   vrfName=args.get( "VRF" ) )

BasicCli.addShowCommandClass( ShowIpBgpAndRegexp )

#-------------------------------------------------------------------------------
# "show (ip | ipv6) bgp summary [<prefix>] [vrf <vrfName>]
#-------------------------------------------------------------------------------
def doShowIpBgpSummaryAcrImpl( mode, prefix=None, vrfName=None, afi=None,
                               nlriAfiSafi=None, extraNlriAfiSafis=None,
                               nlriType=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show summary', vrfName=vrfName,
                               nlriAfiSafi=nlriAfiSafi,
                               extraNlriAfiSafis=extraNlriAfiSafis,
                               transportAfi=afi, nlriType=nlriType,
                               prefix=prefix )
   cmd.run()
   return cliPrinted( summaryVrfModel )

def doShowIpBgpSummary( mode, prefix=None, vrfName=None ):
   return doShowIpBgpSummaryAcrImpl( mode, prefix=prefix,
                                     vrfName=vrfName,
                                     nlriAfiSafi='ipv4Unicast',
                                     extraNlriAfiSafis=[ 'ipv4Lu' ] )

def doShowIpv6BgpSummary( mode, prefix=None, vrfName=None ):
   return doShowIpBgpSummaryAcrImpl( mode, prefix=prefix,
                                     vrfName=vrfName,
                                     nlriAfiSafi='ipv6Unicast',
                                     extraNlriAfiSafis=[ 'ipv6Lu' ] )

arBgpShowCmdDict[ 'doShowIpv6BgpSummary' ] = doShowIpv6BgpSummary
arBgpShowCmdDict[ 'doShowIpBgpSummary' ] = doShowIpBgpSummary

def addressFamilyAndTypeToNlriAfiSafi( addressFamilyAndType ):
   if addressFamilyAndType is None:
      return None
   nlriAf = addressFamilyAndType.get( 'addressFamily' )
   if addressFamilyAndType.get( 'multicast' ):
      nlriAf = ( 'ipv6Multicast' if addressFamilyAndType.get( 'addressFamily' )
            == 'ipv6' else 'ipv4Multicast' )
   if addressFamilyAndType.get( 'mplslabel' ):
      nlriAf += 'Lu'
   if nlriAf == 'ip':
      nlriAf = 'ipv4'
   if nlriAf in ( 'ipv4', 'ipv6' ):
      # show only IP unicast safi
      nlriAf += 'Unicast'
   return nlriAf
#-------------------------------------------------------------------------------
# "show bgp (ipv4 | ipv6) unicast summary [vrf <vrfName>]
#-------------------------------------------------------------------------------

def doShowBgpNlriAfAcrSummary( mode, addressFamilyAndType, vrfName=None ):
   nlriAf = addressFamilyAndTypeToNlriAfiSafi( addressFamilyAndType )
   return doShowIpBgpSummaryAcrImpl( mode, vrfName=vrfName, nlriAfiSafi=nlriAf )

def doShowBgpNlriAfSummary( mode, addressFamilyAndType, vrfName=None ):
   return doShowBgpNlriAfAcrSummary( mode, addressFamilyAndType, vrfName=vrfName )

arBgpShowCmdDict[ 'doShowBgpNlriAfSummary' ] = doShowBgpNlriAfSummary

#-------------------------------------------------------------------------------
# "show [ip | ipv6] bgp neighbors [<ip>] [bfd] [vrf <vrfName>]
#     (hidden options) [internal [verbose] [rib-out] [dump-duplicates]]
#                      [task-stats [reset]]
#
# Note: 'verbose' option prints AdjRibin paths
#-------------------------------------------------------------------------------
def doShowBgpNeighborsAcr( mode, peerAddr=None, vrfName=None, bfd=None, af=None,
                           taskStats=None, iar=None, internal=None, routerId=None,
                           nlriAfiSafi=None, queuedWithdrawals=None,
                           extraNlriAfiSafis=None,
                           cliModel=peerListVrfModel ):
   llIntf = ""
   if not peerAddr:
      peerAddr = ""
   else:
      key = peerAddr
      if key.type == 'peerIpv4':
         peerAddr = key.v4Addr
      elif key.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         peerAddr = key.v6Addr.stringValue
         llIntf = Arnet.IntfId( key.llIntf ).stringValue
   cmdType = 'show peer'
   if queuedWithdrawals:
      cmdType += '-ribout'
   cmd = ArBgpAsyncCliCommand( mode, cmdType, vrfName=vrfName,
                               peerAddr=peerAddr, transportAfi=af,
                               nlriAfiSafi=nlriAfiSafi,
                               extraNlriAfiSafis=extraNlriAfiSafis,
                               routerId=routerId )
   cmd.addParam( 'llIntf', llIntf )
   if queuedWithdrawals:
      cmd.addParam( 'queuedWithdrawals' )
   else:
      cmd.addParam( 'bfd', bfd )
      if taskStats:
         cmd.addParam( 'task-stats' )
         if 'reset' in taskStats:
            cmd.addParam( 'reset' )
      if iar:
         cmd.addParam( 'iar' )
      if internal:
         cmd.addParam( 'internal' )
         if 'verbose' in internal:
            cmd.addParam( 'verbose' )
         if 'rib-out' in internal:
            cmd.addParam( 'rib-out' )
         if 'dump-duplicates' in internal:
            cmd.addParam( 'dump-duplicates' )
   cmd.run()
   return cliPrinted( cliModel )

#-------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def doShowIpBgpUpdateErrors( mode, addr=None, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show peer update errors', vrfName=vrfName,
      peerAddr=addr )
   cmd.run()
   return cliPrinted( updateErrorsVrfModel )

arBgpShowCmdDict[ 'doShowIpBgpUpdateErrors' ] = doShowIpBgpUpdateErrors

def doShowBgpNeighborsImpl( mode, transportAfi=None, addr=None, bfd=None,
                            internal=None, taskStats=None, iar=None, vrfName=None,
                            routerId=None ):

   # No JSON for debug commands
   if ( internal or taskStats or iar ) and not bfd:
      if mode.session_.outputFormat_ == "json":
         mode.addError( "This is an unconverted command" )
         return None

   return doShowBgpNeighborsAcr( mode, peerAddr=addr, vrfName=vrfName,
                                    bfd=bfd, af=transportAfi, taskStats=taskStats,
                                    iar=iar, internal=internal, routerId=routerId )

arBgpShowCmdDict[ 'doShowBgpNeighbors' ] = doShowBgpNeighborsImpl
arBgpShowCmdDict[ 'doShowIpBgpNeighbors' ] = \
    lambda mode, **kwargs: doShowBgpNeighborsImpl( mode, 'ip', **kwargs )
arBgpShowCmdDict[ 'doShowIpv6BgpNeighbors' ] = \
    lambda mode, **kwargs: doShowBgpNeighborsImpl( mode, 'ipv6', **kwargs )

def doShowIpBgpPeerGroup( mode, pgAndVrfName=( None, None ) ):
   cmd = ArBgpAsyncCliCommand( mode, 'show peer-group',
                               vrfName=pgAndVrfName[ 1 ] )
   cmd.addParam( "peerGroup", pgAndVrfName[ 0 ] )
   cmd.run()
   return cliPrinted( peerGroupVrfModel )

arBgpShowCmdDict[ 'doShowIpBgpPeerGroup' ] = doShowIpBgpPeerGroup

#-------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> [(ipv4 unicast) | (ipv6 unicast)]
#       advertised-routes queued-withdrawals"
# "show ipv6 bgp peers <ip> [(ipv4 unicast) | (ipv6 unicast)]
#       advertised-routes queued-withdrawals"
#-------------------------------------------------------------------------------
def doShowIpBgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                           routeFamilyAndType=None, vrfName=None ):

   nlriAfSafi = addressFamilyAndTypeToNlriAfiSafi( routeFamilyAndType )
   if not nlriAfSafi:
      nlriAfSafi = 'ipv4Unicast'
      extraNlriAfiSafis = [ 'ipv4Lu', 'ipv6Unicast', 'ipv6Lu' ]
   else:
      extraNlriAfiSafis = None
   return doShowBgpNeighborsAcr( mode, peerAddr=peerAddr, af='ip', vrfName=vrfName,
                                 nlriAfiSafi=nlriAfSafi,
                                 queuedWithdrawals=queuedWithdrawals,
                                 extraNlriAfiSafis=extraNlriAfiSafis,
                                 cliModel=queuedWithdrawalsVrfModel )

def doShowIpv6BgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                             routeFamilyAndType=None, vrfName=None ):

   nlriAfSafi = addressFamilyAndTypeToNlriAfiSafi( routeFamilyAndType )
   if not nlriAfSafi:
      nlriAfSafi = 'ipv6Unicast'
      extraNlriAfiSafis = [ 'ipv4Unicast', 'ipv4Lu', 'ipv6Lu' ]
   else:
      extraNlriAfiSafis = None
   return doShowBgpNeighborsAcr( mode, peerAddr=peerAddr, af='ipv6', vrfName=vrfName,
                                 nlriAfiSafi=nlriAfSafi,
                                 queuedWithdrawals=queuedWithdrawals,
                                 extraNlriAfiSafis=extraNlriAfiSafis,
                                 cliModel=queuedWithdrawalsVrfModel )

arBgpShowCmdDict[ 'doShowIpBgpNeighborsQueuedWithdrawals' ] = \
   doShowIpBgpNeighborsQueuedWithdrawals
arBgpShowCmdDict[ 'doShowIpv6BgpNeighborsQueuedWithdrawals' ] = \
   doShowIpv6BgpNeighborsQueuedWithdrawals

#-------------------------------------------------------------------------------
# "clear ip bgp [<ip> | *] (([vrf <vrfName>] [reason MESSAGE]) |
#                          ([soft [in | out]] [vrf <vrfName>]))"
#-------------------------------------------------------------------------------

@VrfCli.VrfExecCmdDec( getVrfsFunc=getVrfsFunc )
def doClearIpBgp( mode, transportAddrFamily=None, peer=None, soft=False, 
                  direction=None, vrfName=None, errorsOrCounters=None, resetMsg='' ):

   if ( not peer and config.clearAllDisabled and not errorsOrCounters ):
      mode.addError( "Cannot clear all peering sessions because " +
                     "clear all commands are disabled" )
      return
   #
   # Log what we are attempting to do
   #
   doLogClearIpBgp( mode, transportAddrFamily, peer, soft, direction,
                    vrfName,
                    errorsOrCounters == 'errors',
                    errorsOrCounters == 'counters',
                    resetMsg )

   cmd = ArBgpCliCommand( mode, 'clear', vrfName=vrfName,
                          transportAfi=transportAddrFamily )
   if peer:
      cmd.addParam( 'peer-address', peer.stringValue )
   if soft or direction:
      cmd.addParam( 'soft', direction )
   if errorsOrCounters == 'errors':
      cmd.addParam( 'errors' )
   if errorsOrCounters == 'counters':
      cmd.addParam( 'counters' )

   # transportBasedClear is True in "clear ip(v6) bgp neighbor *" command
   # we derive the transportBasedClear from transportAddrFamily which is set only
   # in 'clear ip(v6) bgp neighbor *' command
   transportBasedClear = transportAddrFamily is not None
   if transportBasedClear:
      cmd.addParam( 'neighbor' )

   if resetMsg:
      cmd.addParam( 'reason', resetMsg )

   cmd.run()

arBgpShowCmdDict[ 'doClearIpBgp' ] = doClearIpBgp

#-------------------------------------------------------------------------------
# "clear (ip | ipv6) bgp neighbor [*] [vrf <vrfName>] [reason MESSAGE]"
#-------------------------------------------------------------------------------

@VrfCli.VrfExecCmdDec( getVrfsFunc=getVrfsFunc )
def doClearIpGenBgpTransport( mode, transportAddrFamily, vrfName=None, resetMsg='' ):
   if config.clearAllDisabled:
      mode.addError( "Cannot clear all peering sessions because "
                     "clear all commands are disabled" )
      return
   cmd = ArBgpCliCommand( mode, 'clear', vrfName=vrfName,
                          transportAfi=transportAddrFamily )
   cmd.addParam( 'neighbor' )
   if resetMsg:
      cmd.addParam( 'reason', resetMsg )
   cmd.run()

arBgpShowCmdDict[ 'doClearIpBgpTransport' ] = doClearIpGenBgpTransport

#-------------------------------------------------------------------------------
# "show tech-support bgp [iar] [verbose] [memory] [importError] [evpn]
#                        [graceful-restart] [file|flash:<file>] [compress]
#                        [dumpLineNumbers] [rcf] [link-state]
#-------------------------------------------------------------------------------
showTechBgpKw = CliMatcher.KeywordMatcher(
   'bgp',
   'Show aggregated status and configuration details for the Bgp agent' )

def showArBgpDump( mode, args ):
   # Run this command only if the protocol model is multiAgent
   if ( not mode.session_.shouldPrint() or
        l3Config.protocolAgentModel != ProtoAgentModel.multiAgent ):
      # shouldPrint is set to False when displaying json and we are
      # not a converted command.
      return
   cmd = ArBgpCliCommand( mode, 'show support' )
   if 'iar' in args:
      cmd = ArBgpAsyncCliCommand( mode, 'show support iar' )
   if 'memory' in args:
      cmd.addParam( 'memory' )
   if 'import-error' in args:
      cmd.addParam( 'importError' )
   if 'verbose' in args:
      cmd.addParam( 'verbose' )
   if 'evpn' in args:
      cmd.addParam( 'evpn' )
   if 'graceful-restart' in args:
      cmd.addParam( 'gracefulRestart' )
   if 'line-numbers' in args:
      cmd.addParam( 'dumpLineNumbers' )
   if 'rcf' in args:
      cmd.addParam( 'rcf' )
   if 'label-management' in args:
      cmd = ArBgpAsyncCliCommand( mode, 'show support label' )
   if 'link-state' in args:
      cmd = ArBgpAsyncCliCommand( mode, 'show support link-state' )
   dumpfile = args.get( 'DUMPFILE' )
   if dumpfile:
      cmd.addParam( 'dumpfile', dumpfile.localFilename() )
      if 'compress' in args:
         cmd.addParam( 'compress' )
      cmd.run( timeout=None )
   else:
      cmd.run()

class ShowTechBgpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support bgp [ iar ] [ verbose ] [ DUMPFILE ] [ compress ] ' \
            '[ memory ] [ import-error ] [ evpn ] [ graceful-restart ] ' \
            '[ line-numbers ] [ rcf ] [ label-management ] [ link-state ]'
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'bgp' : showTechBgpKw,
      'iar' : 'Show IAR related state',
      'verbose' : 'Verbose mode',
      'DUMPFILE' : Url.UrlMatcher( lambda fs : ( fs.realFileSystem() and
                                                fs.supportsWrite() ),
                                   'send output to a file',
                                   acceptSimpleFile=False ),
      'compress' : 'Compress the output',
      'memory' : 'Show memory related info',
      'import-error' : 'Show import error counts',
      'evpn' : 'Show EVPN related state',
      'graceful-restart' : 'Graceful restart information for Bgp agent',
      'line-numbers' : 'Include line numbers in output',
      'rcf' : 'Show router control-functions related state',
      'label-management' : 'Show MPLS label manager information',
      'link-state' : 'Show BGP Link State AFI/SAFI related information'
      }

   privileged = True

   handler = staticmethod( showArBgpDump )

BasicCli.addShowCommandClass( ShowTechBgpCmd )

#-------------------------------------------------------------------------------
# "show tech-support bgp debug [ipUrib]
#-------------------------------------------------------------------------------
def showArBgpDebug( mode, forwardingTarget=None ):
   # Run this command only if the protocol model is multiAgent
   if not mode.session_.shouldPrint() or \
         l3Config.protocolAgentModel != ProtoAgentModel.multiAgent:
      # shouldPrint is set to False when displaying json and we are
      # not a converted command.
      return
   cmd = ArBgpCliCommand( mode, 'show support' )
   cmd.addParam( 'debug' )
   # We default to ipUrib ForwardingTarget when no explicit forwardingTarget is
   # specified is for compatibility with gated - which has 'show tech ribd
   # debug' command
   if not forwardingTarget:
      forwardingTarget = 'ipUrib'
   cmd.addParam( 'forwardingTarget', forwardingTarget )
   cmd.run()

debugKw = CliMatcher.KeywordMatcher( 'debug',
                                     'BGP debugging commands' )
ipUribKw = CliMatcher.KeywordMatcher( 'ipUrib',
                                      'IP Unicast RIB' )

class ShowTechBgpDebugCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support bgp debug [ ipUrib ]'
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'bgp' : showTechBgpKw,
      'debug' : CliCommand.Node( matcher=debugKw, hidden=True ),
      'ipUrib' : CliCommand.Node( matcher=ipUribKw, hidden=True )
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      showArBgpDebug( mode, args.get( 'ipUrib' ) )

BasicCli.addShowCommandClass( ShowTechBgpDebugCmd )

#-------------------------------------------------------------------------------
# "show maintenance bgp ip all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def _doShowMaintenanceBgpAll( mode, afi, vrfName ):

   cmd = ArBgpAsyncCliCommand( mode, 'show maint bgp',
      vrfName=vrfName, transportAfi=afi )
   # The output of the AgentCommandRequest will be json format output. Collect
   # it in a string buffer and then load it into a python object, which we then
   # use to fill in the CAPI model.
   strBuff = StringIO.StringIO()
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = ""
      if vrfName == "all" or (
         vrfNameStatus.nameToIdMap is not None and
         vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName ) is not None ):
         data = json.loads( strBuff.getvalue() )
   except ValueError:
      return None
   else:
      def bgpMaintenanceVrfModelGenerator( data ):
         for vrfName in data:
            model = BgpMaintenanceStatus()
            vrfData = data[ vrfName ]
            model.setAsdot( asnConfig.isAsdotConfigured() )
            model.asn = vrfData[ 'asn' ]
            model.routerId = vrfData[ 'routerId' ]
            model.vrf = vrfData[ 'vrf' ]
            model.peers = peerModelGenerator( vrfData[ 'peers' ] )
            yield vrfName, model

      def peerModelGenerator( peers ):
         for peerAddr in peers:
            peerModel = BgpPeerMaintenanceStatus()
            peer = peers[ peerAddr ]
            peerModel.underMaintenance = True \
               if peer[ 'underMaintenance' ] == 'true' else False
            if peerModel.underMaintenance:
               if 'appliedRouteMapIn' in peer:
                  peerModel.appliedRouteMapIn = peer[ 'appliedRouteMapIn' ]
               if 'appliedRouteMapOut' in peer:
                  peerModel.appliedRouteMapOut = peer[ 'appliedRouteMapOut' ]
               # for backward compatability
               if 'appliedRouteMapIn' in peer and 'appliedRouteMapOut' in peer and \
                  peer[ 'appliedRouteMapIn' ] == peer[ 'appliedRouteMapOut' ]:
                  peerModel.appliedRouteMap = peer[ 'appliedRouteMapIn' ]
            yield peerAddr, peerModel

      model = BgpMaintenanceStatusVrf()
      vrfs = bgpMaintenanceVrfModelGenerator( data )
      if vrfs is not None:
         model.vrfs = vrfs
      return model

def doShowMaintenanceBgpIpAll( mode, vrfName=None ):
   return _doShowMaintenanceBgpAll( mode, afi='ip', vrfName=vrfName )

arBgpShowCmdDict[ 'doShowMaintenanceBgpIpAll' ] = doShowMaintenanceBgpIpAll

#-------------------------------------------------------------------------------
# "show maintenance bgp ipv6 all [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def doShowMaintenanceBgpIp6All( mode, vrfName=None ):
   return _doShowMaintenanceBgpAll( mode, afi='ipv6', vrfName=vrfName )

arBgpShowCmdDict[ 'doShowMaintenanceBgpIp6All' ] = doShowMaintenanceBgpIp6All


#-------------------------------------------------------------------------------
# "show maintenance bgp [<addr>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------
def doShowMaintenanceBgp( mode, peer, vrfName=None ):
   if not vrfName:
      vrfName = 'default'
   llIntf = ""
   afi = 'ip'
   if peer.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
      afi = 'ipv6'
      llIntf = Arnet.IntfId( peer.llIntf ).stringValue
   cmd = ArBgpAsyncCliCommand( mode, 'show maint bgp', vrfName=vrfName,
                               transportAfi=afi, peerAddr=peer.stringValue )
   cmd.addParam( 'llIntf', llIntf )
   # The output of the AgentCommandRequest will be json format output. Collect
   # it in a string buffer and then load it into a python object, which we then
   # use to fill in the CAPI model.
   strBuff = StringIO.StringIO()
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = ""
      if vrfName == "all" or (
         vrfNameStatus.nameToIdMap is not None and
         vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName ) is not None ):
         data = json.loads( strBuff.getvalue() )
   except ValueError:
      return None
   else:
      def getPeerModel( peers, maintPeer ):
         peer = peers[ maintPeer ]
         peerModel = BgpPeerMaintenanceStatus()
         peerModel.underMaintenance = True if peer[ 'underMaintenance' ] == 'true' \
            else False
         if peerModel.underMaintenance:
            if 'appliedRouteMapIn' in peer:
               peerModel.appliedRouteMapIn = peer[ 'appliedRouteMapIn' ]
            if 'appliedRouteMapOut' in peer:
               peerModel.appliedRouteMapOut = peer[ 'appliedRouteMapOut' ]
            # for backward compatability
            if 'appliedRouteMapIn' in peer and 'appliedRouteMapOut' in peer and \
               peer[ 'appliedRouteMapIn' ] == peer[ 'appliedRouteMapOut' ]:
               peerModel.appliedRouteMap = peer[ 'appliedRouteMapIn' ]
         return peerModel

      def getRouteMapInfo( rmName ):
         if rmName.lower() == 'systemgenerated':
            routeMapInfo = routeMapContainerModel(
               defaultInitiatorRmDir, 'SystemGenerated' )
         else:
            routeMapInfo = routeMapContainerModel( mapConfig, rmName )
         return routeMapInfo

      def generatorVrfModel():
         if vrfName == "all":
            vrfs = getVrfsFunc()
            vrfs.append( "default" )
         else:
            vrfs = [ vrfName ]
         for vrf in vrfs:
            maintBgp = BgpPeerMaintenanceInfo()
            maintBgp.setAsdot( asnConfig.isAsdotConfigured() )
            maintBgp.peer = peer.stringValue
            if data and vrf in data:
               vrfData = data[ vrf ]
               peers = vrfData[ 'peers' ]
               bgpPeerStatus = None
               if maintBgp.peer in peers:
                  bgpPeerStatus = BgpMaintenanceStatusPerPeer()
                  bgpPeerStatus.asn = vrfData[ 'asn' ]
                  bgpPeerStatus.routerId = vrfData[ 'routerId' ]
                  bgpPeerStatus.vrf = vrfData[ 'vrf' ]
                  bgpPeerStatus.peers.__setitem__(
                     str( maintBgp.peer ), getPeerModel(
                        vrfData[ 'peers' ], maintBgp.peer ) )
               rmName = None
               rmNameIn = None
               rmNameOut = None
               if bgpPeerStatus and bgpPeerStatus.peers.get( maintBgp.peer ):
                  maintBgp.status = bgpPeerStatus
                  rmName = bgpPeerStatus.peers[ maintBgp.peer ].appliedRouteMap
                  rmNameIn = bgpPeerStatus.peers[
                     maintBgp.peer ].appliedRouteMapIn
                  rmNameOut = bgpPeerStatus.peers[
                      maintBgp.peer ].appliedRouteMapOut
                  if not rmName and ( rmNameIn == rmNameOut ):
                     rmName = rmNameIn
               if rmName:
                  maintBgp.routeMapInfo = getRouteMapInfo( rmName )
               if rmNameIn:
                  maintBgp.routeMapInfoIn = getRouteMapInfo( rmNameIn )
               if rmNameOut:
                  maintBgp.routeMapInfoOut = getRouteMapInfo( rmNameOut )

               getBgpGroupProfileInfo( mode, peer, vrf, maintBgp )
               yield vrf, maintBgp
            else:
               yield vrf, maintBgp

      vrfModel = BgpPeerMaintenanceInfoVrf()
      vrfModel.vrfs = generatorVrfModel()
      return vrfModel

arBgpShowCmdDict[ 'doShowMaintenanceBgp' ] = doShowMaintenanceBgp

#-------------------------------------------------------------------------------
# CliHook for "show maintenance interface <intf> detail" command
#-------------------------------------------------------------------------------
def bgpShowMaintenanceIntf( mode, intfId, status ):
   intfStatus = allIntfStatusDir.intfStatus.get( intfId )
   if not intfStatus or intfStatus.deviceName == "":
      return

   vrfName = getVrf( intfId )
   if not vrfName:
      return

   cmd = ArBgpAsyncCliCommand( mode, 'maint intf', vrfName=vrfName )
   # the str repr of Arnet::IntfId has quotes around the name. Since we need just
   # the name without quotes, use intfId.stringValue
   cmd.addParam( 'interface', intfId.stringValue )

   # need to get the output back from the cmd execution for further parsing
   # the JSON output to populate the MaintenanceInterfaceBgpStatus model,
   # so need a local string buffer to pass to the run method
   strBuff = StringIO.StringIO()

   # need to force json output always, so set the output format explicitly
   # using the forceOutputFormat option
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = json.loads( strBuff.getvalue() )
   except ValueError:
      return
   else:
      model = MaintenanceInterfaceBgpStatus()
      model.maintState = data[ 'maintState' ]
      model.vrf = data[ 'vrf' ]

      for peer in data[ 'peers' ]:
         peerData = data[ 'peers' ][ peer ]
         peerIntfModel = MaintenanceInterfaceBgpPeerStatus()
         if peerData[ 'underMaintenance' ] == 'true':
            if 'appliedRouteMapIn' in peerData:
               peerIntfModel.appliedRouteMapIn = peerData[ 'appliedRouteMapIn' ]
            if 'appliedRouteMapOut' in peerData:
               peerIntfModel.appliedRouteMapOut = peerData[ 'appliedRouteMapOut' ]
            # for backward compatability
            if 'appliedRouteMapIn' in peerData and \
               'appliedRouteMapOut' in peerData and \
               peerData[ 'appliedRouteMapIn' ] == peerData[ 'appliedRouteMapOut' ]:
               peerIntfModel.appliedRouteMap = peerData[ 'appliedRouteMapIn' ]
         model.peers[ peer ] = peerIntfModel

      status.bgpStatus = model

arBgpShowCmdDict[ 'bgpShowMaintenanceIntf' ] = bgpShowMaintenanceIntf

#-------------------------------------------------------------------------------
# "show bgp configuration (active | unsupported | detail)
#-------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpConfig', arBgpModeOnly=True )
def doShowBgpConfig( mode, option=None ):
   if arBgpConfig.asNumber == 0:
      return

   class Mode( object ):
      def __init__( self, parent=None, modeCmd=None ):
         self.modeCmd_ = modeCmd
         self.parent_ = parent
         self.children_ = []
         if not parent:
            self.indent_ = ''
            self.output_ = []
         else:
            parent.children_.append( self )
            self.indent_ = parent.indent_
            self.output_ = parent.output_
         self.asdotConfigured = asnConfig.isAsdotConfigured()

      def __enter__( self ):
         if self.parent_ and len( self.parent_.children_ ) > 1:
            self.addCommand( '!' )
         if self.modeCmd_:
            self.addCommand( self.modeCmd_ )
            self.indent_ += '   '
         return self

      def __exit__( self, t, value, traceback ):
         pass

      def addCommand( self, cmd ):
         self.output_.append( '%s%s' % ( self.indent_, cmd ) )

      # pylint: disable-msg=W0621
      def addRouteMapConfigCommands( self, mapConfig, arBgpMapConfig=None ):
         rmNames = mapConfig.routeMap.keys()
         for rmName in sorted( rmNames ):
            routeMap = mapConfig.routeMap[ rmName ]
            if arBgpMapConfig and rmName not in arBgpMapConfig.routeMap:
               # Skip routeMaps that are not in use by ArBgp
               continue
            seqnos = routeMap.mapEntry.keys()
            for seqno in sorted( seqnos ):
               mapEntry = routeMap.mapEntry[ seqno ]
               with Mode( self, 'route-map %s %s %s' % \
                           ( rmName, matchPermitEnum[ mapEntry.permit ], seqno ) ) \
                     as routeMapMode:
                  cmds = []
                  printRouteMapEntryAttributes( mapEntry, output=cmds )
                  for cmd in cmds:
                     routeMapMode.addCommand( cmd )

      def addBgpConfigCommands( self, bgpConfig, bgpVrfConfigDir,
                                ucmpConfig, vrfUcmpConfigDir ):
         with Mode( self, 'router bgp %s' % bgpFormatAsn( bgpConfig.asNumber,
                           self.asdotConfigured ) ) as routerBgpMode:
            cmds = saveBgpConfig( bgpConfig, bgpConfig, routingHwStatus, 
                                  "", securityConfig, aclCpConfig,
                                  asdotConfigured=self.asdotConfigured )
            for callback in showBgpConfigInstanceCallbacks:
               cmdDict = callback( bgpConfig ) if callback else {}
               for submodeCmd, gloabalCmds in cmdDict.iteritems():
                  if submodeCmd is None:
                     for cmd in gloabalCmds:
                        cmds.append( cmd )

            for cmd in cmds:
               routerBgpMode.addCommand( cmd )

            # UCMP
            cmds = saveUcmpConfig( routingHwStatus, ucmpConfig, ucmpConfig )
            for cmd in cmds:
               routerBgpMode.addCommand( cmd )

            def skipAfMode( af, cmds ):
               if not cmds:
                  if option == 'active':
                     # Skip the af mode cmds when there are no submode cmds
                     # present for the active configuration option.
                     return True
                  elif option == 'detail' or option == 'unsupported':
                     # Skip the af mode cmds if the af is neither ipv4 nor ipv6
                     # for both detail and unsupported configuration options.
                     # Always print both ipv4 and ipv6 af mode cmds since we
                     # know those af modes are supported in both gated and
                     # ArBgp.
                     if af != 'ipv4' and af != 'ipv6':
                        return True
                  else:
                     assert 0
               return False

            # load BGP-LS import config
            importcmds = getLsImportCmds( lsImportConfig )
            for af in bgpConfigAttrAfList():
               cmds = saveAfConfig( bgpConfig, bgpConfig, "", af,
                                    self.asdotConfigured )
               callBack = saveAfConfigCallbackDict.get( af )
               cmds += callBack( bgpConfig, af ) if callBack else []
               if skipAfMode( af, cmds ):
                  continue
               with Mode( routerBgpMode, 'address-family %s' % af ) \
                     as routerBgpAfMode:
                  if af == 'link-state':
                     cmds += importcmds
                     # reset the list, so that it does not get printed again futher
                     # down
                     importcmds = []
                  for cmd in cmds:
                     routerBgpAfMode.addCommand( cmd )

            # if there were no link-state af specific commands just load the
            # import config commands
            if importcmds:
               with Mode( routerBgpMode, 'address-family link-state' ) as lsAfMode:
                  for cmd in importcmds:
                     lsAfMode.addCommand( cmd )

            # Handle 'vrf default' mode
            dreCallBack = saveAfConfigCallbackDict.get( 'default-route-export' )
            defVrfCmds = {}
            defVrfCmds[ None ] = []
            # For 'vrf default' rd is configured in bgpConfig as well as in
            # ip/vrf/routeDistinguisherInputDir/config/bgp.
            # show run fetches from bgpConfig for 'vrf default' and from
            # ip/vrf/routeDistinguisherInputDir/config/bgp for non default vrf
            # Since we are mimicing running config, mirroring the same logic
            if bgpConfig.routeDistinguisher != 'INVALID':
               rd = bgpConfig.routeDistinguisher
               defVrfCmds[ None ].append(
                     'rd %s' % formatRd( rd, self.asdotConfigured ) )
            defVrfCmds[ None ] += saveBgpVpnConfig( DEFAULT_VRF, bgpConfig,
                                       bgpConfig, self.asdotConfigured, False )
            defVrfCmds[ None ] += \
                  dreCallBack( bgpConfig, None ) if dreCallBack else []

            for af in [ 'ipv4', 'ipv6' ]:
               defVrfCmds[ af ] = []
               defVrfCmds[ af ] = saveBgpVpnAfConfig( bgpConfig, bgpConfig, af,
                                          self.asdotConfigured, False )
               defVrfCmds[ af ] += dreCallBack(
                                    bgpConfig, af ) if dreCallBack else []

            cmds_present = not all(
                              skipAfMode( k, v ) for k, v in defVrfCmds.items() )
            if cmds_present:
               with Mode(
                     routerBgpMode, 'vrf %s' % DEFAULT_VRF ) as routerBgpVrfMode:
                  for cmd in defVrfCmds[ None ]:
                     routerBgpVrfMode.addCommand( cmd )
                  for af in [ 'ipv4', 'ipv6' ]:
                     if skipAfMode( af, defVrfCmds[ af ] ):
                        continue
                     with Mode( routerBgpVrfMode, 'address-family %s' % af ) \
                           as routerBgpVrfAfMode:
                        for cmd in defVrfCmds[ af ]:
                           routerBgpVrfAfMode.addCommand( cmd )

            # Handle instance level config submodes that are implemented in other
            # packages (rpki is one).
            for callback in showBgpConfigInstanceCallbacks:
               cmdDict = callback( bgpConfig ) if callback else {}
               for submodeCmd, cmds in cmdDict.iteritems():
                  if submodeCmd is not None:
                     with Mode( routerBgpMode, submodeCmd ) as submode:
                        for cmd in cmds:
                           submode.addCommand( cmd )

            # Handle non default vrf modes
            vrfNames = bgpVrfConfigDir.vrfConfig.keys()
            for vrfName in sorted( vrfNames ):
               bgpVrfConfig = bgpVrfConfigDir.vrfConfig[ vrfName ]
               vrfUcmpConfig = ucmpVrfConfigDir.vrfConfig[ vrfName ]
               rd = rdConfigDir.routeDistinguisher.get( vrfName, None )

               with Mode( routerBgpMode, 'vrf %s' % vrfName ) as routerBgpVrfMode:
                  cmds = []
                  if rd:
                     cmds.append( 'rd %s' % formatRd( rd, self.asdotConfigured ) )
                  cmds += saveBgpConfig( bgpVrfConfig, bgpConfig,
                                         routingHwStatus, vrfName,
                                         securityConfig, aclCpConfig,
                                         asdotConfigured=self.asdotConfigured )
                  cmds += dreCallBack( bgpVrfConfig, None ) if dreCallBack else []
                  for cmd in cmds:
                     routerBgpVrfMode.addCommand( cmd )

                  # UCMP
                  if vrfUcmpConfig:
                     cmds = saveUcmpConfig( routingHwStatus, ucmpConfig,
                                            vrfUcmpConfig )
                     for cmd in cmds:
                        routerBgpVrfMode.addCommand( cmd )

                  for af in bgpConfigAttrAfList( vrfName ):
                     cmds = saveAfConfig( bgpVrfConfig, bgpConfig, vrfName,
                                          af, self.asdotConfigured )
                     cmds += dreCallBack( bgpVrfConfig, af ) if dreCallBack else []
                     if skipAfMode( af, cmds ):
                        continue
                     with Mode( routerBgpVrfMode, 'address-family %s' % af ) \
                           as routerBgpVrfAfMode:
                        for cmd in cmds:
                           routerBgpVrfAfMode.addCommand( cmd )

   activeConfigMode = Mode()
   activeConfigMode.addRouteMapConfigCommands( arBgpMapConfig )
   activeConfigMode.addBgpConfigCommands( arBgpConfig, arBgpVrfConfigDir,
                                          arUcmpConfig, arUcmpVrfConfigDir )

   if option == 'active':
      output = activeConfigMode.output_
   else:
      runningConfigMode = Mode()
      runningConfigMode.addRouteMapConfigCommands( mapConfig, arBgpMapConfig )
      runningConfigMode.addBgpConfigCommands( config, vrfConfigDir,
                                              ucmpConfig, ucmpVrfConfigDir )
      ndiff = difflib.ndiff( runningConfigMode.output_, activeConfigMode.output_ )
      diff = list( ndiff )
      if option == 'detail':
         if len( diff ) != len( activeConfigMode.output_ ):
            output = diff
         else:
            output = activeConfigMode.output_
      else:
         assert option == 'unsupported'
         output = []
         if any( line.startswith( '-' ) for line in diff ):
            def strip( line ):
               return line[ 2 : ]

            def indent( line ):
               return len( strip( line ) ) - len( strip( line ).lstrip() )

            lastLine = ''
            modeCmds = []
            for line in diff: # pylint: disable-msg=too-many-nested-blocks
               if line.lstrip() == "" or \
                     line.startswith( '+' ) or \
                     line.startswith( '?' ):
                  # Ignore empty lines and starting with '+' or '?'.
                  continue
               elif line.find( '!' ) >= 0:
                  # Keep the '!' separator only if we previously appended a
                  # line with greater indent. Otherwise, ignore it.
                  if output and indent( output[ -1 ] ) > indent( line ):
                     output.append( line )
               else:
                  if lastLine:
                     # Track the list of mode commands up to this line.
                     if indent( lastLine ) < indent( line ):
                        # If this line has greater indent than the lastLine,
                        # then we've entered a submode and the lastLine should
                        # be appended to the list of mode commands.
                        if modeCmds and \
                              indent( modeCmds[ -1 ] ) == indent( lastLine ):
                           modeCmds[ -1 ] = lastLine
                        else:
                           if modeCmds:
                              # Sanity check to assert that the last mode
                              # command has greater indent than all other mode
                              # commands up to this point.
                              assert all( [ indent( cmd ) < indent( lastLine ) \
                                             for cmd in modeCmds ] )
                           modeCmds.append( lastLine )
                     else:
                        # Otherwise, remove any previously appended mode
                        # commands with greater indent in case we've instead
                        # exited a submode.
                        modeCmds = [ cmd for cmd in modeCmds \
                                       if indent( cmd ) < indent( line ) ]
                  if line.startswith( '-' ):
                     # Append lines starting with '-' to the output, while also
                     # extending the output with the list of mode commands up
                     # to this line.
                     lastLine = ''
                     if modeCmds:
                        output.extend( modeCmds )
                        modeCmds = []
                     output.append( line )
                  else:
                     lastLine = line

   print "\n".join( output )

class ShowBgpConfiguration( ShowCommand.ShowCliCommandClass ):
   syntax = """show bgp configuration ( active | unsupported | detail )"""
   data = {
         "bgp" : bgpAfterShow,
         "configuration" : RoutingBgpShowCliTokens.bgpConfig,
         "active" : active,
         "unsupported" : unsupported,
         "detail" : detailConfig,
      }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      option = ( args.get( "active" ) or args.get( "unsupported" ) or
            args.get( "detail" ) )
      return doShowBgpConfig( mode, option=option )

BasicCli.addShowCommandClass( ShowBgpConfiguration )

class CommonHistoryTraits( object ):
   '''
   Shared tokens and behavior between the history commands below.
   '''

   history = CliMatcher.KeywordMatcher( 'history', helpdesc='Neighbor history' )
   historyScope = CliCommand.setCliExpression(
      { 'connect-failures' : 'Connection failures' }, name='HISTORY_SCOPE' )

   @staticmethod
   def parse( mode, args, cmdClass, cmdStr ):
      '''
      Parses the common syntax objects and instantiate the command.
      params:
         mode, args: same as in handler
         cmdClass: class object that will run the ACR command 
         cmdStr: command string registered in ArBgp
      returns the command object
      '''
      vrfName = args.get( 'VRF' )
      addr = args.get( 'ADDR' ) or args.get( 'ADDR6' )
      prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )

      cmd = cmdClass( mode, cmdStr, vrfName=vrfName, peerAddr=addr, prefix=prefix )
      if 'PEERGROUP_NAME' in args:
         cmd.addParam( 'PEERGROUP_NAME', args[ 'PEERGROUP_NAME' ] )
      if 'HISTORY_SCOPE' in args and args[ 'HISTORY_SCOPE' ]:
         cmd.addParam( 'HISTORY_SCOPE', args[ 'HISTORY_SCOPE' ] )
      return cmd


#------------------------------------------------------------------------------------
# show bgp neighbors [ <ip> | <prefix> | ( peer-group <peerGroupName> ) ]
#                    history [ connect-failures ] [ ( vrf < vrfName > ) ]
# -- See AID5484 for a description of this command
#-----------------------------------------------------------------------------------
class ShowBgpNeighborsHistoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp neighbors [ ADDR | ADDR6 | PREFIX | PREFIX6 | ' \
            '( peer-group PEERGROUP_NAME ) ] history [ HISTORY_SCOPE ]' \
            '[ VRF ]'

   data = {
      'bgp' : CommonTokens.showBgp,
      'neighbors' : CommonTokens.neighbors,
      'ADDR' : CommonTokens.neighborAddr,
      'ADDR6' : CommonTokens.neighborAddr6,
      'PREFIX' : CommonTokens.neighborPrefix,
      'PREFIX6' : CommonTokens.neighborPrefix6,
      'peer-group' : CommonTokens.peerGroup,
      'PEERGROUP_NAME' : CommonTokens.peerGroupName,
      'history' : CommonHistoryTraits.history,
      'HISTORY_SCOPE' : CommonHistoryTraits.historyScope,
      'VRF' : CommonTokens.vrf,
   }

   cliModel = ShowNeighborsHistoryModel

   @staticmethod
   @ArBgpShowOutput( 'doShowBgpNeighborsHistory', arBgpModeOnly=True )
   def handler( mode, args ):
      cmd = CommonHistoryTraits.parse( mode, args, ArBgpAsyncCliCommand,
         'show neighbors history' )
      cmd.run()
      return cliPrinted( ShowBgpNeighborsHistoryCmd.cliModel )

#-----------------------------------------------------------------------------
# clear bgp [ ip | ip6 | prefix | prefix6 | peer-group peerGroupName ] history
#           [ { connect-failures } ] [ vrf vrfname ]
# -- See AID5484 for a description of this command
#-----------------------------------------------------------------------------
class ClearBgpNeighborsHistoryCmd( CliCommand.CliCommandClass ):
   syntax = 'clear bgp [ ADDR | ADDR6 | PREFIX | PREFIX6 | ' \
            '( peer-group PEERGROUP_NAME ) ] history [ HISTORY_SCOPE ] ' \
            '[ VRF ]'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'bgp' : 'Bgp',
      'ADDR' : CommonTokens.neighborAddr,
      'ADDR6' : CommonTokens.neighborAddr6,
      'PREFIX' : CommonTokens.neighborPrefix,
      'PREFIX6' : CommonTokens.neighborPrefix6,
      'peer-group' : CommonTokens.peerGroup,
      'PEERGROUP_NAME' : CommonTokens.peerGroupName,
      'history' : CommonHistoryTraits.history,
      'HISTORY_SCOPE' : CommonHistoryTraits.historyScope,
      'VRF' : CommonTokens.vrf,
   }

   @staticmethod
   def handler( mode, args ):
      cmd = CommonHistoryTraits.parse( mode, args, ArBgpCliCommand,
         'clear neighbors history' )
      cmd.run()

BasicCli.addShowCommandClass( ShowBgpNeighborsHistoryCmd )
BasicCli.EnableMode.addCommandClass( ClearBgpNeighborsHistoryCmd )

#---------------------------------------------------------------------------
# show dynamic prefix-list [ <dynamic prefix-list name> ] [ vrf <vrf-name> ]
#---------------------------------------------------------------------------
def doArBgpShowDynPfxList( mode, model, args ):
   ret = model()

   def populateDynamicPrefixList():
      def contributingRoutesGenerator():
         contributingRoute = (
               DynamicPrefixListModel.DynamicPrefixListContributingRoutes() )
         # We need to print maximum 10 matched routes
         count = 0
         for matchedRoute in matchedRoutes:
            count += 1
            if count > 10:
               break
            contributingRoute.prefix = matchedRoute
            yield contributingRoute

      for dynPfxListName, dynPfxList in (
            dynPfxListConfigDir.dynamicPrefixList.items() ):
         if not pfxListName or pfxListName == dynPfxListName:
            matchedRoutes = []

            # Dynamic prefix-list state can be one of the following:
            # 1. active
            # 2. inactive
            # 3. matchRouteMapNotConfigured
            # 4. notApplicableToRoutingInstance
            #
            # If dynamic prefix-list configuration does not have any match map
            # configured, then state would be 'matchRouteMapNotConfigured'.
            # If the configuration is complete, then we check the dplState for the
            # VRF in Sysdb. Sysdb only stores the state as 'active' or 'inactive'.
            # If the state for the VRF is not present, that would mean that the DPL
            # is not applicable for that VRF, so state would be
            # 'notApplicableToRoutingInstance'
            # If the show command is issued for a non-existent VRF, then the state
            # would be 'notApplicableToRoutingInstance'
            # The state could be 'inactive' if there are no matching routes or if
            # the match route-map configuration is missing. So, we check for the
            # config if state is 'inactive' and display the state as
            # 'matchRouteMapNotConfigured' if the configuration is missing.
            if vrfNameStatus.nameToIdMap is None:
               continue
            vrfId = vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )

            if dynPfxList.matchMap == '':
               dynPfxListState = 'matchRouteMapNotConfigured'
            else:
               dynPfxListState = 'notApplicableToRoutingInstance'
               # Get the dynamic prefix-list state from sysdb
               dynPfxListStatus = dynPfxListStatusDir.dynamicPrefixList.get(
                  dynPfxListName, None )
               if dynPfxListStatus and vrfId is not None:
                  dynPfxListState = dynPfxListStatus.activeInVrf.get( vrfId,
                     'notApplicableToRoutingInstance' )

               # If dynamic prefix list state is inactive, then check for match route
               # map configuration
               if ( dynPfxListState == 'inactive' and
                    dynPfxList.matchMap not in mapConfig.routeMap.keys() ):
                  dynPfxListState = 'matchRouteMapNotConfigured'

               # We need to populate matched routes only if state is active
               vrfRmStatusSet = None
               if vrfId is not None:
                  vrfRmStatusSet = allVrfRmStatusSet.vrfRmStatusSet.get( vrfId,
                                                                         None )
               if dynPfxListState == 'active' and vrfRmStatusSet:
                  # Avoid transient failures in lookups that could happen immediately
                  # after changing the config (like changing match-map name)
                  rmEvalStatus = None
                  rmEvalStatus = vrfRmStatusSet.rmEvalStatus.get(
                     dynPfxList.matchMap, None )
                  if not rmEvalStatus:
                     continue
                  matchedRoutes = rmEvalStatus.matchedRoute

            dynPfxListInst = DynamicPrefixListModel.DynamicPrefixListModel()
            dynPfxListInst.ipv4PrefixList = dynPfxList.ipv4PrefixList
            dynPfxListInst.ipv6PrefixList = dynPfxList.ipv6PrefixList
            dynPfxListInst.matchRouteMap = dynPfxList.matchMap
            dynPfxListInst.state = dynPfxListState
            dynPfxListInst.numContributingRoutes = len( matchedRoutes )
            dynPfxListInst.contributingRoutes = itertools.chain(
                  dynPfxListInst.contributingRoutes,
                  contributingRoutesGenerator() )
            yield dynPfxListName, dynPfxListInst

      if ( pfxListName and pfxListName not in
            dynPfxListConfigDir.dynamicPrefixList.keys() ):
         dynPfxListInst = DynamicPrefixListModel.DynamicPrefixListModel()
         dynPfxListInst.state = 'notApplicableToRoutingInstance'
         yield pfxListName, dynPfxListInst

   pfxListName = None
   vrfName = args[ 'vrfName' ]
   if 'prefix-list' in args.keys():
      pfxListName = args[ 'prefix-list' ]
   # pylint: disable-msg=W0212
   ret._vrf = vrfName
   ret.dynamicPrefixLists = itertools.chain( ret.dynamicPrefixLists,
                                             populateDynamicPrefixList() )
   return ret

routeMapShowCmdDict[ 'doArBgpShowDynPfxList' ] = doArBgpShowDynPfxList

#-------------------------------------------------------------------------------
# agent Bgp snapshot core
#-------------------------------------------------------------------------------
def bgpAgentSnapshotCore( mode, ns ):
   entityManager = mode.entityManager
   command = 'snapshot core'
   AgentCommandRequest.runSocketCommand( entityManager, BgpAgentName,
                                         command,
                                         '' )

snapshotCoreCallBack[ 'Bgp' ] = bgpAgentSnapshotCore

#-------------------------------------------------------------------------------
# show bgp rt-membership summary
#-------------------------------------------------------------------------------

rtMembershipAfterShowBgp = CliMatcher.KeywordMatcher( "rt-membership",
      helpdesc="Route Target Membership address family" )

class ShowBgpRtMembershipSummary( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rt-membership summary'
   data = {
         "bgp" : bgpAfterShow,
         "rt-membership" : rtMembershipAfterShowBgp,
         "summary" : RoutingBgpShowCliTokens.summary,
   }
   cliModel = summaryVrfModel

   @staticmethod
   @ArBgpShowOutput( 'doShowBgpRtMembershipSummary', arBgpModeOnly=True )
   def handler( mode, args ):
      cmd = ArBgpAsyncCliCommand( mode, 'show summary', vrfName=DEFAULT_VRF,
                                  nlriAfiSafi='rt-membership' )
      cmd.run()
      return cliPrinted( summaryVrfModel )

if Toggles.BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   BasicCli.addShowCommandClass( ShowBgpRtMembershipSummary )

#-------------------------------------------------------------------------------
# show bgp rt-membership floodlist [<rt>]
#-------------------------------------------------------------------------------

routeTargetFloodlist = CliMatcher.KeywordMatcher( 'floodlist',
                                                  'Route Target floodlist' )

class ShowRtMembershipFloodlist( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rt-membership floodlist [ ROUTE_TARGET ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'rt-membership' : CommonTokens.rtMembership,
      'floodlist' : routeTargetFloodlist,
      'ROUTE_TARGET' : RtSooExtCommCliMatcher( "Route Target" ),
   }

   cliModel = ShowRtMembershipFloodlistModel

   @staticmethod
   @ArBgpShowOutput( 'doShowRtMembershipFloodlist', arBgpModeOnly=True )
   def handler( mode, args ):
      cmd = ArBgpAsyncCliCommand( mode, 'show rt-membership floodlist' )
      if 'ROUTE_TARGET' in args:
         cmd.addParam( 'routeTarget', args[ 'ROUTE_TARGET' ] )
      cmd.run()
      return cliPrinted( ShowRtMembershipFloodlist.cliModel )

if Toggles.BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   BasicCli.addShowCommandClass( ShowRtMembershipFloodlist )

#-------------------------------------------------------------------------------
# show bgp neighbors <nbr> rt-membership filter
#-------------------------------------------------------------------------------

routeTargetFilter = CliMatcher.KeywordMatcher( 'filter',
                                               'Route Target filter' )

class ShowNeighborRtMembershipFilter( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp neighbors ( ADDR | ADDR6 ) rt-membership filter'
   data = {
      'bgp' : CommonTokens.showBgp,
      'neighbors' : CommonTokens.neighbors,
      'ADDR' : CommonTokens.neighborAddr,
      'ADDR6' : CommonTokens.neighborAddr6,
      'rt-membership' : CommonTokens.rtMembership,
      'filter' : routeTargetFilter,
   }

   cliModel = ShowRtMembershipNeighborFilterModel

   @staticmethod
   @ArBgpShowOutput( 'doShowBgpNeighborsRtMembershipFilter', arBgpModeOnly=True )
   def handler( mode, args ):
      peerAddr = args.get( 'ADDR' ) or args.get( 'ADDR6' )
      cmd = ArBgpAsyncCliCommand( mode, 'show neighbors rt-membership filter',
                                  peerAddr=peerAddr )
      cmd.run()
      return cliPrinted( ShowNeighborRtMembershipFilter.cliModel )

if Toggles.BgpCommonToggleLib.toggleBgpRtMembershipEnabled():
   BasicCli.addShowCommandClass( ShowNeighborRtMembershipFilter )

#-------------------------------------------------------------------------------
# show bgp rcf [ <func-name> ]
#-------------------------------------------------------------------------------

class ShowBgpRcf( ShowCommand.ShowCliCommandClass ):
   syntax = 'show bgp rcf [ FUNC_NAME ]'
   data = {
      'bgp' : CommonTokens.showBgp,
      'rcf' : RcfCliLib.rcfKw,
      'FUNC_NAME' : RcfCliLib.rcfFunctionMatcher,
   }

   cliModel = ShowBgpRcfModel

   @staticmethod
   def handler( mode, args ):
      if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
         return doShowBgpOutputNotSupported( mode, *args )
      cmd = ArBgpAsyncCliCommand( mode, 'show bgp rcf' )
      if 'FUNC_NAME' in args:
         funcNameParam = args[ 'FUNC_NAME' ]
         if not funcNameParam.endswith( '()' ):
            funcNameParam = funcNameParam + '()'
         cmd.addParam( 'functionName', funcNameParam )
      cmd.run()
      return cliPrinted( ShowBgpRcf.cliModel )

BasicCli.addShowCommandClass( ShowBgpRcf )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHwStatus
   global allVrfConfig
   global config, vrfConfigDir, mapConfig
   global asnConfig
   global arBgpConfig, arBgpVrfConfigDir, arBgpMapConfig
   global defaultInitiatorRmDir, allIntfStatusDir
   global l3Config
   global securityConfig
   global vrfNameStatus, dynPfxListConfigDir, dynPfxListStatusDir, allVrfRmStatusSet
   global ucmpConfig, ucmpVrfConfigDir
   global arUcmpConfig, arUcmpVrfConfigDir
   global aclCpConfig
   global rdConfigDir
   global lsImportConfig

   routingHwStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
   config = LazyMount.mount( entityManager, 'routing/bgp/config',
                             'Routing::Bgp::Config', 'r' )
   vrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/vrf/config',
                                   'Routing::Bgp::VrfConfigDir', 'r' )
   mapConfig = LazyMount.mount( entityManager, 'routing/routemap/config',
                                'Routing::RouteMap::Config', 'r' )
   asnConfig = LazyMount.mount( entityManager, 'routing/bgp/asn/config',
                                  'Routing::AsnConfig', 'r' )
   arBgpConfig = LazyMount.mount( entityManager, 'routing/arbgp/bgp/config',
                                  'Routing::Bgp::Config', 'r' )
   arBgpVrfConfigDir = LazyMount.mount( entityManager, 'routing/arbgp/vrf/config',
                                       'Routing::Bgp::VrfConfigDir', 'r' )
   arBgpMapConfig = LazyMount.mount( entityManager, 'routing/arbgp/routemap/config',
                                     'Routing::RouteMap::Config', 'r' )
   securityConfig = LazyMount.mount( entityManager, 'mgmt/security/config',
                                    'Mgmt::Security::Config', 'r' )

   defaultInitiatorRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/initiatorRM/',
      'Routing::RouteMap::Config', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'r' )

   # initialize the global ext comm parser state machinery, or get the existing one
   ArBgpCliCommon.getBgpExtCommParserContainerSm()

   # Mounts for dynamic prefix list, maintenance bgp show commands
   vrfNameStatus = LazyMount.mount( entityManager,
                                    Cell.path( 'vrf/vrfNameStatus' ),
                                    'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
   dynPfxListConfigDir = LazyMount.mount( entityManager, 'routing/dynPfxList/config',
                                          'Routing::DynamicPrefixList::Config', 'r' )
   dynPfxListStatusDir = LazyMount.mount( entityManager, 'routing/dynPfxList/status',
         'Routing::DynamicPrefixList::AllDynamicPrefixListVrfSet', 'r' )
   allVrfRmStatusSet = LazyMount.mount( entityManager, 'routing/rmeval/status',
                                        'Routing::RmEval::AllVrfRmStatusSet', 'r' )

   ucmpConfig = LazyMount.mount( entityManager, 'routing/ucmp/bgp/config',
                                 'RoutingLib::Ucmp::UcmpConfig', 'r' )
   ucmpVrfConfigDir = LazyMount.mount( entityManager,
                                       'routing/ucmp/bgp/vrf/config',
                                       'RoutingLib::Ucmp::VrfUcmpConfigDir',
                                       'r' )
   arUcmpConfig = LazyMount.mount( entityManager, 'routing/arbgp/ucmp/config',
                                   'RoutingLib::Ucmp::UcmpConfig', 'r' )
   arUcmpVrfConfigDir = LazyMount.mount( entityManager,
                                         'routing/arbgp/ucmp/vrf/config',
                                         'RoutingLib::Ucmp::VrfUcmpConfigDir',
                                         'r' )
   aclCpConfig = LazyMount.mount( entityManager, 'acl/cpconfig/cli',
                                  'Acl::Input::CpConfig', 'r' )
   rdConfigDir = LazyMount.mount( entityManager,
                              'ip/vrf/routeDistinguisherInputDir/config/bgp',
                              'Ip::BgpRouteDistinguisherInput', 'r' )
   lsImportConfig = LazyMount.mount( entityManager, 'routing/bgp/lsImportConfig',
                        'Routing::Bgp::LinkStateImportConfig', 'w' )
