# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import sys

import AgentDirectory
import AllocTrackShowHelper
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliPlugin.AgentCli as AgentCli
import CliPlugin.AgentResourceCli as AgentResourceCli
from CliPlugin.AllocTrackModel import AllocTrackModel
import CliToken.Agent
import LazyMount
import ShowCommand
import Tac

allocTrackShowHelper = None

def showImpl( mode, args, agent, snapshot, filterTacc ):
   typesFromLastRun = allocTrackShowHelper.typesFromLastRun( mode )
   renderOptions = {}
   renderOptions[ '_maxNameLen' ] = 100
   renderOptions[ '_tableWidth' ] = 200
   renderOptions[ '_limit' ] = args.get( 'LIMIT', 0 )
   noDeltaBase = False
   delta = args.get( 'DELTA' )
   sort = args.get( 'SORT' )

   if sort:
      if sort == 'high-watermark':
         renderOptions[ '_sortOrder' ] = 'highWatermarkMemory'
      elif sort == 'total-allocations':
         renderOptions[ '_sortOrder' ] = 'totalAllocations'
      else:
         renderOptions[ '_sortOrder' ] = 'currentMemory'
   elif delta:
      baseTypes = typesFromLastRun.get( agent )
      if baseTypes is None:
         noDeltaBase = True
         baseTypes = AllocTrackModel( types={} ).types
      renderOptions[ '_fullOrDelta' ] = 'delta'
      renderOptions[ '_baseTypes' ] = baseTypes
      sortOrder = 'currentMemory' if delta == 'current' else 'transientMemory'
      renderOptions[ '_sortOrder' ] = sortOrder
   model = AllocTrackShowHelper.allocTrackSnapshotToModel( snapshot,
         typeFilter=args.get( 'TYPE_PREFIX' ),
         filterTacc=filterTacc, **renderOptions )
   typesFromLastRun[ agent ] = model.types
   if noDeltaBase:
      print( 'No base allocation data to diff against.' )
      print( 'Setting base for next command invocation.' )
      return AllocTrackModel()
   return model

def getAgents():
   # this logic to create the agent name comes from Agent.py
   def sysdbAgentName( ( agentName, subsystemId ) ):
      if subsystemId is None or subsystemId == "":
         return agentName
      return agentName + "-" + subsystemId
   return set( map( sysdbAgentName,
                    AgentDirectory.agents( allocTrackShowHelper.sysname ).keys() ) )

#--------------------------------------------------------------------------------
# show agent [ AGENT ] memory allocations [ summary ] [ ( sort SORT | delta DELTA )
#                                   [ limit LIMIT ] ] [ filter TYPE_PREFIX ] [ all ]
#--------------------------------------------------------------------------------
class AgentMemoryAllocationsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show agent [ AGENT ] memory allocations [ summary ] '
              '[ ( ( sort SORT ) | ( delta DELTA ) ) [ limit LIMIT ] ] '
              '[ filter TYPE_PREFIX ] [ all ]' )
   data = {
      'agent': CliToken.Agent.agentKwForShow,
      'AGENT': AgentCli.agentNameNewMatcher,
      'memory': AgentResourceCli.matcherMemory,
      'allocations': 'Show per-type memory allocation info',
      'summary': 'Show sum of memory allocations across all running agents',
      'sort': 'Type information sort order',
      'SORT': CliMatcher.EnumMatcher( {
         'high-watermark': 'Sort types by memory use high watermark',
         'current': 'Sort types by current memory use',
         'total-allocations': 'Sort types by total allocations',
      } ),
      'delta': 'Allocation changes over a period of time',
      'DELTA': CliMatcher.EnumMatcher( {
         'current': 'Display delta in current memory use',
         'transient': 'Display transient allocations',
      } ),
      'limit': 'Number of types to display',
      'LIMIT': CliMatcher.IntegerMatcher( 1, sys.maxint, helpdesc='Type limit' ),
      'filter': 'Display only type names starting with prefix',
      'TYPE_PREFIX': CliMatcher.PatternMatcher( '[^|].*',
                                                helpdesc='Type name prefix',
                                                helpname='PREFIX' ),
      'all': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'all',
                                            helpdesc='Disable default filtering' ),
         hidden=True )
   }

   @staticmethod
   def handler( mode, args ):
      filterTacc = 'all' not in args
      agent = args.get( 'AGENT', '*' )
      if agent != '*':
         snapshot = allocTrackShowHelper.getSnapshot( agent )
         if snapshot is None:
            return AllocTrackModel()
      else:
         sumSnapshot = Tac.newInstance( 'Tac::AllocTrackSnapshotEnt', 0 )
         allAgents = getAgents()
         # Skip all Smbus agents and Stp/StpTxRx, we hit a bug in
         # EntityManager for these agents triggered by having a non
         # default listenSocketName. See BUG353782
         allAgents = [ agent for agent in allAgents
                       if not agent.startswith( 'Smbus' ) and agent != 'Stp'
                       and agent != 'StpTxRx' ]
         snapshots, err = allocTrackShowHelper.getSnapshots( agents=allAgents,
                                                             blocking=False )
         for snapshot in snapshots.itervalues():
            sumSnapshot.add( snapshot )
         for _err in err.itervalues():
            print( _err )
         snapshot = sumSnapshot
      return showImpl( mode, args, agent, snapshot, filterTacc )

   cliModel = AllocTrackModel

BasicCli.addShowCommandClass( AgentMemoryAllocationsCmd )

def Plugin( entityManager ):
   global allocTrackShowHelper
   agentConfigDir = LazyMount.mount( entityManager,
                                     Cell.path( 'agent/config' ),
                                     'Tac::Dir', 'wi' )
   agentStatusDir = LazyMount.mount( entityManager,
                                     Cell.path( 'agent/status' ),
                                     'Tac::Dir', 'ri' )
   allocTrackShowHelper = AllocTrackShowHelper.AllocTrackShowHelper(
      entityManager.sysname(), agentConfigDir, agentStatusDir )
