# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re
import sys

import BasicCli
import BasicCliSession
import CliCommand
import CliMatcher
from CliMode.Alias import AliasMode
import CliParser
import CliPlugin.CliCliModel as CliCliModel
import CliSession
import CliToken.Cli
import ConfigMount
import ShowCommand
import Tac
from TypeFuture import TacLazyType

Alias = TacLazyType( 'Cli::Config::Alias' )

cliConfig = None

def maybeAddAliasSessionOnCommitHandler( mode ):
   if mode.session.inConfigSession():
      callback = BasicCliSession.resetRegexAliases
      em = mode.entityManager
      CliSession.registerSessionOnCommitHandler( em, 'regexAliases', callback )

class AliasConfigMode( AliasMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'alias configuration'
   modeParseTree = CliParser.ModeParseTree()
   # We already have a previous implementation for 'show active'.
   showActiveCmdRegistered_ = True

   # Maximum sequence number for command.
   MAX_SEQ = 0xFFFFFFFF # 32-bit integer
   # Maximum number of commands we allow for each alias.
   MAX_CMD = 32
   # The incremental step, default is 10.
   INC_STEP = 10

   #----------------------------------------------------------------------------
   # Constructs a new AliasConfigMode instance for an alias of multi-line
   # commands. This is called every time the user enters "alias-name" mode.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, cmdAlias=None, regex=None ):
      self.compiledRegex = regex
      if regex:
         self.commentKeyTemplate = 'alias-"%s"'
         self.aliases = cliConfig.regexAlias

         # Check if we are editing an existing alias.
         for idx, alias in self.aliases.items():
            if alias.cmdAlias == cmdAlias:
               self.key = idx
               break
         else:
            self.key = 0

         # Prompt with parens (likely in Regex) leads to issues with tests.
         prompt = 'regex'
      else:
         self.commentKeyTemplate = 'alias-%s'
         self.aliases = cliConfig.alias
         self.key = cmdAlias.lower()
         prompt = cmdAlias

      try:
         self.alias = Tac.nonConst( self.aliases[ self.key ] )
      except KeyError:
         self.alias = Alias( cmdAlias )

      AliasMode.__init__( self, prompt )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def _commitAlias( self ):
      maybeAddAliasSessionOnCommitHandler( self )
      if self.alias is not None:
         if not self.alias.originalCmd:
            # Remove associated comment if any.
            self.removeComment()
            self.addError( "Empty alias not allowed" )
         else:
            if not self.key:
               # Committing a brand new regex alias.
               self.key = BasicCliSession.addCompiledRegex( self.compiledRegex,
                                                            self.aliases )
            self.aliases[ self.key ] = self.alias

   def _lastSeq( self ):
      return self.alias.originalCmd.keys()[ -1 ] if self.alias.originalCmd else 0

   def onExit( self ):
      self._commitAlias()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self, args ):
      # Remove associated comment if any.
      self.removeComment()
      self.alias = None
      self.session_.gotoParentMode( )

   def commentKey( self ):
      # Comments are stored in Sysdb under the mode's key.
      # Regex alias config mode has a plain, 'regex', prompt,
      # because parens are likely in regex and that confuses tests,
      # but using 'regex' for this key puts all regex alias modes' comments
      # under the same key. So we override this.
      result = self.commentKeyTemplate % self.alias.cmdAlias
      return result

   def addCmd( self, args ):
      originalCmd = args[ 'COMMAND' ]
      if len( self.alias.originalCmd ) < self.MAX_CMD:
         theSeq = self._lastSeq() + self.INC_STEP
         self.alias.originalCmd[ theSeq ] = originalCmd
      else:
         self.addError( "Exceeding maximum number of commands allowed: %d"
                        % self.MAX_CMD )

   def editCmd( self, args ):
      self.alias.originalCmd[ args[ 'SEQ' ] ] = args[ 'COMMAND' ]

   def removeCmd( self, args ):
      del self.alias.originalCmd[ args[ 'SEQ' ] ]

   def getAliasConfig( self, name ):
      return cliConfig.alias.get( name, None )

def verifyAllowedAliasOrStop( mode, alias ):
   '''We don't allow aliasing of certain keywords or empty regex aliases.'''
   alias = alias.lower()
   if alias in ( 'no', 'default', 'alias', '' ):
      mode.addErrorAndStop( '%r is not allowed as an alias name' % alias )

#------------------------------------------------------------------------------------
# alias ALIAS | REGEX
#------------------------------------------------------------------------------------
aliasKwMatcher = CliMatcher.KeywordMatcher( 'alias',
      helpdesc='Add a command alias' )
aliasRegexMatcher = CliMatcher.QuotedStringMatcher(
      helpdesc='Python-compatible RegEx command alias. Use ^V to escape \'?\'',
      helpname='QUOTED STRING',
      requireQuote=True,
      priority=CliParser.PRIO_HIGH )
cmdAliasMatcher = CliMatcher.PatternMatcher( r'.+',
      helpname='WORD',
      helpdesc='Command alias string' )
originalCmdMatcher = CliMatcher.StringMatcher(
      helpname='LINE',
      helpdesc='Original command string' )

class GotoAliasMode( CliCommand.CliCommandClass ):
   syntax = 'alias ALIAS'
   data = {
      'alias': aliasKwMatcher,
      'ALIAS': cmdAliasMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      alias = args[ 'ALIAS' ]
      verifyAllowedAliasOrStop( mode, alias )
      childMode = mode.childMode( AliasConfigMode, cmdAlias=alias )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( GotoAliasMode )

def compileRegexOrStop( mode, pattern ):
   '''Compile regex pattern to make sure it's valid.
   Stop further handler execusion if it's not.
   '''
   verifyAllowedAliasOrStop( mode, pattern )
   try:
      return re.compile( pattern )
   except re.error as e:
      mode.addErrorAndStop( str( e ) )

class GotoRegexAliasMode( CliCommand.CliCommandClass ):
   syntax = 'alias REGEX'
   data = {
      'alias': aliasKwMatcher,
      'REGEX': aliasRegexMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      pattern = args[ 'REGEX' ]
      regex = compileRegexOrStop( mode, pattern )
      childMode = mode.childMode( AliasConfigMode, cmdAlias=pattern, regex=regex )
      mode.session_.gotoChildMode( childMode )

BasicCli.GlobalConfigMode.addCommandClass( GotoRegexAliasMode )

#------------------------------------------------------------------------------------
# alias commands path PATH
#------------------------------------------------------------------------------------
def onSessionCommitFn( mode, onSessionCommit ):
   if onSessionCommit and mode.session_ and not mode.session_.isEapiClient():
      mode.session_.loadDynamicAliases()

class CommandsPathCmd( CliCommand.CliCommandClass ):
   syntax = 'alias commands path PATH'
   noOrDefaultSyntax = 'alias commands path ...'
   data = {
      'alias': aliasKwMatcher,
      'commands': CliMatcher.KeywordMatcher( 'commands',
                      helpdesc='Create Cli aliases for all executable files '
                               'under a given path',
                      priority=CliParser.PRIO_HIGH ),
      'path': 'Load commands from path',
      'PATH': CliMatcher.PatternMatcher( '.+',
                                          helpdesc='Path to load commands',
                                          helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      cmdsPath = args.get( 'PATH', '' )
      if mode.session_:
         cliConfig.commandsPath = cmdsPath
         if not mode.session_.isEapiClient():
            mode.session_.loadDynamicAliases()

         mode.session_.maybeCallConfigSessionOnCommitHandler( "commandsPath",
                                                              onSessionCommitFn )
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( CommandsPathCmd )

#------------------------------------------------------------------------------------
# alias ALIAS COMMAND
#------------------------------------------------------------------------------------
def removeAlias( mode, cmdAlias, collection ):
   '''Remove an alias from a given collection and any associated comments.
   '''
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, AliasConfigMode ) and lastMode.alias
        and lastMode.alias.cmdAlias == cmdAlias ):
      # If we are deleting ourselves, make sure we don't write it back when
      # we exit.
      lastMode.alias = None
   # Remove associated comment if any. commentKey should be same as what user
   # entered initially.
   alias = collection.get( cmdAlias )
   if alias:
      mode.removeCommentWithKey( 'alias-' + alias.cmdAlias )
      del collection[ cmdAlias ]

class ModifyAliasCmd( CliCommand.CliCommandClass ):
   syntax = 'alias ALIAS COMMAND'
   noOrDefaultSyntax = 'alias ALIAS ...'
   data = {
      'alias': aliasKwMatcher,
      'ALIAS': cmdAliasMatcher,
      'COMMAND': originalCmdMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      alias = args[ 'ALIAS' ]
      command = args[ 'COMMAND' ]
      verifyAllowedAliasOrStop( mode, alias )
      theAlias = Alias( alias )
      theAlias.originalCmd[ AliasConfigMode.INC_STEP ] = command
      cliConfig.alias[ alias.lower() ] = theAlias

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      removeAlias( mode, args[ 'ALIAS' ].lower(), cliConfig.alias )

BasicCli.GlobalConfigMode.addCommandClass( ModifyAliasCmd )

#------------------------------------------------------------------------------------
# alias REGEX TEMPLATE
#------------------------------------------------------------------------------------
class ModifyRegexAliasCmd( CliCommand.CliCommandClass ):
   syntax = 'alias REGEX TEMPLATE'
   noOrDefaultSyntax = 'alias REGEX ...'
   data = {
      'alias': aliasKwMatcher,
      'REGEX': aliasRegexMatcher,
      'TEMPLATE': CliMatcher.StringMatcher(
                        helpname='LINE',
                        helpdesc='A string template for this alias' ),
   }

   @staticmethod
   def handler( mode, args ):
      pattern = args[ 'REGEX' ]
      regex = compileRegexOrStop( mode, pattern )
      index = BasicCliSession.getRegexAliasIndex( pattern, cliConfig.regexAlias )
      if index:
         theAlias = Tac.nonConst( cliConfig.regexAlias[ index ] )
      else:
         index = BasicCliSession.addCompiledRegex( regex, cliConfig.regexAlias )
         theAlias = Alias( regex.pattern )

      theAlias.originalCmd[ AliasConfigMode.INC_STEP ] = args[ 'TEMPLATE' ]
      cliConfig.regexAlias[ index ] = theAlias
      maybeAddAliasSessionOnCommitHandler( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      BasicCliSession.delRegexAlias( args[ 'REGEX' ], cliConfig.regexAlias )
      maybeAddAliasSessionOnCommitHandler( mode )

BasicCli.GlobalConfigMode.addCommandClass( ModifyRegexAliasCmd )

#------------------------------------------------------------------------------------
# command COMMAND
#------------------------------------------------------------------------------------
class AliasAddCmd( CliCommand.CliCommandClass ):
   syntax = 'command COMMAND'
   data = {
      'command': 'Add a command to the end of the sequence',
      'COMMAND': originalCmdMatcher
   }

   handler = AliasConfigMode.addCmd

AliasConfigMode.addCommandClass( AliasAddCmd )

#------------------------------------------------------------------------------------
# [ no | default ] SEQ COMMAND
#------------------------------------------------------------------------------------
class AliasEditCmd( CliCommand.CliCommandClass ):
   syntax = 'SEQ COMMAND'
   noOrDefaultSyntax = 'SEQ ...'
   data = {
      'SEQ': CliMatcher.IntegerMatcher( 1, AliasConfigMode.MAX_SEQ,
         helpdesc='Unsigned sequence number' ),
      'COMMAND': originalCmdMatcher
   }

   handler = AliasConfigMode.editCmd
   noOrDefaultHandler = AliasConfigMode.removeCmd

AliasConfigMode.addCommandClass( AliasEditCmd )

#------------------------------------------------------------------------------------
# abort
#------------------------------------------------------------------------------------
class AliasAbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': CliToken.Cli.abortMatcher
   }

   handler = AliasConfigMode.abort

AliasConfigMode.addCommandClass( AliasAbortCmd )

#------------------------------------------------------------------------------------
# show [ pending ]
#------------------------------------------------------------------------------------
def _showList( mode, alias, output=None ):
   if output is None:
      output = sys.stdout
   if alias is None or not alias.originalCmd:
      return
   template = 'alias "%s"\n' if mode.compiledRegex else 'alias %s\n'
   output.write( template % alias.cmdAlias )
   for seq, oCmd in alias.originalCmd.iteritems():
      output.write( '   %d\t%s\n' % ( seq, oCmd ) )

class ShowPending( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending ]'
   data = {
      'pending': 'Show pending list in this session'
   }

   @staticmethod
   def handler( mode, args ):
      _showList( mode, mode.alias )

AliasConfigMode.addShowCommandClass( ShowPending )

#------------------------------------------------------------------------------------
# show active
#------------------------------------------------------------------------------------
class ShowActive( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active'
   data = {
      'active': 'Show the list in the current running-config'
   }

   @staticmethod
   def handler( mode, args ):
      _showList( mode, mode.getAliasConfig( mode.alias.cmdAlias.lower() ) )

AliasConfigMode.addShowCommandClass( ShowActive )

#------------------------------------------------------------------------------------
# show diff
#------------------------------------------------------------------------------------
class ShowDiff( ShowCommand.ShowCliCommandClass ):
   syntax = 'show diff'
   data = {
      'diff': 'Show the difference between active and pending list'
   }

   @staticmethod
   def handler( mode, args ):
      # generate diff between active and pending
      import difflib
      import cStringIO
      activeOutput = cStringIO.StringIO()
      _showList( mode, mode.getAliasConfig( mode.alias.cmdAlias.lower() ),
                 output=activeOutput )
      pendingOutput = cStringIO.StringIO()
      _showList( mode, mode.alias, output=pendingOutput )
      diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                   pendingOutput.getvalue().splitlines(),
                                   lineterm='' )
      print( '\n'.join( diff ) )

AliasConfigMode.addShowCommandClass( ShowDiff )

#------------------------------------------------------------------------------------
# show aliases
#------------------------------------------------------------------------------------
def _createAliases( aliasCollection ):
   AliasModel = CliCliModel.Aliases.Alias
   return { alias.cmdAlias: AliasModel( commands=dict( alias.originalCmd ) )
            for alias in aliasCollection.values() }

class ShowAliases( ShowCommand.ShowCliCommandClass ):
   syntax = 'show aliases [ match STRING ]'
   data = {
      'aliases': 'List all configured aliases',
      'match': 'List all RegEx aliases matching the following string',
      'STRING': CliMatcher.QuotedStringMatcher( helpdesc='String to test against' ),
   }
   cliModel = CliCliModel.Aliases

   @staticmethod
   def handler( mode, args ):
      testString = args.get( 'STRING' )
      if testString:
         aliases = {}
         dynamicAliases = {}
      else:
         aliases = _createAliases( cliConfig.alias )
         dynamicAliases = _createAliases( cliConfig.dynamicAlias )

      # For regex aliases, order matters,
      # and we filter them if 'STRING' has been specified.
      regexIndices = [ i
                       for i, alias in cliConfig.regexAlias.items()
                       if not testString or re.match( alias.cmdAlias, testString ) ]

      regexAliases = []
      RegexAlias = CliCliModel.Aliases.RegexAlias
      for index in sorted( regexIndices ):
         alias = cliConfig.regexAlias[ index ]
         regexAliases.append( RegexAlias( regex=alias.cmdAlias,
                                          commands=dict( alias.originalCmd ) ) )

      return CliCliModel.Aliases( commandsPath=cliConfig.commandsPath,
                                  aliases=aliases,
                                  regexAliases=regexAliases,
                                  dynamicAliases=dynamicAliases )

BasicCli.addShowCommandClass( ShowAliases )

#------------------------------------------------------------------------------------
# Plugin Func
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, 'cli/config', 'Cli::Config', 'w' )
