#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from cStringIO import StringIO
import json

import AgentCommandRequest
from AleVerifyCliLib import VerifyPlatformStateCapiModel, VerifyPlatformStateHelper
import AleCliLib
import BasicCli
import CliMatcher
import CliParser
import CliToken.Platform
import LazyMount
import ShowCommand
import Tac

agentConfigDir = None

allAgentCompletion = [ CliParser.Completion( 'WORD', 'Forwarding Agent name', 
                                             literal=False ) ]
def allAgentNames( mode ):
   agents = [ agentName[ 0 ] for agentName in agentConfigDir.items() ]
   return agents

agentNameMatcher = CliMatcher.DynamicKeywordMatcher(
   allAgentNames,
   emptyTokenCompletion=allAgentCompletion )

#--------------------------------------------------------------------------------
# show platform [ AGENT ] pending-state [ quiet ]
#--------------------------------------------------------------------------------
def _getPendingStateCommand( quiet='quiet' ):
   cmd = 'AATS#' + quiet
   return cmd

class PlatformPendingStateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform [ AGENT ] pending-state [ quiet ]'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'AGENT' : agentNameMatcher,
      'pending-state' : 'Show agent state',
      'quiet' : 'Print only in transient state',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      quiet = args.get( 'quiet', '' )
      agent = args.get( 'AGENT' )
      agents = [ agent ] if agent else allAgentNames( mode )
      if not agents:
         print 'No agents support this command'
         return
      cmd = _getPendingStateCommand( quiet )
      for agent in agents:
         AgentCommandRequest.runSocketCommand( AleCliLib.entityManager, 'ale',
                                               ( 'aleVerify::%s' % agent ), cmd )

BasicCli.addShowCommandClass( PlatformPendingStateCmd )

#--------------------------------------------------------------------------------
# show platform agent-state verify ( AGENT | all )
#--------------------------------------------------------------------------------
def _runVerifyCliSocketCommand( agent, checkPending=True ):
   if checkPending:
      cmd = _getPendingStateCommand()
   else:
      # need a ending '#' to prevent agent name being
      # substr of another agent name. we skipped agents that have '#'
      cmd = 'VTS#' + agent + '#'
   cmdBuf = StringIO()
   AgentCommandRequest.runSocketCommand( AleCliLib.entityManager, 'ale',
                                         ( 'aleVerify::%s' % agent ), cmd,
                                         stringBuff=cmdBuf )
   return cmdBuf.getvalue()

class PlatformAgentStateVerifyAllCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform agent-state verify ( AGENT | all )'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'agent-state' : 'The agent state',
      'verify' : 'Verify the agent state',
      'all' : 'Verify all agents',
      'AGENT' : agentNameMatcher,
   }
   hidden = True
   privileged = True
   cliModel = VerifyPlatformStateCapiModel

   @staticmethod
   def handler( mode, args ):
      agent = args.get( 'AGENT' )
      agents = []
      results = dict()
      if agent:
         agents.append( agent )
      else:
         agents.extend( allAgentNames( mode ) )
      if not agents:
         mode.addError( 'No agents support this command' )
      for agent in agents:
         if '#' in agent:
            mode.addError( 'Skipping agent: %s' % agent )
            continue
         pendingStateOutput = _runVerifyCliSocketCommand( agent )
         if pendingStateOutput.strip():
            output = { agent: [ { '_' : VerifyPlatformStateHelper.retry() } ] }
         else:
            output = _runVerifyCliSocketCommand( agent, checkPending=False )
            try:
               output = json.loads( output )
            except ValueError:
               mode.addError( VerifyPlatformStateHelper.garbled( agent ) )
               output = { agent: 
                     [ { '_' : VerifyPlatformStateHelper.crash( output ) } ] }
         results.update( output )
      capiResults = VerifyPlatformStateCapiModel()
      capiResults.setDict( results )
      return capiResults

BasicCli.addShowCommandClass( PlatformAgentStateVerifyAllCmd )

def Plugin( entityManager ):
   global agentConfigDir
   agentConfigDir = LazyMount.mount( entityManager,
                                     'hardware/ale/verifyAgentConfigDir',
                                     'Tac::Dir', 'ri' )
