#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AleCliLib
import CliParser
import CliCommand
import BasicCli
import CliToken.Ip
import CliToken.Ipv6
from CliMatcher import (
   DynamicIntegerMatcher,
   FloatMatcher,
   IntegerMatcher,
   KeywordMatcher,
)
import Tac
from IraCommonCli import fibMatcher

# Global tokens or keyword matcher
ipForConfigMatcher = CliToken.Ip.ipMatcherForConfig
ipv6ForConfigMatcher = CliToken.Ipv6.ipv6MatcherForConfig

hardwareMatcher = CliToken.Hardware.hardwareForConfigMatcher

nextHopMatcher = KeywordMatcher(
                     'next-hop',
                     helpdesc='Next-hop configuration' )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop sharing"
#-------------------------------------------------------------------------------
def setAdjSharing( mode, args ):
   if CliCommand.isDefaultCmd( args ):
      val = 'platformDefault'
   elif CliCommand.isNoCmd( args ):
      val = 'disabled'
   else:
      val = 'enabled'
   AleCliLib.routingHwConfig.adjSharing = val

class AleAdjSharingCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop sharing'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'sharing': "Configure next-hop sharing mode",
   }
   handler = setAdjSharing
   noOrDefaultHandler = setAdjSharing

BasicCli.GlobalConfigMode.addCommandClass( AleAdjSharingCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop multi-path maximum <max>"
#-------------------------------------------------------------------------------
def maxAleEcmpRange( mode ):
   upperBound = AleCliLib.routingHwStatus.maxLogicalEcmp
   if upperBound == 0:
      upperBound = 0xFFFFFFFF
   return ( 1, upperBound )

class AleMultiPathMaxCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop multi-path maximum MAX_VAL'
   noOrDefaultSyntax = 'ip hardware fib next-hop multi-path maximum ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'multi-path': "Multi-path configuration",
      'maximum': "Configure max ale ecmp",
      'MAX_VAL': DynamicIntegerMatcher( maxAleEcmpRange,
                                        helpdesc="Max Ale Ecmp" )
   }

   @staticmethod
   def handler( mode, args ):
      AleCliLib.routingHwConfig.maxAleEcmp = args[ 'MAX_VAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # set to default value which maps to maxEcmp
      AleCliLib.routingHwConfig.maxAleEcmp = \
         AleCliLib.routingHwConfig.maxAleEcmpDefault

BasicCli.GlobalConfigMode.addCommandClass( AleMultiPathMaxCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib default route protected"
#-------------------------------------------------------------------------------
def setDefaultRouteProtection( mode, args ):
   enabled = not CliCommand.isNoOrDefaultCmd( args )
   AleCliLib.routingHwConfig.protectDefaultRoute = enabled
   AleCliLib.routing6HwConfig.protectDefaultRoute = enabled
   if not AleCliLib.routingHwStatus.protectDefaultRouteHitless:
      mode.addWarning( "Please restart layer 3 forwarding agent to ensure " \
                       "routes are programmed properly" )

class AleDefaultRouteProtectCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib default route protected'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'default': "Configure default route",
      'route': "Configure route",
      'protected': "Configure default route protection"
   }
   handler = setDefaultRouteProtection
   noOrDefaultHandler = setDefaultRouteProtection

BasicCli.GlobalConfigMode.addCommandClass( AleDefaultRouteProtectCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib route delete delay <seconds>
#-------------------------------------------------------------------------------
class AleRouteDeleteDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib route delete delay SECONDS'
   noOrDefaultSyntax = 'ip hardware fib route delete delay ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'route': "Routes",
      'delete': "Delete",
      'delay': 'Delay',
      'SECONDS': IntegerMatcher( 0, 60, helpdesc="Number of seconds" )
   }

   @staticmethod
   def handler( mode, args ):
      timeDelay = args[ 'SECONDS' ]
      AleCliLib.routingHwConfig.deletionDelay = timeDelay
      AleCliLib.routing6HwConfig.deletionDelay = timeDelay

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      AleCliLib.routingHwConfig.deletionDelay = \
                              AleCliLib.routingHwConfig.defaultDeletionDelay
      AleCliLib.routing6HwConfig.deletionDelay = \
                              AleCliLib.routingHwConfig.defaultDeletionDelay

BasicCli.GlobalConfigMode.addCommandClass( AleRouteDeleteDelayCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib next-hop update event bfd
#-------------------------------------------------------------------------------
def setNhUpdateEvent( mode, args ):
   # Default results in enabling this feature.
   enabled = not CliCommand.isNoCmd( args )
   AleCliLib.routingHwConfig.bfdPeerEventEnabled = enabled

class AleUpdateEventBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop update event bfd'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'update': "Update Fib routes",
      'event': "Event type",
      'bfd': "BFD status"
   }
   handler = setNhUpdateEvent
   noOrDefaultHandler = setNhUpdateEvent

BasicCli.GlobalConfigMode.addCommandClass( AleUpdateEventBfdCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib maximum routes <num-routes>"
# "[no|default] ipv6 hardware fib maximum routes <num-routes>"
#-------------------------------------------------------------------------------
class AleMaxRoutesCmd( CliCommand.CliCommandClass ):
   syntax = '( ip | ipv6 ) hardware fib maximum routes MAX_ROUTES'
   noOrDefaultSyntax = '( ip | ipv6 ) hardware fib maximum routes ...'
   data = {
      'ip': ipForConfigMatcher,
      'ipv6': ipv6ForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'maximum': "Softlimit of Max routes",
      'routes': "IP routes in this address family",
      'MAX_ROUTES': IntegerMatcher( 1, 0xFFFFFFFF,
                                    helpdesc="number of max Routes" )
   }

   @staticmethod
   def _getConfig( args ):
      return ( AleCliLib.routing6HwConfig if 'ipv6' in args
               else AleCliLib.routingHwConfig )

   @staticmethod
   def handler( mode, args ):
      AleMaxRoutesCmd._getConfig( args ).maxNumRoutes = args[ 'MAX_ROUTES' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      AleMaxRoutesCmd._getConfig( args ).maxNumRoutes = 0

BasicCli.GlobalConfigMode.addCommandClass( AleMaxRoutesCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop weight-deviation <percentage>"
#-------------------------------------------------------------------------------
class AleWeightDeviationCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop weight-deviation PERCENT'
   noOrDefaultSyntax = 'ip hardware fib next-hop weight-deviation ...'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'weight-deviation': "Configure weight deviation",
      'PERCENT': FloatMatcher( 0.01, 50.0,
                               helpdesc='Weight deviation percentage',
                               precisionString='%.2f' )
   }

   @staticmethod
   def handler( mode, args ):
      AleCliLib.routingHwConfig.ucmpWeightDeviation = args[ 'PERCENT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      AleCliLib.routingHwConfig.ucmpWeightDeviation = 0.0

BasicCli.GlobalConfigMode.addCommandClass( AleWeightDeviationCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop flattening weight-expansion
#              ucmp [ecmp]
#-------------------------------------------------------------------------------

def setEcmpHfecWeightExpansion( enabled ):
   AleCliLib.routingHwConfig.hfecExpandEcmpNexthopWeight = enabled

hierarchicalMatcher = KeywordMatcher( 'hierarchical',
                                      helpdesc="Hardware resource configuration" )
hierarchcialNextHopMatcher = KeywordMatcher( 'next-hop',
                                             helpdesc='Next-hop configuration' )

class AleHierarchicalWeightExpansionCmd( CliCommand.CliCommandClass ):
   syntax = '''ip hardware fib hierarchical next-hop flattening weight-expansion
               ucmp [ ecmp ]
            '''
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'flattening': "Flattening of next-hop configuration",
      'weight-expansion': "Follow configured next-hop weights by replication",
      'ucmp': "Unequal cost multipath expansion",
      'ecmp': "Equal cost multipath expansion"
   }

   @staticmethod
   def handler( mode, args ):
      setEcmpHfecWeightExpansion( 'ecmp' in args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setEcmpHfecWeightExpansion( False )

BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalWeightExpansionCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop disabled
#-------------------------------------------------------------------------------
def hierarchicalFecSupported( mode, token ):
   if AleCliLib.routingHwStatus.hierarchicalFecSupported():
      return None
   else:
      return CliParser.guardNotThisPlatform

def setHfecDisabled( mode, args ):
   disabled = not CliCommand.isNoOrDefaultCmd( args )
   AleCliLib.routingHwConfig.hierarchicalFecsDisabled = disabled

class AleHierarchicalNextHopDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib hierarchical next-hop disabled'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'disabled': "Disable hierarchical FECs"
   }
   handler = setHfecDisabled
   noOrDefaultHandler = setHfecDisabled

BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalNextHopDisabledCmd )

#-------------------------------------------------------------------------------
# [no|default] ip hardware fib hierarchical next-hop max-level <level>
#-------------------------------------------------------------------------------
unconfigured = Tac.Type( "Routing::Hardware::HierarchicalFecsMaxLevel" ).invalid

def maxLevelRange( mode ):
   if AleCliLib.routingHwStatus.maxHierarchicalFecChainLength == 1:
      # Normally not possible thanks to the guard, but needed on startup where
      # we parse the config even if the feature agents have not populated
      # routingHwStatus yet. The configured value is bounded by the actual
      # hardware limit in L3FeatureConfigSm.
      return ( 1, unconfigured )
   return ( 1, AleCliLib.routingHwStatus.maxHierarchicalFecChainLength )

def setHfecMaxLevel( maxLevel ):
   AleCliLib.routingHwConfig.hierarchicalFecsMaxLevel = maxLevel

class AleHierarchicalNextHopMaxLevelCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib hierarchical next-hop max-level MAX_LEVEL'
   noOrDefaultSyntax = 'ip hardware fib hierarchical next-hop max-level'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'hierarchical': hierarchicalMatcher,
      'next-hop': hierarchcialNextHopMatcher,
      'max-level': CliCommand.Node(
                      matcher=KeywordMatcher(
                         'max-level',
                         helpdesc='Max FEC hierarchy level configuration' ),
                      guard=hierarchicalFecSupported ),
      'MAX_LEVEL': DynamicIntegerMatcher( maxLevelRange,
                                          helpdesc='FEC hierarchy max level' )
   }

   @staticmethod
   def handler( mode, args ):
      setHfecMaxLevel( args[ 'MAX_LEVEL' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setHfecMaxLevel( unconfigured )

BasicCli.GlobalConfigMode.addCommandClass( AleHierarchicalNextHopMaxLevelCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop resource optimization threshold
#               low <percentage> high <percentage>"
#-------------------------------------------------------------------------------
def nextHopResourceOptimizationSupported( mode, token ):
   if AleCliLib.routingHwStatus.adjResourceOptimizationSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def setResourceOptimizationThresholds( mode, args ):
   thresholds = AleCliLib.routingHwConfig.adjResourceOptimizationThresholds
   thresholds.stopOptimizationThreshold = args[ 'THRESHOLD_LOW' ]
   thresholds.startOptimizationThreshold = args[ 'THRESHOLD_HIGH' ]
   thresholds.configured = True

def resetResourceOptimizationThresholds( mode, args ):
   thresh = Tac.newInstance(
         'Routing::Hardware::AdjResourceOptimizationThresholds', 'thresholds' )
   thresholds = AleCliLib.routingHwConfig.adjResourceOptimizationThresholds
   thresholds.startOptimizationThreshold = thresh.defaultStartOptimizationThreshold
   thresholds.stopOptimizationThreshold = thresh.defaultStopOptimizationThreshold
   thresholds.configured = False

resourceMatcher = KeywordMatcher( "resource",
                     helpdesc="Hardware resource configuration" )

nodeOptimization = CliCommand.Node(
      matcher=KeywordMatcher( "optimization",
                     helpdesc="Optimize hardware resource" ),
      guard=nextHopResourceOptimizationSupported )

matcherThresholdValue = IntegerMatcher( 0, 100,
                                 helpdesc="Resource utilization percentage "
                                          "for starting/stopping optimization." )

matcherLow = KeywordMatcher( "low",
                     helpdesc="Stop optimization threshold" )

matcherHigh = KeywordMatcher( "high",
                     helpdesc="Start optimization threshold" )

class AdjResourceOptimizationThresholdsCmd( CliCommand.CliCommandClass ):
   syntax = ( 'ip hardware fib next-hop resource optimization thresholds '
              'low THRESHOLD_LOW high THRESHOLD_HIGH' )
   noOrDefaultSyntax = 'ip hardware fib next-hop resource optimization thresholds'
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'resource': resourceMatcher,
      'optimization': nodeOptimization,
      'thresholds': "Utilization percentage for "
                    "starting or stopping optimization",
      'low': matcherLow,
      'THRESHOLD_LOW': matcherThresholdValue,
      'high': matcherHigh,
      'THRESHOLD_HIGH': matcherThresholdValue,
   }

   handler = setResourceOptimizationThresholds
   noOrDefaultHandler = resetResourceOptimizationThresholds

BasicCli.GlobalConfigMode.addCommandClass( AdjResourceOptimizationThresholdsCmd )

#-------------------------------------------------------------------------------
# "[no|default] ip hardware fib next-hop resource optimization disabled"
#-------------------------------------------------------------------------------
def setAdjResourceOptimization( mode, args ):
   enabled = CliCommand.isNoOrDefaultCmd( args )
   AleCliLib.routingHwConfig.adjResourceOptimizationEnabled = enabled

class AdjResourceOptimizationDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop resource optimization disabled'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      'resource': resourceMatcher,
      'optimization': nodeOptimization,
      'disabled': "Disable hardware resource optimization for adjacency programming",
   }

   handler = setAdjResourceOptimization
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( AdjResourceOptimizationDisabledCmd )

#-------------------------------------------------------------------------------
# hidden
# "[ no | default ] ip hardware fib next-hop proxy disabled"
#-------------------------------------------------------------------------------
class AdjProxyDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'ip hardware fib next-hop proxy disabled'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipForConfigMatcher,
      'hardware': hardwareMatcher,
      'fib': fibMatcher,
      'next-hop': nextHopMatcher,
      # Add CLI save plugin if you want to unhide this command
      'proxy': CliCommand.Node( matcher=KeywordMatcher(
             "proxy", helpdesc="Proxy configuration for adjacencies" ),
             hidden=True ),
      'disabled': "Disable proxy adjacency programming",
   }

   @staticmethod
   def handler( mode, args ):
      AleCliLib.routingHwConfig.adjProxyEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      AleCliLib.routingHwConfig.adjProxyEnabled = True

BasicCli.GlobalConfigMode.addCommandClass( AdjProxyDisabledCmd )
