#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliUtil
import CliCommand
import CliMatcher
import CliParser
import CliParserCommon
import CliToken.Hardware
import ConfigMount
import LazyMount
import ShowCommand
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'PortGroupCli' )
t0 = __defaultTraceHandle__.trace0

portGroupDir = None
portGroupCliConfig = None

def portGroupSupported( mode, token ):
   for portGroup in portGroupDir.portGroup.values():
      if portGroup.defaultMode:
         return None
   return CliParser.guardNotThisPlatform

nodePortGroup = CliCommand.guardedKeyword( 'port-group',
      helpdesc='Port-group of QSFP or SFP ports', guard=portGroupSupported )

def getPortList( groupName, groupMode ):
   group = portGroupDir.portGroup.get( groupName )
   if group:
      portList = group.mode.get( groupMode )
      return portList
   return None

def getDefaultGroupMode( groupName ):
   groupPort = portGroupDir.portGroup.get( groupName )
   if groupPort:
      return groupPort.defaultMode
   return None

def promptConfirmed( mode ):
   warning = '''
                  WARNING!
   Changing the port-group mode will cause all
   interfaces on the switch to flap.
   '''
   prompt = "Do you wish to proceed with this command? [y/N]"

   mode.addWarning( warning ) 
   return BasicCliUtil.confirm( mode, prompt, answerForReturn=False )

def isInDefaultMode( portGroup ):
   ''' is the portGroup currently in it's default mode?  '''
   currentPortMode = portGroupCliConfig.config.get( portGroup )
   currentPortList = getPortList( portGroup, currentPortMode ) if currentPortMode \
                     else None
   return not currentPortList or currentPortMode == getDefaultGroupMode( portGroup )

def isStartupConfig( mode ):
   return mode.session_.startupConfig()

class PortGroupMatcher( CliMatcher.Matcher ):
   def match( self, mode, context, token ):
      if not isStartupConfig( mode ):
         if token in portGroupDir.portGroup:
            return CliParserCommon.MatchResult( token, token )
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      if portGroupDir.portGroup:
         ret = []
         if token == '':
            for portGroup in sorted( portGroupDir.portGroup ):
               ret.append( CliParser.Completion( portGroup,
                  'Set mode for port group %s' % portGroup ) )
         else:
            if token in portGroupDir.portGroup:
               ret.append( CliParser.Completion( token,
                  'Set mode for port group %s' % token ) )
         return ret
      return [ CliParser.Completion( 'WORD', 'Portgroup name' ) ]

class PortModeMatcher( CliMatcher.Matcher ):
   def match( self, mode, context, token ):
      if not isStartupConfig( mode ):
         portGroup = context.sharedResult[ 'PORT_GROUP' ]
         portGroupConfig = portGroupDir.portGroup.get( portGroup )
         portModes = None
         if portGroupConfig:
            portModes = portGroupConfig.mode
         if portModes and token in portModes:
            return CliParserCommon.MatchResult( token, token )
         return CliParserCommon.noMatch
      return CliParserCommon.MatchResult( token, token )

   def completions( self, mode, context, token ):
      if portGroupDir.portGroup:
         ret = []
         portGroup = context.sharedResult[ 'PORT_GROUP' ]
         portGroupConfig = portGroupDir.portGroup.get( portGroup )
         portModes = None
         if portGroupConfig:
            portModes = portGroupConfig.mode
         if portModes:
            if token == '':
               for portMode in portModes:
                  ret.append( CliParser.Completion( portMode,
                     'Activate ports %s' % portMode ) )
            else:
               matchingPortModes = [ portMode for portMode in portModes if token in
                     portMode ]
               if matchingPortModes:
                  for portMode in matchingPortModes:
                     ret.append( CliParser.Completion( portMode,
                        'Activate ports %s' % portMode ) )
            return ret
         return []
      return [ CliParser.Completion( 'WORD', 'Portgroup mode' ) ]

#--------------------------------------------------------------------------------
# [ no | default ] hardware port-group PORT_GROUP select PORT_MODE
#--------------------------------------------------------------------------------
class ConfigPortGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware port-group PORT_GROUP select PORT_MODE'
   noOrDefaultSyntax = 'hardware port-group PORT_GROUP ...'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForConfig,
      'port-group' : nodePortGroup,
      'PORT_GROUP' : CliCommand.Node( matcher=PortGroupMatcher(),
         storeSharedResult=True ),
      'select' : 'Set the operational mode for port group',
      'PORT_MODE' : PortModeMatcher(),
   }

   @staticmethod
   def handler( mode, args ):
      portGroup = args[ 'PORT_GROUP' ]
      portMode = args[ 'PORT_MODE' ]

      # at startup, any portGroup or portMode strings are allowed:
      if isStartupConfig( mode ):
         portGroupCliConfig.config[ portGroup ] = portMode
         return

      # if the portMode stays the same:
      currentPortMode = portGroupCliConfig.config.get( portGroup )
      if currentPortMode == portMode:
         return

      # the portGroup may be in default because of a bad setting string.
      # do not prompt in that case
      skipPrompt = ( portMode == getDefaultGroupMode( portGroup ) and
                     isInDefaultMode( portGroup ) )
      if skipPrompt or promptConfirmed( mode ):
         t0( "Setting cli port group config for group %s to %s" % ( 
            portGroup, portMode ) )
         portGroupCliConfig.config[ portGroup ] = portMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      portGroup = args[ 'PORT_GROUP' ]

      # if we are already in default mode, do not prompt
      if isInDefaultMode( portGroup ) or promptConfirmed( mode ):
         del portGroupCliConfig.config[ portGroup ]

BasicCli.GlobalConfigMode.addCommandClass( ConfigPortGroupCmd )

#--------------------------------------------------------------------------------
# show hardware port-group
#--------------------------------------------------------------------------------
class HardwarePortGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware port-group'
   data = {
      'hardware' : CliToken.Hardware.hardwareMatcherForShow,
      'port-group' : nodePortGroup,
   }

   @staticmethod
   def handler( mode, args ):
      pgHeader = "\n%-14s\tActive Ports: %4s"
      pgColumn = "%-14s\t%-12s" % ( "Port", "State" )
      pgFooter = "------------------------------------------"
      portHeader = "%-14s\t%-12s"

      for groupName, portGroup in portGroupDir.portGroup.iteritems():
         portGroupModeName = portGroupCliConfig.config.get( groupName )
         portGroupList = portGroup.mode.get( portGroupModeName ) if \
                         portGroupModeName else None
         if not portGroupModeName or not portGroupList:
            portGroupModeName = getDefaultGroupMode( groupName )
            portGroupList = portGroup.mode[ portGroupModeName ]

         portGroupName = 'Portgroup: %s' % groupName
         print pgHeader % ( portGroupName, portGroupModeName )
         print pgColumn
         print pgFooter
         for modeName, _mode in portGroup.mode.iteritems():
            if modeName == portGroupModeName:
               state = "Active"
            else:
               state = "Inactive"
            for port in sorted( _mode.ports ):
               print portHeader % ( port, state )

BasicCli.addShowCommandClass( HardwarePortGroupCmd )

def Plugin( em ):
   global portGroupCliConfig, portGroupDir
   portGroupDir = LazyMount.mount( # all slices configs
      em, 'hardware/ale/portGroup', 'Ale::PortGroupDir', 'ri' )

   # there are two config sources for the port groups.
   # 1 - the user, with the Cli and the Cli's startup config
   # 2 - default group, set by fdl and Fru.
   # the forwaring agent creats a portGroupConfig with those two as input
   
   # here we mount portGroupCliConfig where we set the CLI's portgroup config
   portGroupCliConfig = ConfigMount.mount( 
      em, 'hardware/ale/portGroupCliConfig', 'Ale::PortGroupConfig', 'rw' )
