# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Arnet
import BasicCliUtil
import CliPlugin.IntfCli as IntfCli
from CliPlugin.MasterResourceManager import getRcdToIntfListMap, getMrm
import LazyMount
from TypeFuture import TacLazyType

ResourceType = TacLazyType( "AlePhy::ResourceType" )

promptTemplate = \
"""The following interfaces will be disabled due to insufficient {reason}:

{interfaces}"""

# Create the prompt string with list of interfaces that will be affected
def createPrompt( intfNames, reason ):
   prompt = ""
   if intfNames:
      intfNamesStr = "".join( ( "  %s\n" % i ) for i in intfNames )
      prompt = promptTemplate.format( reason=reason,
                                      interfaces=intfNamesStr )
   return prompt

# Ask the user if they would like to continue
def transitionPrompt( intfModes, warningPrompt ):
   if warningPrompt:
      mode = intfModes[ 0 ]
      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         intfNamesStr = ", ".join( i.intf.name for i in intfModes )
         abortMsg = "Command aborted by user for %s" % intfNamesStr
         mode.addMessage( abortMsg )
         return False

   return True

# ---------------------------------------
# Logical Ports Checks
# ---------------------------------------
LogicalPortAllocationPriority = TacLazyType(
   "AlePhy::LogicalPortAllocationPriority::LogicalPortAllocationPriority" )
PRIORITY_INTFID_AND_INTF_ENABLED = LogicalPortAllocationPriority.intfIdAndIntfEnabled

resourceConsumerSliceDir = None

def getLogicalPortsAffectedIntfs( intfList ):
   allIntfNames = set()
   rcdToIntfListMap = getRcdToIntfListMap( intfList, resourceConsumerSliceDir )
   for rcd, intfsInRcd in rcdToIntfListMap.iteritems():
      # Skip products not enabling releaseOnshut.
      if rcd.logicalPortAllocationPriority != PRIORITY_INTFID_AND_INTF_ENABLED:
         continue

      mrm = getMrm( rcd )
      if not mrm:
         continue

      mrm.synchronize()

      # Apply configs.
      for intf in intfsInRcd:
         mrm.enableInterface( intf.name )

      mrm.processIntfs()
      intfNames = mrm.getDisabledIntfs()
      allIntfNames |= intfNames

   return Arnet.sortIntf( allIntfNames )

def canNoShutdownLogicalPortsCheck( intfModes ):
   intfList = [ i.intf for i in intfModes ]
   intfNames = getLogicalPortsAffectedIntfs( intfList )
   warningPrompt = createPrompt( intfNames, "logical ports" )
   ans = transitionPrompt( intfModes, warningPrompt )
   return ( ans, warningPrompt )

# ---------------------------------------
# Serdes checks
# ---------------------------------------
def getSerdesAffectedIntfs( intfList ):
   allIntfNames = set()
   rcdToIntfListMap = getRcdToIntfListMap( intfList, resourceConsumerSliceDir )
   for rcd, intfsInRcd in rcdToIntfListMap.iteritems():
      # Skip products not enabling releaseOnshut.
      if not rcd.releaseOnShutIntfs:
         continue

      mrm = getMrm( rcd )
      if not mrm:
         continue

      mrm.synchronize()

      # Apply configs.
      for intf in intfsInRcd:
         if intf.name not in rcd.releaseOnShutIntfs:
            continue
         mrm.enableInterface( intf.name )

      mrm.processIntfs()
      intfNames = mrm.getDisabledIntfs( reason=ResourceType.intfPermissions )
      # BUG488999: remove intfRestrictions after everything is converted
      # to use intf permissions
      intfNames |= mrm.getDisabledIntfs( reason=ResourceType.intfRestricted )
      allIntfNames |= intfNames

   return Arnet.sortIntf( allIntfNames )

def canNoShutdownSerdesCheck( intfModes ):
   intfList = [ i.intf for i in intfModes ]
   intfNames = getSerdesAffectedIntfs( intfList )
   warningPrompt = createPrompt( intfNames, "SerDes" )
   ans = transitionPrompt( intfModes, warningPrompt )
   return ( ans, warningPrompt )

def canNoShutdownCheck( intfModes ):
   answer = True
   serdesAnswer, serdesWarn = canNoShutdownSerdesCheck( intfModes )
   logicalAnswer, logicalWarn = canNoShutdownLogicalPortsCheck( intfModes )
   if serdesWarn:
      assert not logicalWarn
   if logicalWarn:
      assert not serdesWarn
   answer &= serdesAnswer
   answer &= logicalAnswer
   return answer

IntfCli.canNoShutdownIfHook.addExtension( canNoShutdownCheck )

def Plugin( em ):
   global resourceConsumerSliceDir
   resourceConsumerSliceDir = LazyMount.mount( em,
                                               "interface/resources/consumers/slice",
                                               "Tac::Dir", "ri" )
