# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""
Implementation of the CLI command(s):

   show interfaces [ <interface> ] interactions
"""
import re

import AlePhyIntfInteractionsModel as IxnModel
import Arnet
import BasicCli
import CliCommand
import CliParser
import CliPlugin.XcvrAllStatusDir
import CliPlugin.IntfInteractionReferenceMatcher as RefMatcher
import EthIntfCli
import IntfCli
import IntfInteractionsReferenceLib as IxnRef
import LazyMount
import Tracing
import Tac

traceHandle = Tracing.Handle( "AlePhyCli" )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

# Declare global names that will be used to refer to mounted paths
ethPhyDefaultCapsSliceDir = None
l1MappingDir = None
l1TopoDir = None
l1TopoRootSliceDir = None
resourceConsumerBaseDir = None
entmibStatus = None
xcvrAllStatusDir = None

# Expose TAC-types with shorter names
EthIntfId = Tac.Type( "Arnet::EthIntfId" )
EthLinkModeSet = Tac.Type( "Interface::EthLinkModeSet" )
PhySide = Tac.Type( "PhyEee::PhySide" )
SerdesGroupMode = Tac.Type( "Hardware::Phy::SerdesGroupMode" )
TraversalHelperFactory = Tac.Type( "Hardware::L1Topology::TraversalHelperFactory" )

#----------------------------------------------------------------------------
# show interfaces [ <interface> ] interactions
#----------------------------------------------------------------------------

# BUG290864: regexes for products+variants that I have validated interactions
# CLI against.
supportedFixedProducts = [
   re.compile( "DCS-7280QRA?-C72" ), # Capitola(?:Stat)?
   re.compile( "DCS-7280QRA-C36S" ), # Nice
   re.compile( "DCS-7280[PD]R3K?-24" ),  # Lyonsville(?:DD)?(?:BK)?
   re.compile( "DCS-7280CR3M?K?-32[PD]4" ), # Smartsville(?:DD)?(?:BK)?(?:MS)?
   re.compile( "DCS-7060[PD]X4-32" ), # Blackhawk(?:O|DD)
   re.compile( "DCS-7280CR3K?-96" ), # Torrance(?:BK)?
]
supportedLinecardProducts = [
   re.compile( "7368-16C" ), # Sperry
   re.compile( "7368-4[DP]" ), # Grinnel[DP]
   re.compile( "7500R3K?-36CQ-LC" ), # MtSheridan(?:BK)?
   re.compile( "7800R3K?-48CQ-LC" ), # Clearwater(?:BK)?
]

def entmibModelName( cardSlot=None ):
   """
   Returns a model name from EntityMib::Status. Desired behavior:

   * For fixed-systems, returns the product model name (e.g. DCS-XXXX-...)
   * For modular systems, if cardSlot not provided, returns the chassis product
     name (e.g. DCS-7304, 7368, etc.)
   * For modular systems, if cardSlot IS provided and cardSlot is populated on
     chassis, return the card's (presumably linecard) product model name
     (e.g. 7368-4D)
   * Else, return the empty string

   Parameters
   ----------
   cardSlot: int, position of card in chassis
   """
   modelName = ""
   # A reactor in EntityMib::Status updates root to correctly point to
   # 'fixedSystem' or 'chassis', depending on the product.
   if entmibStatus.root:
      if cardSlot and entmibStatus.root == entmibStatus.chassis:
         # This is the code path to get the model name of a specific linecard.
         # If the slot isn't populated, we still return "".
         slot = entmibStatus.root.cardSlot.get( cardSlot )
         card = getattr( slot, "card", None )
         modelName = getattr( card, "modelName", "" )
      else:
         # Fall down here to return fixed-system or modular chassis name
         modelName = entmibStatus.root.modelName
   return modelName

def intfIxnsSupported( cardSlot=None ):
   """
   Command is supported under the following conditions:
   * For fixed-systems, product must match one of the supportedFixedProducts regexes
   * For modular-systems, at least one installed linecard must match one of the 
     supportedLinecardProducts regexes

   If cardSlot is provided, for a modular chassis, the specific linecard indicated
   by the argument must match one of the supportedLinecardProducts
   """
   supported = False
   if entmibStatus.fixedSystem:
      modelName = entmibModelName()
      supported = any( r.search( modelName ) for r in supportedFixedProducts )
   elif entmibStatus.chassis:
      slots = entmibStatus.chassis.cardSlot.values()
      # If cardSlot is provided, filter the total list of slot objects down to
      # just the specified cardSlot. .keys() makes entmibStatus.chassis.cardSlot
      # a python list that I can check membership of None against.
      if cardSlot in entmibStatus.chassis.cardSlot.keys():
         slots = [ entmibStatus.chassis.cardSlot[ cardSlot ] ]
      cardModelNames = set( entmibModelName( slot.relPos ) for slot in slots )
      # The following four lines could possibly be more pythonic, but I think this
      # is more readable than anything else I came up with.
      for cardName in cardModelNames:
         supported = any( r.search( cardName ) for r in supportedLinecardProducts )
         if supported:
            break
   return supported

def intfIxnGuard( mode, token ):
   # Infer that this platform supports the cmd if an agent (the fwd-agent) has
   # instantiated a slicified path and is in the product whitelist
   if len( resourceConsumerBaseDir ) and intfIxnsSupported():
      return None
   return CliParser.guardNotThisPlatform

# BUG479124: there's no good reason for these getters to search the entire world
# for the entry they're looking for. We should at least reduce the search space
# to a single slice. We might still have to loop over all the subSlices.
def getSerdesResourceConsumer( intfId ):
   for sliceDir in resourceConsumerBaseDir.itervalues():
      for subSliceDir in sliceDir.itervalues():
         src = subSliceDir.serdesResourceConsumer.get( intfId )
         if src:
            return src
   t1( "SerdesResourceConsumer not found for", intfId )
   return None

def getSpeedGroupName( intfId ):
   for sliceDir in resourceConsumerBaseDir.itervalues():
      for subSliceDir in sliceDir.itervalues():
         if subSliceDir.speedGroupStatusDir:
            sgName = subSliceDir.speedGroupStatusDir.intfSpeedGroup.get( intfId )
            if sgName:
               return sgName 
   t1( "Speed-group not found for", intfId )
   return None

def getLogicalPortPoolDesc( intfId ):
   for sliceDir in resourceConsumerBaseDir.itervalues():
      for subSliceDir in sliceDir.itervalues():
         if subSliceDir.logicalPortPoolDescDir:
            poolId = subSliceDir.logicalPortPoolDescDir.poolIdByIntf.get( intfId )
            if poolId is not None:  # poolId can legitimately be 0
               lppDesc = subSliceDir.logicalPortPoolDescDir.pool.get( poolId )
               if lppDesc:
                  return lppDesc
   t1( "Logical port pool description not found for", intfId )
   return None

def generateSerdesDomains( intfs ):
   """
   Generates a list of SerdesDomainBase-derived objects. A "serdes domain" is
   defined as a set of interfaces that share a set of serdes (typically switch-
   chip serdes).

   Returns
   -------
   serdesDomains: list of SerdesDomainBase-derived objects

   Note
   ----
   I found this to be a deceptively complex problem due to cases where a serdes
   domain can span multiple front-panel ports. I don't consider my approach to
   be efficient by any means, but I believe it to be correct. I'll leave the
   task of optimization to someone smarter than me...
   """
   # First, create a mapping of front-panel port number to the superset of
   # serdes used by the port (typically switch-chip serdes). Also map the
   # port number to the list of member intfIds, which will be used later.
   portToSerdesSuperset = {}
   portToIntfIds = {}
   for intf in intfs:
      intfId = intf.name
      src = getSerdesResourceConsumer( intfId )
      if not src:
         continue
      port = EthIntfId.port( intfId )
      portToIntfIds.setdefault( port, [] ).append( intfId )
      serdesSuperset = portToSerdesSuperset.setdefault( port, set() )
      for sg in src.serdesGroup.itervalues():
         serdesSuperset.update( sg.serdesId )

   # Next, invert portToSerdesSuperset to map individual serdes IDs to the
   # ports that use them.
   serdesIdToPorts = {}
   for port, serdesSet in portToSerdesSuperset.items():
      for serdes in serdesSet:
         serdesIdToPorts.setdefault( serdes, set() ).add( port )

   # Finally, reverse-sort the previous map by the length of the values. The
   # group of ports that make up a serdes domain will move to the front of the
   # iteration.
   portToSerdesDomain = {}
   intfSets = []
   for portSet in sorted( serdesIdToPorts.values(), key=len, reverse=True ):
      if all( portId in portToSerdesDomain for portId in portSet ):
         # Most of the ports will be processed in the beginning of the loop
         # because of how the entries are sorted. No need to process them again.
         #
         # BUG478057: there is an assumption here that each serdes-domain contains
         # a serdes that is used by all physical ports that are members of the
         # serdes-domain. If that assumption is broken, we need to revisit this.
         continue
      intfSet = set()
      # The following two loops must be separate because the intfSet is not
      # finalized until each port in the portSet has been processed. The
      # serdesDomain list must contain only finalized serdes domains.
      for portId in portSet:
         intfSet.update( portToIntfIds[ portId ] )
      for portId in portSet:
         portToSerdesDomain[ portId ] = intfSet
      # Happens outside the loop because we only need to register each intfSet once
      intfSets.append( intfSet )

   # Construct a SerdesDomain object from each serdes domain. Each interface in
   # the domain must be "named" correctly to be understood correctly by the
   # IntfInteractionReference objects.
   serdesDomains = []
   for domain in intfSets:
      serdesDomain = None
      numPorts = len( set( EthIntfId.port( intf ) for intf in domain ) )
      if numPorts == 1:
         # Multi-lane ports will have their members named by lane. Single-lane
         # ports will have its member named "lane0"
         intfDict = { "lane%d" % EthIntfId.lane( intf ): intf
                      for intf in domain }
         serdesDomain = IxnRef.SerdesDomain( intfDict )
      elif numPorts == 2:
         # For now, dual port is the only multi-port domain we have. Need
         # to determine the master and slave port relationship. Total number
         # of serdes used should be a decent heuristic to deterimine a master port.
         ports = sorted( set( EthIntfId.port( i ) for i in domain ) )
         if( len( portToSerdesSuperset[ ports[ 0 ] ] ) >=
             len( portToSerdesSuperset[ ports[ 1 ] ] ) ):
            # The usage of >= instead of strictly > is to handle the case where both
            # ports use the same number of serdes. I'm assuming that in such a case
            # we would define the master port to be the one with a lower power
            # number. 'ports' is sorted, so the lower number should be guaranteed to
            # be first.
            master, slave = ports
         else:
            # Otherwise, the latter port uses more serdes and is likely the master
            slave, master = ports
         intfDict = { "master%d" % EthIntfId.lane( intf ): intf
                      for intf in domain if EthIntfId.port( intf ) == master }
         intfDict.update( { "slave%d" % EthIntfId.lane( intf ): intf
                            for intf in domain
                            if EthIntfId.port( intf ) == slave } )
         assert len( intfDict ) == len( domain )
         serdesDomain = IxnRef.SerdesDomain( intfDict )
      else:
         # Consciously do not add a serdes domain to avoid presenting bad input
         # to downstream code. We might still be able to generate useful
         # information if we keep going instead of asserting here.
         t1( "Unhandled number of ports for serdes domain: intfIds =", domain )
      if serdesDomain:
         # If this is defined, we were able to classify the serdesDomain
         serdesDomains.append( serdesDomain )
         t1( "Created SerdesDomain: memberIntfs =", Arnet.sortIntf( domain ) )
   t1( "Num serdes domains:", len( serdesDomains ) )
   return serdesDomains

def getL1TopoPhys( topoHelper, intfId ):
   module = EthIntfId.module( intfId )
   port = EthIntfId.port( intfId )
   lane = EthIntfId.lane( intfId )
   if lane > 0:
      # IntfId lanes start at 1, but topology lanes start at 0. EthIntfId.lane
      # returns the intfId lane or 0 if the intfId doesn't have a lane. This
      # means we want to subtract 1 from lane unless EthIntfId.lane returns 0.
      # Not sure if there is a way to do this that feels less hacky. 
      lane -= 1
   # BUG479126: I can likely use EthPhyIntf.sliceName() here, though
   # I don't think it does the "FixedSystem" part quite correctly. I'll see
   # about switching over in a follow-up MUT.
   sliceId = "FixedSystem"
   if module:
      sliceId = "Linecard%d" % module
   # Seems like the convention is to trace the TX path (that's what the boolean
   # argument represents).
   xcvrTopo = topoHelper.xcvrHelper.xcvrLaneTopology( sliceId, port, lane,
                                                      True )
   intfTopo = topoHelper.xcvrHelper.intfTopology( sliceId, intfId )
   # If L1 topology exists for this interface, take the union of phys used at
   # any configuration of the interface, in case they might be different.
   phys = set()
   if xcvrTopo and intfTopo:
      serdesGroupModes = [ SerdesGroupMode.fromEthIntfMode( i )
                           for i in intfTopo.intfSpeedTopology.keys() ]
      serdesGroups = [ topoHelper.lastComponentGroup( sliceId, xcvrTopo, mode,
                                                      PhySide.phySideSystem )
                       for mode in serdesGroupModes ]
      map( phys.update, [ group.phy.values() for group in serdesGroups ] )
   return phys

def identifyXcvrSlotType( intfId ):
   xcvrSlot = ""
   # BUG474967: need to define behavior for when xcvrAllStatusDir.xcvrStatus
   # is empty (e.g. early start-up). Will address this along with btests for
   # this implementation.
   module = EthIntfId.module( intfId )
   port = EthIntfId.port( intfId )
   # XXX michaelchin: XcvrLib.getXcvrSlotName does this, but in a way that I
   # disagree with, so writing my own.
   xcvrSlotName = ( "Ethernet%d/%d" % ( module, port )
                    if module else "Ethernet%d" % port )
   xStatus = xcvrAllStatusDir.xcvrStatus.get( xcvrSlotName )
   if xStatus:
      # We are looking for physical slot form-factor, so we can take xcvrType
      # as is (i.e. don't need to look at swizzled xcvrStatus)
      xcvrSlot = xStatus.xcvrType
   return xcvrSlot

def showInterfacesInteractions( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   # Get all intfs since we have to consider the relationships between all
   # interfaces.
   allIntfs = IntfCli.Intf.getAll( mode, None, mod, EthIntfCli.EthPhyIntf,
                                   exposeInactive=True )
   intfIdToEthPhyIntfObj = { intf.name: intf for intf in allIntfs }
   intfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthPhyIntf,
                                exposeInactive=True )
   displayIntfNames = [ intf.name for intf in intfs ]
   t0( "Running `show interfaces interactions` for intfs:", intfs )
   model = IxnModel.InterfacesInteractions()

   # Create L1 Topo traversal helper to help identify criteria used to match
   # against interaction templates. Use globally-scoped traversal helper instead
   # of managing one helper per slice
   pcsd = Tac.newInstance( "AlePhy::PhyCommonSmData", "pcsd" )
   pcsd.topoDir = LazyMount.force( l1TopoDir )
   pcsd.mappingDir = LazyMount.force( l1MappingDir )
   topoHelper = TraversalHelperFactory().create( pcsd )
   # We will want to cache whether or not a particular linecard on a modular
   # system supports interactions. There's a helper method for it, but it
   # does O(N) regex matches, so it'd be nice to reduce the number of times
   # we have to call it.
   moduleSupportsIxns = {}
   # Identify serdes domains (i.e. interfaces that share a set of switch-chip
   # serdes). They will be used to match interfaces to the appropriate
   # interaction reference template. SerdesDomains must be computed per slice.
   # Once the serdes-domains have been created, they can be operated on
   # without regards to slices.
   intfsBySlice = {}
   for intf in allIntfs:
      intfId = intf.name
      module = EthIntfId.module( intfId )
      # 'module' is either 0 for fixed systems or a card slot for modular systems.
      # If a particular linecard does not support the interactions CLI, do not
      # attempt to generate interactions for it.
      if module not in moduleSupportsIxns:
         moduleSupportsIxns[ module ] = intfIxnsSupported( module )
      if moduleSupportsIxns.get( module ):
         intfsBySlice.setdefault( module, [] ).append( intf )
   serdesDomains = []
   for intfs in intfsBySlice.values():
      serdesDomains.extend( generateSerdesDomains( intfs ) )
   # For each serdes domain, identify criteria that would be used to match against
   # a template.
   for serdesDomain in serdesDomains: 
      # First identify phys on interface via L1Topo. Use the first memberIntf as
      # a base-line.
      #
      # There is an assumption here that all intfIds in a serdes domain will pass
      # through the same phys as the others in the domain. If this assumption is
      # broken, we will have to loop over all intfIds in the domain and use the
      # union of all of their phys.
      domainIntf = next( iter( serdesDomain.memberIntfs.values() ) )
      groupPhys = getL1TopoPhys( topoHelper, domainIntf )

      # Next identify the xcvr slot type. Between this and groupPhys, we can
      # correctly match against a majority of the port types on our products.
      xcvrSlot = identifyXcvrSlotType( domainIntf )

      # Gather the remaining information that may be used to refine a reference match
      # XXX michaelchin: it may make sense in the future to fetch the linecard model
      # instead of the chassis model by passing in EthIntfId.module( domainIntf )
      productModel = entmibModelName()
      portRange = set( map( EthIntfId.port, serdesDomain.memberIntfs.values() ) )
      refId = RefMatcher.InteractionReferenceId( groupPhys, xcvrSlot,
                                                 productModel, portRange )
      template = RefMatcher.findMatch( refId )
      if not template:
         continue
      # There is a risk of an internal error here if the serdesDomain is not
      # compatible with the reference template. This could happen if there is
      # a mistake in the matcher's registry. The pythonic thing to do would
      # be to wrap this in a try-except -- I'll leave that as a TODO when there
      # is a test that could exercise it (BUG479133).
      ixnRef = template( serdesDomain )

      # Populate optional parameters in ixnRef before generating the model
      # For capabilities, take the union of capabilities of all intfs in the
      # serdes domain
      domainCapabilities = EthLinkModeSet()
      for intfId in serdesDomain.memberIntfs.values():
         intfCaps = intfIdToEthPhyIntfObj[ intfId ].ethPhyDefaultCaps()
         domainCapabilities |= intfCaps.linkModeCapabilities
      ixnRef.groupLinkModes = domainCapabilities
      # For speed-groups, if the intf is associated with a speed-group name
      # according to SpeedGroupStatusDir, use that name.
      speedGroupName = getSpeedGroupName( domainIntf )
      if speedGroupName:
         ixnRef.speedGroupName = speedGroupName
      # For logical ports, if the intf is associated with a logical port pool
      # description, use the description to populate the reference.
      lppDesc = getLogicalPortPoolDesc( domainIntf )
      if lppDesc:
         # lppDesc.memberIntf is a set-void collection; reduce that to something
         # that easier for the rest of the command to work with.
         desc = IxnRef.LogicalPortInteractionDesc( lppDesc.poolId,
                                                   lppDesc.memberIntf.keys(),
                                                   lppDesc.maxPorts )
         ixnRef.logicalPortPoolDesc = desc

      # This is terribly inefficient, but its correct. Given this current
      # architecture, we would have to only loop over the serdes domains that
      # are relevant to the interfaces we want to display.
      model.interfaces.update( { k: v for k, v in ixnRef.model().interfaces.items()
                                 if k in displayIntfNames } )
   return model

# Register Cli command
interactionsKw = CliCommand.guardedKeyword(
   "interactions",
   "Show interactions with other Ethernet interfaces",
   intfIxnGuard )

class ShowIntfInteractions( IntfCli.ShowIntfCommand ):
   syntax = "show interfaces interactions"
   data = dict( interactions=interactionsKw )
   cliModel = IxnModel.InterfacesInteractions
   handler = showInterfacesInteractions
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfInteractions )

#----------------------------------------------------------------------------
# Plugin
#----------------------------------------------------------------------------
def Plugin( entityManager ):
   global ethPhyDefaultCapsSliceDir
   global l1MappingDir
   global l1TopoDir
   global l1TopoRootSliceDir
   global resourceConsumerBaseDir
   global xcvrAllStatusDir

   ethPhyDefaultCapsSliceDir = LazyMount.mount( entityManager,
                        "interface/archer/status/eth/phy/capabilities/default/slice",
                        "Tac::Dir", "ri" )
   l1MappingDir = LazyMount.mount( entityManager,
                                   "hardware/l1/mapping", "Tac::Dir", "ri" )
   l1TopoDir = LazyMount.mount( entityManager,
                                "hardware/l1/topology", "Tac::Dir", "ri" )
   l1TopoRootSliceDir = LazyMount.mount( entityManager,
                                         "hardware/l1/topology/slice",
                                         "Tac::Dir", "ri" )
   resourceConsumerBaseDir = LazyMount.mount( entityManager,
                                              "interface/resources/consumers/slice",
                                              "Tac::Dir", "ri" )
   xcvrAllStatusDir = CliPlugin.XcvrAllStatusDir.xcvrAllStatusDir( entityManager )

   # BUG290864
   global entmibStatus
   entmibStatus = LazyMount.mount( entityManager, "hardware/entmib/",
                                   "EntityMib::Status", "ri" )
