# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from AleHaloCliModel import (
      AlgoMatchHwAclInfo,
      AlgoMatchAclMapping,
      AlgoMatchMaskGroup,
      AlgoMatchTableLayout,
      AlgoMatchTableUsage,
)
import AleHaloCliModel
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliToken.Platform
from HaloLib import TableType
import LazyMount
import ShowCommand
import Tac

aclStatus = None
rootConfigCollSysdb = None
rootStatusCollSysdb = None

def guardAlgoMatch( mode, token ):
   if aclStatus.haloSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

matcherAcl = CliMatcher.KeywordMatcher( 'acl',
      helpdesc='ACL' )
matcherLastAttempt = CliMatcher.KeywordMatcher( 'last-attempt',
      helpdesc='Information from the last attempt' )
matcherTable = CliMatcher.KeywordMatcher( 'table',
      helpdesc='Table information' )
nodeAlgomatch = CliCommand.guardedKeyword( 'algomatch', helpdesc='Algomatch',
      guard=guardAlgoMatch )
matcherChipName = CliMatcher.PatternMatcher( '.*', helpname='WORD',
      helpdesc='Chip name' )

def _getRootConfig( chipName, lastAttempt=None ):
   rootConfig = None
   sysdbColl = None
   if lastAttempt:
      sysdbColl = rootConfigCollSysdb.chipRootConfigLastAttempt
   else:
      sysdbColl = rootConfigCollSysdb.chipRootConfig
   for rootConfigSysdb in sysdbColl.values():
      if not lastAttempt and not rootConfigSysdb.active:
         continue
      if rootConfigSysdb.chipName != chipName:
         continue
      rootConfig = rootConfigSysdb
      break
   return rootConfig

def _getRootStatus( chipName, lastAttempt=None ):
   rootStatus = None
   # We need to get the active root config to determine which corresponding
   # root status to get
   rootConfig = _getRootConfig( chipName, lastAttempt )
   if rootConfig is not None:
      if lastAttempt:
         rootStatus = \
            rootStatusCollSysdb.chipRootStatusLastAttempt.get( rootConfig.chipId,
                                                               None )
      else:
         rootStatus = \
            rootStatusCollSysdb.chipRootStatus.get( rootConfig.rootConfigId, None )
      assert rootStatus.chipName == chipName
   return rootStatus

def _getTcpFlagStr( tcpFlag ):
   flags = []
   if tcpFlag.fin:
      flags.append( 'f' )
   if tcpFlag.syn:
      flags.append( 's' )
   if tcpFlag.rst:
      flags.append( 'r' )
   if tcpFlag.psh:
      flags.append( 'p' )
   if tcpFlag.ack:
      flags.append( 'a' )
   if tcpFlag.urg:
      flags.append( 'u' )
   return ','.join( flags )

def _getIpMaskGroupModel( maskGroup ):
   assert maskGroup.aclType != TableType.macAcl
   return AleHaloCliModel.IpMaskGroup(
            srcIp=maskGroup.srcIpPrefixMask,
            dstIp=maskGroup.dstIpPrefixMask,
            srcPort=maskGroup.srcPortRangeMask,
            dstPort=maskGroup.dstPortRangeMask,
            tcpFlag=_getTcpFlagStr( maskGroup.tcpFlag ),
            established=maskGroup.establishedPresent,
            ttl=maskGroup.ttlPresent,
            dscp=maskGroup.dscpPresent,
            fragment=maskGroup.ipFragmentBitPresent,
            protocol=maskGroup.ipProtoFieldPresent )

def _getMacMaskGroupModel( maskGroup ):
   assert maskGroup.aclType == TableType.macAcl
   return AleHaloCliModel.MacMaskGroup(
            srcMac=maskGroup.sourceMacMask,
            dstMac=maskGroup.destMacMask,
            ethProto=maskGroup.ethProtoPresent )

def _getMaskGroupModel( maskGroupIndex, maskGroup, entries ):
   mgId = Tac.nonConstPtr( maskGroupIndex ).index( maskGroup )
   ipMaskGroup = None
   macMaskGroup = None
   if maskGroup.aclType == TableType.macAcl:
      macMaskGroup = _getMacMaskGroupModel( maskGroup )
   else:
      ipMaskGroup = _getIpMaskGroupModel( maskGroup )
   return AleHaloCliModel.MaskGroup( id=mgId,
                                     ip=ipMaskGroup,
                                     mac=macMaskGroup,
                                     entries=entries )

def _getMaskGroupsModel( maskGroupIndex, table ):
   maskGroups = []
   if table:
      for maskGroup, entries in table.maskGroup.items():
         maskGroups.append( _getMaskGroupModel( maskGroupIndex,
                                                maskGroup,
                                                entries ) )
   return AleHaloCliModel.MaskGroups( maskGroups=maskGroups )

def _getTableLayoutModel( maskGroupIndex, stageHashMask ):
   layout = {}
   for stage, maskGroup in stageHashMask.items():
      # BUG213974
      layout[ stage ] = _getMaskGroupModel( maskGroupIndex, maskGroup, 0 )
   return AleHaloCliModel.TableLayout( layout=layout )

def _getHwAclType( tableType ):
   if tableType == TableType.ipv4PAcl or tableType == TableType.ipv6PAcl:
      return 'port'
   elif tableType == TableType.ipv4RAcl or tableType == TableType.ipv6RAcl:
      return 'routed'
   elif tableType == TableType.macAcl:
      return 'mac'
   return 'unknown'

def _addHwAclInfoModel( model, aclType, name, direction, hwAclType, hwAclId ):
   def _addHwAclSubmodel( model, submodelTypeName, keyName, key ):
      objDict = getattr( model, keyName )
      if key not in objDict:
         objDict[ key ] = getattr( AleHaloCliModel, submodelTypeName )()
      return objDict[ key ]

   aclTypeInfoModel = _addHwAclSubmodel( model,
                                         'AclTypeHwAclInfo',
                                         'aclType',
                                         aclType )
   nameInfoModel = _addHwAclSubmodel( aclTypeInfoModel,
                                      'NameHwAclInfo',
                                      'name',
                                      name )
   dirInfoModel = _addHwAclSubmodel( nameInfoModel,
                                     'DirectionHwAclInfo',
                                     'direction',
                                     direction )
   hwAclTypeInfoModel = _addHwAclSubmodel( dirInfoModel,
                                           'HwAclTypeHwAclInfo',
                                           'hwAclType',
                                           hwAclType )
   hwAclInfoModel = _addHwAclSubmodel( hwAclTypeInfoModel,
                                       'HwAclInfo',
                                       'hwAclId',
                                       hwAclId )
   return hwAclInfoModel

def _addMaskGroupHwAclInfo( maskGroupTableColl,
                            maskGroupIndex,
                            maskGroupInfoModel,
                            aclKey,
                            aclId ):
   hwAclInfoModel = _addHwAclInfoModel( maskGroupInfoModel,
                                        aclKey.type,
                                        aclKey.name,
                                        aclKey.direction,
                                        _getHwAclType( aclId.tableType ),
                                        aclId.chipRuleListId )
   table = maskGroupTableColl.maskGroupTable.get( aclId, None )
   hwAclInfoModel.maskGroups = \
         _getMaskGroupsModel( maskGroupIndex, table )

#--------------------------------------------------------------------------------
# show platform algomatch CHIP_NAME acl mapping [ last-attempt ]
#--------------------------------------------------------------------------------
class ShowMappingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform algomatch CHIP_NAME acl mapping [ last-attempt ]'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'algomatch' : nodeAlgomatch,
      'CHIP_NAME' : matcherChipName,
      'acl' : matcherAcl,
      'mapping' : 'Mapping between an ACL and its Hw ACL IDs',
      'last-attempt' : matcherLastAttempt,
   }
   cliModel = AlgoMatchAclMapping

   @staticmethod
   def handler( mode, args ):
      chipName = args[ 'CHIP_NAME' ]
      lastAttempt = 'last-attempt' in args
      hwAclMappingInfo = AlgoMatchHwAclInfo()
      rootConfig = _getRootConfig( chipName, lastAttempt )
      if rootConfig is None:
         mode.addError( 'Chip: %s is invalid' % chipName )
         raise CliParser.AlreadyHandledError()

      for aclKey, aclIdSet in rootConfig.aclKeyToAclId.keyToIdTypeSet.items():
         for aclId in aclIdSet.aclIdType.values():
            _addHwAclInfoModel( hwAclMappingInfo,
                                aclKey.type,
                                aclKey.name,
                                aclKey.direction,
                                _getHwAclType( aclId.tableType ),
                                aclId.chipRuleListId )
      return AlgoMatchAclMapping( chipName=chipName,
                                  hwAclMappingInfo=hwAclMappingInfo )

BasicCli.addShowCommandClass( ShowMappingCmd )

#--------------------------------------------------------------------------------
# show platform algomatch CHIP_NAME acl mask-group [ last-attempt ]
#--------------------------------------------------------------------------------
class ShowMaskGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform algomatch CHIP_NAME acl mask-group [ last-attempt ]'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'algomatch' : nodeAlgomatch,
      'CHIP_NAME' : matcherChipName,
      'acl' : matcherAcl,
      'mask-group' : 'Mask group information',
      'last-attempt' : matcherLastAttempt,
   }
   cliModel = AlgoMatchMaskGroup

   @staticmethod
   def handler( mode, args ):
      chipName = args[ 'CHIP_NAME' ]
      lastAttempt = 'last-attempt' in args
      rootStatus = _getRootStatus( chipName, lastAttempt )
      rootConfig = _getRootConfig( chipName, lastAttempt )
      initMaskGroupInfo = AlgoMatchHwAclInfo()
      finalMaskGroupInfo = AlgoMatchHwAclInfo()
      if rootStatus is None or rootConfig is None:
         mode.addError( 'Chip: %s is invalid' % chipName )
         raise CliParser.AlreadyHandledError()

      for aclKey, aclIdSet in rootConfig.aclKeyToAclId.keyToIdTypeSet.items():
         for aclId in aclIdSet.aclIdType.values():
            _addMaskGroupHwAclInfo( rootStatus.maskGroupTableColl,
                                    rootStatus.maskGroupIndex,
                                    initMaskGroupInfo,
                                    aclKey,
                                    aclId )
            _addMaskGroupHwAclInfo( rootStatus.maskGroupSummaryTableColl,
                                    rootStatus.maskGroupIndex,
                                    finalMaskGroupInfo,
                                    aclKey,
                                    aclId )
      return AlgoMatchMaskGroup( chipName=chipName,
                                 initMaskGroupInfo=initMaskGroupInfo,
                                 finalMaskGroupInfo=finalMaskGroupInfo )

BasicCli.addShowCommandClass( ShowMaskGroupCmd )

#--------------------------------------------------------------------------------
# show platform algomatch CHIP_NAME acl table layout [ last-attempt ]
#--------------------------------------------------------------------------------
class ShowTableLayoutCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform algomatch CHIP_NAME acl table layout [ last-attempt ]'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'algomatch' : nodeAlgomatch,
      'CHIP_NAME' : matcherChipName,
      'acl' : matcherAcl,
      'table' : matcherTable,
      'layout' : 'Layout information',
      'last-attempt' : matcherLastAttempt,
   }
   cliModel = AlgoMatchTableLayout

   @staticmethod
   def handler( mode, args ):
      chipName = args[ 'CHIP_NAME' ]
      lastAttempt = 'last-attempt' in args
      rootStatus = _getRootStatus( chipName, lastAttempt )
      rootConfig = _getRootConfig( chipName, lastAttempt )
      finalLayoutInfo = AlgoMatchHwAclInfo()
      if rootStatus is None or rootConfig is None:
         mode.addError( 'Chip: %s is invalid' % chipName )
         raise CliParser.AlreadyHandledError()
      # Once we support snapshotting, to harmonize the codepath, we should not use
      # proxyChipConfig, and instead the new collection populated by packer, keyed
      # probably by epoch (where the last epoch- if ordered- will match pdConfig
      # on success)
      cdf = rootStatus.proxyChipConfig.chipDataFlowConfig
      for aclKey, aclIdSet in rootConfig.aclKeyToAclId.keyToIdTypeSet.items():
         for aclId in aclIdSet.aclIdType.values():
            hwAclInfoModel = _addHwAclInfoModel( finalLayoutInfo,
                                                   aclKey.type,
                                                   aclKey.name,
                                                   aclKey.direction,
                                                   _getHwAclType( aclId.tableType ),
                                                   aclId.chipRuleListId )
            stageToHashMask = \
                  cdf.chipRuleListIdToStageMask.get( aclId.chipRuleListId, None )
            if stageToHashMask:
               hwAclInfoModel.tableLayout = \
                     _getTableLayoutModel( rootStatus.maskGroupIndex,
                                           stageToHashMask.stageHashMask )
      return AlgoMatchTableLayout( chipName=chipName,
            finalLayoutInfo=finalLayoutInfo )

BasicCli.addShowCommandClass( ShowTableLayoutCmd )

#--------------------------------------------------------------------------------
# show platform algomatch CHIP_NAME acl table usage [ last-attempt ]
#--------------------------------------------------------------------------------
class ShowTableUsageCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform algomatch CHIP_NAME acl table usage [ last-attempt ]'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'algomatch' : nodeAlgomatch,
      'CHIP_NAME' : matcherChipName,
      'acl' : matcherAcl,
      'table' : matcherTable,
      'usage' : 'Usage information',
      'last-attempt' : matcherLastAttempt,
   }
   cliModel = AlgoMatchTableUsage

   @staticmethod
   def handler( mode, args ):
      chipName = args[ 'CHIP_NAME' ]
      lastAttempt = 'last-attempt' in args
      rootStatus = _getRootStatus( chipName, lastAttempt )
      if rootStatus is None:
         mode.addError( 'Chip: %s is invalid' % chipName )
         raise CliParser.AlreadyHandledError()
      usage = {}
      for stage, stageStats in rootStatus.stats.stageStat.items():
         usage[ stage ] = stageStats.entryCount
      return AlgoMatchTableUsage( chipName=chipName, usage=usage )

BasicCli.addShowCommandClass( ShowTableUsageCmd )

#--------------------------------------------------------------------------------
# show platform acl
#--------------------------------------------------------------------------------
class ShowHaloAclCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform acl'
   data = {
      'platform' : CliToken.Platform.platformMatcherForShow,
      'acl' : matcherAcl,
   }
   privileged = True
   cliModel = AleHaloCliModel.RootConfigColl

   @staticmethod
   def handler( mode, args ):
      '''Main entry for show platform acl'''
      rootConfigSysdb = rootConfigCollSysdb
      rootConfigColl = AleHaloCliModel.RootConfigColl()
      for key, rootConfigSysdb in rootConfigCollSysdb.chipRootConfig.items():
         if not rootConfigSysdb.active:
            continue
         rootConfig = AleHaloCliModel.RootConfig()
         populateHaloRootConfig( rootConfig, rootConfigSysdb )
         rootConfigColl.configs[ key ] = rootConfig
      for key, rootConfigSysdb in \
               rootConfigCollSysdb.chipRootConfigLastAttempt.items():
         rootConfig = AleHaloCliModel.RootConfig()
         populateHaloRootConfig( rootConfig, rootConfigSysdb )
         rootConfigColl.lastAttemptConfigs[ key ] = rootConfig

      return rootConfigColl

BasicCli.addShowCommandClass( ShowHaloAclCmd )

#-----------------------------------------------------------------
#-----------------------------------------------------------------

def populateMacFilter( macRuleConfig, macRuleConfigSysdb ):
   '''populate MacFilter'''
   macFilter = AleHaloCliModel.MacFilter()
   macFilterSysdb = macRuleConfigSysdb.filter
   macFilter.sourceAddr = macFilterSysdb.sourceAddr
   macFilter.sourceMask = macFilterSysdb.sourceMask
   macFilter.destAddr = macFilterSysdb.destinationAddr
   macFilter.destMask = macFilterSysdb.destinationMask
   macFilter.proto = macFilterSysdb.proto
   macRuleConfig.macFilter = macFilter

def populateIpFilter( ipRuleConfig, ipRuleConfigSysdb ):
   '''populate IpFilter'''
   ipFilter = AleHaloCliModel.IpFilter()
   ipFilterSysdb = ipRuleConfigSysdb.filter
   ipFilter.source = AleHaloCliModel.IpGenAddrWithFullMask()
   ipFilter.source.ip = ipRuleConfigSysdb.filter.source
   ipFilter.destination = AleHaloCliModel.IpGenAddrWithFullMask()
   ipFilter.destination.ip = ipRuleConfigSysdb.filter.destination
   ipFilter.sportRange = AleHaloCliModel.L4PortRange()
   ipFilter.sportRange.rangeStart = ipFilterSysdb.sportRange.rangeStart
   ipFilter.sportRange.rangeEnd = ipFilterSysdb.sportRange.rangeEnd
   ipFilter.dportRange = AleHaloCliModel.L4PortRange()
   ipFilter.dportRange.rangeStart = ipFilterSysdb.dportRange.rangeStart
   ipFilter.dportRange.rangeEnd = ipFilterSysdb.dportRange.rangeEnd

   if ipFilterSysdb.fragments:
      ipFilter.fragments = ipFilterSysdb.fragments
   if ipFilterSysdb.matchDscp:
      ipFilter.dscp = ipFilterSysdb.dscp
      ipFilter.matchDscp = ipFilterSysdb.matchDscp
   if ipFilterSysdb.matchProto:
      ipFilter.proto = ipFilterSysdb.proto
      ipFilter.matchProto = ipFilterSysdb.matchProto
   if ipFilterSysdb.includeTtl:
      ipFilter.ttl = ipFilterSysdb.ttl
      ipFilter.includeTtl = ipFilterSysdb.includeTtl
   if ipFilterSysdb.tcpFlag.value is not 0:
      ipFilter.tcpFlag = AleHaloCliModel.TcpFlag()
      for attr in [ 'fin', 'syn', 'rst', 'psh', 'ack', 'urg' ]:
         if getattr( ipFilterSysdb, attr ):
            setattr( ipFilter.tcpFlag, attr, getattr( ipFilterSysdb, attr ) )
   if ipFilterSysdb.established:
      ipFilter.established = ipFilterSysdb.established
   if ipFilterSysdb.icmpType != ipFilterSysdb.icmpAll:
      ipFilter.icmpType = ipFilterSysdb.icmpType
   if ipFilterSysdb.icmpCode != ipFilterSysdb.icmpAll:
      ipFilter.icmpCode = ipFilterSysdb.icmpCode
   if ipFilterSysdb.l4Rule:
      ipFilter.l4Rule = ipFilterSysdb.l4Rule
   ipRuleConfig.ipFilter = ipFilter

def populateRuleConfig( acl, ruleById ):
   macAcl = acl.aclId.tableType == TableType.macAcl
   for key, ruleConfigSysdb in ruleById.items():
      keyStr = '%d.%d.%d' % ( key.ruleId, key.subRuleId, key.subIndex )
      if macAcl:
         ruleConfig = AleHaloCliModel.MacRuleConfig()
         populateMacFilter( ruleConfig, ruleConfigSysdb )
         acl.macRules[ keyStr ] = ruleConfig
      else:
         ruleConfig = AleHaloCliModel.IpRuleConfig()
         populateIpFilter( ruleConfig, ruleConfigSysdb )
         acl.ipRules[ keyStr ] = ruleConfig
      ruleConfig.action = ruleConfigSysdb.action
      if ruleConfigSysdb.log:
         ruleConfig.log = ruleConfigSysdb.log
      if ruleConfigSysdb.count:
         ruleConfig.count = ruleConfigSysdb.count

def populateHaloAcl( acl, aclSysdb ):
   ''' populate HaloAcl '''
   acl.aclId = AleHaloCliModel.AclId()
   acl.aclId.tableType = aclSysdb.aclId.tableType
   acl.aclId.chipRuleListId = aclSysdb.aclId.chipRuleListId
   acl.aclId.direction = aclSysdb.aclId.direction
   acl.aclType = aclSysdb.aclType
   acl.name = aclSysdb.cliName

   populateRuleConfig( acl, aclSysdb.macRuleById )
   populateRuleConfig( acl, aclSysdb.ipRuleById )

def populateHaloRootConfig( rootConfig, rootConfigSysdb ):
   '''populate HaloRootConfig'''
   rootConfig.chipName = rootConfigSysdb.chipName
   for key, aclSysdb in rootConfigSysdb.haloAclColl.acl.items():
      keyStr = '%s:%d:%s' % ( key.tableType, key.chipRuleListId, key.direction )
      acl = AleHaloCliModel.HaloAcl()
      rootConfig.acls[ keyStr ] = acl
      populateHaloAcl( acl, aclSysdb )
   keyToIdList = list()
   for aclKeySysdb, aclIdSetSysdb in \
         rootConfigSysdb.aclKeyToAclId.keyToIdTypeSet.items():
      keyToId = AleHaloCliModel.AclKeyToAclId()
      aclKey = AleHaloCliModel.AclKey()
      aclKey.name = aclKeySysdb.name
      aclKey.aclType = aclKeySysdb.type
      aclKey.direction = aclKeySysdb.direction
      aclKey.standard = aclKeySysdb.standard
      aclKey.vlanId = aclKeySysdb.vlanId
      keyToId.aclKey = aclKey
      for proto, aclIdSysdb in aclIdSetSysdb.aclIdType.items():
         aclId = AleHaloCliModel.AclId()
         aclId.tableType = aclIdSysdb.tableType
         aclId.chipRuleListId = aclIdSysdb.chipRuleListId
         aclId.direction = aclIdSysdb.direction
         keyToId.aclIds[ proto ] = aclId
      keyToIdList.append( keyToId )
   rootConfig.keyToIds = keyToIdList


#-------------------------------------------------------------------------------
# Mount necessary mountpoints for show commands
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global aclStatus
   global rootConfigCollSysdb
   global rootStatusCollSysdb

   # AclStatus (used for CLI guard)
   aclStatus = LazyMount.mount( entityManager, 'acl/status/all', 'Acl::Status', 'r' )

   # RootConfigColl
   rootConfigCollSysdb = LazyMount.mount( entityManager,
                                          'acl/status/rootConfigColl',
                                          'Halo::RootConfigColl', 'r' )
   # RootStatusColl
   rootStatusCollSysdb = LazyMount.mount( entityManager,
                                          'acl/status/rootStatusColl',
                                          'Halo::RootStatusColl', 'r' )
