#!/usr/bin/env python
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, datetime
from CliModel import (
      Model,
      Str,
      Int,
      Bool,
      List,
      Dict,
      Enum,
      Float,
      Submodel,
      DeferredModel,
   )
from ArnetModel import MacAddress, IpGenericPrefix
from IntfModel import Interface

fibRouteTypeEnum = tuple( set( Tac.Type( 'Routing::Fib::RouteType' ).attributes ) |
                          set( Tac.Type( 'Routing6::Fib::RouteType' ).attributes ) )
l2AdjTypeEnum = tuple( Tac.Type( 'Ale::NextHopEntryType' ).attributes )
l2AdjStateEnum = tuple( Tac.Type( 'Ale::L2AdjState' ).attributes )
fecInfoUnavailable = Tac.Type( 'Ale::CliFecTypeStrings' ).fecInfoUnavailable
routeTypeEnum = tuple( set( Tac.Type( 'Ale::RouteType' ).attributes ).union(
                                                        { fecInfoUnavailable } ) )
decimalPlaces = 5

# optimized - Routes in LEM ( for Sand )
# optimizedAndExpanded - Routes Expanded in Hardware
# optimizedAndCompressed - Routes Compressed in Hardware
routeOptimizedEnum = ( 'unoptimized', 'optimized',
                       'optimizedAndExpanded', 'optimizedAndCompressed' )

stateEnum = ( 'enabled', 'disabled' )
support = ( 'supported', 'notSupported' )
labelOperationEnum = ( 'push', 'pop', 'swap', 'forward' )

# This is same as Ark.py:timeStampToStr but also gives microseconds
def timestampToStr( timestamp ):
   td = datetime.datetime.fromtimestamp( Tac.utcNow() - ( Tac.now() - timestamp ) )
   return td.strftime( '%Y-%m-%d %H:%M:%S.%f' )

class AleStats( Model ):
   insertTime = Float( help='Insert timestamp' )
   lastUpdateTime = Float( help='Update timestamp' )
   numUpdates = Int( help='Number of updates' )

class AlePrevLevelFecs( Model ):
   fecId = Int( help='Previous level FEC ID' )
   tableId = Int( help='FEC table ID' )

class GreEncapInfo( Model ):
   destination = IpGenericPrefix(
             help='Location in the IP network where the tunnel terminates' )
   source = IpGenericPrefix(
             help='Location in the IP network where the tunnel starts' )
   tos = Int( optional=True, help='Type of service' )
   ttl = Int( optional=True, help='Time to live' )

class MplsEncapInfo( Model ):
   operation = Enum( values=labelOperationEnum, help='Label Operation' )
   labelStack = List( valueType=int, help='MPLS Label Stack' )

class AleL2AdjVia( Model ):
   weight = Int( help='Metric representing the hop cost' )
   adjType = Enum( values=l2AdjTypeEnum, help='L2 Adjacency Type' )
   macAddr = MacAddress( help='Next-Hop Mac Address' )
   l3IntfId = Interface( help='L3 Interface Id' )
   l2IntfId = Interface( help='L2 Interface Id' )
   nextLevelL2AdjIndex = Int(
         optional=True,
         help='L2 Adjacency ID of the next FEC in a hierarchical FEC' )
   counterId = Int( optional=True, help='Next-Hop Counter Id' )
   mplsEncapInfo = Submodel( valueType=MplsEncapInfo, help="MPLS Encap information",
                             optional=True )
   greEncapInfo = Submodel( valueType=GreEncapInfo, help="GRE Encap information",
                             optional=True )

class AleL3AdjViaTunnel( Model ):
   tunnelType = Str( help="Tunnel type" )
   tunnelAddressFamily = Enum( values=( 'IPv4', 'IPv6' ), help="Address family",
                               optional=True )
   tunnelIndex = Int( help="Tunnel table index" )

class AleL3AdjVia( Model ):
   nextHopIp = IpGenericPrefix( help='Ip Address of Next Hop' )
   weight = Int( help='Metric representing the hop cost' )
   l3IntfId = Interface( help='L3 Interface Id' )
   tunnelDescriptor = Submodel( valueType=AleL3AdjViaTunnel,
                                help="Tunnel information", optional=True )

class FecProxyInfo( Model ):
   sourceFec = Int( help='Source Fec Id for this proxy set' )
   cmdErr = Str( optional=True, help="Error Message" )
   proxyFecList = List( valueType=long, help='List of proxy FECs for source' )

class FecProxyInfoList( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   fecAliases = List( optional=True,
                      valueType=FecProxyInfo, help='List of FEC proxies' )

class AleL2AdjInfo( Model ):
   adjCount = Int( optional=True, help='Number of L3 Adjacencies' )
   l2AdjState = Enum( values=l2AdjStateEnum, help='L2 Adjacency State' )
   l2AdjVia = List( valueType=AleL2AdjVia, optional=True,
                    help='List storing the L2 Adjacency Vias' )
   backupL2AdjVia = List( valueType=AleL2AdjVia, optional=True,
                          help='List storing the backup L2 Adjacency Vias' )
   platformFecIndex = Int( optional=True,
                           help='Platform FEC table index used by the L2 Adjacency' )
   l2AdjStats = Submodel( valueType=AleStats, optional=True,
                          help='L2 Adjacency timestamps' )

class AlePrefix( Model ):
   prefix = Str( help='Prefix' )

class AleV4Prefixes( Model ):
   v4Prefixes = List( valueType=AlePrefix, optional=True,
      help="A mapping between VRF ID and V4 Prefixes" )

class AleV6Prefixes( Model ):
   v6Prefixes = List( valueType=AlePrefix, optional=True,
      help="A mapping between VRF ID and V6 Prefixes" )

class AleL3AdjInfo( Model ):
   routeType = Enum( values=routeTypeEnum, optional=True,
                     help='Specifies Route Type' )
   l2AdjIndex = Int( optional=True, help='Layer 2 Adjacency Index' )
   prefixCount = Int( optional=True, help='Number of Prefixes' )
   adjRefCount = Int( optional=True, help='Inter Adjacency references' )
   pbrRefCount = Int( optional=True, help='PBR references' )
   extRefCount = Int( optional=True, help='External FEC references' )
   proxyRefCount = Int( optional=True, help='proxy FEC references' )
   totalRefCount = Int( optional=True, help='Total references' )
   v4PrefixesByVrf = Dict( keyType=str, valueType=AleV4Prefixes,
                           optional=True,
                           help='Dict to list V4 Prefix references by VRF' )
   v6PrefixesByVrf = Dict( keyType=str, valueType=AleV6Prefixes,
                           optional=True,
                           help='Dict to list V6 Prefix references by VRF' )
   adjStats = Submodel( valueType=AleStats, optional=True,
                        help='SubModel storing Adjacency Timestamps' )
   l3AdjVia = List( valueType=AleL3AdjVia, optional=True,
                    help='List storing the L3 V4 Adjacency Vias' )
   backupL3AdjVia = List( valueType=AleL3AdjVia, optional=True,
                          help='List storing the backup L3 V4 Adjacency Vias' )
   l3AdjVia6 = List( valueType=AleL3AdjVia, optional=True,
                    help='List storing the L3 V6 Adjacency Vias' )
   backupL3AdjVia6 = List( valueType=AleL3AdjVia, optional=True,
                           help='List storing the backup L3 V6 Adjacency Vias' )
   transientAdj = Bool( optional=True,
                        help="ECMP Adj programmed as non ECMP in hardware" )
   orderedNexthops = Bool( optional=True,
                           help="The ordering of next-hops is honored" )
   sourceFec = Int( optional=True, help="Source FEC being used by this proxy" )
   fecRefs = List( valueType=AlePrevLevelFecs, optional=True,
                         help='List of previous level fec refs' )

class AleRouteInfo( Model ):
   cmdErr = Str( optional=True, help="Error Message" )
   routeProgrammed = Bool( optional=True,
                           help='Flag to indiacte Route is programmed or not' )
   routeOptimized = Enum( values=routeOptimizedEnum, optional=True,
                          help='Details of Route Optimization' )
   parent = Str( optional=True, help='Parent prefix' )
   l3AdjInfo = Dict( keyType=long, valueType=AleL3AdjInfo,
                     optional=True, help='Dict to handle L3 Adjacency Info' )
   l2AdjInfo = Dict( keyType=long, valueType=AleL2AdjInfo,
                     optional=True, help='Dict to handle L2 Adjacency Info' )
   routeStats = Submodel( valueType=AleStats, optional=True,
                      help='Stats of a route' )

class AleRoute( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   vrfName = Str( optional=True, help="VRF name" )
   numUnprog = Int( optional=True, help="Count of unprogrammed routes" )
   fibRoute = Dict( keyType=IpGenericPrefix, valueType=AleRouteInfo, optional=True,
                    help="Mapping prefix to its FibRoute Info" )

class AleL2Adj( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   l2Adj = Dict( keyType=long, valueType=AleL2AdjInfo, optional=True,
                 help="Dictionary to map l2AdjIndex to l2Adj Info" )

class AleAdj( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   adj = Dict( keyType=long, valueType=AleL3AdjInfo,
               optional=True,
               help="Dictionary to map fecId to L3Adj" )
   transientAdjsPresent = Bool( optional=True,
                                help='Output contains some transient adjs' )

class FibFecInfo( Model ):
   cmdErr = Str( optional=True, help="Error flag" )
   fecType = Enum( values=routeTypeEnum, optional=True, help="FEC type" )
   vias = Int( optional=True, help="Number of vias for the route" )
   fecVias = List( valueType=AleL3AdjVia, optional=True,
                   help="Via list" )
   backupFecVias = List( valueType=AleL3AdjVia, optional=True,
                         help="Backup Via list" )

class FibFec( DeferredModel ):
   cmdErr = Str( optional=True, help="Error flag" )
   routeType = Enum( values=fibRouteTypeEnum, optional=True, help="Route type" )
   fec = Dict( keyType=long, valueType=FibFecInfo, optional=True,
               help="Mapping fecId to its FEC Info" )
   routeCompressed = Bool( optional=True,
                           help="Route is compressed" )
   compressedByRoute = IpGenericPrefix( optional=True,
                                        help="IP Address of the parent route" )

class FibRoute( DeferredModel ):
   cmdErr = Str( optional=True, help="Error flag" )
   vrfName = Str( optional=True, help="VRF Name" )
   route = Dict( keyType=IpGenericPrefix, valueType=FibFec, optional=True,
                 help="Mapping prefix to its FibRoute Info" )

class AdjResourceOptimizationThresholds( Model ):
   low = Int( help="Threshold for starting resource optimization" )
   high = Int( help="Threshold for stopping resource optimization" )

class AdjResourceOptimization( Model ):
   supported = Enum( values=support,
                     help="Adjacency resource optimization support" )
   enabled = Enum( values=stateEnum,
                   optional=True,
                   help="Adjacency resource optimization enabled" )
   thresholds = Submodel( valueType=AdjResourceOptimizationThresholds,
                          optional=True,
                          help="Adjacency resource optimization thresholds" )

class FibSummaryStatus( Model ):
   __revision__ = 2
   adjShare = Enum( values=stateEnum, optional=True,
                    help="Adjacency sharing status" )
   bfdEvent = Enum( values=stateEnum, optional=True, help="BFD peer event" )
   deletionDelay = Float( help="Deletion delay" )
   pbrSupport = Enum( values=support, optional=True, help="PBR support" )
   urpfSupport = Enum( values=support, help="URPF support" )
   icmpStatus = Enum( values=stateEnum, help="ICMP status" )
   maxAleEcmp = Int( optional=True, help="Max Ale ECMP" )
   ucmpWeightDeviation = Float( optional=True, help="UCMP weight deviation" )
   maxRoutes = Int( help="Maximum number of routes" )
   protectDefaultRoute = Enum( values=stateEnum,
                               help="Protect default route status" )
   fibCompression = Enum( values=stateEnum,
                          help="FIB compression status" )
   adjResourceOptimization = Submodel( valueType=AdjResourceOptimization,
                           optional=True,
                           help="Adjacency resource optimization" )

   def degrade( self, dictRepr, revision ):
      # Removed 'parentDrop' in revision 2. By default is disabled
      if 1 == revision:
         dictRepr[ 'parentDrop' ] = 'disabled'
      return dictRepr

   def render( self ):
      print 'Fib summary'
      print '-----------'
      if self.adjShare:
         print 'Adjacency sharing: %s' % self.adjShare
      if self.bfdEvent:
         print 'BFD peer event: %s' % self.bfdEvent
      print 'Deletion Delay: %d' % self.deletionDelay
      if self.protectDefaultRoute:
         print 'Protect default route: %s' % self.protectDefaultRoute
      if self.pbrSupport:
         print 'PBR: %s' % self.pbrSupport
      if self.urpfSupport:
         print 'URPF: %s' % self.urpfSupport
      if self.icmpStatus:
         print 'ICMP unreachable: %s' %  self.icmpStatus
      if self.maxAleEcmp is not None:
         print 'Max Ale ECMP: %s' % self.maxAleEcmp
      if self.ucmpWeightDeviation is not None:
         print 'UCMP weight deviation: %s' % \
               str( round( self.ucmpWeightDeviation, decimalPlaces ) )
      print 'Maximum number of routes: %d' % self.maxRoutes
      if self.fibCompression:
         print 'Fib compression: %s' % self.fibCompression
      if self.adjResourceOptimization is not None:
         if self.adjResourceOptimization.supported == 'supported':
            if self.adjResourceOptimization.enabled is not None:
               print 'Resource optimization for adjacency programming: %s' % (
                  self.adjResourceOptimization.enabled )
            if self.adjResourceOptimization.thresholds is not None:
               print ( 'Adjacency resource optimization thresholds: '
                       'low %d, high %d' % (
                        self.adjResourceOptimization.thresholds.low,
                        self.adjResourceOptimization.thresholds.high ) )
         else:
            print 'Resource optimization for adjacency programming: %s' % (
                  self.adjResourceOptimization.supported )

class AleResEcmpSummary( Model ):
   vrfName = Str( help="VRF Name" )
   numPrefixes = Int( help="Number of programmed resilient ECMP prefixes" )
   numProgrammed = Int( help="Number of routes programmed as resilient" )
   numUnprogrammed = Int( help="Number of routes not programmed as resilient" )

   def render( self ):
      print 'VRF: %s' % self.vrfName
      print 'Number of resilient prefixes: %d' % self.numPrefixes
      print 'Number of programmed resilient routes: %d' % self.numProgrammed
      print 'Number of unprogrammed resilient routes: %d' % self.numUnprogrammed

class AleResEcmpRouteInfo( Model ):
   prefix = IpGenericPrefix( help="The route's prefix" )
   resilient = Bool( help="The route is programmed as resilient" )
   fecId = Int( help="The non-resilient FEC ID" )
   resFecId = Int( help="The resilient FEC ID" )
   reason = Str( help="Resason why route is not programmed as resilient" )

class AleResEcmpRouteInfos( Model ):
   capacity = Int( optional=True, help="The resilient capacity" )
   redundancy = Int( optional=True, help="The resilient redundancy" )
   routes = List( valueType=AleResEcmpRouteInfo,
                  help="List of resilient routes information" )

class AleResEcmpAllOrSpecific( Model ):
   vrfName = Str( help="VRF name" )
   resPrefixesToRouteInfos = Dict( keyType=IpGenericPrefix,
                                   valueType=AleResEcmpRouteInfos,
                                   help="Map from resilient ECMP prefixes to their \
                                         covered routes" )
   _oneRoute = Bool( help="Render one specific route" )

   def render( self ):
      print 'VRF: %s' % self.vrfName

      if not self._oneRoute:
         print 'Codes: * - route not resilient'
         # Printing all routes
         for resPrefix in self.resPrefixesToRouteInfos:
            routeInfos = self.resPrefixesToRouteInfos[ resPrefix ]
            print '%s: capacity: %d, redundancy: %d' % ( resPrefix,
                                                         routeInfos.capacity,
                                                         routeInfos.redundancy )
            for routeInfo in routeInfos.routes:
               unprogrammedStr = ''
               resFecIdStr = ''
               if routeInfo.resilient:
                  resFecIdStr = ', resilient FEC ID: %d' % ( routeInfo.resFecId )
               else:
                  unprogrammedStr = '*'
               print '\t%s%s, FEC ID: %d%s' % ( unprogrammedStr, routeInfo.prefix,
                                                routeInfo.fecId, resFecIdStr )
      else:
         # Printing a specific route
         coveringPrefix, routeInfos = self.resPrefixesToRouteInfos.items()[ 0 ]
         routeInfo = routeInfos.routes[ 0 ]

         print '%s:' % routeInfo.prefix
         if routeInfo.resilient:
            print '\tResilient: yes, capacity: %d, redundancy: %d' % (
               routeInfos.capacity, routeInfos.redundancy )
         else:
            print '\tResilient: no'
         print '\tCovered by: %s' % coveringPrefix
         print '\tFEC ID: %d' % routeInfo.fecId
         print '\tResilient FEC ID: %d' % routeInfo.resFecId
         print '\tReason: %s' % routeInfo.reason
