#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import AgentCommandRequest
import Arnet
import BasicCli
import CliCommand
from CliMatcher import (
   IntegerMatcher,
   KeywordMatcher
)
import CliModel
import CliParser
import CliPlugin.VrfCli as VrfCli
import CliToken.Hardware
import CliToken.Ip
import CliToken.Ipv6
import CommonGuards
import IpAddrMatcher
import Ip6AddrMatcher
import IraCommonCli
import ShowCommand
import TableOutput
import Tac
import TechSupportCli
import Tracing

import AleCliLib
import AleFibCliDiffUtilLib
from AleFibCliModel import (
      AdjResourceOptimization,
      AdjResourceOptimizationThresholds,
      AleRoute,
      AleL2Adj,
      AleAdj,
      AleResEcmpAllOrSpecific,
      AleResEcmpRouteInfo,
      AleResEcmpRouteInfos,
      AleResEcmpSummary,
      FecProxyInfoList,
      FibRoute,
      FibFec,
      FibSummaryStatus,
      stateEnum,
      support,
)
from ArnetModel import IpGenericPrefix
import SharedMem

__defaultTraceHandle__ = Tracing.Handle( 'AleFibCli' )

#------------------------------------------------------------------------------------
# Common tokens used in multiple commands
#------------------------------------------------------------------------------------

ipMatcherForShow = CliToken.Ip.ipMatcherForShow
ipv6MatcherForShow = CliToken.Ipv6.ipv6MatcherForShow

hardwareMatcherForShow = CliToken.Hardware.hardwareMatcherForShow

aleKwForShow = CliCommand.guardedKeyword( 'ale',
      helpdesc="Ale-specific information",
      guard=CommonGuards.standbyGuard )
ecmpMatcherForShow = KeywordMatcher( 'ecmp',
      helpdesc='Show ECMP routes' )
resilienceMatcherForShow = KeywordMatcher( 'resilience',
      helpdesc='Show Resilience in ECMP routes' )
summaryMatcher = KeywordMatcher( 'summary',
      helpdesc="configuration parameters summary" )
vrfExprFactory = VrfCli.VrfExprFactory( helpdesc='Specify the VRF' )

def vrfRootSmGuard( mode, token ):
   vrfTableStatus = AleCliLib.vrfTableStatus
   if not vrfTableStatus.vrfNameToFibTableId:
      return CliParser.guardNotThisPlatform
   return None

#------------------------------------------------------------------------------------
# "show ip hardware fib summary
# "show ipv6 hardware fib summary
#------------------------------------------------------------------------------------

class ShowIpIpv6HardwareFibSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware fib summary'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'summary': summaryMatcher,
          }
   cliModel = FibSummaryStatus
   privileged = True

   @staticmethod
   def handler( mode, args ):
      v4 = 'ip' in args
      if v4:
         routingHwConfig = AleCliLib.routingHwConfig
         routingHwStatus = AleCliLib.routingHwStatus
         routingHwRouteStatus = AleCliLib.routingHardwareRouteStatus
      else:
         routingHwConfig = AleCliLib.routing6HwConfig
         routingHwStatus = AleCliLib.routing6HwStatus
         routingHwRouteStatus = AleCliLib.routing6HardwareRouteStatus

      def _enabledState( enabled ):
         return stateEnum[ 0 ] if enabled else stateEnum[ 1 ]

      def _supportedState( supported ):
         return support[ 0 ] if supported else support[ 1 ]

      fibSummary = FibSummaryStatus()

      # Attributes in both V4 and V6
      fibSummary.deletionDelay = routingHwRouteStatus.deletionDelay
      fibSummary.protectDefaultRoute = _enabledState(
            routingHwConfig.protectDefaultRoute )
      fibSummary.urpfSupport = _supportedState( routingHwStatus.urpfSupported )
      fibSummary.icmpStatus = _enabledState( routingHwConfig.icmpUnreachable )
      fibSummary.maxRoutes = routingHwRouteStatus.maxNumRoutes
      fibSummary.fibCompression = _enabledState( routingHwConfig.fibSuppression )

      if v4:
         fibSummary.adjShare = _enabledState( routingHwRouteStatus.adjSharing )
         fibSummary.bfdEvent = _enabledState(
               routingHwRouteStatus.bfdPeerEventEnabled )
         fibSummary.pbrSupport = _supportedState( routingHwStatus.pbrSupported )
         fibSummary.maxAleEcmp = routingHwRouteStatus.effectiveMaxLogicalEcmp
         fibSummary.ucmpWeightDeviation = routingHwConfig.ucmpWeightDeviation
         fibSummary.adjResourceOptimization = AdjResourceOptimization()
         adjResourceOpt = fibSummary.adjResourceOptimization
         adjResourceOpt.supported = _supportedState(
            routingHwStatus.adjResourceOptimizationSupported )
         if routingHwStatus.adjResourceOptimizationSupported:
            adjResourceOpt.enabled = _enabledState(
                  routingHwConfig.adjResourceOptimizationEnabled )
            adjResourceOpt.thresholds = AdjResourceOptimizationThresholds()
            thresholds = fibSummary.adjResourceOptimization.thresholds
            thresholdConfig = routingHwConfig.adjResourceOptimizationThresholds
            thresholds.low = thresholdConfig.stopOptimizationThreshold
            thresholds.high = thresholdConfig.startOptimizationThreshold

      return fibSummary

BasicCli.addShowCommandClass( ShowIpIpv6HardwareFibSummaryCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib diff
#------------------------------------------------------------------------------------

def doShowL3FibRoutesV4Diff( mode, args ):
   cmd = [ '/usr/bin/AleFibCliDiffUtil' ]

   if os.environ.get( 'SIMULATION_VMID' ):
      em = mode.session_.entityManager
      ( routeStatus, vrfMap, routeHwStatus, arpSmash, routing6HwRouteStatus,
        arpVrfMap ) = AleFibCliDiffUtilLib.mountEntities( em )

      AleFibCliDiffUtilLib.runCmd( em, routeStatus, vrfMap, routeHwStatus, arpSmash,
                                   routing6HwRouteStatus, arpVrfMap )
   else:
      Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )

diffMatcher = KeywordMatcher(
      "diff",
      helpdesc="Display routing table differences between fib and platform" )

class ShowIpHardwareFibDiffCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib diff'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'diff': diffMatcher,
          }
   privileged = True
   handler = doShowL3FibRoutesV4Diff

BasicCli.addShowCommandClass( ShowIpHardwareFibDiffCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib routes [vrf vrfName] [prefix]
#------------------------------------------------------------------------------------

def doShowL3FibRoutesV4( mode, args ):
   cmd = "FR4#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += "#"
   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode,
                                                 asyncCommand=True )
   return FibRoute

v4RoutesMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "routes", helpdesc="Display IPv4 routes" ),
   guard=vrfRootSmGuard )

class IpPrefixOrAddrMatcher( CliCommand.CliExpression ):
   expression = 'ADDR | PREFIX'
   data = { 'ADDR': IpAddrMatcher.IpAddrMatcher( 'Match this IP address' ),
            'PREFIX': IpAddrMatcher.ipPrefixExpr(
                        'Match this IP address',
                        'Match this subnet mask',
                        'Match this IP prefix',
                        partial=False,
                        overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''Consistently provides PREFIX as an Arnet::Prefix'''
      # If an address is provided, convert it to /32 prefix
      if 'ADDR' in args:
         args[ 'PREFIX' ] = Arnet.Prefix( args.pop( 'ADDR' ) )
      elif 'PREFIX' in args:
         args[ 'PREFIX' ] = Arnet.Prefix( args[ 'PREFIX' ] )

class ShowIpHardwareFibRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib routes [ VRF ] [ PREFIX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'routes': v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'PREFIX': IpPrefixOrAddrMatcher,
          }
   cliModel = FibRoute
   privileged = True
   handler = doShowL3FibRoutesV4

BasicCli.addShowCommandClass( ShowIpHardwareFibRoutesCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware fib routes [vrf vrfName] [prefix]
#------------------------------------------------------------------------------------

def doShowL3FibRoutesV6( mode, args ):
   cmd = "FR6#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += "#"
   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode,
                                                 asyncCommand=True )
   return FibRoute

v6RoutesMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "routes", helpdesc="Display IPv6 routes" ),
   guard=vrfRootSmGuard )

class Ip6PrefixOrAddrMatcher( CliCommand.CliExpression ):
   expression = 'ADDR | PREFIX'
   data = { 'ADDR': Ip6AddrMatcher.Ip6AddrMatcher( 'Match this IPv6 address' ),
            'PREFIX': Ip6AddrMatcher.ip6PrefixExpr(
               'Match this IPv6 address',
               'Match this subnet mask',
               'Match this IPv6 prefix',
               overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
               fullMaskValid=False ),
          }

   @staticmethod
   def adapter( mode, args, argsList ):
      '''Consistently provides PREFIX as an Ip6AddrWithMask'''
      # If an address is provided, convert it to /128 prefix
      if 'ADDR' in args:
         args[ 'PREFIX' ] = Arnet.Ip6AddrWithMask( args.pop( 'ADDR' ) )
      elif 'PREFIX' in args:
         # Even though fullMaskValid is False, we still get back a
         # Ip6AddrWithFullMask. Convert, since the mask is required to be contiguous,
         # and it is easier to return a consistent type.
         pfx = args[ 'PREFIX' ]
         args[ 'PREFIX' ] = Arnet.Ip6AddrWithMask( pfx.address,
                                                   mask=pfx.prefixLen() )

class ShowIp6HardwareFibRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib routes [ VRF ] [ PREFIX ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'routes': v6RoutesMatcher,
            'VRF': vrfExprFactory,
            'PREFIX': Ip6PrefixOrAddrMatcher,
          }
   cliModel = FibRoute
   privileged = True
   handler = doShowL3FibRoutesV6

BasicCli.addShowCommandClass( ShowIp6HardwareFibRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib fec [vrf vrfName] [fecId]
# "show ipv6 hardware fib fec [vrf vrfName] [fecId]
#------------------------------------------------------------------------------------

def doShowL3FibFec( mode, args ):
   cmd = "FF4#" if 'ip' in args else "FF6#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   fecId = args.get( 'FECID' )
   if fecId:
      cmd += str( fecId )
   cmd += "#"

   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode,
                                                 asyncCommand=True )
   return FibFec

fecMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "fec",
                           helpdesc="Display adjacencies in this address family" ),
   guard=vrfRootSmGuard )

fecIdMatcher = IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF, helpdesc='fec identifier' )

class ShowIpIp6HardwareFibFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware fib fec [ VRF ] [ FECID ]'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'fib': IraCommonCli.fibMatcher,
            'fec': fecMatcher,
            'VRF': vrfExprFactory,
            'FECID': fecIdMatcher,
          }
   cliModel = FibFec
   privileged = True
   handler = doShowL3FibFec

BasicCli.addShowCommandClass( ShowIpIp6HardwareFibFecCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib ecmp resilience [vrf vrfname] summary"
#------------------------------------------------------------------------------------

def getVrfNameAndResilientEcmpStatus( vrfName, ipv4 ):
   defaultVrfName = Tac.Type( 'L3::VrfName' ).defaultVrf
   if not vrfName:
      vrfName = defaultVrfName

   vrfTableStatus = AleCliLib.vrfTableStatus
   if vrfName not in vrfTableStatus.vrfNameToFibTableId:
      return ( vrfName, None )

   if ipv4:
      resilientEcmpStatus = AleCliLib.resilientEcmpStatus
      mountPath = 'routing/vrf/resilientEcmpStatus/%s'
      mountType = 'Ale::ResilientEcmpStatus'
   else:
      resilientEcmpStatus = AleCliLib.resilientEcmpStatus6
      mountPath = 'routing6/vrf/resilientEcmpStatus/%s'
      mountType = 'Ale::ResilientEcmpStatus6'

   if vrfName != defaultVrfName:
      shmemEm = SharedMem.entityManager( sysdbEm=AleCliLib.entityManager )
      resilientEcmpStatus = shmemEm.doMount( mountPath % vrfName,
                                             mountType,
                                             AleCliLib.smashReaderInfo )

   return ( vrfName, resilientEcmpStatus )

def doShowResilientEcmpSummary( mode, args ):
   vrfName = args.get( 'VRF' )
   ipv4 = 'ip' in args

   vrfName, resilientEcmpStatus = getVrfNameAndResilientEcmpStatus( vrfName, ipv4 )
   if not resilientEcmpStatus:
      mode.addError( "VRF %s not found" % vrfName )
      return None

   coveringPrefixes = set()
   programmedRoutes = 0
   unprogrammedRoutes = 0
   routePrefixToResEcmpInfo = resilientEcmpStatus.routePrefixToResilientEcmpInfo
   for pfx in routePrefixToResEcmpInfo:
      resilientEcmpInfo = routePrefixToResEcmpInfo[ pfx ]
      coveringPrefixes.add( resilientEcmpInfo.coveringResEcmpPrefix )
      if resilientEcmpInfo.resilientAdjIndex > 0:
         programmedRoutes += 1
      else:
         unprogrammedRoutes += 1

   aleResEcmpSummary = AleResEcmpSummary()
   aleResEcmpSummary.vrfName = vrfName
   aleResEcmpSummary.numPrefixes = len( coveringPrefixes )
   aleResEcmpSummary.numProgrammed = programmedRoutes
   aleResEcmpSummary.numUnprogrammed = unprogrammedRoutes

   return aleResEcmpSummary

def resilientEcmpGuard( mode, token ):
   if AleCliLib.routingHwStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

class ShowIpEcmpResilienceSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib ecmp resilience [ VRF ] summary'
   data = { "ip": ipMatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpGuard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "summary": summaryMatcher,
            }
   cliModel = AleResEcmpSummary
   privileged = True
   handler = doShowResilientEcmpSummary

BasicCli.addShowCommandClass( ShowIpEcmpResilienceSummaryCmd )

#------------------------------------------------------------------------------------
# "show ip hardware fib ecmp resilience [vrf vrfname] [prefix]"
#------------------------------------------------------------------------------------

def doShowResilientEcmpStatus( mode, args ):
   # pylint: disable=protected-access
   def createAleResEcmpRouteInfo( info ):
      def reasonStr( result ):
         resultEnum = Tac.Type( 'Ale::ReviewResilientAdjResult' )
         if result == resultEnum.resEcmpSuccess:
            return 'Success'
         elif result == resultEnum.noLongerResilientFailure:
            return 'Route no longer resilient'
         elif result == resultEnum.noFecIdFailure:
            return 'No FEC ID found for route'
         elif result == resultEnum.reviewAdjFailure:
            return 'Failed to review adjacency for route'
         elif result == resultEnum.invalidViasFailure:
            return 'Route has invalid vias'
         elif result == resultEnum.nonForwardRouteFailure:
            return 'Non-forward route not made resilient'
         elif result == resultEnum.emptyViasFecFailure:
            return 'FEC has empty via set'
         elif result == resultEnum.kernelRouteFailure:
            return 'Kernel route not made resilient'
         elif result == resultEnum.unprogrammedRouteFailure:
            return 'Route is not programmed'
         else:
            return ''

      result = AleResEcmpRouteInfo()
      result.prefix = Arnet.IpGenPrefix( str( info.key ) )
      result.resilient = info.resilientAdjIndex != 0
      result.fecId = info.adjIndex
      result.resFecId = info.resilientAdjIndex if result.resilient else 0
      result.reason = reasonStr( info.reviewResilientAdjResult )

      return result

   def updateAllOrSpecificOutput( prefix ):
      routePrefixToResEcmpInfo = \
         resilientEcmpStatus.routePrefixToResilientEcmpInfo[ prefix ]

      coveringPrefix = \
         Arnet.IpGenPrefix( str( routePrefixToResEcmpInfo.coveringResEcmpPrefix ) )

      routeInfo = createAleResEcmpRouteInfo( routePrefixToResEcmpInfo )

      routeInfoList = AleResEcmpRouteInfos()
      if coveringPrefix in output.resPrefixesToRouteInfos:
         routeInfoList = output.resPrefixesToRouteInfos[ coveringPrefix ]

      resEcmpConfig = \
         routingHwConfig.resilientEcmpPrefix[ coveringPrefix ]
      if not output._oneRoute or routeInfo.resilient:
         routeInfoList.capacity = resEcmpConfig.capacity
         routeInfoList.redundancy = resEcmpConfig.redundancy

      routeInfoList.routes.append( routeInfo )

      output.resPrefixesToRouteInfos[ coveringPrefix ] = routeInfoList

   vrfName = args.get( 'VRF' )
   prefix = args.get( 'PREFIX' )
   ipv4 = 'ip' in args

   vrfName, resilientEcmpStatus = getVrfNameAndResilientEcmpStatus( vrfName, ipv4 )
   if not resilientEcmpStatus:
      mode.addError( "VRF %s not found" % vrfName )
      return None

   output = AleResEcmpAllOrSpecific()
   output._oneRoute = prefix is not None

   output.vrfName = vrfName
   if ipv4:
      routingHwConfig = AleCliLib.routingHwConfig
   else:
      routingHwConfig = AleCliLib.routing6HwConfig

      # The matcher consistently returns an Ip6AddrWithMask
      prefix = Arnet.Ip6Prefix( str( prefix ) ) if output._oneRoute else None

   if output._oneRoute:
      # Specific prefix case
      if prefix not in resilientEcmpStatus.routePrefixToResilientEcmpInfo:
         mode.addError( "Route %s not found" % prefix )
         return None

      updateAllOrSpecificOutput( prefix )
   else:
      # All prefixes case
      for pfx in resilientEcmpStatus.routePrefixToResilientEcmpInfo:
         updateAllOrSpecificOutput( pfx )

   # pylint: enable=protected-access
   return output

class ShowIpEcmpResilienceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware fib ecmp resilience [ VRF ] [ PREFIX ]'
   data = { "ip": ipMatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpGuard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "PREFIX": IpPrefixOrAddrMatcher,
            }
   cliModel = AleResEcmpAllOrSpecific
   privileged = True
   handler = doShowResilientEcmpStatus

BasicCli.addShowCommandClass( ShowIpEcmpResilienceCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware fib ecmp resilience [vrf vrfname] summary"
#------------------------------------------------------------------------------------

def resilientEcmpIp6Guard( mode, token ):
   if AleCliLib.routing6HwStatus.resilientEcmpSupported:
      return None
   return CliParser.guardNotThisPlatform

class ShowIpv6EcmpResilienceSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib ecmp resilience [ VRF ] summary'
   data = { "ipv6": ipv6MatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpIp6Guard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "summary": summaryMatcher,
            }
   cliModel = AleResEcmpSummary
   privileged = True
   handler = doShowResilientEcmpSummary

BasicCli.addShowCommandClass( ShowIpv6EcmpResilienceSummaryCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware fib ecmp resilience [vrf vrfname] [prefix]"
#------------------------------------------------------------------------------------

class ShowIpv6EcmpResilienceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware fib ecmp resilience [ VRF ] [ PREFIX ]'
   data = { "ipv6": ipv6MatcherForShow,
            "hardware": hardwareMatcherForShow,
            "fib": IraCommonCli.fibMatcher,
            "ecmp": CliCommand.Node( matcher=ecmpMatcherForShow,
                                     guard=resilientEcmpIp6Guard ),
            "resilience": resilienceMatcherForShow,
            "VRF": vrfExprFactory,
            "PREFIX": Ip6PrefixOrAddrMatcher,
            }
   cliModel = AleResEcmpAllOrSpecific
   privileged = True
   handler = doShowResilientEcmpStatus

BasicCli.addShowCommandClass( ShowIpv6EcmpResilienceCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale routes [vrf vrfname] drop
# "show ipv6 hardware ale routes [vrf vrfname] drop
#------------------------------------------------------------------------------------

class VrfDropRoute( CliModel.Model ):
   vrfName = CliModel.Str( help="Name of a VRF" )
   insertTime = CliModel.Float( help='Drop route creation time' )
   prefix = IpGenericPrefix( help="The route's prefix" )

class DropRouteStatus( CliModel.Model ):
   vrfDropRoutes = CliModel.List( optional=True, valueType=VrfDropRoute,
                              help="List of all drop routes" )

   def render( self ):
      if not self.vrfDropRoutes:
         return
      table = TableOutput.createTable( ( "Prefix", "VRF Name", "Creation Time" ) )
      tableMaxWidth = table.printableWidth()
      if tableMaxWidth < 80:
         tableMaxWidth = 80

      f1 = TableOutput.Format( justify="left", maxWidth=20, minWidth=12 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=30, minWidth=6 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="left", maxWidth=30, minWidth=6 )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      table.formatColumns( f1, f2 )

      from Ark import timestampToStr
      for vrfDropRoute in self.vrfDropRoutes:
         creationTime = timestampToStr( vrfDropRoute.insertTime, False )
         table.newRow( vrfDropRoute.prefix, vrfDropRoute.vrfName, creationTime )
      print table.output()

def doShowL3AleDropRoutes( mode, args ):
   vrfDropRoutes = []
   af = 'ipv4' if 'ip' in args else 'ipv6'
   argsVrfName = args.get( 'VRF' )
   for dropRoute in AleCliLib.dropRouteStatus.dropRoute.values():
      vrfName = AleCliLib.vrfIdMap.vrfIdToName[ dropRoute.key.vrfId ].vrfName
      if ( not argsVrfName ) or ( argsVrfName and ( argsVrfName == vrfName ) ):
         prefix = dropRoute.key.prefix
         if prefix.af != af:
            continue
         vrfDropRoutes.append(
               VrfDropRoute(
                  vrfName=vrfName,
                  insertTime=dropRoute.insertTime,
                  prefix=prefix ) )
   vrfDropRoutes = sorted(
         vrfDropRoutes, key=lambda x:( x[ 'vrfName' ], x[ 'prefix' ] ) )
   return DropRouteStatus( vrfDropRoutes=vrfDropRoutes )

dropMatcher = KeywordMatcher( "drop", "Show the routes pointing to drop fec" )

class ShowIpHardwareAleDropRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( ip | ipv6 ) hardware ale routes [ VRF ] drop'
   data = { 'ip': ipMatcherForShow,
            'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'drop': dropMatcher,
          }
   cliModel = DropRouteStatus
   privileged = True
   handler = doShowL3AleDropRoutes

BasicCli.addShowCommandClass( ShowIpHardwareAleDropRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale routes [vrf vrfname] [unprogrammed] [prefix]]
#------------------------------------------------------------------------------------

def doShowL3AleRoutesV4( mode, args ):
   cmd = 'AR4#'
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += '#'
   if 'unprogrammed' in args:
      cmd += 'u'
   cmd += '#'
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += '#'

   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode,
                                                 asyncCommand=True )
   return AleRoute

unprogrammedMatcher = KeywordMatcher( "unprogrammed", "Show unprogrammed routes" )

class ShowIpHardwareAleRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale routes [ VRF ] [ unprogrammed ] [ PREFIX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': v4RoutesMatcher,
            'VRF': vrfExprFactory,
            'unprogrammed': unprogrammedMatcher,
            'PREFIX': IpPrefixOrAddrMatcher,
          }
   cliModel = AleRoute
   privileged = True
   handler = doShowL3AleRoutesV4

BasicCli.addShowCommandClass( ShowIpHardwareAleRoutesCmd )

#------------------------------------------------------------------------------------
# "show ipv6 hardware ale routes [vrf vrfName] [unprogrammed] [prefix]"
#------------------------------------------------------------------------------------

def doShowL3AleRoutesV6( mode, args ):
   cmd = "AR6#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   if 'unprogrammed' in args:
      cmd += "u"
   cmd += "#"
   prefix6 = args.get( 'PREFIX' )
   if prefix6 is not None:
      cmd += "%s" % prefix6.stringValue
   cmd += "#"

   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode,
                                                 asyncCommand=True )
   return AleRoute

class ShowIp6HardwareAleRoutesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ipv6 hardware ale routes [ VRF ] [ unprogrammed ] [ PREFIX ]'
   data = { 'ipv6': ipv6MatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'routes': v6RoutesMatcher,
            'VRF': vrfExprFactory,
            'unprogrammed': unprogrammedMatcher,
            'PREFIX': Ip6PrefixOrAddrMatcher,
          }
   cliModel = AleRoute
   privileged = True
   handler = doShowL3AleRoutesV6

BasicCli.addShowCommandClass( ShowIp6HardwareAleRoutesCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale adj [vrf vrfName] [fecId]
#------------------------------------------------------------------------------------

def doShowL3AleAdj( mode, args ):
   cmd = 'AA3#'
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += '#'
   fecId = args.get( 'FECID' )
   if fecId:
      cmd += str( fecId )
   cmd += '#'
   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode )
   return AleAdj

adjMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "adj", helpdesc="Display ale adjacencies" ),
   guard=vrfRootSmGuard )

class ShowIpHardwareAleAdjCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale adj [ VRF ] [ FECID ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'adj': adjMatcher,
            'VRF': vrfExprFactory,
            'FECID': fecIdMatcher,
          }
   cliModel = AleAdj
   privileged = True
   handler = doShowL3AleAdj

BasicCli.addShowCommandClass( ShowIpHardwareAleAdjCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale adj proxy
#------------------------------------------------------------------------------------

def doShowL3AleAdjProxy( mode, args ):
   cmd = 'AAP#'
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += '#'
   fecId = args.get( 'FECID' )
   if fecId:
      cmd += str( fecId )
   cmd += '#'
   if 'proxy' in args:
      cmd += 'proxy'
   cmd += '#'
   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode )
   return FecProxyInfoList

class ShowIpHardwareAleAdjProxyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale adj proxy [ VRF ] [ FECID ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'adj': adjMatcher,
            'proxy': 'Display ale FEC aliases',
            'VRF': vrfExprFactory,
            'FECID': fecIdMatcher,
          }
   cliModel = FecProxyInfoList
   privileged = True
   handler = doShowL3AleAdjProxy

BasicCli.addShowCommandClass( ShowIpHardwareAleAdjProxyCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale l2-adj [vrf vrfName] [adjIndex]
#------------------------------------------------------------------------------------

def doShowL3AleL2Adj( mode, args ):

   cmd = "AA2#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   adjIndex = args.get( 'ADJ_INDEX' )
   if adjIndex:
      cmd += str( adjIndex )
   cmd += '#'

   AgentCommandRequest.runCliPrintSocketCommand( AleCliLib.entityManager, "ale",
                                                 "VrfRootSmCliCallbackWithFormat",
                                                 cmd, mode=mode,
                                                 asyncCommand=True )
   return AleL2Adj

l2AdjMatcher = CliCommand.Node(
   matcher=KeywordMatcher( "l2-adj", helpdesc="Display ale layer2 adjacencies" ),
   guard=vrfRootSmGuard )

l2AdjIndexMatcher = IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF, helpdesc='adj index' )

class ShowIpHardwareAleL2AdjCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale l2-adj [ VRF ] [ ADJ_INDEX ]'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'l2-adj': l2AdjMatcher,
            'VRF': vrfExprFactory,
            'ADJ_INDEX': l2AdjIndexMatcher,
          }
   cliModel = AleL2Adj
   privileged = True
   handler = doShowL3AleL2Adj

BasicCli.addShowCommandClass( ShowIpHardwareAleL2AdjCmd )

#------------------------------------------------------------------------------------
# "show ip hardware ale vrf
#------------------------------------------------------------------------------------

def doShowL3AleVrf( mode, args ):
   vrfStatus = {}
   for vs in AleCliLib.routingHardwareRouteStatus.vrfStatus.itervalues():
      tableId = 0
      if vs.vrfName in AleCliLib.vrfTableStatus.vrfNameToFibTableId:
         tableId = AleCliLib.vrfTableStatus.vrfNameToFibTableId[ vs.vrfName ]
      vrfStatus[ vs.vrfName ] = VrfStatus( vrfName=vs.vrfName, vrfId=vs.vrfId,
                                           tableId=tableId )
   return RouteStatus( vrfStatus=vrfStatus )

class VrfStatus( CliModel.Model ):
   vrfName = CliModel.Str( help="Name of a VRF" )
   vrfId = CliModel.Int( help="ID used on hardware for the VRF" )
   tableId = CliModel.Int( help="ID used in Ale for the FIB table" )

class RouteStatus( CliModel.Model ):
   vrfStatus = CliModel.Dict( keyType=str, valueType=VrfStatus,
                              help="List of all Vrf names and IDs" )

   def render( self ):
      if not self.vrfStatus:
         return
      table = TableOutput.createTable( ( "VRF Name", "VRF ID", "Table ID" ) )
      tableMaxWidth = table.printableWidth()
      if tableMaxWidth < 80:
         tableMaxWidth = 80

      f1 = TableOutput.Format( justify="left", maxWidth=64, minWidth=20 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=10, minWidth=6 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="left", maxWidth=10, minWidth=6 )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      table.formatColumns( f1, f2 )

      for _, vrfStatus in sorted( self.vrfStatus.items() ):
         table.newRow( vrfStatus.vrfName, vrfStatus.vrfId, vrfStatus.tableId )
      print table.output()

class ShowIpHardwareAleVrfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip hardware ale vrf'
   data = { 'ip': ipMatcherForShow,
            'hardware': hardwareMatcherForShow,
            'ale': aleKwForShow,
            'vrf': KeywordMatcher( "vrf", helpdesc="Displays the vrf table" ),
          }
   cliModel = RouteStatus
   handler = doShowL3AleVrf

BasicCli.addShowCommandClass( ShowIpHardwareAleVrfCmd )


#--------------------------------------------------------------------------
# register show tech-support to include Ale command outputs
#--------------------------------------------------------------------------
def _showTechAleCmds():
   cmds = [ 'show ip hardware ale vrf' ]
   if vrfRootSmGuard( mode=None, token=None ) is None:
      cmds += [ 'show ip hardware ale routes',
                'show ipv6 hardware ale routes',
                'show ip hardware ale adj',
                'show ip hardware ale l2-adj',
      ]
   return cmds

TechSupportCli.registerShowTechSupportCmdCallback( '2019-07-19 14:24:51',
                                                   _showTechAleCmds )
