# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliPlugin.IraIpCli as IraIpCli
from CliPlugin.VrfCli import vrfExists, VrfExprFactory, DEFAULT_VRF
import ShowCommand
import SmashLazyMount
import RouteCountersModel
import Tac
from AleCountersCli import counterFeatureSupported

featureIdEnum = Tac.Type( 'FlexCounters::FeatureId' )
featureIdEnumVal = Tac.Type( 'FlexCounters::FeatureIdEnumVal' )
allFapsId = Tac.Type( 'FlexCounters::FapId' ).allFapsId

featureSupportedGuard = counterFeatureSupported( 'Route' )

routeCounterTable = None
routeCounterSnapshotTable = None
routeCounterStatus = None
l3RouteConfig = None

matcherCounters = CliMatcher.KeywordMatcher( 'counters',
                                             helpdesc='Show packet and byte count' )
matcherRoute = CliMatcher.KeywordMatcher( 'route', helpdesc='Route information' )
nodeRoute = CliCommand.Node( matcher=matcherRoute,
                           guard=featureSupportedGuard )
def populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                              vrfNameAndPrefix, counterIndex ):
   routeCounterEntry.prefix = vrfNameAndPrefix.ipGenPrefix.stringValue
   if routeCounterTable is None:
      routeCounterEntry.packetCount = 0
      routeCounterEntry.byteCount = 0
   elif routeCounterSnapshotTable is None or \
         counterIndex not in routeCounterSnapshotTable.counter:
      routeCounterEntry.packetCount = routeCounterTable.counter[ counterIndex ].pkts
      routeCounterEntry.byteCount = routeCounterTable.counter[ counterIndex ].octets
   else:
      routeCounterEntry.packetCount = \
         routeCounterTable.counter[ counterIndex ].pkts - \
         routeCounterSnapshotTable.counter[ counterIndex ].pkts
      routeCounterEntry.byteCount = \
         routeCounterTable.counter[ counterIndex ].octets - \
         routeCounterSnapshotTable.counter[ counterIndex ].octets

   if vrfNameAndPrefix.vrfName not in vrfRouteCounters.vrfs:
      routeCounters = RouteCountersModel.RouteCounters()
      routeCounters.counters.append( routeCounterEntry )
      vrfRouteCounters.vrfs[ vrfNameAndPrefix.vrfName ] = routeCounters
   else:
      vrfRouteCounters.vrfs[ vrfNameAndPrefix.vrfName ].\
         counters.append( routeCounterEntry )

def showRouteCounters( mode, args ):
   if 'ipv4' in args:
      routeType = 'ipv4'
      prefix = args.get( 'PREFIX4' )
   else:
      routeType = 'ipv6'
      prefix = args.get( 'ADDR6' ) or args.get( 'PREFIX6' )

   vrfName = args.get( 'VRF', DEFAULT_VRF )
   vrfRouteCounters = RouteCountersModel.VrfRouteCounters()
   if not vrfExists( vrfName ):
      mode.addError( 'No such VRF %s' % vrfName )
      return None
   if prefix is not None:
      if isinstance( prefix, Tac.Type( 'Arnet::IpAddrWithMask' ) ):
         prefix = IpAddrMatcher.prefixFormat( prefix.address, prefix.len,
                     overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO )

      prefixObj = Tac.newInstance( 'Arnet::IpGenPrefix', str( prefix ) )
      vrfNameObj = Tac.newInstance( 'L3::VrfName', vrfName )
      vrfNameAndIpGenPrefix = Tac.Value(
         'Ale::FlexCounter::VrfNameAndIpGenPrefix', vrfNameObj, prefixObj )
      found = routeCounterStatus.hasCounterInfo( vrfNameAndIpGenPrefix )
      if not found:
         mode.addError( 'No counter attached to route %s in vrf %s'
                        % ( prefix, vrfName ) )
         return None
      else:
         counterInfo = routeCounterStatus.getCounterInfo( vrfNameAndIpGenPrefix )
         routeCounterEntry = RouteCountersModel.RouteCounterEntry()
         populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                                   vrfNameAndIpGenPrefix, counterInfo.counterIndex )
   else:
      for vKey, vVal in routeCounterStatus.vrfCounterInfo.iteritems():
         for pKey, pVal in vVal.prefixCounterInfo.iteritems():
            counterIndex = pVal.counterIndex
            routeCounterEntry = RouteCountersModel.RouteCounterEntry()
            vrfNameAndIpGenPrefix = Tac.Value(
               'Ale::FlexCounter::VrfNameAndIpGenPrefix', vKey, pKey )
            if routeType == pKey.af:
               if vrfName is None or vKey == vrfName:
                  populateVrfRouteCounters( vrfRouteCounters, routeCounterEntry,
                                            vrfNameAndIpGenPrefix, counterIndex )
   return RouteCountersModel.VrfRouteCounters( vrfs=vrfRouteCounters.vrfs )

class ShowRouteIpCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show route ( ( ipv4 [ VRF ] [ PREFIX4 ] ) | ' \
                         '( ipv6 [ VRF ] [ ADDR6 | PREFIX6 ] ) ) ' \
                         'counters'
   data = {
      'route' : nodeRoute,
      'ipv4' : 'Show IPv4 route counters',
      'ipv6' : 'Show IPv6 route counters',
      'VRF' : VrfExprFactory( helpdesc='Show route counters with given VRF name',
                              guard=IraIpCli.vrfRoutingSupportedGuard ),
      'PREFIX4' : IpAddrMatcher.ipAddrWithMaskExpr(
                  'Match this IP address', 'Match this subnet mask',
                  'Match this IP prefix',
                  overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'ADDR6' : Ip6AddrMatcher.ip6AddrMatcher,
      'PREFIX6' : Ip6AddrMatcher.ip6PrefixMatcher,
      'counters' : matcherCounters,
   }
   cliModel = RouteCountersModel.VrfRouteCounters
   handler = showRouteCounters

BasicCli.addShowCommandClass( ShowRouteIpCounters )

def Plugin( em ):
   global routeCounterTable, routeCounterSnapshotTable, routeCounterStatus
   global l3RouteConfig

   smir = SmashLazyMount.mountInfo( 'reader' )

   mountPath = 'flexCounter/counterTable/%u/%u' % (
         featureIdEnumVal.enumVal( featureIdEnum.Route ), allFapsId )
   routeCounterTable = SmashLazyMount.mount( em, mountPath,
         "FlexCounters::CounterTable", smir )

   mountPath = 'flexCounter/snapshotCounterTable/%u/%u' % (
         featureIdEnumVal.enumVal( featureIdEnum.Route ), allFapsId )
   routeCounterSnapshotTable = SmashLazyMount.mount( em, mountPath,
         "FlexCounters::CounterTable", smir )

   routeCounterStatus = LazyMount.mount( em,
                                 'flexCounter/stratal3/l3RouteFlexCounterStatus',
                                 'Ale::FlexCounter::L3RouteFlexCounterStatus', 'r' )

   mountPath = 'hardware/counter/l3route/config'
   l3RouteConfig = LazyMount.mount( em, mountPath,
                                    'Ale::FlexCounter::L3RouteConfig', 'r' )
