# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime

from AleFlexCounterTypes import featureNameAndDirection
import Ark
import Arnet
from CliModel import Int
from CliModel import Dict
from CliModel import Enum
from CliModel import Float
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
import IntfCli
from IntfModel import InterfaceCountersRateBase
from IntfModels import Interface
from TableOutput import TableFormatter, Headings, Format
import Tac

TimeConstants = Tac.Type( 'Ale::Counters::TimeConstants' )
isSubIntfId = Tac.Type( 'Arnet::SubIntfId' ).isSubIntfId

# pylint fails to follow the Model class structure, causing many false errors

def timestampToIsoformat( timestamp ):
   return datetime.fromtimestamp( int( timestamp ) ).isoformat( ' ' )

class ChipEvent( Model ):
   firstOccurrence = Float( help='Time of first occurrence' )
   lastOccurrence = Float( help='Time of last occurrence' )
   eventCount = Int( help='Number of times the event has occurred' )
   zScore = Float( help='Number of standard deviations away from mean the current '
                        'counter event value is' )

class CounterEvent( Model ):
   counterEventName = Str( help='Name of the counter causing the event' )
   fiveMinuteEvents = Dict( keyType=str, valueType=ChipEvent, help='Mapping between '
                                'chip name and the events over five minutes for a '
                                'chip' )
   tenMinuteEvents = Dict( keyType=str, valueType=ChipEvent, help='Mapping between '
                                'chip name and the events over ten minutes for a '
                                'chip' )
   oneHourEvents = Dict( keyType=str, valueType=ChipEvent, help='Mapping between '
                                'chip name and the events over one hour for a chip' )
   oneDayEvents = Dict( keyType=str, valueType=ChipEvent, help='Mapping between '
                                'chip name and the events over one day for a chip' )
   oneWeekEvents = Dict( keyType=str, valueType=ChipEvent, help='Mapping between '
                                'chip name and the events over one week for a chip' )

class FlowTrendEventModel( Model ):
   counterEvents = Dict( keyType=int, valueType=CounterEvent,
                          help='Mapping between counter Id and information for '
                          'events for a counter' )

   def render( self ):
      header = "-----------------------------------------------------------------" \
            "-------------------------------------------"
      print header
      # TODO Increase the width for event name
      print "  Interval  |   Event Name   | Chip Name |         First         |" \
            "         Last          |  Count | Z-Score "
      print "            |                |           |       Occurrence      |" \
            "      Occurrence       |        |         "
      print header
      fmtStr = ' %10s | %14s | %9s | %21s | %21s | %6s | %3.4f '
      for counter in sorted( self.counterEvents.keys() ):
         counterEvent = self.counterEvents.get( counter )
         intervalStrs = [ '5 Minutes', '10 Minutes', '1 Hour', '1 Day', '1 Week' ]
         intervalEvents = [ counterEvent.fiveMinuteEvents,
               counterEvent.tenMinuteEvents, counterEvent.oneHourEvents,
               counterEvent.oneDayEvents, counterEvent.oneWeekEvents ]
         for interval, intervalEvent in zip( intervalStrs, intervalEvents ):
            for chipName, chipEvent in intervalEvent.iteritems():
               print fmtStr % ( interval, counterEvent.counterEventName, chipName,
                     Ark.timestampToStr( chipEvent.firstOccurrence, False ),
                     Ark.timestampToStr( chipEvent.lastOccurrence, False ),
                     chipEvent.eventCount, chipEvent.zScore )
      print header

class DropCounterModel( Model ):
   _printFmt = Str( help="Type of print format" )

   class DropEvents( Model ):
      class DropEvent( Model ):
         counterId = Int( help='Counter ID' )
         counterName = Str( help='Counter name' )
         counterType = Str( help='Counter type' )
         dropCount = Int( help='Drop count' )

         eventCount = Int( help='Drop event count' )
         initialEventTime = Str( help='UTC timestamp of first occurrence' )
         lastEventTime = Str( help='UTC timestamp of most recent occurrence' )

         dropInLastMinute = Int( help='The number of drops that happend in '
                                 'the last minute', optional=True )
         dropInLastTenMinute = Int( help='The number of drops that happend in '
                                    'the last ten minutes', optional=True )
         dropInLastOneHour = Int( help='The number of drops that happend in '
                                  'the last one hour', optional=True )
         dropInLastOneDay = Int( help='The number of drops that happend in '
                                 'the last one day', optional=True )
         dropInLastOneWeek = Int( help='The number of drops that happend in '
                                  'the last one week', optional=True )

         thresholdEventCount = Int( optional=True,
                                    help='Drop count above configured threshold '
                                    'count' )
         initialThresholdEventTime = Str( optional=True,
                                          help='UTC timestamp of first occurrence '
                                          'above configured threshold' )
         lastThresholdEventTime = Str( optional=True,
                                       help='UTC timestamp of most recent '
                                       'occurrence above configured threshold' )

         lastSyslogTime = Str( optional=True,
                               help='UTC timestamp of most recent syslog' )

      dropEvent = List( valueType=DropEvent, help='List of drop events' )
   totalAdverseDrops = Int( help='Total count of adverse drop counters' )
   totalCongestionDrops = Int( help='Total count of congestion drop counters' )
   totalPacketProcessorDrops = Int( help='Total count of packet processor drop '
                                    'counters' )
   dropEvents = Dict( valueType=DropEvents,
                      help='Mapping from chipName to drop events' )

   def render( self ):
      if self._printFmt == 'rates':
         self._renderDropRates()
      else:
         self._renderDropEvents()

   def _renderDropEvents( self ):
      print 'Summary:'
      print 'Total Adverse (A) Drops:', self.totalAdverseDrops
      print 'Total Congestion (C) Drops:', self.totalCongestionDrops
      print 'Total Packet Processor (P) Drops:', self.totalPacketProcessorDrops

      formatStr = '%-4s  %-10s   %-30s : %15s : %-19s : %-19s '
      print formatStr % ( 'Type', 'Chip', 'CounterName', 'Count',
                          'First Occurrence', 'Last Occurrence' )
      print '-' * 110

      # flatten the dropEvents for sorting based on last event time stamp
      eventList = []
      for chipName in self.dropEvents.keys():
         for event in self.dropEvents[ chipName ].dropEvent:
            eventList.append( ( chipName, event ) )

      # sort by counterType, lastEventTime
      for chipName, event in sorted( eventList,
                                     key=lambda e: ( e[ 1 ].lastEventTime ),
                                     reverse=True ):
         print formatStr % ( event.counterType[ 0 ], chipName,
                             event.counterName,
                             event.dropCount,
                             event.initialEventTime,
                             event.lastEventTime )

   def _renderDropRates( self ):
      formatStr = '%-4s  %-10s  %-30s  %10s %10s %10s %10s %10s %10s'
      print formatStr % ( 'Type', 'Chip', 'CounterName', 'Count',
                          '1-Min', '10-Min', '1-Hour', '1-Day', '1-Week' )
      print '-' * 115

      # flatten the dropEvents for sorting based on last event time stamp
      eventList = []
      for chipName in self.dropEvents.keys():
         for event in self.dropEvents[ chipName ].dropEvent:
            eventList.append( ( chipName, event ) )

      def formatRate( rateValue ):
         return 'NA' if rateValue is None else rateValue

      # sort by counterType, lastEventTime
      for chipName, event in sorted( eventList,
                                     key=lambda e: ( e[ 1 ].lastEventTime ),
                                     reverse=True ):
         print formatStr % (
                  event.counterType[ 0 ], chipName,
                  event.counterName, event.dropCount,
                  formatRate( event.dropInLastMinute ),
                  formatRate( event.dropInLastTenMinute ),
                  formatRate( event.dropInLastOneHour ),
                  formatRate( event.dropInLastOneDay ),
                  formatRate( event.dropInLastOneWeek ) )

class QueueInfo( Model ):
   queueLength = Int( help="Current queue length in bytes" )

class SourceQueueInfo( Model ):
   sources = Dict( keyType=str, valueType=QueueInfo,
         help="Maps traffic source name to queue information" )

class QueueTypeQueueInfo( Model ):
   def renderQueue( self, intf, fmt ):
      # Must be implemented by derived models, and the derived implementation
      # should call renderSource().
      raise NotImplementedError

   def renderSource( self, intf, queueType, fmt ):
      for ( tc, sourceQueue ) in sorted( queueType.items() ):
         sourceQueueInfo = sourceQueue.sources
         for source in Arnet.sortIntf( sourceQueueInfo ):
            lengthStr = str( sourceQueueInfo[ source ].queueLength )
            print fmt % ( intf, tc, source, lengthStr )

class SandQueueTypeQueueInfo( QueueTypeQueueInfo ):
   voqs = Dict( keyType=str, valueType=SourceQueueInfo,
      help="Maps traffic class to queue information for virtual output queue type" )

   def renderQueue( self, intf, fmt ):
      for queueType in [ self.voqs ]:
         if queueType is None:
            continue
         self.renderSource( intf, queueType, fmt )

class InterfaceQueueLengthInfo( Model ):
   frontPanelPorts = Dict( keyType=Interface, valueType=QueueTypeQueueInfo,
         help="Maps front panel port name to queue information" )
   lagSubIntfs = Dict( keyType=Interface, valueType=QueueTypeQueueInfo,
         help="Maps LAG sub interface name to queue information" )
   cpuPorts = Dict( keyType=Interface, valueType=QueueTypeQueueInfo,
         help="Maps CPU port name to queue information" )
   _cpuPortsOrderedList = List( valueType=str,
         help="Maps internal port ID to CPU port name" )
   fabricPorts = Dict( keyType=Interface, valueType=QueueTypeQueueInfo,
         help="Maps fabric port name to queue information" )
   _fabricPortsOrderedList = List( valueType=str,
         help="Maps internal port ID to fabric port name" )
   recyclePorts = Dict( keyType=Interface, valueType=QueueTypeQueueInfo,
         help="Maps recycle port name to queue information" )
   _recyclePortsOrderedList = List( valueType=str,
         help="Maps internal port ID to recycle port name" )

   def getCpuPortsOrderedList( self ):
      return self._cpuPortsOrderedList

   def setCpuPortsOrderedList( self, value ):
      self._cpuPortsOrderedList = value

   def headers( self ):
      return ( "Egress interface", "Queue",
               "Source", "Queue length (bytes)" )

   def render( self ):
      fmtWidths = ( 30, 6, 15, 20 )
      fmt = "%%-%ds %%-%ds %%-%ds %%%ds" % fmtWidths

      def printDropThresholdHeader():
         print fmt % self.headers()
         dashes = [ '-' * x for x in fmtWidths ]
         print fmt % tuple( dashes )
      printDropThresholdHeader()
      for intf in Arnet.sortIntf( self.frontPanelPorts ):
         intfQueue = self.frontPanelPorts[ intf ]
         intfQueue.renderQueue( intf, fmt )
      for intf in Arnet.sortIntf( self.lagSubIntfs ):
         intfQueue = self.lagSubIntfs[ intf ]
         intfQueue.renderQueue( intf, fmt )
      for intf in self._cpuPortsOrderedList:
         intfQueue = self.cpuPorts[ intf ]
         intfQueue.renderQueue( intf, fmt )
      for intf in self._fabricPortsOrderedList:
         intfQueue = self.fabricPorts[ intf ]
         intfQueue.renderQueue( intf, fmt )
      for intf in self._recyclePortsOrderedList:
         intfQueue = self.recyclePorts[ intf ]
         intfQueue.renderQueue( intf, fmt )

class InterfaceIpv4Ipv6CountersRates( Model ):

   class InterfaceIpv4Ipv6CounterRates( InterfaceCountersRateBase ):
      interval = Float( help="Interval in seconds" )

      ipv4InOctetsRate = Int( help="Number of input IPv4 octets per second",
                              optional=True )
      ipv4InPktsRate = Int( help="Number of input IPv4 packets per second",
                            optional=True )
      ipv6InOctetsRate = Int( help="Number of input IPv6 octets per second",
                              optional=True )
      ipv6InPktsRate = Int( help="Number of input IPv6 packets per second",
                            optional=True )
      ipv4OutOctetsRate = Int( help="Number of output IPv4 octets per second",
                               optional=True )
      ipv4OutPktsRate = Int( help="Number of output IPv4 packets per second",
                             optional=True )
      ipv6OutOctetsRate = Int( help="Number of ouput IPv6 octets per second",
                               optional=True )
      ipv6OutPktsRate = Int( help="Number of ouput IPv6 packets per second",
                             optional=True )
      lastUpdateTimestamp = Float( help="Time of last update", optional=True )

   interfaces = Dict( keyType=Interface, valueType=InterfaceIpv4Ipv6CounterRates,
                      help="Mapping between an interface and its rate counters" )

   def renderRates( self, intf, interval, v4InOctetsRate, v4InPktsRate,
                    v6InOctetsRate, v6InPktsRate ):

      def formatRate( counterRate ):
         return 'n/a' if counterRate is None else counterRate

      print "{:10} {:5} {:>20} {:>18} {:>20} {:>18}".format( intf,
         formatRate( interval ),
         formatRate( v4InOctetsRate ),
         formatRate( v4InPktsRate ),
         formatRate( v6InOctetsRate ),
         formatRate( v6InPktsRate ) )

   def render( self ):
      if not self.interfaces:
         return
      self.renderRates(  'Interface', 'Intvl', 'IPv4InOctets/sec', 'IPv4InPkts/sec',
                         'IPv6InOctets/sec', 'IPv6InPkts/sec' )
      intfs = Arnet.sortIntf( self.interfaces )
      for intf in intfs:
         counterRate = self.interfaces[ intf ]
         self.renderRates( IntfCli.Intf.getShortname( intf ), counterRate.interval,
                           counterRate.ipv4InOctetsRate, counterRate.ipv4InPktsRate,
                           counterRate.ipv6InOctetsRate, counterRate.ipv6InPktsRate )

      print
      self.renderRates( 'Interface', 'Intvl', 'IPv4OutOctets/sec', 'IPv4OutPkts/sec',
                        'IPv6OutOctets/sec', 'IPv6OutPkts/sec' )
      for intf in intfs:
         counterRate = self.interfaces[ intf ]
         self.renderRates( IntfCli.Intf.getShortname( intf ), counterRate.interval,
                           counterRate.ipv4OutOctetsRate,
                           counterRate.ipv4OutPktsRate,
                           counterRate.ipv6OutOctetsRate,
                           counterRate.ipv6OutPktsRate )

class InterfacesIpv4Ipv6Counters( Model ):
   __revision__ = 2
   # Per interface ingress/egress ipv4, ipv6 counters
   class InterfaceIpv4Ipv6Counters( Model ):
      ipv4InOctets = Int( help="Number of input IPv4 octets", optional=True )
      ipv4InPkts = Int( help="Number of input IPv4 packets", optional=True )
      ipv6InOctets = Int( help="Number of input IPv6 octets", optional=True )
      ipv6InPkts = Int( help="Number of input IPv6 packets", optional=True )
      ipv4OutOctets = Int( help="Number of output IPv4 octets", optional=True )
      ipv4OutPkts = Int( help="Number of output IPv4 packets", optional=True )
      ipv6OutOctets = Int( help="Number of ouput IPv6 octets", optional=True )
      ipv6OutPkts = Int( help="Number of ouput IPv6 packets", optional=True )

   interfaces = Dict( keyType=Interface, valueType=InterfaceIpv4Ipv6Counters,
                      help="Mapping of an interface to its counters" )

   def printRow( self, intf, v4Octets, v4Pkts, v6Octets, v6Pkts,
                 printEmptyEntries=True ):
      def formatCounter( counter ):
         return 'n/a' if counter is None else counter

      if ( not printEmptyEntries and
           v4Octets is None and
           v4Pkts is None and
           v6Octets is None and
           v6Pkts is None ):
         return

      print "{:10} {:>18} {:>15} {:>18} {:>15}".format( intf,
            formatCounter( v4Octets ),
            formatCounter( v4Pkts ),
            formatCounter( v6Octets ),
            formatCounter( v6Pkts ) )

   def separateL3Interfaces( self, interfaces ):
      l2Intfs = [ intf for intf in interfaces if not isSubIntfId( intf ) ]
      l3Intfs = [ intf for intf in interfaces if isSubIntfId( intf ) ]
      return Arnet.sortIntf( l2Intfs ), Arnet.sortIntf( l3Intfs )

   def render( self ):
      if not self.interfaces:
         return
      l2Interfaces, l3Interfaces = self.separateL3Interfaces( self.interfaces )
      # Filter out non-'Ethernet' L2 interfaces.
      l2Interfaces = [ i for i in l2Interfaces if i.startswith( 'Ethernet' ) ]
      self.printRow( "Interface", "IPv4InOctets", "IPv4InPkts", "IPv6InOctets",
                     "IPv6InPkts" )
      for intf in l2Interfaces:
         counter = self.interfaces[ intf ]
         self.printRow( IntfCli.Intf.getShortname( intf ), counter.ipv4InOctets,
                        counter.ipv4InPkts, counter.ipv6InOctets,
                        counter.ipv6InPkts )

      print
      self.printRow( "Interface", "IPv4OutOctets", "IPv4OutPkts", "IPv6OutOctets",
                     "IPv6OutPkts" )
      for intf in l2Interfaces:
         counter = self.interfaces[ intf ]
         self.printRow( IntfCli.Intf.getShortname( intf ), counter.ipv4OutOctets,
                        counter.ipv4OutPkts, counter.ipv6OutOctets,
                        counter.ipv6OutPkts )

      if not l3Interfaces:
         return
      print
      self.printRow( "L3 Interface", "IPv4InOctets", "IPv4InPkts", "IPv6InOctets",
                     "IPv6InPkts" )
      for intf in l3Interfaces:
         counter = self.interfaces[ intf ]
         self.printRow( IntfCli.Intf.getShortname( intf ), counter.ipv4InOctets,
                        counter.ipv4InPkts, counter.ipv6InOctets,
                        counter.ipv6InPkts, printEmptyEntries=False )

      print
      self.printRow( "L3 Interface", "IPv4OutOctets", "IPv4OutPkts", "IPv6OutOctets",
                     "IPv6OutPkts" )
      for intf in l3Interfaces:
         counter = self.interfaces[ intf ]
         self.printRow( IntfCli.Intf.getShortname( intf ), counter.ipv4OutOctets,
                        counter.ipv4OutPkts, counter.ipv6OutOctets,
                        counter.ipv6OutPkts, printEmptyEntries=False )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         for intf in dictRepr[ 'interfaces' ].itervalues():
            intf[ 'ipv4InOctets' ] = intf.get( 'ipv4InOctets', 0 )
            intf[ 'ipv4InPkts' ] = intf.get( 'ipv4InPkts', 0 )
            intf[ 'ipv6InOctets' ] = intf.get( 'ipv6InOctets', 0 )
            intf[ 'ipv6InPkts' ] = intf.get( 'ipv6InPkts', 0 )
            intf[ 'ipv4OutOctets' ] = intf.get( 'ipv4OutOctets', 0 )
            intf[ 'ipv4OutPkts' ] = intf.get( 'ipv4OutPkts', 0 )
            intf[ 'ipv6OutOctets' ] = intf.get( 'ipv6OutOctets', 0 )
            intf[ 'ipv6OutPkts' ] = intf.get( 'ipv6OutPkts', 0 )
      return dictRepr

# --------------------------------------------------------------------
#  show hardware counter feature
# --------------------------------------------------------------------

# Maps to allow featureId and model attribute name conversion.
featureIdToModelAttrMap = {
   "IpIngress" : "routedPortIngress",
   "Mcast" : "multicastIpv4",
   "Route" : "routeIpv4",
}
modelAttrToFeatureIdMap = { v : k for k, v in featureIdToModelAttrMap.iteritems() }

class CounterFeatureResource( Model ):
   resourceIds = List( valueType=int, help="Counter resource list" )

class CounterFeatureInfo( Model ):
   status = Enum( values=( "unknown", "up", "down", "degraded" ), optional=True,
                  help="Functional status of the counter feature" )
   statusDetail = Str( optional=True,
                       help="Detailed description of the counter feature status" )
   resourceTypes = Dict( keyType=str, valueType=CounterFeatureResource,
                     optional=True,
                     help="A mapping between a resource type and counter resources" )

class CounterFeatureModel( Model ):
   aclIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress ACL counters" )
   aclEgressIpv4 = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress IPv4 ACL counters" )
   aclEgressIpv6 = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress IPv6 ACL counters" )
   aclEgressMac = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress MAC ACL counters" )
   cpuTrafficPolicyIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for CPU traffic policy counters" )
   decapGroup = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for decap group counters" )
   directFlow = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for DirectFlow counters" )
   ecn = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress ECN counters" )
   flowspec = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for Flowspec counters" )
   greTunnelInterfaceIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress GRE tunnel interface "
           "counters" )
   greTunnelInterfaceEgress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress GRE tunnel interface "
           "counters" )
   intfTrafficPolicyIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for interface traffic policy counters" )
   ipv4v6Ingress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress IP counters" )
   ipv4v6Egress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress IP counters" )
   l2SubInterfaceEgress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress layer 2 subinterface "
           "counters" )
   l2SubInterfaceIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress layer 2 subinterface "
           "counters" )
   l3Ipv4v6Egress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for layer 3 egress IPv4, IPv6 counters" )
   l3Ipv4v6Ingress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for layer 3 ingress IPv4, IPv6 "
           "counters" )
   maxQueueSize = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for LANZ counters" )
   mplsInterfaceIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for MPLS interface ingress counters" )
   mplsLfib = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for MPLS LFIB counters" )
   mplsTunnel = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for MPLS tunnel counters" )
   multicastIpv4 = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for IPv4 multicast counters" )
   multicastIpv6 = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for IPv6 multicast counters" )
   nexthop = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for nexthop group counters" )
   pbrIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress PBR counters" )
   pdp = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for PDP counters" )
   qosIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress QoS counters" )
   qosMeterIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress QoS meters" )
   qosTrTcmIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress QoS trTCM counters" )
   queueEgress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress queue counters" )
   realTimeTrafficBasedWfq = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for real-time traffic-based "
           "weighted fair queueing counters" )
   routedPortIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress routed port counters" )
   routeIpv4 = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for IPv4 route counters" )
   routeIpv6 = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for IPv6 route counters" )
   sampledFlowTracking = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for sampled flow tracking counters" )
   sflowAccel = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for sFlow accel counters" )
   stormControlMeterIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress storm control metering" )
   subInterfaceEgress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress layer 3 subinterface "
           "counters" )
   subInterfaceIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress layer 3 subinterface "
           "counters" )
   sviEgress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress SVI counters" )
   sviIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress SVI counters" )
   tapAggIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress Tap Aggregation counters" )
   vlanEgress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for egress VLAN counters" )
   vlanIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress VLAN counters" )
   vlanTcIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress traffic-class counters" )
   vniDecap = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for VNI decapsulation counters" )
   vniEncap = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for VNI encapsulation counters" )
   voqIngress = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for ingress VOQ counters" )
   vtepDecap = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for VTEP decapsulation counters" )
   vtepEncap = Submodel( valueType=CounterFeatureInfo, optional=True,
      help="Resource and status information for VTEP encapsulation counters" )

   _counterResourceTitle = Enum( values=( "engine", "pool" ),
                                 help="Type of counter resource" )

   def counterFeatureInfo( self, featureId ):
      """Returns class feature info entity based on featureId."""

      # Get the correct model attribute name (first letter is lowercase).
      attr = featureIdToModelAttrMap.get( featureId,
                                          featureId[ 0 ].lower() + featureId[ 1 : ] )

      # Point to the counter feature info entity.
      counterFeatureInfo = getattr( self, attr )

      return ( attr, counterFeatureInfo )

   def counterFeatureInfoIs( self, featureId, resourceType ):
      """Selects (or allocates if needed) class feature entity based on featureId and
      resourceType."""

      # Point to the counter feature info entity.
      ( attr, counterFeatureInfo ) = self.counterFeatureInfo( featureId )

      if counterFeatureInfo is None:
         # Allocate one if this has not been done yet.
         counterFeatureInfo = CounterFeatureInfo()

      # If there is no resource structure, allocate one.
      if resourceType not in counterFeatureInfo.resourceTypes:
         counterFeatureInfo.resourceTypes[ resourceType ] = CounterFeatureResource()
      # Store the info structure into the matching attribute in the model.
      setattr( self, attr, counterFeatureInfo )
      return counterFeatureInfo

   def renderFeatureInfo( self, counterFeatureInfo, attr, table ):
      # Get the featureId (first letter is uppercase).
      featureId = modelAttrToFeatureIdMap.get( attr,
                                               attr[ 0 ].upper() + attr[ 1 : ] )

      # Get the parts, and format the output for the resources.
      featureName, direction = featureNameAndDirection( featureId )
      resourceStr = ""
      for resourceType, resource in counterFeatureInfo.resourceTypes.iteritems():
         resourceIds = ", ".join( str( k ) for k in sorted( resource.resourceIds ) )
         resourceStr += resourceType.capitalize() + ': ' + resourceIds + '\n'

      # Set up the table row.
      table.newRow( featureName, direction, resourceStr.strip(),
                    counterFeatureInfo.status, counterFeatureInfo.statusDetail )

   def render( self ):
      noFeaturesEnabled = True
      t = TableFormatter()
      f1 = Format( justify="left", wrap=False, minWidth=1, padding=1 )
      f1.noPadLeftIs( True )
      counterResourceStr = "Counter Resource (" + \
                           self._counterResourceTitle.capitalize() + ")"
      headings = ( "Feature", "Direction", counterResourceStr, "Status", "Detail" )
      th = Headings( ( ( h, "l" ) for h in headings ) )

      th.doApplyHeaders( t )

      for attr in sorted( self.__dict__ ):
         if not attr.startswith( '_' ):
            counterFeatureInfo = getattr( self, attr )
            if counterFeatureInfo:
               self.renderFeatureInfo( counterFeatureInfo, attr, t )
               noFeaturesEnabled = False

      t.formatColumns( *[ f1 for _ in headings ] )

      if noFeaturesEnabled:
         print 'No counter features configured.'
      else:
         print t.output()

   def counterResourceTitleIs( self, title ):
      self._counterResourceTitle = title

class PollInfo( Model ):
   pollInterval = Float( help='Configured polling period in seconds' )
   lastPollTimestamp = Float( help='Timestamp of at the end of last poll interval' )
   totalPollCount = Int( help='Numer of times the polling has completed' )
   latePollCount = Int( help='Number of times poll interval has been missed' )

   totalFetchCount = Int( help='Number of times the event has occurred' )
   averageFetchTime = Float( help='Average number of seconds to collect counters' )
   maximumFetchTime = Float( help='Maximum number of seconds to collect counters' )

   reEnqueueCount = Int( help='Number of times update requests saturated Schan' )
   retryDequeueCount = Int( help='Number of times waited for request completion' )

   def render( self ):
      print 'Poll period (in seconds):', self.pollInterval
      print 'Last poll time (in seconds):', self.lastPollTimestamp
      print 'Total poll count:', self.totalPollCount
      print 'Late poll count:', self.latePollCount
      print 'Average fetch time (in seconds):', self.averageFetchTime
      print 'Maximum fetch time (in seconds):', self.maximumFetchTime
      print 'Total fetch count:', self.totalFetchCount
      print 'Re-enqueue count:', self.reEnqueueCount
      print 'Retry dequeue count:', self.retryDequeueCount

class LatePollEvent( Model ):
   timestamp = Float( help='Timestamp of the late poll' )
   delay = Float( help='Polling delay (in milliseconds)' )

   def render( self ):
      print "%s: %.1f ms delay" % (
         timestampToIsoformat( self.timestamp ), self.delay )

class FastPollStats( Model ):
   statsStartTimestamp = Float( help='Start of the monitored period' )

   totalPollCount = Int( help='Numer of times the polling has completed' )
   latePollCount = Int( help='Number of times poll interval has been missed' )
   averageFetchTime = Float(
      help='Average time needed to collect counters (ms)' )
   maximumFetchTime = Float(
      help='Maximum time needed to collect counters (ms)' )

   def render( self ):
      print 'Start of the monitored period:', timestampToIsoformat(
            self.statsStartTimestamp )
      print 'Total poll count:', self.totalPollCount
      print 'Late poll count:', self.latePollCount
      print 'Average fetch time (in milliseconds): %.1f' % self.averageFetchTime
      print 'Maximum fetch time (in milliseconds): %.1f' % self.maximumFetchTime

class FastPollInfoDetail( Model ):
   latePollHistory = List( valueType=LatePollEvent, help='Last late poll events' )
   statsLastHour = Submodel(
         valueType=FastPollStats,
         help='Statistics over the last hour',
         optional=True )
   statsLastDay = Submodel(
         valueType=FastPollStats,
         help='Statistics over the last day',
         optional=True )

   def render( self ):
      if self.statsLastHour:
         print "\nStatistics over the last hour:"
         self.statsLastHour.render()

      if self.statsLastDay:
         print "\nStatistics over the last day:"
         self.statsLastDay.render()

      print "\nLast late poll events:"
      for latePollEvent in self.latePollHistory:
         latePollEvent.render()

class FastPollInfo( Model ):
   pollFastInterval = Int( help='Configured fast polling period in milliseconds' )
   lastPollTimestamp = Float( help='Timestamp of at the end of last poll interval' )
   totalPollCount = Int( help='Numer of times the polling has completed' )
   latePollCount = Int( help='Number of times poll interval has been missed' )

   totalFetchCount = Int( help='Number of times the event has occurred' )
   averageFetchTime = Float(
      help='Average number of milliseconds to collect counters' )
   maximumFetchTime = Float(
      help='Maximum number of milliseconds to collect counters' )

   reEnqueueCount = Int( help='Number of times update requests saturated Schan' )
   retryDequeueCount = Int( help='Number of times waited for request completion' )

   detail = Submodel( valueType=FastPollInfoDetail,
                      help='Late poll details',
                      optional=True )

   def render( self ):
      print 'Fast poll period (in milliseconds):', self.pollFastInterval
      print 'Last poll time (in seconds):', self.lastPollTimestamp
      print 'Total poll count:', self.totalPollCount
      print 'Late poll count:', self.latePollCount
      print 'Average fetch time (in milliseconds):', self.averageFetchTime
      print 'Maximum fetch time (in milliseconds):', self.maximumFetchTime
      print 'Total fetch count:', self.totalFetchCount
      print 'Re-enqueue count:', self.reEnqueueCount
      print 'Retry dequeue count:', self.retryDequeueCount
      if self.detail:
         self.detail.render()
