# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliToken.Hardware
import ConfigMount
import ShowCommand
import Tac
import AleCapacityModel
import TechSupportCli

capacityConfig = None
disableAlertThreshold = 0

def getThreshold( mode ):
   tableThresholds = {}
   statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

   for rowName in capacityConfig.threshold:
      tableThreshold = AleCapacityModel.TableThreshold()
      tableThreshold.configThreshold = capacityConfig.threshold[ rowName ]
      tableThresholds[ rowName ] = tableThreshold

   for agentName in statusDir:
      agentDir = mode.sysdbRoot.entity[ "hardware/capacity/status/%s" % (
         agentName ) ]

      for tableKey in agentDir.entry:
         if tableKey.feature != '':
            rowName = '-'.join( [ tableKey.table, tableKey.feature ] )
         else:
            rowName = tableKey.table

         if rowName not in tableThresholds:
            # If the entry doesn't exist in capacityConfig, it is set to default
            tableThreshold = AleCapacityModel.TableThreshold()
            tableThreshold.configThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold
            tableThreshold.defaultThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold
            tableThresholds[ rowName ] = tableThreshold
         else:
            # If the entry exists in capacityConfig, update the default
            tableThresholds[ rowName ].defaultThreshold = \
                  agentDir.entry[ tableKey ].defaultThreshold

   return tableThresholds

#------------------------------------------------------------------------------------
# show hardware capacity [ utilization percent exceed ( <percentVal> | threshold ) ]
#------------------------------------------------------------------------------------
class ShowCapacityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = """show hardware capacity [ utilization percent exceed
               ( <percentVal> | threshold ) ]"""
   data = {
      'hardware': CliToken.Hardware.hardwareForShowMatcher,
      'capacity': 'Capacity and usage of hardware resources',
      'utilization': 'Filter based on current utilization',
      'percent': 'Use percentage values as unit for filtering',
      'exceed': 'Values that are greater than or equal to',
      '<percentVal>': CliMatcher.IntegerMatcher( 0, 100,
                                helpdesc='Utilization value in percentage <0-100>' ),
      'threshold': 'Configured or default threshold'
   }
   cliModel = AleCapacityModel.CapacityUsage

   @staticmethod
   def handler( mode, args ):
      tableEntries = []
      statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

      if args.get( 'threshold' ) == 'threshold':
         tableThresholds = getThreshold( mode )

      for agentName in statusDir:
         agentDir = mode.sysdbRoot.entity[ "hardware/capacity/status/%s" % (
            agentName ) ]

         for tableKey in agentDir.entry:
            if args.get( '<percentVal>' ) is not None:
               if ( agentDir.entry[ tableKey ].usedPercent() <
                    args.get( '<percentVal>' ) ):
                  continue

            if args.get( 'threshold' ) == 'threshold':
               if tableKey.feature != '':
                  rowName = '-'.join( [ tableKey.table, tableKey.feature ] )
               else:
                  rowName = tableKey.table
               # Skip the row if threshold is set to 0 ( basically disabled )
               if ( ( tableThresholds[ rowName ].configThreshold ==
                      disableAlertThreshold ) or
                    ( agentDir.entry[ tableKey ].usedPercent() <
                      tableThresholds[ rowName ].configThreshold ) ):
                  continue

            tableEntry = AleCapacityModel.TableEntry()
            tableEntry.table = agentDir.entry[ tableKey ].tableKey.table
            tableEntry.feature = agentDir.entry[ tableKey ].tableKey.feature
            tableEntry.chip = agentDir.entry[ tableKey ].tableKey.chip
            tableEntry.used = agentDir.entry[ tableKey ].used
            tableEntry.free = agentDir.entry[ tableKey ].free
            tableEntry.usedPercent = agentDir.entry[ tableKey ].usedPercent()
            tableEntry.committed = agentDir.entry[ tableKey ].committed
            tableEntry.maxLimit = agentDir.entry[ tableKey ].maxLimit
            tableEntry.highWatermark = agentDir.entry[ tableKey ].highWatermark
            tableEntries.append( tableEntry )

      return AleCapacityModel.CapacityUsage( tables=tableEntries )

BasicCli.addShowCommandClass( ShowCapacityCmd )

#------------------------------------------------------------------------------------
# show hardware capacity alert threshold
#------------------------------------------------------------------------------------
class ShowThresholdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show hardware capacity alert threshold'
   data = {
      'hardware': CliToken.Hardware.hardwareForShowMatcher,
      'capacity': 'Capacity and usage of hardware resources',
      'alert': 'Configured alerts for hardware resource utilization',
      'threshold': 'Display all thresholds for generating alerts'
   }
   cliModel = AleCapacityModel.CapacityThreshold

   @staticmethod
   def handler( mode, args ):
      tableThresholds = getThreshold( mode )
      return AleCapacityModel.CapacityThreshold( thresholds=tableThresholds )

BasicCli.addShowCommandClass( ShowThresholdCmd )

#------------------------------------------------------------------------------------
# [ no | default ] hardware capacity alert table <table-name> [ feature
#                  <feature-name> ] threshold THRESHOLD
#------------------------------------------------------------------------------------
def getTableNames( mode ):
   tableList = {}
   statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

   for agentName in statusDir:
      agentDir = mode.sysdbRoot.entity[
         "hardware/capacity/status/%s" % ( agentName ) ]
      if agentDir.tableList is None:
         continue
      for tableName in agentDir.tableList.featureMapping:
         tableList[ tableName ] = "Set threshold for %s table" % ( tableName )

   return tableList

def getTableFeatureNames( mode, context ):
   featureList = {}
   tableName = context.sharedResult[ 'TABLE_NAME' ]
   statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

   for agentName in statusDir:
      # If two agentDir has the same tableName and featureName ( basically the same
      # tableKey ), we would try to go ahead and display duplicate entries. It is
      # upto the developers populating the agentDir to keep it exclusive
      agentDir = mode.sysdbRoot.entity[
         "hardware/capacity/status/%s" % ( agentName ) ]
      if agentDir.tableList is None:
         continue
      if tableName not in agentDir.tableList.featureMapping:
         continue
      for featureName in agentDir.tableList.featureMapping[ tableName ].feature:
         if featureName != "":
            featureList[ featureName ] = "Set threshold for %s feature" % (
                  featureName )

   return featureList

class CfgCapacityThreshCmd( CliCommand.CliCommandClass ):
   syntax = ( "hardware capacity alert table TABLE_NAME "
              "[ feature ( FEATURE_NAME | all ) ] "
              "threshold THRESHOLD" )

   noOrDefaultSyntax = ( "hardware capacity alert table TABLE_NAME "
                         "[ feature ( FEATURE_NAME | all ) ] "
                         "[ threshold ... ]" )

   data = {
      'hardware': CliToken.Hardware.hardwareForConfigMatcher,
      'capacity': 'Capacity and usage of hardware resources',
      'alert': 'Trigger syslog for hardware resource utilization',
      'table': 'Specify the table to monitor utilization',
      'TABLE_NAME': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( getTableNames,
                                           alwaysMatchInStartupConfig=True ),
         storeSharedResult=True ), # stores the result for featureName matcher
      'feature': 'Specify the feature to monitor utilization',
      'FEATURE_NAME': CliMatcher.DynamicKeywordMatcher(
         getTableFeatureNames,
         alwaysMatchInStartupConfig=True,
         passContext=True ),
      'all': 'Monitor utilization for all features',
      'threshold': 'Set threshold to trigger alert based on percentage utilization',
      'THRESHOLD': CliMatcher.IntegerMatcher(
         0, 100, helpdesc='Threshold value in percentage <0-100>' )
   }

   @staticmethod
   def handler( mode, args ):
      tableName = args.get( 'TABLE_NAME' )

      if args.get( 'all' ) == 'all':
         # Set the threshold for all the features in the table
         statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]

         for agentName in statusDir:
            agentDir = mode.sysdbRoot.entity[
               "hardware/capacity/status/%s" % ( agentName ) ]
            if agentDir.tableList is None:
               continue
            featureMapping = agentDir.tableList.featureMapping

            if tableName in featureMapping:
               for featureName in featureMapping[ tableName ].feature:
                  if featureName != '':
                     rowName = '-'.join( [ tableName, featureName ] )
                  else:
                     rowName = tableName
                  capacityConfig.threshold[ rowName ] = args.get( 'THRESHOLD' )

      else:
         if args.get( 'FEATURE_NAME' ):
            tableName = '-'.join( [ tableName, args.get( 'FEATURE_NAME' ) ] )
         else:
            statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]
            emptyFeature = False

            for agentName in statusDir:
               agentDir = mode.sysdbRoot.entity[
                  "hardware/capacity/status/%s" % ( agentName ) ]
               if agentDir.tableList is None:
                  continue
               featureMapping = agentDir.tableList.featureMapping
               if tableName in featureMapping:
                  if '' in featureMapping[ tableName ].feature:
                     emptyFeature = True
                     break

            if not emptyFeature and not mode.session_.startupConfig():
               mode.addError( "Enter a feature name" )
               return

         capacityConfig.threshold[ tableName ] = args.get( 'THRESHOLD' )

      return

   @staticmethod
   def _noOrDefaultThreshold( args, rowName ):
      if CliCommand.isDefaultCmd( args ):
         # If set to default, delete the entry from the config
         del capacityConfig.threshold[ rowName ]
      else:
         assert CliCommand.isNoCmd( args )
         # If no, set threshold to 0
         capacityConfig.threshold[ rowName ] = 0

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tableName = args.get( 'TABLE_NAME' )

      if 'all' in args:
         # Set the default threshold for all the features in the table
         statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]
         for agentName in statusDir:
            agentDir = mode.sysdbRoot.entity[
               "hardware/capacity/status/%s" % ( agentName ) ]
            if agentDir.tableList is None:
               continue
            featureMapping = agentDir.tableList.featureMapping

            if tableName in featureMapping:
               for featureName in featureMapping[ tableName ].feature:
                  if featureName != '':
                     rowName = '-'.join( [ tableName, featureName ] )
                  else:
                     rowName = tableName
                  CfgCapacityThreshCmd._noOrDefaultThreshold( args, rowName )
      else:
         featureName = args.get( 'FEATURE_NAME' )
         if featureName:
            tableName = '-'.join( [ tableName, featureName ] )
         else:
            statusDir = mode.sysdbRoot.entity[ "hardware/capacity/status" ]
            emptyFeature = False

            for agentName in statusDir:
               agentDir = mode.sysdbRoot.entity[
                  "hardware/capacity/status/%s" % ( agentName ) ]
               if agentDir.tableList is None:
                  continue
               featureMapping = agentDir.tableList.featureMapping
               if tableName in featureMapping:
                  if '' in featureMapping[ tableName ].feature:
                     emptyFeature = True
                     break

            if not emptyFeature and not mode.session_.startupConfig():
               mode.addError( "Enter a feature name" )
               return

         CfgCapacityThreshCmd._noOrDefaultThreshold( args, tableName )

      return

BasicCli.GlobalConfigMode.addCommandClass( CfgCapacityThreshCmd )

#------------------------------------------------------------------------------------
# show tech-support hooks
#------------------------------------------------------------------------------------

# Timestamp is added to maintain historical order whithin show tech-support
TechSupportCli.registerShowTechSupportCmdCallback(
      "2016-12-19 08:45:00",
      lambda: [ "show hardware capacity" ],
      summaryCmdCallback=lambda: [ "show hardware capacity" ] )

#------------------------------------------------------------------------------------
# plugin definition
#------------------------------------------------------------------------------------

def Plugin( entityManager ):
   global capacityConfig

   capacityConfig = ConfigMount.mount( entityManager,
                                       "hardware/capacity/config",
                                       "AleCapacity::TableThreshold", "w" )

   mg = entityManager.mountGroup()
   mg.mount( "hardware/capacity/status", "Tac::Dir", "ri" )

   def _finish():
      pass

   mg.close( _finish )

