# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#------------------------------------------------------
# This module implements the following configuration.
#
# [ no | default ] agent <agentName> shutdown 
# [ no | default ] agent <agentName-sliceId> shutdown supervisor 
# <active/standby/all>
#-------------------------------------------------------

import BasicCli
import CliToken.Agent
import LazyMount
import ConfigMount
import Tac
import LauncherLib
import AgentCli
import re
import CliCommand
import CliMatcher

agentCfgDir = None
agentShutdownCfgDir = None

configMode = BasicCli.GlobalConfigMode

# Get agent names from agentConfigDir, but allow any string for startup-config
# parsing. We allow all agent names and launchNames.
#
# We cache those names but keep track of the number of entries in agentConfigDir
# and once it changes we recalculate the agent names.
def allAgentNames( mode ):
   allAgents = AgentCli.procMgrAgents.copy()

   # Get the list of Agent names from Sysdb's launcher config, so that
   # we do not require the agent is running.
   for name, config in agentCfgDir.iteritems():
      allAgents[ name ] = '%s agent' % name
      if config.launchName:
         allAgents[ config.launchName ] = '%s agent' % config.launchName

   return allAgents

matcherSupervisor = CliMatcher.KeywordMatcher( 'supervisor',
                    helpdesc='Shutdown the agent on a per cell basis' )

agentMatcher = CliMatcher.DynamicNameMatcher( AgentCli.allAgentNames,
                  helpdesc='Agent name', pattern='.+' )

def _agentShutdownCfgIsDefault( agentShutdownCfg ):
   if agentShutdownCfg.shutdown:
      return False
   if len( agentShutdownCfg.perSliceShutdown ):
      return False
   if len( agentShutdownCfg.perCellShutdown ):
      return False
   if len( agentShutdownCfg.perCellSliceShutdown ):
      return False
   return True

def getAgentShutdownParams( args ):
   supervisor = None
   agent = args[ 'AGENT' ]
   if 'supervisor' in args:
      if 'active' in args:
         supervisor = args.get( 'active' )
      elif 'standby' in args:
         supervisor = args.get( 'standby' )
      else:
         supervisor = args.get( 'all' )
         supervisor = None
   return( agent, supervisor )

def setAgentShutdown( mode, args ):
   agent, supervisor = getAgentShutdownParams( args )
   agent, sliceId = agentToAgentSlice( agent )
   
   agentShutdownCfg = agentShutdownCfgDir.agent.newMember( agent )
   if sliceId and supervisor:
      cellSlice = Tac.newInstance( "Launcher::CellSliceIndex",
                                   supervisor, sliceId )
      agentShutdownCfg.perCellSliceShutdown[ cellSlice ] = True
   elif sliceId is None and supervisor:
      agentShutdownCfg.perCellShutdown[ supervisor ] = True
   elif sliceId and supervisor is None:
      agentShutdownCfg.perSliceShutdown[ sliceId ] = True
   else:
      agentShutdownCfg.shutdown = True

def noAgentShutdown( mode, args ):
   agent, supervisor = getAgentShutdownParams( args )
   agent, sliceId = agentToAgentSlice( agent )
   
   agentShutdownCfg = agentShutdownCfgDir.agent.get( agent )
   if not agentShutdownCfg:
      return

   if sliceId and supervisor:
      cellSlice = Tac.newInstance( "Launcher::CellSliceIndex",
                                   supervisor, sliceId )
      del agentShutdownCfg.perCellSliceShutdown[ cellSlice ]
   elif sliceId is None and supervisor:
      del agentShutdownCfg.perCellShutdown[ supervisor ]
   elif sliceId and supervisor is None:
      del agentShutdownCfg.perSliceShutdown[ sliceId ]
   else:
      agentShutdownCfg.shutdown = False
   if _agentShutdownCfgIsDefault( agentShutdownCfg ):
      del agentShutdownCfgDir.agent[ agent ]

def agentToAgentSlice( agent ):
   if agent in agentCfgDir:
      return agent, None

   # Check whether there are multiple dashes in the agent name and slice them.
   if ( agent.count( '-' ) >= 3 ):
      agentSlice = re.findall( r'^\w+-\w+|-|\S+', agent )
      agentSlice.remove( '-' )
      return agentSlice
   if agent.startswith( "Strata-" ):
      # Different cases : Strata-FixedSystem, Strata-Linecard0-0, Strata-Linecard0
      agentSlice = agent.split( "-", 1 )
   else:
      # Below are two cases on how slicing is handled.
      # Case 1: If agent name is 'foo', it will be stored in the vrf-list as 
      # 'Rib-vrf-foo'. So it should be sliced as 'Rib-vrf' and 'foo'.
      # Case 2: If agent name is 'foo1-foo2', it will be stored in the vrf-list as
      # 'Rib-vrf-foo1-foo2'. So it should be sliced as 'Rib-vrf' and 'foo1-foo2'.
      agentSlice = agent.rsplit( "-", 1 )
   if len( agentSlice ) == 1:
      return agentSlice[0], None
   return agentSlice

#--------------------------------------------------------------------------------
# [ no | default ] agent AGENT shutdown [ supervisor ( active | standby | all ) ]
#--------------------------------------------------------------------------------
class AgentShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'agent AGENT shutdown [ supervisor ( active | standby | all ) ]'
   noOrDefaultSyntax = syntax
   data = {
      'agent': CliToken.Agent.agentKwForConfig,
      'AGENT': agentMatcher,
      'shutdown': 'Shutdown the agent process',
      'supervisor': matcherSupervisor,
      'active': 'Shutdown the agent on the active supervisor',
      'standby': 'Shutdown the agent on the standby supervisor',
      'all': 'Shutdown the agent on the all supervisors',
   }
   handler = setAgentShutdown
   noOrDefaultHandler = noAgentShutdown

configMode.addCommandClass( AgentShutdownCmd )

def Plugin( entityManager ):
   global agentCfgDir
   global agentShutdownCfgDir
   agentCfgDir = LazyMount.mount( entityManager,
                                  "sys/config/agentConfigDir",
                                  "Launcher::AgentConfigDir", "r" )
   agentShutdownCfgDir = ConfigMount.mount( entityManager,
                                            LauncherLib.agentShutdownConfigDirPath,
                                           "Launcher::AgentShutdownConfigDir", "wi" )
