#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from operator import itemgetter

import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliPlugin.AgentCli as AgentCli
from CliPlugin.AgentMonitorModel import (
   AgentUptime,
   AgentUptimeEntry,
   AgentOomScores,
   AgentOomScoresEntry,
)
from CliPlugin.CliCliModel import AgentMemoryUsage
import CliPlugin.TechSupportCli
import CliToken.Agent
import CliToken.Clear
import LazyMount
import PyClient
import PyClientBase
import ShowCommand
import Tac

# pkgdeps: rpm AgentMonitor-lib

def agentHealthConfigSysdbPath():
   return Cell.path( 'agent/health/config' )
def agentHealthStatusSysdbPath():
   return Cell.path( 'agent/health/status' )
def agentStatusSysdbPath():
   return Cell.path( 'agent/status' )

ERR_NOT_RUNNING_TEMPLATE = 'Could not connect to %s. Is it running?'
ERR_PY_ONLY = 'Only Python agents are supported'
PYCLIENT_TIMEOUT = 10

agentHealthConfig = None
agentHealthStatus = None
agentBaseStatus = None

#--------------------------------------------------------------------------------
# show agent [ { AGENTS } ] memory
#--------------------------------------------------------------------------------
matcherMemory = CliMatcher.KeywordMatcher( 'memory',
      helpdesc='Memory statistics for the agent' )

class AgentMemoryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ { AGENTS } ] memory'
   data = {
      'agent' : CliToken.Agent.agentKwForShow,
      'AGENTS' : AgentCli.agentNameNewMatcher,
      'memory' : matcherMemory,
   }

   @staticmethod
   def handler( mode, args ):
      agentList = args.get( 'AGENTS' )
      agMem = []
      print 'Agent Name             Last Memory(Kb) Max Memory(Kb) '
      print '---------------------- --------------- --------------'
      for agentName, agentStatus in agentHealthStatus.agent.iteritems():
         if not agentStatus.active:
            continue
         if agentList and agentName not in agentList:
            continue
         agMem.append( [ agentName, agentStatus.lastMemory, agentStatus.maxMemory ] )
      agMem.sort( key=itemgetter( 2 ), reverse=True )
      for x in agMem:
         print "%-22s %-15u %u" % ( x[0], x[1], x[2] )

BasicCli.addShowCommandClass( AgentMemoryCmd )

#--------------------------------------------------------------------------------
# show agent AGENT memory detail
#--------------------------------------------------------------------------------
class AgentAgentMemoryDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent AGENT memory detail'
   data = {
      'agent' : CliToken.Agent.agentKwForShow,
      'AGENT' : AgentCli.agentNameNewMatcher,
      'memory' : matcherMemory,
      'detail' : 'Private memory count, thread count, and 100 most common objects',
   }
   cliModel = AgentMemoryUsage

   @staticmethod
   def handler( mode, args ):
      agent = args[ 'AGENT' ]
      status = agentHealthStatus.agent.get( agent )
      if status is None or not status.active:
         mode.addError( ERR_NOT_RUNNING_TEMPLATE % agent )
         return None

      try:
         pyC = PyClient.PyClient( mode.sysname, agent,
               connectTimeout=PYCLIENT_TIMEOUT )
      except PyClientBase.RpcConnectionError:
         mode.addError( ERR_NOT_RUNNING_TEMPLATE % agent )
         return None

      # Currently, this only applies to Python agents
      if pyC.eval( 'hasattr( sys.modules[ "__main__" ], "%s" )' % agent ):
         data = pyC.eval(
               'sys.modules["__main__"].%s.Agent.getDataForModel()' % agent )
         return AgentMemoryUsage( **data )
      else:
         mode.addError( ERR_PY_ONLY )
      return None

BasicCli.addShowCommandClass( AgentAgentMemoryDetailCmd )

#-----------------------------------------------------------------------------
#
# Agent uptime cli commands:
#    show agent [ <agentName> ] uptime
#
#----------------------------------------------------------------------------
class ShowAgentUptimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show agent [ AGENT ] uptime"
   data = {
      'agent' : CliToken.Agent.agentKwForShow,
      'AGENT' : AgentCli.agentNameNewMatcher,
      'uptime' : 'Uptime and restart statistics for the agent',
   }
   cliModel = AgentUptime

   @staticmethod
   def handler( mode, args ):
      agentUptimeEntries = AgentUptime()
      agentRequested = args.get( 'AGENT' )
      if not agentRequested:
         agentHealthCollection = agentHealthStatus.agent
      else:
         agentHealthCollection = {}
         agentRequestedStatus = agentHealthStatus.agent.get( agentRequested )
         if agentRequestedStatus:
            agentHealthCollection[ agentRequested ] = agentRequestedStatus

      for agentName, healthStatus in agentHealthCollection.iteritems():
         if not healthStatus.active:
            continue
         status = agentBaseStatus[ agentName ]
         if status.startCount == 0:
            continue
         agentUptime = Tac.now() - status.startTime
         startTimeUTC = int( Tac.utcNow() - agentUptime )
         agentUptimeEntries.agents[ agentName ] = AgentUptimeEntry(
            restartCount=status.startCount - 1, agentStartTime=startTimeUTC )

      return agentUptimeEntries

BasicCli.addShowCommandClass( ShowAgentUptimeCmd )

#--------------------------------------------------------------------------------
# show agent [ { AGENTS } ] cpu
#--------------------------------------------------------------------------------
class AgentCpuCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ { AGENTS } ] cpu'
   data = {
      'agent' : CliToken.Agent.agentKwForShow,
      'AGENTS' : AgentCli.agentNameNewMatcher,
      'cpu' : 'CPU statistics for the agent',
   }

   @staticmethod
   def handler( mode, args ):
      agentList = args.get( 'AGENTS' )
      print 'Agent Name             Last Cpu(%) Max Cpu(%/secs)'
      print '---------------------- ----------- ---------------'
      agCpu = []
      for agentName, agentStatus in agentHealthStatus.agent.items():
         if not agentStatus.active:
            continue
         if agentList and agentName not in agentList:
            continue
         agCpu.append( [ agentName, agentStatus.lastCpu, 
                         agentStatus.maxCpu, agentStatus.secsAtMaxCpu ] ) 
      agCpu.sort( key = lambda x : x[ 2 ], reverse = True )
      for x in agCpu:
         print "%-22s %-11u %-3u/%u" % ( x[0], x[1], x[2], x[3] )

BasicCli.addShowCommandClass( AgentCpuCmd )

#--------------------------------------------------------------------------------
# clear agent cpu
#--------------------------------------------------------------------------------
class ClearAgentCpuCmd( CliCommand.CliCommandClass ):
   syntax = 'clear agent cpu'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'agent' : CliToken.Agent.agentKwForClear,
      'cpu' : 'Clear agent CPU statistics',
   }

   @staticmethod
   def handler( mode, args ):
      clearRequestTime = Tac.now()
      agentHealthConfig.clearCpuRequest = clearRequestTime
      # Wait for the clear to be acknowledged
      try:
         Tac.waitFor( lambda: agentHealthStatus.clearCpuResponse >= clearRequestTime,
                      timeout=10,
                      warnAfter=5,
                      maxDelay=0.1,
                      sleep=True,
                      description="agent cpu statistics to clear" )
      except Tac.Timeout:
         mode.addWarning( "Failed to clear agent cpu statistics" )

BasicCli.EnableMode.addCommandClass( ClearAgentCpuCmd )

#-----------------------------------------------------------------------------
#
# Agent oom score cli commands:
#    show agent [ { <agentName> } ] oom scores
#
#----------------------------------------------------------------------------
class ShowAgentOomScores( ShowCommand.ShowCliCommandClass ):
   syntax = "show agent [ { AGENTS } ] oom scores"
   data = {
      'agent' : CliToken.Agent.agentKwForShow,
      'AGENTS' : AgentCli.agentNameNewMatcher,
      'oom' : 'Show agent oom parameters',
      'scores' : 'Show agent oom score related parameters',
   }
   cliModel = AgentOomScores

   @staticmethod
   def handler( mode, args ):
      agentOomScoresEntries = AgentOomScores()
      agentsRequested = args.get( 'AGENTS' )
      if not agentsRequested:
         agentHealthCollection = agentHealthStatus.agent
      else:
         agentHealthCollection = {}
         for agent in agentsRequested:
            agentRequestedStatus = agentHealthStatus.agent.get( agent )
            if agentRequestedStatus:
               agentHealthCollection[ agent ] = agentRequestedStatus

      for agentName, healthStatus in agentHealthCollection.iteritems():
         if not healthStatus.active:
            continue
         status = agentBaseStatus[ agentName ]
         if status.startCount == 0:
            continue
         agentOomScoresEntries.agents[ agentName ] = AgentOomScoresEntry(
               rssAnon=healthStatus.rssAnon, vmSize=healthStatus.lastMemory,
               maxVmSize=healthStatus.maxMemory, oomScore=healthStatus.oomScore,
               oomScoreAdj=healthStatus.oomScoreAdj )

      return agentOomScoresEntries

BasicCli.addShowCommandClass( ShowAgentOomScores )

#--------------------------------------------------------------
# show tech support
#--------------------------------------------------------------
def _showTechCommands():
   return [ 'show agent oom scores',
          ]

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2019-04-12 13:58:49', _showTechCommands )

def Plugin( entityManager ):
   global agentHealthConfig, agentHealthStatus, agentBaseStatus
   agentHealthConfig = LazyMount.mount( entityManager,
                                        agentHealthConfigSysdbPath(),
                                        "Agent::Health::Config", "wc" )
   agentHealthStatus = LazyMount.mount( entityManager,
                                        agentHealthStatusSysdbPath(),
                                        "Agent::Health::Status", "r" )
   agentBaseStatus = LazyMount.mount( entityManager,
                                      agentStatusSysdbPath(),
                                      "Tac::Dir", "ri" )
