#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliPlugin.AgentCli as AgentCli
import CliToken.Agent
import CliToken.Clear
import LazyMount
import ShowCommand
import Tac

agentHealthConfig = None
agentHealthStatus = None

#--------------------------------------------------------------------------------
# agent monitor ping-interval INTERVAL
#--------------------------------------------------------------------------------
class SetPingIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'agent monitor ping-interval INTERVAL'
   data = {
      'agent' : CliToken.Agent.agentKwForConfig,
      'monitor' : 'Configure the agent monitor settings',
      'ping-interval' : 'Set the agent monitor ping interval',
      'INTERVAL' : CliMatcher.IntegerMatcher( 0, 3600,
         helpdesc='Ping interval in seconds' ),
   }

   @staticmethod
   def handler( mode, args ):
      agentHealthConfig.pingInterval = args[ 'INTERVAL' ]

BasicCli.EnableMode.addCommandClass( SetPingIntervalCmd )

#--------------------------------------------------------------------------------
# show agent [ { AGENTS } ] ping
#--------------------------------------------------------------------------------
class ShowAgentPingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show agent [ { AGENTS } ] ping'
   data = {
      'agent' : CliToken.Agent.agentKwForShow,
      'AGENTS' : AgentCli.agentNameNewMatcher,
      'ping' : 'Ping statistics for the agent',
   }

   @staticmethod
   def handler( mode, args ):
      agentList = args.get( 'AGENTS' )
      print ( 'Agent Name             Last Ping     Max Ping     '
               'Max Ping Response Seen  Last Ping Response Seen ' )
      print ( '---------------------- ------------- ------------ '
               '----------------------- ------------------------' )
      for agentName, agentStatus in sorted( agentHealthStatus.agent.items() ):
         if not agentStatus.active:
            continue
         if agentList and agentName not in agentList:
            continue
         def _pingStr( ping ):
            if ping and ping != Tac.endOfTime:
               return '%8.3f ms' % ( ping * 1000 )
            else:
               return 'unknown'
         print '%-22s %-13s %-13s %-23s %-21s' % ( agentName,
                                            _pingStr( agentStatus.lastPing ),
                                            _pingStr( agentStatus.maxPing ),
                                            Ark.timestampToStr(
                                               agentStatus.maxPingResponseAt,
                                               relative=False ),
                                            Ark.timestampToStr(
                                               agentStatus.lastPingResponseAt,
                                               relative=False ) )

BasicCli.addShowCommandClass( ShowAgentPingCmd )

#--------------------------------------------------------------------------------
# clear agent ping
#--------------------------------------------------------------------------------
class ClearAgentPingCmd( CliCommand.CliCommandClass ):
   syntax = 'clear agent ping'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'agent' : CliToken.Agent.agentKwForClear,
      'ping' : 'Clear agent ping statistics',
   }

   @staticmethod
   def handler( mode, args ):
      clearRequestTime = Tac.now()
      agentHealthConfig.clearPingRequest = clearRequestTime
      # Wait for the clear to be acknowledged
      try:
         Tac.waitFor( lambda: ( agentHealthStatus.clearPingResponse >=
                                clearRequestTime ),
                      timeout=10,
                      warnAfter=5,
                      maxDelay=0.1,
                      sleep=True,
                      description='agent ping statistics to clear' )
      except Tac.Timeout:
         mode.addWarning( 'Failed to clear agent ping statistics' )

BasicCli.EnableMode.addCommandClass( ClearAgentPingCmd )

def Plugin( entityManager ):
   global agentHealthConfig, agentHealthStatus
   agentHealthConfig = LazyMount.mount( entityManager,
                                        Cell.path( 'agent/health/config' ),
                                        'Agent::Health::Config', 'wc' )
   agentHealthStatus = LazyMount.mount( entityManager,
                                        Cell.path( 'agent/health/status' ),
                                        'Agent::Health::Status', 'r' )
